/**
 * GoValid QR Analytics — Charts, Map, and QR Drilldown.
 *
 * @package GoValid_QR
 */

/* global jQuery, govalidQR, Chart, L */
(function ($) {
	'use strict';

	var charts = {};
	var ddCharts = {};
	var map = null;
	var markerGroup = null;

	/* ---------- palette ---------- */
	var PALETTE = [
		'#2563eb', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6',
		'#06b6d4', '#ec4899', '#14b8a6', '#f97316', '#6366f1'
	];

	/* ---------- helpers ---------- */
	function num(v) {
		return (v || 0).toLocaleString();
	}

	function getFlagEmoji(cc) {
		if (!cc || cc.length !== 2) return '';
		var codePoints = cc.toUpperCase().split('').map(function (c) {
			return 0x1f1e6 - 65 + c.charCodeAt(0);
		});
		return String.fromCodePoint.apply(null, codePoints);
	}

	/* ---------- init ---------- */
	function init() {
		if ( ! govalidQR.isConnected ) {
			showEmpty();
			return;
		}
		$.ajax({
			url: govalidQR.restUrl + 'analytics',
			method: 'GET',
			beforeSend: function (xhr) {
				xhr.setRequestHeader('X-WP-Nonce', govalidQR.nonce);
			},
			success: function (res) {
				var d = (res && res.data) ? res.data : res;
				if (!d || !d.summary) {
					showEmpty();
					return;
				}
				removeSkeleton();
				renderSummary(d.summary);
				renderMap(d.locations || []);
				renderDailyTrend(d.daily_trend || []);
				renderCountries(d.countries || []);
				renderDevices(d.devices || {});
				renderQRTypes(d.qr_by_type || {});
				renderTopQR(d.top_qr_codes || []);
			},
			error: function () {
				showEmpty();
			}
		});

		$('#drilldown-close').on('click', function () {
			$('#qr-drilldown-card').slideUp(200);
		});
	}

	function removeSkeleton() {
		$('.govalid-stat-card').removeClass('govalid-shimmer');
	}

	function showEmpty() {
		removeSkeleton();
		$('#stat-qr-codes, #stat-total-scans, #stat-month-scans, #stat-countries').text('0');
	}

	/* ---------- summary ---------- */
	function renderSummary(s) {
		$('#stat-qr-codes').text(num(s.total_qr_codes));
		$('#stat-total-scans').text(num(s.total_scans));
		$('#stat-month-scans').text(num(s.scans_this_month));
		$('#stat-countries').text(num(s.unique_countries));
	}

	/* ---------- map ---------- */
	function renderMap(locations) {
		if (!locations.length) {
			$('#govalid-analytics-map').hide();
			$('#map-empty').show();
			return;
		}

		map = L.map('govalid-analytics-map').setView([0, 20], 2);

		/* --- Tile layers --- */
		var streetLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
			attribution: '&copy; OpenStreetMap',
			maxZoom: 19
		});

		var satelliteLayer = L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}', {
			attribution: '&copy; Esri &mdash; Earthstar Geographics',
			maxZoom: 18
		});

		var topoLayer = L.tileLayer('https://{s}.tile.opentopomap.org/{z}/{x}/{y}.png', {
			attribution: '&copy; OpenTopoMap',
			maxZoom: 17
		});

		streetLayer.addTo(map);

		L.control.layers({
			'Street': streetLayer,
			'Satellite': satelliteLayer,
			'Terrain': topoLayer
		}, null, { position: 'topright' }).addTo(map);

		markerGroup = L.featureGroup();

		locations.forEach(function (loc) {
			var circle = L.circleMarker([loc.latitude, loc.longitude], {
				radius: Math.min(6 + loc.scan_count * 2, 20),
				fillColor: '#2563eb',
				color: '#1e40af',
				weight: 1,
				opacity: 0.8,
				fillOpacity: 0.5
			});

			circle.bindPopup(
				'<strong>' + escapeHtml(loc.qr_code_name || '') + '</strong><br>' +
				escapeHtml(loc.location_name || '') + '<br>' +
				loc.scan_count + ' ' + (govalidQR.i18n && govalidQR.i18n.scans_label ? govalidQR.i18n.scans_label : 'scans')
			);

			markerGroup.addLayer(circle);
		});

		markerGroup.addTo(map);
		map.fitBounds(markerGroup.getBounds().pad(0.15));
	}

	/* ---------- daily trend (line) ---------- */
	function renderDailyTrend(trend) {
		if (!trend.length) {
			$('#chart-daily-trend').closest('.govalid-chart-wrap').hide();
			$('#trend-empty').show();
			return;
		}

		var ctx = document.getElementById('chart-daily-trend').getContext('2d');
		charts.dailyTrend = new Chart(ctx, {
			type: 'line',
			data: {
				labels: trend.map(function (t) { return t.date; }),
				datasets: [{
					label: govalidQR.i18n && govalidQR.i18n.scans_label ? govalidQR.i18n.scans_label : 'Scans',
					data: trend.map(function (t) { return t.count; }),
					borderColor: '#2563eb',
					backgroundColor: 'rgba(37,99,235,0.08)',
					fill: true,
					tension: 0.35,
					pointRadius: 2,
					pointHoverRadius: 5
				}]
			},
			options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: { legend: { display: false } },
				scales: {
					x: { grid: { display: false }, ticks: { maxTicksLimit: 8 } },
					y: { beginAtZero: true, ticks: { precision: 0 } }
				}
			}
		});
	}

	/* ---------- countries (doughnut) ---------- */
	function renderCountries(countries) {
		if (!countries.length) {
			$('#chart-countries').closest('.govalid-chart-wrap').hide();
			$('#countries-empty').show();
			return;
		}

		var top = countries.slice(0, 8);
		var ctx = document.getElementById('chart-countries').getContext('2d');
		charts.countries = new Chart(ctx, {
			type: 'doughnut',
			data: {
				labels: top.map(function (c) { return getFlagEmoji(c.country_code) + ' ' + c.country_code; }),
				datasets: [{
					data: top.map(function (c) { return c.count; }),
					backgroundColor: PALETTE.slice(0, top.length)
				}]
			},
			options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: {
					legend: { position: 'right', labels: { boxWidth: 14, padding: 10 } }
				}
			}
		});
	}

	/* ---------- devices (horizontal bar) ---------- */
	function renderDevices(devices) {
		var platforms = devices.platforms || [];
		var browsers = devices.browsers || [];
		var items = platforms.concat(browsers).sort(function (a, b) { return b.count - a.count; }).slice(0, 10);

		if (!items.length) {
			$('#chart-devices').closest('.govalid-chart-wrap').hide();
			$('#devices-empty').show();
			return;
		}

		var ctx = document.getElementById('chart-devices').getContext('2d');
		charts.devices = new Chart(ctx, {
			type: 'bar',
			data: {
				labels: items.map(function (i) { return i.name; }),
				datasets: [{
					data: items.map(function (i) { return i.count; }),
					backgroundColor: PALETTE.slice(0, items.length)
				}]
			},
			options: {
				indexAxis: 'y',
				responsive: true,
				maintainAspectRatio: false,
				plugins: { legend: { display: false } },
				scales: {
					x: { beginAtZero: true, ticks: { precision: 0 } },
					y: { grid: { display: false } }
				}
			}
		});
	}

	/* ---------- QR types (doughnut) ---------- */
	function renderQRTypes(types) {
		var labels = Object.keys(types);
		var values = Object.values(types);

		if (!labels.length || values.every(function (v) { return v === 0; })) {
			$('#chart-qr-types').closest('.govalid-chart-wrap').hide();
			$('#types-empty').show();
			return;
		}

		var ctx = document.getElementById('chart-qr-types').getContext('2d');
		charts.qrTypes = new Chart(ctx, {
			type: 'doughnut',
			data: {
				labels: labels,
				datasets: [{
					data: values,
					backgroundColor: PALETTE.slice(0, labels.length)
				}]
			},
			options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: {
					legend: { position: 'right', labels: { boxWidth: 14, padding: 10 } }
				}
			}
		});
	}

	/* ---------- top QR codes table ---------- */
	function renderTopQR(topQR) {
		if (!topQR.length) {
			$('#top-qr-empty').show();
			return;
		}

		var $body = $('#top-qr-body');
		$body.empty();

		topQR.forEach(function (qr) {
			var $row = $('<tr>');
			$row.append($('<td>').text(qr.name || '—'));
			$row.append($('<td>').text(qr.type || '—'));
			$row.append($('<td style="text-align:right;">').text(num(qr.scans)));
			var $btn = $('<button type="button" class="button button-small govalid-drilldown-btn">')
				.text(govalidQR.i18n && govalidQR.i18n.view_detail ? govalidQR.i18n.view_detail : 'View')
				.data('uuid', qr.uuid)
				.data('name', qr.name);
			$row.append($('<td style="text-align:center;">').append($btn));
			$body.append($row);
		});

		$('#top-qr-table').show();

		$body.on('click', '.govalid-drilldown-btn', function () {
			var uuid = $(this).data('uuid');
			var name = $(this).data('name');
			loadDrilldown(uuid, name);
		});
	}

	/* ---------- drilldown ---------- */
	function loadDrilldown(uuid, name) {
		$('#drilldown-title').text(name || 'QR Code Detail');
		$('#qr-drilldown-card').slideDown(200);
		$('#drilldown-loading').show();
		$('#drilldown-content').hide();

		// Scroll to drilldown
		$('html, body').animate({ scrollTop: $('#qr-drilldown-card').offset().top - 40 }, 300);

		$.ajax({
			url: govalidQR.restUrl + 'analytics/' + encodeURIComponent(uuid),
			method: 'GET',
			beforeSend: function (xhr) {
				xhr.setRequestHeader('X-WP-Nonce', govalidQR.nonce);
			},
			success: function (res) {
				$('#drilldown-loading').hide();
				$('#drilldown-content').show();

				var d = (res && res.data) ? res.data : res;
				renderDrilldownContent(d);
			},
			error: function () {
				$('#drilldown-loading').html('<p style="color:#ef4444;">' +
					(govalidQR.i18n && govalidQR.i18n.error ? govalidQR.i18n.error : 'An error occurred.') +
					'</p>');
			}
		});
	}

	function renderDrilldownContent(d) {
		$('#dd-total-scans').text(num(d.total_scans));
		$('#dd-unique-scanners').text(num(d.unique_scanners));

		// Destroy previous drilldown charts
		if (ddCharts.daily) ddCharts.daily.destroy();
		if (ddCharts.geo) ddCharts.geo.destroy();

		// Daily scans chart
		var daily = d.daily_scans || [];
		if (daily.length) {
			var ctx1 = document.getElementById('dd-chart-daily').getContext('2d');
			ddCharts.daily = new Chart(ctx1, {
				type: 'line',
				data: {
					labels: daily.map(function (i) { return i.date; }),
					datasets: [{
						data: daily.map(function (i) { return i.total; }),
						borderColor: '#2563eb',
						backgroundColor: 'rgba(37,99,235,0.08)',
						fill: true,
						tension: 0.35,
						pointRadius: 2
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { display: false } },
					scales: {
						x: { grid: { display: false }, ticks: { maxTicksLimit: 8 } },
						y: { beginAtZero: true, ticks: { precision: 0 } }
					}
				}
			});
		}

		// Geographic distribution chart
		var geo = d.geographic_distribution || [];
		if (geo.length) {
			var top = geo.slice(0, 6);
			var ctx2 = document.getElementById('dd-chart-geo').getContext('2d');
			ddCharts.geo = new Chart(ctx2, {
				type: 'doughnut',
				data: {
					labels: top.map(function (g) { return getFlagEmoji(g.country_code) + ' ' + (g.country || g.country_code); }),
					datasets: [{
						data: top.map(function (g) { return g.scan_count; }),
						backgroundColor: PALETTE.slice(0, top.length)
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { position: 'right', labels: { boxWidth: 14, padding: 10 } } }
				}
			});
		}
	}

	/* ---------- escape ---------- */
	function escapeHtml(str) {
		var div = document.createElement('div');
		div.appendChild(document.createTextNode(str));
		return div.innerHTML;
	}

	/* ---------- boot ---------- */
	$(document).ready(init);

})(jQuery);
