<?php

/**
 * @file plugins/generic/goValidOJS/pages/GoValidOJSHandler.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class GoValidOJSHandler
 *
 * Handle requests for GoValidOJS plugin pages.
 */

namespace APP\plugins\generic\goValidOJS\pages;

use APP\core\Application;
use APP\facades\Repo;
use APP\handler\Handler;
use APP\template\TemplateManager;
use PKP\core\JSONMessage;
use PKP\plugins\PluginRegistry;
use PKP\core\PKPRequest;
use PKP\submission\PKPSubmission;
use PKP\file\PKPPublicFileManager;
use APP\plugins\generic\goValidOJS\GoValidOJSPlugin;
use APP\plugins\generic\goValidOJS\classes\EmailTemplateManager;

class GoValidOJSHandler extends Handler
{
    /** @var GoValidOJSPlugin The plugin associated with this request */
    public GoValidOJSPlugin $plugin;

    /**
     * Constructor
     * OJS 3.5: Accept plugin instance from the LoadHandler hook
     */
    public function __construct(GoValidOJSPlugin $plugin)
    {
        parent::__construct();
        $this->plugin = $plugin;
    }

    /**
     * @copydoc PKPHandler::authorize()
     */
    public function authorize($request, &$args, $roleAssignments)
    {
        $debugFile = dirname(__FILE__) . '/../debug_submissions.log';
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " === authorize() called ===\n", FILE_APPEND);
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Request path: " . $request->getRequestPath() . "\n", FILE_APPEND);

        // Define all operations that this handler supports
        $allOperations = [
            'designer',
            'ajax',
            'saveTemplate',
            'getIssues',
            'getSubmissions',
            'emailTemplateManager',
            'emailTemplateDesigner',
            'saveEmailTemplate',
            'deleteTemplate',
            'activateTemplate',
            'previewTemplate',
            'getDefaultTemplate',
            'getEmailTemplate',
            'sendTestEmail',
            'testEmail',
            'searchUsers',
            'sendCertificateEmail'
        ];

        // Get context and plugin settings to determine allowed roles
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : 0;

        // Build allowed roles based on FAB visibility settings
        $allowedRoles = [\PKP\security\Role::ROLE_ID_SITE_ADMIN]; // Site admin always has access

        if ($this->plugin) {
            $showFabForManagers = $this->plugin->getSetting($contextId, 'showFabForManagers') ?? true;
            $showFabForEditors = $this->plugin->getSetting($contextId, 'showFabForEditors') ?? true;
            $showFabForAssistants = $this->plugin->getSetting($contextId, 'showFabForAssistants') ?? true;

            if ($showFabForManagers) {
                $allowedRoles[] = \PKP\security\Role::ROLE_ID_MANAGER;
            }
            if ($showFabForEditors) {
                $allowedRoles[] = \PKP\security\Role::ROLE_ID_SUB_EDITOR;
            }
            if ($showFabForAssistants) {
                $allowedRoles[] = \PKP\security\Role::ROLE_ID_ASSISTANT;
            }
        } else {
            // Fallback: allow all roles if plugin not loaded
            $allowedRoles = [
                \PKP\security\Role::ROLE_ID_SITE_ADMIN,
                \PKP\security\Role::ROLE_ID_MANAGER,
                \PKP\security\Role::ROLE_ID_SUB_EDITOR,
                \PKP\security\Role::ROLE_ID_ASSISTANT
            ];
        }

        $this->addPolicy(new \PKP\security\authorization\RoleBasedHandlerOperationPolicy(
            $request,
            $allowedRoles,
            $allOperations
        ));

        return parent::authorize($request, $args, $roleAssignments);
    }

    /**
     * Display the certificate designer or handle AJAX requests
     *
     * @param array $args
     * @param PKPRequest $request
     */
    public function designer($args, $request)
    {
        $debugFile = dirname(__FILE__) . '/../debug_submissions.log';
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " === designer() called ===\n", FILE_APPEND);

        // Check if this is an AJAX request
        $action = $request->getUserVar('action');
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Action: " . ($action ?: 'NULL') . "\n", FILE_APPEND);

        if ($action) {
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Routing to handleAjaxAction...\n", FILE_APPEND);
            // Handle AJAX actions
            return $this->handleAjaxAction($action, $request);
        }
        
        // OJS 3.5: Plugin is passed via constructor, no need to load it here
        $templateMgr = TemplateManager::getManager($request);
        $templateMgr->assign('pluginName', $this->plugin->getName());
        $templateMgr->assign('baseUrl', $request->getBaseUrl());

        $context = $request->getContext();
        $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;

        // Assign context path for AJAX URLs
        $contextPath = $context ? $context->getPath() : 'index';
        $templateMgr->assign('contextPath', $contextPath);

        // Simple journal info
        if ($context) {
            // Get logo URL using robust method from EmailTemplateManager
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
            $logoUrl = $emailTemplateManager->getJournalLogoUrl($request);
            if (!$logoUrl) $logoUrl = '';

            $journalInfo = [
                'name' => $context->getLocalizedData('name') ?: 'Journal Name',
                'logo' => $logoUrl,
                'url' => $request->getBaseUrl() . '/' . $context->getPath(),
                'onlineIssn' => $context->getData('onlineIssn') ?: '',
                'printIssn' => $context->getData('printIssn') ?: ''
            ];
        } else {
            $journalInfo = ['name' => 'Journal Name', 'logo' => '', 'url' => '', 'onlineIssn' => '', 'printIssn' => ''];
        }
        $templateMgr->assign('journalInfo', $journalInfo);
        
        // Load existing template if available
        $existingTemplate = $this->plugin->getSetting($contextId, 'certificateTemplate');
        $templateMgr->assign('existingTemplate', $existingTemplate);

        // Display the template
        $templateMgr->display($this->plugin->getTemplateResource('designer.tpl'));
    }

    /**
     * Save certificate template
     */
    public function saveTemplate($args, $request)
    {
        $context = $request->getContext();
        $templateData = $request->getUserVar('templateData');

        if (!$templateData) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'No template data provided']);
            exit;
        }

        // Parse the template data
        $config = json_decode($templateData, true);
        if (!$config) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'Invalid template data']);
            exit;
        }

        // Extract template name and subject
        $templateName = $config['name'] ?? 'Untitled Template';
        $subject = $config['subject'] ?? '';

        // Generate a unique template key from the name
        $templateKey = 'CUSTOM_' . strtoupper(preg_replace('/[^a-zA-Z0-9]/', '_', $templateName));

        try {
            // Use OJSEmailTemplateIntegration to save as OJS email template
            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);

            // Build the email body from the content fields
            $greeting = $config['content']['greeting'] ?? '';
            $message = $config['content']['message'] ?? '';
            $articleDetails = $config['content']['articleDetails'] ?? '';
            $closing = $config['content']['closing'] ?? '';

            // Combine content parts into body (content is already safe from designer)
            $bodyContent = '';
            if ($greeting) $bodyContent .= '<p>' . nl2br($greeting) . '</p>';
            if ($message) $bodyContent .= '<p>' . nl2br($message) . '</p>';
            if ($articleDetails) $bodyContent .= '<div style="background: ' . ($config['content']['detailsBgColor'] ?? '#f8f9fa') . '; padding: 15px; margin: 15px 0; border-radius: 4px;">' . nl2br($articleDetails) . '</div>';
            if ($closing) $bodyContent .= '<p>' . nl2br($closing) . '</p>';

            // Use reflection to call the private wrapContentWithDesign method
            $reflectionClass = new \ReflectionClass($integration);
            $method = $reflectionClass->getMethod('wrapContentWithDesign');
            $method->setAccessible(true);
            $emailBody = $method->invoke($integration, $bodyContent, $config);

            // Save as OJS email template
            $result = $integration->saveCustomTemplate([
                'key' => $templateKey,
                'name' => $templateName,
                'subject' => $subject,
                'body' => $emailBody,
                'description' => 'Custom email template created with Email Designer',
                'designConfig' => json_encode($config)
            ]);

            if ($result['success']) {
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'message' => 'Template saved successfully!', 'templateKey' => $templateKey]);
            } else {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'message' => $result['error'] ?? 'Failed to save template']);
            }
        } catch (\Exception $e) {
            error_log('GoValid Plugin: Error saving template: ' . $e->getMessage());
            error_log('GoValid Plugin: Stack trace: ' . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }

        exit;
    }

    /**
     * Handle AJAX requests for issues
     */
    public function getIssues($args, $request)
    {
        $context = $request->getContext();
        
        try {
            $issues = [];
            
            if ($context) {
                // Use Repository pattern to get published issues
                $publishedIssues = Repo::issue()->getCollector()
                    ->filterByContextIds([$context->getId()])
                    ->filterByPublished(true)
                    ->orderBy(Repo::issue()->getCollector()::ORDERBY_PUBLISHED_ISSUES)
                    ->getMany();
                
                foreach ($publishedIssues as $issue) {
                    $issues[] = [
                        'id' => $issue->getId(),
                        'title' => $issue->getLocalizedTitle() ?: 'Vol. ' . $issue->getVolume() . ' No. ' . $issue->getNumber() . ' (' . $issue->getYear() . ')',
                        'volume' => $issue->getVolume() ?: '',
                        'number' => $issue->getNumber() ?: '',
                        'year' => $issue->getYear() ?: '',
                        'datePublished' => $issue->getDatePublished(),
                        'description' => $issue->getLocalizedDescription() ?: '',
                        'published' => true
                    ];
                }
            }
            
            header('Content-Type: application/json');
            echo json_encode(['issues' => $issues]);
            exit;
            
        } catch (\Exception $e) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Error loading issues: ' . $e->getMessage(),
                'issues' => []
            ]);
            exit;
        }
    }

    /**
     * Handle AJAX requests for submissions
     */
    public function getSubmissions($args, $request)
    {
        $debugFile = dirname(__FILE__) . '/../debug_submissions.log';
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " === getSubmissions HANDLER called ===\n", FILE_APPEND);

        // Enable error reporting
        error_reporting(E_ALL);
        ini_set('display_errors', 0);
        
        $context = $request->getContext();
        $issueId = $request->getUserVar('issueId');

        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Context: " . ($context ? $context->getId() : 'NULL') . ", IssueId: $issueId\n", FILE_APPEND);

        try {
            $submissions = [];
            
            if ($context) {
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 1: Getting submission collector...\n", FILE_APPEND);
                
                // Use Repository pattern to get published submissions
                $submissionCollector = Repo::submission()->getCollector()
                    ->filterByContextIds([$context->getId()])
                    ->filterByStatus([PKPSubmission::STATUS_PUBLISHED]);
                
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 2: Basic collector created\n", FILE_APPEND);
                
                if ($issueId) {
                    @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 3: Filtering by issue ID: " . (int)$issueId . "\n", FILE_APPEND);
                    $submissionCollector->filterByIssueIds([(int)$issueId]);
                }
                
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 4: About to call getMany()...\n", FILE_APPEND);
                $publishedSubmissions = $submissionCollector->getMany();
                
                $count = is_object($publishedSubmissions) && method_exists($publishedSubmissions, 'count') 
                    ? $publishedSubmissions->count() 
                    : (is_countable($publishedSubmissions) ? count($publishedSubmissions) : 'unknown');
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 5: getMany() returned. Count: $count\n", FILE_APPEND);
                
                foreach ($publishedSubmissions as $submission) {
                    @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 6: Processing submission ID: " . $submission->getId() . "\n", FILE_APPEND);
                    
                    $publication = $submission->getCurrentPublication();
                    if ($publication) {
                        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 7: Got publication for submission " . $submission->getId() . "\n", FILE_APPEND);
                        
                        // Get authors for this publication
                        $authorsData = [];
                        $authors = $publication->getData('authors');
                        if ($authors) {
                            foreach ($authors as $author) {
                                $authorsData[] = [
                                    'id' => $author->getId(),
                                    'name' => $author->getFullName(),
                                    'email' => $author->getEmail(),
                                    'affiliation' => $author->getLocalizedData('affiliation') ?: '',
                                    'country' => $author->getCountry() ?: ''
                                ];
                            }
                        }

                        $submissions[] = [
                            'id' => $submission->getId(),
                            'title' => $publication->getLocalizedTitle() ?: '',
                            'doi' => $publication->getStoredPubId('doi') ?: '',
                            'pages' => $publication->getData('pages') ?: '',
                            'dateSubmitted' => $submission->getData('dateSubmitted'),
                            'datePublished' => $publication->getData('datePublished'),
                            'abstract' => $publication->getLocalizedData('abstract') ?: '',
                            'issueId' => $publication->getData('issueId'),
                            'sectionId' => $publication->getData('sectionId'),
                            'authors' => $authorsData
                        ];
                        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 8: Added submission " . $submission->getId() . " to results\n", FILE_APPEND);
                    }
                }
                
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 9: Total submissions found: " . count($submissions) . "\n", FILE_APPEND);
            } else {
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " ERROR: No context available\n", FILE_APPEND);
            }
            
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 10: Sending JSON response...\n", FILE_APPEND);
            header('Content-Type: application/json');
            echo json_encode(['submissions' => $submissions]);
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Step 11: Response sent successfully\n", FILE_APPEND);
            exit;
            
        } catch (\Exception $e) {
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " EXCEPTION: " . $e->getMessage() . "\n", FILE_APPEND);
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Stack: " . $e->getTraceAsString() . "\n", FILE_APPEND);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Error loading submissions: ' . $e->getMessage(),
                'submissions' => []
            ]);
            exit;
        } catch (\Error $e) {
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " FATAL ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
            @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Stack: " . $e->getTraceAsString() . "\n", FILE_APPEND);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Fatal error loading submissions: ' . $e->getMessage(),
                'submissions' => []
            ]);
            exit;
        }
    }

    /**
     * Get a single submission by ID (works for unpublished submissions too)
     */
    public function getSubmissionById($args, $request)
    {
        $context = $request->getContext();
        $submissionId = $request->getUserVar('submissionId');

        try {
            if (!$context || !$submissionId) {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'message' => 'Missing context or submission ID',
                    'submission' => null
                ]);
                exit;
            }

            // Use OJS 3.5 Repo pattern to get submission
            $submission = Repo::submission()->get((int)$submissionId);

            if (!$submission || $submission->getData('contextId') != $context->getId()) {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'message' => 'Submission not found with ID: ' . $submissionId,
                    'submission' => null
                ]);
                exit;
            }

            // Get the current publication
            $publication = $submission->getCurrentPublication();

            // Get authors from publication
            $authorsData = [];
            if ($publication) {
                $authors = $publication->getData('authors');
                if ($authors) {
                    foreach ($authors as $author) {
                        $authorsData[] = [
                            'id' => $author->getId(),
                            'name' => $author->getFullName(),
                            'email' => $author->getEmail(),
                            'affiliation' => $author->getLocalizedData('affiliation') ?: ''
                        ];
                    }
                }
            }

            // Get reviewers for this submission using Repo pattern
            $reviewersData = [];
            try {
                $reviewAssignments = Repo::reviewAssignment()
                    ->getCollector()
                    ->filterBySubmissionIds([(int) $submissionId])
                    ->getMany();
                
                foreach ($reviewAssignments as $reviewAssignment) {
                    $reviewerId = $reviewAssignment->getReviewerId();
                    $reviewer = Repo::user()->get($reviewerId);
                    if ($reviewer) {
                        $reviewersData[] = [
                            'id' => $reviewer->getId(),
                            'name' => $reviewer->getFullName(),
                            'email' => $reviewer->getEmail()
                        ];
                    }
                }
            } catch (\Exception $e) {
                // Silently fail
            }

            // Get editors for this submission
            $editorsData = [];
            try {
                $stageAssignments = \PKP\stageAssignment\StageAssignment::withSubmissionIds([(int) $submissionId])
                    ->get();
                
                $processedUserIds = [];
                foreach ($stageAssignments as $assignment) {
                    $userId = $assignment->userId;
                    if (!in_array($userId, $processedUserIds)) {
                        $user = Repo::user()->get($userId);
                        if ($user) {
                            // Check if user has editor role
                            $userGroups = Repo::userGroup()->userUserGroups($userId);
                            foreach ($userGroups as $userGroup) {
                                $roleId = $userGroup->getRoleId();
                                if (in_array($roleId, [\PKP\security\Role::ROLE_ID_MANAGER, \PKP\security\Role::ROLE_ID_SUB_EDITOR])) {
                                    $editorsData[] = [
                                        'id' => $user->getId(),
                                        'name' => $user->getFullName(),
                                        'email' => $user->getEmail()
                                    ];
                                    $processedUserIds[] = $userId;
                                    break;
                                }
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                // Silently fail
            }

            // Get issue info if assigned
            $issueId = null;
            $issueData = null;
            if ($publication) {
                $issueId = $publication->getData('issueId');
                if ($issueId) {
                    $issue = Repo::issue()->get($issueId);
                    if ($issue) {
                        $issueData = [
                            'id' => $issue->getId(),
                            'title' => $issue->getLocalizedTitle() ?: 'Vol. ' . $issue->getVolume() . ' No. ' . $issue->getNumber() . ' (' . $issue->getYear() . ')',
                            'volume' => $issue->getVolume() ?: '',
                            'number' => $issue->getNumber() ?: '',
                            'year' => $issue->getYear() ?: '',
                            'published' => $issue->getPublished() ? true : false
                        ];
                    }
                }
            }

            // Get stage info
            $stageId = $submission->getData('stageId');
            $stageLabels = [
                1 => 'Submission',
                2 => 'Internal Review',
                3 => 'External Review',
                4 => 'Copyediting',
                5 => 'Production'
            ];
            $stageName = isset($stageLabels[$stageId]) ? $stageLabels[$stageId] : 'Unknown';

            // Get status info
            $status = $submission->getData('status');
            $statusLabels = [
                \PKP\submission\PKPSubmission::STATUS_QUEUED => 'Queued',
                \PKP\submission\PKPSubmission::STATUS_PUBLISHED => 'Published',
                \PKP\submission\PKPSubmission::STATUS_DECLINED => 'Declined',
                \PKP\submission\PKPSubmission::STATUS_SCHEDULED => 'Scheduled'
            ];
            $statusName = isset($statusLabels[$status]) ? $statusLabels[$status] : 'Unknown';

            // Get title from publication
            $title = '';
            if ($publication) {
                $title = $publication->getLocalizedTitle() ?: $publication->getLocalizedFullTitle() ?: '';
            }

            $submissionData = [
                'id' => $submission->getId(),
                'title' => $title,
                'doi' => $publication ? ($publication->getStoredPubId('doi') ?: '') : '',
                'pages' => $publication ? ($publication->getData('pages') ?: '') : '',
                'dateSubmitted' => $submission->getData('dateSubmitted'),
                'datePublished' => $publication ? $publication->getData('datePublished') : null,
                'abstract' => $publication ? ($publication->getLocalizedData('abstract') ?: '') : '',
                'issueId' => $issueId,
                'issue' => $issueData,
                'sectionId' => $publication ? $publication->getData('sectionId') : null,
                'stageId' => $stageId,
                'stageName' => $stageName,
                'status' => $status,
                'statusName' => $statusName,
                'authors' => $authorsData,
                'reviewers' => $reviewersData,
                'editors' => $editorsData,
                'locale' => $submission->getData('locale'),
                'contextLocale' => $context->getPrimaryLocale()
            ];

            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'submission' => $submissionData
            ]);
            exit;

        } catch (\Exception $e) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Error loading submission: ' . $e->getMessage(),
                'submission' => null
            ]);
            exit;
        }
    }


    /**
     * Handle AJAX actions
     *
     * @param string $action
     * @param PKPRequest $request
     * @return void
     */
    protected function handleAjaxAction($action, $request)
    {
        $debugFile = dirname(__FILE__) . '/../debug.log';
        @file_put_contents($debugFile, date('Y-m-d H:i:s') . " HANDLER: handleAjaxAction called with action: $action\n", FILE_APPEND);

        // OJS 3.5: Plugin is passed via constructor
        // Map actions to plugin methods
        switch ($action) {
            case 'getIssues':
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Calling getIssues...\n", FILE_APPEND);
                $this->getIssues([], $request);
                break;

            case 'getSubmissionById':
                $this->getSubmissionById([], $request);
                break;

            case 'getSubmissions':
                @file_put_contents($debugFile, date('Y-m-d H:i:s') . " Calling getSubmissions...\n", FILE_APPEND);
                $this->getSubmissions([], $request);
                break;
                
            case 'saveTemplate':
                $this->saveTemplate([], $request);
                break;

            case 'sendTestEmail':
                $this->sendTestEmail([], $request);
                break;

            case 'sendCertificateEmail':
                $this->sendCertificateEmail([], $request);
                break;

            case 'searchUsers':
                $this->searchUsers([], $request);
                break;

            case 'pinStatus':
                $this->proxyPinStatus($request);
                break;

            case 'verifyPin':
                $this->proxyVerifyPin($request);
                break;

            case 'list':
            case 'login':
            case 'verify2FA':
            case 'verifyEmailOTP':
            case 'logout':
            case 'checkAuth':
            case 'save':
            case 'load':
            case 'delete':
            case 'preview':
            case 'getSubscription':
            case 'checkCapability':
            case 'getUserInfo':
            case 'getReviewers':
            case 'getReviewersForSubmission':
            case 'getEditorsForSubmission':
            case 'getPatterns':
            case 'previewPattern':
            case 'generatePattern':
            case 'testNumberingAPI':
            case 'generateCertificateQR':
            case 'searchReviewers':
            case 'searchAuthors':
            case 'searchEditors':
            case 'refreshAuth':
            case 'sendOtp':
            case 'verifyOtp':
            case 'testEndpoints':
            case 'proxyQRImage':
            case 'verifyPIN':
            case 'checkPINStatus':
            case 'listDefaultTemplates':
            case 'loadDefaultTemplate':
            case 'loadSavedTemplates':
            case 'saveDesignerTemplate':
            case 'deleteTemplate':
            case 'getAntiCounterfeitTags':
                // These actions should be handled by the plugin's designer handler
                if ($this->plugin) {
                    // Call the plugin's _handleDesignerAjax method directly
                    try {
                        $reflectionClass = new \ReflectionClass($this->plugin);
                        $method = $reflectionClass->getMethod('_handleDesignerAjax');
                        $method->setAccessible(true);
                        $result = $method->invoke($this->plugin, $request);

                        if ($result && is_a($result, 'PKP\\core\\JSONMessage')) {
                            header('Content-Type: application/json');
                            echo $result->getString();
                            exit;
                        }
                    } catch (\Exception $e) {
                        error_log('Error calling plugin method: ' . $e->getMessage());
                        header('Content-Type: application/json');
                        echo json_encode(['status' => false, 'error' => 'Plugin method error: ' . $e->getMessage()]);
                        exit;
                    }
                }
                break;

            // OJS Email Template Integration Actions
            case 'getOJSEmailTemplates':
                $this->getOJSEmailTemplates($request);
                break;

            case 'getOJSEmailTemplate':
                $this->getOJSEmailTemplate($request);
                break;

            case 'saveOJSEmailTemplate':
                $this->saveOJSEmailTemplate($request);
                break;

            case 'resetOJSEmailTemplate':
                $this->resetOJSEmailTemplate($request);
                break;

            case 'getOJSEmailVariables':
                $this->getOJSEmailVariables($request);
                break;

            case 'applyDesignToOJSTemplate':
                $this->applyDesignToOJSTemplate($request);
                break;

            // Custom Email Template Actions
            case 'getCustomEmailTemplates':
                $this->getCustomEmailTemplates($request);
                break;

            case 'getCustomEmailTemplate':
                $this->getCustomEmailTemplate($request);
                break;

            // Design Template Library Actions
            case 'getDesignTemplates':
                $this->getDesignTemplates($request);
                break;

            case 'getDesignTemplate':
                $this->getDesignTemplate($request);
                break;

            default:
                header('Content-Type: application/json');
                echo json_encode(['error' => 'Invalid action: ' . $action]);
                exit;
        }
    }
    
    /**
     * Email template manager
     */
    public function emailTemplateManager($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, $args);
    }
    
    /**
     * Email template designer (new visual designer)
     */
    public function emailTemplateDesigner($args, $request)
    {
        // Check if this is an AJAX request
        $action = $request->getUserVar('action');
        if ($action) {
            // Handle AJAX actions
            return $this->handleAjaxAction($action, $request);
        }

        $templateMgr = TemplateManager::getManager($request);
        $context = $request->getContext();

        // Get template ID if editing
        $templateId = $request->getUserVar('templateId');

        // Get journal logo URL
        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $logoUrl = $emailTemplateManager->getJournalLogoUrl($request);
        $logoData = $emailTemplateManager->getJournalLogoData();

        $templateMgr->assign([
            'pluginName' => $this->plugin->getName(),
            'baseUrl' => $request->getBaseUrl(),
            'contextPath' => $context->getPath(),
            'templateId' => $templateId ?? '',
            'templateName' => '',
            'journalLogoUrl' => $logoUrl ?? '',
            'journalLogoAlt' => $logoData['altText'] ?? 'Journal Logo',
            'journalLogoWidth' => $logoData['width'] ?? 180,
            'journalLogoHeight' => $logoData['height'] ?? 90,
            'journalName' => $context->getLocalizedData('name') ?? 'Journal Name',
            'lb' => '{{',  // Left brace for Smarty
            'rb' => '}}'   // Right brace for Smarty
        ]);

        // Display the template
        $templateMgr->display($this->plugin->getTemplateResource('emailTemplateDesigner.tpl'));
    }

    /**
     * Get email template for editing
     */
    public function getEmailTemplate($args, $request)
    {
        $templateId = $request->getUserVar('templateId');
        $context = $request->getContext();

        if (!$templateId || !$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
            exit;
        }

        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $template = $emailTemplateManager->getJSONTemplate($templateId);

        header('Content-Type: application/json');
        if ($template) {
            echo json_encode(['status' => 'success', 'template' => $template]);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Template not found']);
        }
        exit;
    }

    /**
     * Send test email
     */
    public function sendTestEmail($args, $request)
    {
        $context = $request->getContext();
        $testEmail = $request->getUserVar('testEmail');
        $subject = $request->getUserVar('subject');
        $htmlContent = $request->getUserVar('htmlContent');
        $cc = $request->getUserVar('cc');
        $bcc = $request->getUserVar('bcc');

        if (!$context || !$testEmail || !$subject || !$htmlContent) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
            exit;
        }

        // Handle file attachments
        $attachments = [];

        // Debug logging to file
        $debugMsg = "\n\n=== ATTACHMENT DEBUG " . date('Y-m-d H:i:s') . " ===\n";
        $debugMsg .= "Checking for attachments...\n";
        $debugMsg .= "\$_FILES content: " . print_r($_FILES, true) . "\n";
        file_put_contents('/tmp/govalid_attachments.log', $debugMsg, FILE_APPEND);

        error_log("GoValid Plugin: Checking for attachments...");
        error_log("GoValid Plugin: \$_FILES content: " . print_r($_FILES, true));

        if (isset($_FILES['attachments']) && is_array($_FILES['attachments']['name'])) {
            $fileCount = count($_FILES['attachments']['name']);
            error_log("GoValid Plugin: Found {$fileCount} attachments");

            for ($i = 0; $i < $fileCount; $i++) {
                // Check if file was uploaded without errors
                if ($_FILES['attachments']['error'][$i] === UPLOAD_ERR_OK) {
                    $tmpPath = $_FILES['attachments']['tmp_name'][$i];
                    $fileName = $_FILES['attachments']['name'][$i];
                    $fileSize = $_FILES['attachments']['size'][$i];
                    $fileType = $_FILES['attachments']['type'][$i];

                    // Validate file size (5MB max)
                    if ($fileSize > 5 * 1024 * 1024) {
                        header('Content-Type: application/json');
                        echo json_encode(['status' => 'error', 'message' => "File {$fileName} exceeds 5MB limit"]);
                        exit;
                    }

                    // Validate file type
                    $allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                    $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                    $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png', 'gif'];

                    if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, $allowedExtensions)) {
                        header('Content-Type: application/json');
                        echo json_encode(['status' => 'error', 'message' => "File {$fileName} has invalid type. Only PDF and images allowed."]);
                        exit;
                    }

                    // Read file contents
                    $fileContents = file_get_contents($tmpPath);

                    if ($fileContents !== false) {
                        $attachments[] = [
                            'name' => $fileName,
                            'data' => $fileContents,
                            'type' => $fileType
                        ];
                        error_log("GoValid Plugin: Added attachment: {$fileName} ({$fileSize} bytes, {$fileType})");
                        file_put_contents('/tmp/govalid_attachments.log', "Added attachment: {$fileName} ({$fileSize} bytes, {$fileType})\n", FILE_APPEND);
                    }
                }
            }
        } else {
            error_log("GoValid Plugin: No attachments found in \$_FILES");
            file_put_contents('/tmp/govalid_attachments.log', "No attachments found in \$_FILES\n", FILE_APPEND);
        }

        error_log("GoValid Plugin: Total attachments prepared: " . count($attachments));
        file_put_contents('/tmp/govalid_attachments.log', "Total attachments prepared: " . count($attachments) . "\n", FILE_APPEND);

        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $result = $emailTemplateManager->sendTestEmail($testEmail, $subject, $htmlContent, $cc, $bcc, $attachments);

        header('Content-Type: application/json');
        echo json_encode($result);
        exit;
    }

    /**
     * Search users for email autocomplete
     */
    public function searchUsers($args, $request)
    {
        $context = $request->getContext();
        $searchTerm = $request->getUserVar('q');

        if (!$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'No context']);
            exit;
        }

        // Require at least 2 characters to search
        if (empty($searchTerm) || strlen($searchTerm) < 2) {
            header('Content-Type: application/json');
            echo json_encode(['results' => []]);
            exit;
        }

        try {
            // Use OJS 3.4 Repository pattern
            $collector = \APP\facades\Repo::user()->getCollector();
            $collector->filterByContextIds([$context->getId()])
                ->searchPhrase($searchTerm)
                ->limit(10);

            $users = $collector->getMany();
            $results = [];

            foreach ($users as $user) {
                $results[] = [
                    'id' => $user->getId(),
                    'email' => $user->getEmail(),
                    'name' => $user->getFullName(),
                    'label' => $user->getFullName() . ' <' . $user->getEmail() . '>'
                ];
            }

            header('Content-Type: application/json');
            echo json_encode(['results' => $results]);
            exit;
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error searching users: ' . $e->getMessage());
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
            exit;
        }
    }

    /**
     * Save email template
     */
    public function saveEmailTemplate($args, $request)
    {
        $context = $request->getContext();
        $templateData = $request->getUserVar('templateData');
        $templateId = $request->getUserVar('templateId');

        // Check if this is a JSON-based template (from new designer)
        if ($templateData) {
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
            $result = $emailTemplateManager->saveJSONTemplate([
                'templateId' => $templateId,
                'templateData' => $templateData
            ]);

            header('Content-Type: application/json');
            echo json_encode($result);
            exit;
        }

        // Otherwise, use old handler
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['saveEmailTemplate'], $args));
    }
    
    /**
     * Delete email template
     */
    public function deleteTemplate($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['deleteTemplate'], $args));
    }
    
    /**
     * Activate email template
     */
    public function activateTemplate($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['activateTemplate'], $args));
    }
    
    /**
     * Preview email template
     */
    public function previewTemplate($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['previewTemplate'], $args));
    }
    
    /**
     * Get default template
     */
    public function getDefaultTemplate($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['getDefaultTemplate'], $args));
    }
    
    /**
     * Test email template
     */
    public function testEmail($args, $request)
    {
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['testEmail'], $args));
    }

    /**
     * Handle AJAX requests for designer
     *
     * @param array $args
     * @param PKPRequest $request
     */
    public function ajax($args, $request)
    {
        // This is essentially the same as designer() but specifically for AJAX
        $action = $request->getUserVar('action');
        if ($action) {
            return $this->handleAjaxAction($action, $request);
        }

        header('Content-Type: application/json');
        echo json_encode(['status' => false, 'error' => 'No action specified']);
        exit;
    }

    // ========================================================================
    // OJS Email Template Integration Methods
    // ========================================================================

    /**
     * Get all OJS email templates
     */
    protected function getOJSEmailTemplates($request)
    {
        try {
            error_log("GoValid Plugin: getOJSEmailTemplates handler called");
            $context = $request->getContext();
            error_log("GoValid Plugin: Context ID: " . ($context ? $context->getId() : 'NULL'));

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            error_log("GoValid Plugin: OJSEmailTemplateIntegration class loaded");

            $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
            error_log("GoValid Plugin: Integration instance created");

            $templates = $integration->getOJSEmailTemplates();
            error_log("GoValid Plugin: Templates loaded - count: " . count($templates));

            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
            exit;
        } catch (\Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getOJSEmailTemplates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    /**
     * Get a specific OJS email template
     */
    protected function getOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
        $template = $integration->getOJSEmailTemplate($templateKey);

        if ($template) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['template' => $template]]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
        }
        exit;
    }

    /**
     * Save/update an OJS email template
     */
    protected function saveOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');
        $templateName = $request->getUserVar('templateName'); // Optional - for new templates
        $alternateTo = $request->getUserVar('alternateTo'); // Optional - for alternate templates
        $subject = $request->getUserVar('subject');
        $body = $request->getUserVar('body');

        if (!$templateKey || !$subject || !$body) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Missing required parameters']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
        $success = $integration->saveOJSEmailTemplate($templateKey, $subject, $body, $templateName, $alternateTo);

        if ($success) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['success' => true, 'message' => 'Template saved successfully']]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Failed to save template']]);
        }
        exit;
    }

    /**
     * Reset OJS email template to default
     */
    protected function resetOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
        $success = $integration->resetOJSEmailTemplate($templateKey);

        if ($success) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['success' => true, 'message' => 'Template reset to default']]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Failed to reset template']]);
        }
        exit;
    }

    /**
     * Get available OJS email template variables
     */
    protected function getOJSEmailVariables($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
        $variables = $integration->getAvailableVariables($templateKey);

        header('Content-Type: application/json');
        echo json_encode(['status' => true, 'content' => ['variables' => $variables]]);
        exit;
    }

    /**
     * Apply design to an OJS email template
     */
    protected function applyDesignToOJSTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');
        $designConfigJson = $request->getUserVar('designConfig');

        if (!$templateKey || !$designConfigJson) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Missing required parameters']]);
            exit;
        }

        $designConfig = json_decode($designConfigJson, true);
        if (!$designConfig) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Invalid design configuration']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);
        $result = $integration->applyDesignToOJSTemplate($templateKey, $designConfig);

        if ($result['status'] === 'success') {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => $result]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => $result]);
        }
        exit;
    }

    // ========================================================================
    // Custom Email Template Methods
    // ========================================================================

    /**
     * Get all custom email templates (saved via "Save As")
     */
    protected function getCustomEmailTemplates($request)
    {
        try {
            error_log("GoValid Plugin: getCustomEmailTemplates handler called");
            $context = $request->getContext();

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);

            $templates = $integration->getCustomEmailTemplates();
            error_log("GoValid Plugin: Custom templates loaded - count: " . count($templates));

            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
            exit;
        } catch (\Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getCustomEmailTemplates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    /**
     * Get a specific custom email template by ID
     */
    protected function getCustomEmailTemplate($request)
    {
        try {
            $context = $request->getContext();
            $templateId = $request->getUserVar('templateId');

            if (!$templateId) {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'content' => ['error' => 'Template ID required']]);
                exit;
            }

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);

            $template = $integration->getCustomEmailTemplate($templateId);

            if ($template) {
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'content' => ['template' => $template]]);
            } else {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
            }
            exit;
        } catch (\Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getCustomEmailTemplate: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    // ========================================================================
    // Design Template Library Methods
    // ========================================================================

    /**
     * Get all design templates
     */
    protected function getDesignTemplates($request)
    {
        require_once($this->plugin->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
        $templates = \APP\plugins\generic\goValidOJS\classes\EmailDesignTemplateLibrary::getDesignTemplates();

        header('Content-Type: application/json');
        echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
        exit;
    }

    /**
     * Get a specific design template
     */
    protected function getDesignTemplate($request)
    {
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
        $template = \APP\plugins\generic\goValidOJS\classes\EmailDesignTemplateLibrary::getDesignTemplate($templateKey);

        if ($template) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['template' => $template]]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
        }
        exit;
    }

    /**
     * Send certificate via email with attachment
     */
    public function sendCertificateEmail($args, $request)
    {
        $context = $request->getContext();

        // Get parameters
        $templateName = $request->getUserVar('templateName');
        $emailSubject = $request->getUserVar('emailSubject'); // Edited subject from preview
        $emailBody = $request->getUserVar('emailBody'); // Edited body from preview
        $toEmail = $request->getUserVar('toEmail');
        $ccEmail = $request->getUserVar('ccEmail');
        $bccEmail = $request->getUserVar('bccEmail');
        $attachmentFormat = $request->getUserVar('attachmentFormat'); // 'pdf' or 'jpg'
        $certificateImageData = $request->getUserVar('certificateImageData'); // Legacy: Direct canvas rendering (base64)
        $canvasData = $request->getUserVar('canvasData'); // Legacy support
        $replacementsJson = $request->getUserVar('replacements');
        $orientation = $request->getUserVar('orientation');

        // Check for file upload (new method - Blob instead of base64 string)
        $hasCertificateFile = isset($_FILES['certificateImageFile']) && $_FILES['certificateImageFile']['error'] === UPLOAD_ERR_OK;

        if (!$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'No context']);
            exit;
        }

        // Check if we have certificate data in any form
        if (empty($toEmail) || (!$hasCertificateFile && empty($certificateImageData) && empty($canvasData)) || empty($replacementsJson)) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
            exit;
        }

        // Use edited content if provided, otherwise load from template
        if (empty($emailSubject) || empty($emailBody)) {
            if (empty($templateName)) {
                header('Content-Type: application/json');
                echo json_encode(['status' => 'error', 'message' => 'Missing email content']);
                exit;
            }
        }

        try {
            // Parse replacements
            $replacements = json_decode($replacementsJson, true);
            if (!$replacements) {
                throw new \Exception('Invalid replacements data');
            }

            // Initialize email template manager
            require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);

            // Use edited email content if provided, otherwise load from template
            if (!empty($emailSubject) && !empty($emailBody)) {
                // Use the edited content from the preview
                $emailHtml = $emailBody;
                $finalEmailSubject = $emailSubject;
            } else {
                // Load from template (fallback)
                require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this->plugin, $context);

                $ojsTemplate = $integration->getTemplateByKey($templateName);
                if (!$ojsTemplate) {
                    throw new \Exception('Email template not found: ' . $templateName);
                }

                $emailHtml = $ojsTemplate['body'] ?? $ojsTemplate['html'] ?? '';
                $finalEmailSubject = $ojsTemplate['subject'] ?? 'Certificate';
            }

            // Generate or process certificate file
            $certificateData = null;
            $attachmentFilename = '';
            $mimeType = '';

            if ($hasCertificateFile) {
                // New method: File upload (Blob) - direct binary data
                error_log("GoValid Plugin: Using uploaded certificate file (Blob method)");

                $certificateData = file_get_contents($_FILES['certificateImageFile']['tmp_name']);

                if ($certificateData === false) {
                    throw new \Exception('Failed to read uploaded certificate file');
                }

                $attachmentFilename = 'certificate_' . time() . '.jpg';
                $mimeType = 'image/jpeg';

                error_log("GoValid Plugin: Certificate file uploaded successfully, size: " . number_format(strlen($certificateData) / 1024 / 1024, 2) . " MB");

                // Verify it's a valid JPEG by checking magic bytes
                $magicBytes = bin2hex(substr($certificateData, 0, 4));
                error_log("GoValid Plugin: JPEG magic bytes: {$magicBytes} (should start with ffd8ff)");

                if (substr($magicBytes, 0, 6) !== 'ffd8ff') {
                    error_log("GoValid Plugin: WARNING - File does not appear to be a valid JPEG!");
                }

            } elseif (!empty($certificateImageData)) {
                // Legacy method: Direct canvas rendering - decode base64 JPG data
                error_log("GoValid Plugin: Using legacy base64 method (may have encoding issues)");

                // Remove data URL prefix if present (data:image/jpeg;base64,)
                if (strpos($certificateImageData, 'data:image/jpeg;base64,') === 0) {
                    $certificateImageData = substr($certificateImageData, strlen('data:image/jpeg;base64,'));
                } elseif (strpos($certificateImageData, 'data:image/jpg;base64,') === 0) {
                    $certificateImageData = substr($certificateImageData, strlen('data:image/jpg;base64,'));
                }

                $certificateData = base64_decode($certificateImageData);

                if ($certificateData === false) {
                    throw new \Exception('Failed to decode certificate image data');
                }

                $attachmentFilename = 'certificate_' . time() . '.jpg';
                $mimeType = 'image/jpeg';

                error_log("GoValid Plugin: Certificate JPG decoded successfully, size: " . number_format(strlen($certificateData) / 1024 / 1024, 2) . " MB");

                // Verify it's a valid JPEG by checking magic bytes
                $magicBytes = bin2hex(substr($certificateData, 0, 4));
                error_log("GoValid Plugin: JPEG magic bytes: {$magicBytes} (should start with ffd8ff)");

            } else {
                // Legacy support: Generate from canvas data (PDF/JPG)
                error_log("GoValid Plugin: Using legacy PDF/JPG generation from canvas data");

                require_once($this->plugin->getPluginPath() . '/classes/CertificateGenerator.php');
                $certificateGenerator = new \APP\plugins\generic\goValidOJS\classes\CertificateGenerator($this->plugin, $context);

                $canvasArray = json_decode($canvasData, true);
                if (!$canvasArray) {
                    throw new \Exception('Invalid canvas data');
                }

                // Generate certificate
                if ($attachmentFormat === 'jpg') {
                    $certificateData = $certificateGenerator->generateJPG($canvasArray, $replacements, $orientation);
                    $attachmentFilename = 'certificate_' . time() . '.jpg';
                    $mimeType = 'image/jpeg';
                } else {
                    // Default to PDF
                    $certificateData = $certificateGenerator->generatePDF($canvasArray, $replacements, $orientation);
                    $attachmentFilename = 'certificate_' . time() . '.pdf';
                    $mimeType = 'application/pdf';
                }
            }

            if (!$certificateData) {
                throw new \Exception('Failed to generate certificate');
            }

            // Check certificate size
            $certificateSizeMB = strlen($certificateData) / 1024 / 1024;
            error_log("GoValid Plugin: Certificate size: " . number_format($certificateSizeMB, 2) . " MB");

            // Limit: Certificate max 5MB
            if (strlen($certificateData) > 5 * 1024 * 1024) {
                throw new \Exception("Certificate file is too large (" . number_format($certificateSizeMB, 2) . " MB). Please simplify your design, reduce image sizes, or remove some elements. Maximum allowed: 5MB");
            }

            // Handle additional attachments
            $additionalAttachments = [];
            $totalAdditionalSize = 0;

            if (isset($_FILES['additionalAttachments']) && is_array($_FILES['additionalAttachments']['name'])) {
                $fileCount = count($_FILES['additionalAttachments']['name']);
                error_log("GoValid Plugin: Found {$fileCount} additional attachments for certificate");

                for ($i = 0; $i < $fileCount; $i++) {
                    if ($_FILES['additionalAttachments']['error'][$i] === UPLOAD_ERR_OK) {
                        $tmpPath = $_FILES['additionalAttachments']['tmp_name'][$i];
                        $fileName = $_FILES['additionalAttachments']['name'][$i];
                        $fileSize = $_FILES['additionalAttachments']['size'][$i];
                        $fileType = $_FILES['additionalAttachments']['type'][$i];

                        // Validate file size (5MB max per file)
                        if ($fileSize > 5 * 1024 * 1024) {
                            throw new \Exception("File {$fileName} exceeds 5MB limit");
                        }

                        // Validate file type
                        $allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                        $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                        $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png', 'gif'];

                        if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, $allowedExtensions)) {
                            throw new \Exception("File {$fileName} has invalid type. Only PDF and images allowed.");
                        }

                        // Read file contents
                        $fileContents = file_get_contents($tmpPath);

                        if ($fileContents !== false) {
                            $totalAdditionalSize += strlen($fileContents);

                            $additionalAttachments[] = [
                                'name' => $fileName,
                                'data' => $fileContents,
                                'type' => $fileType
                            ];
                            error_log("GoValid Plugin: Added additional attachment: {$fileName} ({$fileSize} bytes, {$fileType})");
                        }
                    }
                }
            }

            // Check total email size (Certificate + Additional Attachments)
            // Limit: Total 10MB for all attachments combined
            $totalSize = strlen($certificateData) + $totalAdditionalSize;
            $totalSizeMB = $totalSize / 1024 / 1024;

            if ($totalSize > 10 * 1024 * 1024) {
                $certSizeMB = number_format(strlen($certificateData) / 1024 / 1024, 2);
                $addSizeMB = number_format($totalAdditionalSize / 1024 / 1024, 2);
                throw new \Exception("Total email size is too large (" . number_format($totalSizeMB, 2) . " MB). Certificate: {$certSizeMB} MB + Additional files: {$addSizeMB} MB. Maximum allowed: 10MB total. Please reduce file sizes or remove some attachments.");
            }

            error_log("GoValid Plugin: Total email size: " . number_format($totalSizeMB, 2) . " MB (Certificate: " . number_format($certificateSizeMB, 2) . " MB + Additional: " . number_format($totalAdditionalSize / 1024 / 1024, 2) . " MB)");

            // Send email with certificate attachment and additional attachments
            $result = $emailTemplateManager->sendEmailWithAttachment(
                $toEmail,
                $finalEmailSubject,
                $emailHtml,
                $ccEmail,
                $bccEmail,
                $certificateData,
                $attachmentFilename,
                $mimeType,
                $additionalAttachments
            );

            if ($result['status'] === 'success') {
                header('Content-Type: application/json');
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Certificate sent successfully'
                ]);
            } else {
                throw new \Exception($result['message']);
            }

        } catch (\Exception $e) {
            error_log('GoValidOJS: Error sending certificate email: ' . $e->getMessage());
            header('Content-Type: application/json');
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }

        exit;
    }

    /**
     * Proxy request to GoValid API for PIN status
     */
    private function proxyPinStatus($request)
    {
        $token = $request->getUserVar('token');

        if (!$token) {
            // Try to get token from Authorization header
            $headers = getallheaders();
            if (isset($headers['Authorization'])) {
                $token = str_replace('Bearer ', '', $headers['Authorization']);
            }
        }

        if (!$token) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No authentication token provided']);
            exit;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://my.govalid.org/api/v1/ojs/pin-status/');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Content-Type: application/json'
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log('GoValidOJS: CURL error in proxyPinStatus: ' . $error);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'Network error: ' . $error]);
            exit;
        }

        header('Content-Type: application/json');
        http_response_code($httpCode);
        echo $response;
        exit;
    }

    /**
     * Proxy request to GoValid API for PIN verification
     */
    private function proxyVerifyPin($request)
    {
        $token = $request->getUserVar('token');
        $pin = $request->getUserVar('pin');

        if (!$token) {
            // Try to get token from Authorization header
            $headers = getallheaders();
            if (isset($headers['Authorization'])) {
                $token = str_replace('Bearer ', '', $headers['Authorization']);
            }
        }

        if (!$token) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No authentication token provided']);
            exit;
        }

        if (!$pin) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No PIN provided']);
            exit;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://my.govalid.org/api/v1/ojs/verify-pin/');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['pin' => $pin]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Content-Type: application/json'
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log('GoValidOJS: CURL error in proxyVerifyPin: ' . $error);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'Network error: ' . $error]);
            exit;
        }

        header('Content-Type: application/json');
        http_response_code($httpCode);
        echo $response;
        exit;
    }
}