    // Load default template on page load (user's saved default or LoA)
    async function loadDefaultLoATemplate() {
        // Only load if canvas is empty
        if (canvas.getObjects().length > 0) {
            return;
        }

        // Check if user has set a custom default template
        const userDefault = localStorage.getItem('govalidDefaultTemplate');

        if (userDefault) {
            // Load user's saved template as default
            console.log('Loading user default template:', userDefault);
            try {
                const response = await fetch(getAjaxURL(), {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                        'Accept': 'application/json',
                    },
                    credentials: 'same-origin',
                    body: 'action=load&templateName=' + encodeURIComponent(userDefault)
                });

                const data = await response.json();
                const result = data.content || data;

                if (result.templateData) {
                    // Set orientation first if specified
                    if (result.orientation && result.orientation !== currentOrientation) {
                        currentOrientation = result.orientation;
                        const orientationRadio = document.querySelector(`input[name="orientation"][value="${result.orientation}"]`);
                        if (orientationRadio) {
                            orientationRadio.checked = true;
                        }
                        const A4_WIDTH_PX = 794;
                        const A4_HEIGHT_PX = 1123;
                        const width = result.orientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
                        const height = result.orientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
                        canvas.setWidth(width * currentZoom);
                        canvas.setHeight(height * currentZoom);
                        canvas.setZoom(currentZoom);
                    }

                    const templateData = typeof result.templateData === 'string'
                        ? JSON.parse(result.templateData)
                        : result.templateData;

                    canvas.loadFromJSON(templateData, () => {
                        // Track journal logo placeholders for replacement
                        const objectsToRemove = [];
                        const logoReplacements = [];

                        canvas.getObjects().forEach(function (obj) {
                            // Ensure object is selectable and moveable
                            obj.set({
                                selectable: true,
                                evented: true,
                                hasControls: true,
                                hasBorders: true,
                                lockMovementX: false,
                                lockMovementY: false
                            });

                            if (obj.fieldType && typeof obj.fieldType === 'string' && obj.fieldType.startsWith('{')) {
                                // Handle {JOURNAL_LOGO} placeholder
                                if (obj.fieldType === '{JOURNAL_LOGO}') {
                                    if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                                        logoReplacements.push({
                                            left: obj.left,
                                            top: obj.top,
                                            width: obj.width * (obj.scaleX || 1),
                                            height: obj.height * (obj.scaleY || 1),
                                            originX: obj.originX || 'left',
                                            originY: obj.originY || 'top'
                                        });
                                        objectsToRemove.push(obj);
                                    }
                                } else if (obj.fieldType === '{ISSN}') {
                                // Keep ISSN text as-is with placeholders visible
                                // Text already contains "ISSN: {ISSN_ONLINE} (Online)\nISSN: {ISSN_PRINT} (Print)"
                            } else if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
                                    obj.set('text', obj.fieldType);
                                } else if (obj.type === 'group' && obj._objects) {
                                    obj._objects.forEach(function(child) {
                                        if (child.type === 'text' && child.text) {
                                            child.set('text', obj.fieldType);
                                        }
                                    });
                                }
                            }
                        });

                        // Remove journal logo placeholders
                        objectsToRemove.forEach(function(obj) {
                            canvas.remove(obj);
                        });

                        // Load journal logo images
                        logoReplacements.forEach(function(pos) {
                            fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                                if (img) {
                                    const targetWidth = pos.width || 150;
                                    const scaleX = targetWidth / img.width;
                                    img.set({
                                        left: pos.left,
                                        top: pos.top,
                                        scaleX: scaleX,
                                        scaleY: scaleX,
                                        originX: pos.originX,
                                        originY: pos.originY,
                                        fieldType: 'journal_logo',
                                        selectable: true,
                                        evented: true,
                                        hasControls: true,
                                        hasBorders: true,
                                        lockMovementX: false,
                                        lockMovementY: false,
                                        cornerColor: '#ff6b6b',
                                        cornerSize: 8,
                                        transparentCorners: false
                                    });
                                    canvas.add(img);
                                    canvas.renderAll();
                                }
                            }, { crossOrigin: 'anonymous' });
                        });

                        canvas.renderAll();
                        updateLayerUI();
                        console.log('User default template loaded:', userDefault);
                    });
                    return;
                }
            } catch (error) {
                console.error('Error loading user default template:', error);
                // Fall through to load LoA
            }
        }

        // Load LoA as fallback default
        try {
            console.log('Loading default LoA template...');

            const response = await fetch(getAjaxURL(), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'Accept': 'application/json',
                },
                credentials: 'same-origin',
                body: 'action=loadDefaultTemplate&templateFile=01_letter_of_acceptance.json'
            });

            const data = await response.json();
            console.log('Default LoA template response:', data);

            if (data.status && data.content && data.content.templateData) {
                // Set orientation FIRST, before loading canvas data
                if (data.content.orientation && data.content.orientation !== currentOrientation) {
                    currentOrientation = data.content.orientation;
                    const orientationRadio = document.querySelector(`input[name="orientation"][value="${data.content.orientation}"]`);
                    if (orientationRadio) {
                        orientationRadio.checked = true;
                    }
                    // Set canvas dimensions directly
                    const A4_WIDTH_PX = 794;
                    const A4_HEIGHT_PX = 1123;
                    const width = data.content.orientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
                    const height = data.content.orientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
                    canvas.setWidth(width * currentZoom);
                    canvas.setHeight(height * currentZoom);
                    canvas.setZoom(currentZoom);
                }

                // Parse the template data
                const templateData = typeof data.content.templateData === 'string'
                    ? JSON.parse(data.content.templateData)
                    : data.content.templateData;

                // Load onto canvas
                canvas.loadFromJSON(templateData, () => {
                    // Track journal logo placeholders for replacement
                    const objectsToRemove = [];
                    const logoReplacements = [];

                    // Restore placeholder text for objects that have fieldType property
                    canvas.getObjects().forEach(function (obj) {
                        // Ensure object is selectable and moveable
                        obj.set({
                            selectable: true,
                            evented: true,
                            hasControls: true,
                            hasBorders: true,
                            lockMovementX: false,
                            lockMovementY: false
                        });

                        if (obj.fieldType && typeof obj.fieldType === 'string' && obj.fieldType.startsWith('{')) {
                            // Handle {JOURNAL_LOGO} placeholder - replace with actual logo image
                            if (obj.fieldType === '{JOURNAL_LOGO}') {
                                if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                                    logoReplacements.push({
                                        left: obj.left,
                                        top: obj.top,
                                        width: obj.width * (obj.scaleX || 1),
                                        height: obj.height * (obj.scaleY || 1),
                                        originX: obj.originX || 'left',
                                        originY: obj.originY || 'top'
                                    });
                                    objectsToRemove.push(obj);
                                }
                            } else if (obj.fieldType === '{ISSN}') {
                                // Keep ISSN text as-is with placeholders visible
                                // Text already contains "ISSN: {ISSN_ONLINE} (Online)\nISSN: {ISSN_PRINT} (Print)"
                            } else if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
                                obj.set('text', obj.fieldType);
                            } else if (obj.type === 'group' && obj._objects) {
                                obj._objects.forEach(function(child) {
                                    if (child.type === 'text' && child.text) {
                                        child.set('text', obj.fieldType);
                                    }
                                });
                            }
                        }
                    });

                    // Remove journal logo placeholders
                    objectsToRemove.forEach(function(obj) {
                        canvas.remove(obj);
                    });

                    // Load journal logo images at placeholder positions
                    logoReplacements.forEach(function(pos) {
                        fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                            if (img) {
                                const targetWidth = pos.width || 150;
                                const scaleX = targetWidth / img.width;

                                img.set({
                                    left: pos.left,
                                    top: pos.top,
                                    scaleX: scaleX,
                                    scaleY: scaleX,
                                    originX: pos.originX,
                                    originY: pos.originY,
                                    fieldType: 'journal_logo',
                                    selectable: true,
                                    evented: true,
                                    hasControls: true,
                                    hasBorders: true,
                                    lockMovementX: false,
                                    lockMovementY: false,
                                    cornerColor: '#ff6b6b',
                                    cornerSize: 8,
                                    transparentCorners: false
                                });

                                canvas.add(img);
                                canvas.renderAll();
                            }
                        }, { crossOrigin: 'anonymous' });
                    });

                    canvas.renderAll();
                    updateLayerUI();
                    console.log('LoA template loaded successfully');
                });
            } else {
                // Fallback to default objects if template loading fails
                console.log('Failed to load LoA template, using default objects');
                addDefaultObjects();
            }
        } catch (error) {
            console.error('Error loading default LoA template:', error);
            // Fallback to default objects
            addDefaultObjects();
        }
    }

    function addDefaultObjects() {
        // Only add if canvas is empty
        if (canvas.getObjects().length > 0) {
            return;
        }

        // Calculate positions based on canvas size (landscape by default)
        // Landscape: width=1123px, height=794px
        const canvasWidth = canvas.width || 1123;
        const canvasHeight = canvas.height || 794;
        const qrSize = 125;

        // QR Code at center bottom
        const qrLeft = (canvasWidth / 2) - (qrSize / 2);
        const qrTop = canvasHeight - qrSize - 30;

        // Add QR Code as a placeholder box (125x125px) at center bottom
        const qrBox = new fabric.Rect({
            left: 0,
            top: 0,
            width: qrSize,
            height: qrSize,
            fill: '#f0f0f0',
            stroke: '#333',
            strokeWidth: 2,
            cornerColor: '#ff6b6b',
            cornerSize: 8,
            transparentCorners: false,
            fieldType: '{QR_CODE}'
        });

        const qrLabel = new fabric.Text('{QR_CODE}', {
            left: qrSize / 2,
            top: qrSize / 2,
            fontSize: 12,
            fill: '#333',
            originX: 'center',
            originY: 'center',
            textBaseline: 'alphabetic',
            selectable: false,
            evented: false
        });

        const qrGroup = new fabric.Group([qrBox, qrLabel], {
            left: qrLeft,
            top: qrTop,
            fieldType: '{QR_CODE}',
            cornerColor: '#ff6b6b',
            cornerSize: 8,
            transparentCorners: false
        });

        canvas.add(qrGroup);

        // Add QR ID below the QR code (centered)
        const qrId = new fabric.Text('{QR_ID}', {
            left: qrLeft + (qrSize / 2),
            top: qrTop + qrSize + 10,
            fontSize: 12,
            fill: '#d97706',
            backgroundColor: '#fffbea',
            padding: 6,
            fontWeight: 'bold',
            textAlign: 'center',
            originX: 'center',
            textBaseline: 'alphabetic',
            fieldType: '{QR_ID}',
            cornerColor: '#f59e0b',
            cornerSize: 8,
            transparentCorners: false
        });

        canvas.add(qrId);

        // Add QR Identifier at top-right corner
        const qrIdentifier = new fabric.Text('{QR_IDENTIFIER}', {
            left: canvasWidth - 200,
            top: 30,
            fontSize: 14,
            fill: '#d97706',
            backgroundColor: '#fffbea',
            padding: 8,
            fontWeight: 'bold',
            textBaseline: 'alphabetic',
            fieldType: '{QR_IDENTIFIER}',
            cornerColor: '#f59e0b',
            cornerSize: 8,
            transparentCorners: false
        });

        canvas.add(qrIdentifier);
        canvas.renderAll();
        updateLayerUI();
    }


    // Initialize all functionality after functions are defined
    document.addEventListener('DOMContentLoaded', function() {
        try {
            // Hide login modal by default - will show only if needed
            const loginModal = document.getElementById('govalidLoginModal');
            if (loginModal) {
                loginModal.style.setProperty('display', 'none', 'important');
            }

            setupCanvasEventHandlers();
            setupDragAndDrop();
            setupBackgroundControls();
            // Layer controls removed in favor of drag and drop
            updateLayerUI();
            // loadTemplateList() removed - templates handled by Load Template button
            loadJournalLogo();
            handleOrientationChange();
            handleZoom();
            setupPreview();
            setupAuthentication();
            setupTabs();
            setupQRSettings();
            setupIdentifierGenerator();
            setupContentTab();
            setupLoadTemplate();
            setupOTPVerification();
            setupKeyboardShortcuts(); // Initialize keyboard shortcuts for undo/redo/copy/paste/delete

            // Update UI to show auth overlay if not authenticated
            updateUIAuthState();
            
            // Initialize auto-populated dropdowns
            initializeDropdowns();
            loadIssuesOnPageLoad();

            // Load LoA template by default
            setTimeout(() => {
                loadDefaultLoATemplate();
            }, 200);

            // Mark page as fully loaded after a delay to ensure everything is initialized
            setTimeout(() => {
                console.log('Page marked as fully loaded - OTP modal now allowed');
            }, 1000);
        } catch (error) {
            console.error('Error during initialization:', error);
            console.error('Stack trace:', error.stack);
            // Continue initialization even if some parts fail
            // Still mark as loaded even if there are errors
            setTimeout(() => {
                console.log('Page marked as fully loaded (with errors) - OTP modal now allowed');
            }, 2000);
        }
        
        // Auto-fit canvas after a short delay to ensure container dimensions are ready
        setTimeout(autoFitCanvas, 100);
        
        // Re-fit on window resize
        let resizeTimeout;
        window.addEventListener('resize', function() {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(autoFitCanvas, 250);
        });
    });
