<?php

/**
 * @file plugins/generic/govalidOJS/classes/GoValidOJSSettingsForm.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class GoValidOJSSettingsForm
 *
 * @ingroup plugins_generic_govalidOJS
 *
 * @brief Form for journal managers to modify GoValid OJS plugin settings
 */

namespace APP\plugins\generic\goValidOJS\classes;

use APP\plugins\generic\goValidOJS\GoValidOJSPlugin;
use APP\template\TemplateManager;
use PKP\form\Form;

class GoValidOJSSettingsForm extends Form
{
    /** @var int Context ID */
    public int $_contextId;

    /** @var GoValidOJSPlugin Plugin instance */
    public GoValidOJSPlugin $_plugin;

    /**
     * Constructor
     *
     * @param \APP\plugins\generic\goValidOJS\GoValidOJSPlugin $plugin
     * @param int $contextId
     */
    public function __construct($plugin, $contextId)
    {
        $this->_contextId = $contextId;
        $this->_plugin = $plugin;

        parent::__construct($plugin->getTemplateResource('settingsForm.tpl'));

        // Add form checks
        $this->addCheck(new \PKP\form\validation\FormValidatorPost($this));
        $this->addCheck(new \PKP\form\validation\FormValidatorCSRF($this));
    }

    /**
     * Initialize form data.
     */
    public function initData()
    {
        $this->_data = [
            // FAB visibility settings - default all to true
            'showFabForManagers' => $this->_plugin->getSetting($this->_contextId, 'showFabForManagers') ?? true,
            'showFabForEditors' => $this->_plugin->getSetting($this->_contextId, 'showFabForEditors') ?? true,
            'showFabForCopyeditors' => $this->_plugin->getSetting($this->_contextId, 'showFabForCopyeditors') ?? true,
            'showFabForAssistants' => $this->_plugin->getSetting($this->_contextId, 'showFabForAssistants') ?? true,
            // Smart Verify Widget settings
            'enableSmartVerifyWidget' => $this->_plugin->getSetting($this->_contextId, 'enableSmartVerifyWidget') ?? false,
            'smartVerifyWidgetPosition' => $this->_plugin->getSetting($this->_contextId, 'smartVerifyWidgetPosition') ?? 'bottom-right',
        ];
    }

    /**
     * Assign form data to user-submitted data.
     */
    public function readInputData()
    {
        $this->readUserVars([
            'showFabForManagers',
            'showFabForEditors',
            'showFabForCopyeditors',
            'showFabForAssistants',
            'enableSmartVerifyWidget',
            'smartVerifyWidgetPosition',
        ]);
    }

    /**
     * @copydoc Form::fetch()
     *
     * @param null|mixed $template
     */
    public function fetch($request, $template = null, $display = false)
    {
        $templateMgr = TemplateManager::getManager($request);
        $templateMgr->assign('pluginName', $this->_plugin->getName());

        // FAB visibility settings
        $templateMgr->assign('showFabForManagers', $this->getData('showFabForManagers'));
        $templateMgr->assign('showFabForEditors', $this->getData('showFabForEditors'));
        $templateMgr->assign('showFabForCopyeditors', $this->getData('showFabForCopyeditors'));
        $templateMgr->assign('showFabForAssistants', $this->getData('showFabForAssistants'));

        // Smart Verify Widget settings
        $templateMgr->assign('enableSmartVerifyWidget', $this->getData('enableSmartVerifyWidget'));
        $templateMgr->assign('smartVerifyWidgetPosition', $this->getData('smartVerifyWidgetPosition'));

        // Designer URLs
        $context = $request->getContext();
        $contextPath = $context ? $context->getPath() : 'index';

        $designerUrl = $request->getBaseUrl() . '/index.php/' . $contextPath . '/govalidojs/designer';
        $templateMgr->assign('designerUrl', $designerUrl);

        $emailTemplateManagerUrl = $request->getBaseUrl() . '/index.php/' . $contextPath . '/govalidojs/emailTemplateDesigner';
        $templateMgr->assign('emailTemplateManagerUrl', $emailTemplateManagerUrl);

        // Get plugin version directly from version.xml
        $versionFile = $this->_plugin->getPluginPath() . '/version.xml';
        $pluginVersion = '1.0.0';
        if (file_exists($versionFile)) {
            $versionXml = simplexml_load_file($versionFile);
            if ($versionXml && isset($versionXml->release)) {
                $pluginVersion = (string)$versionXml->release;
            }
        }
        $templateMgr->assign('pluginVersion', $pluginVersion);

        return parent::fetch($request, $template, $display);
    }

    /**
     * @copydoc Form::execute()
     */
    public function execute(...$functionArgs)
    {
        // Save FAB visibility settings
        $this->_plugin->updateSetting($this->_contextId, 'showFabForManagers', $this->getData('showFabForManagers') ? true : false, 'bool');
        $this->_plugin->updateSetting($this->_contextId, 'showFabForEditors', $this->getData('showFabForEditors') ? true : false, 'bool');
        $this->_plugin->updateSetting($this->_contextId, 'showFabForCopyeditors', $this->getData('showFabForCopyeditors') ? true : false, 'bool');
        $this->_plugin->updateSetting($this->_contextId, 'showFabForAssistants', $this->getData('showFabForAssistants') ? true : false, 'bool');

        // Save Smart Verify Widget settings
        $this->_plugin->updateSetting($this->_contextId, 'enableSmartVerifyWidget', $this->getData('enableSmartVerifyWidget') ? true : false, 'bool');
        $this->_plugin->updateSetting($this->_contextId, 'smartVerifyWidgetPosition', $this->getData('smartVerifyWidgetPosition') ?: 'bottom-right', 'string');

        parent::execute(...$functionArgs);
    }
}

if (!PKP_STRICT_MODE) {
    class_alias('\APP\plugins\generic\goValidOJS\classes\GoValidOJSSettingsForm', '\GoValidOJSSettingsForm');
}
