<?php

/**
 * @file plugins/generic/goValidOJS/classes/EmailTemplateManager.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class EmailTemplateManager
 * @ingroup plugins_generic_goValidOJS
 *
 * @brief Manages email templates for the GoValid OJS plugin
 */

namespace APP\plugins\generic\goValidOJS\classes;

use PKP\core\Core;
use PKP\config\Config;
use PKP\db\DAORegistry;
use PKP\core\PKPApplication;

class EmailTemplateManager {
    
    /** @var array Default email templates */
    private $defaultTemplates = [
        'professional' => [
            'name' => 'Professional',
            'subject' => 'Certificate of Review - {$journal_name}',
            'body' => '
<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #ffffff;">
    <div style="background: #2c3e50; color: white; padding: 30px; text-align: center;">
        {if $journal_logo_url}
        <div style="margin-bottom: 20px;">
            <img src="{$journal_logo_url}"
                 alt="{$journal_logo_alt}"
                 style="max-width: 200px; height: auto; display: block; margin: 0 auto;"
                 width="{$journal_logo_width}"
                 height="{$journal_logo_height}">
        </div>
        {/if}
        <h1 style="margin: 0; color: white; font-size: 28px;">{$journal_name}</h1>
        <h2 style="margin: 15px 0 0 0; color: #ecf0f1; font-size: 20px; font-weight: normal;">Certificate of Review</h2>
    </div>

    <div style="padding: 40px 30px; background: white; line-height: 1.6;">
        <p style="font-size: 16px; margin-bottom: 20px;">Dear <strong>{$reviewer_name}</strong>,</p>

        <p style="font-size: 16px; margin-bottom: 20px;">Thank you for your valuable contribution to the peer review process for our journal. Your expertise and dedication help maintain the high standards of scholarly publishing.</p>

        <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 25px 0;">
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Article:</strong> {$article_title}</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Authors:</strong> {$article_authors}</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Issue:</strong> {$issue_title}</p>
            <p style="margin: 0; font-size: 14px;"><strong>Review completed:</strong> {$review_date}</p>
        </div>

        <p style="font-size: 16px; margin-bottom: 20px;">Your certificate of review is attached to this email as recognition of your professional contribution.</p>

        <p style="font-size: 16px; margin-bottom: 20px;">{$custom_message}</p>
    </div>

    <div style="background: #ecf0f1; padding: 25px; text-align: center; border-top: 3px solid #2c3e50;">
        <p style="margin: 0 0 10px 0; font-size: 16px; color: #2c3e50;">Best regards,</p>
        <p style="margin: 0 0 15px 0; font-size: 16px; color: #2c3e50;"><strong>{$journal_name} Editorial Team</strong></p>
        <p style="margin: 0; font-size: 14px; color: #7f8c8d;">
            <a href="{$journal_url}" style="color: #3498db; text-decoration: none;">{$journal_url}</a>
        </p>
    </div>
</div>'
        ],
        'modern' => [
            'name' => 'Modern',
            'subject' => 'Review Certificate - {$journal_name}',
            'body' => '
<div style="font-family: \'Segoe UI\', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; background: #f8f9fa;">
    <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 40px; text-align: center; border-radius: 0;">
        {if $journal_logo_url}
        <div style="margin-bottom: 25px;">
            <img src="{$journal_logo_url}"
                 alt="{$journal_logo_alt}"
                 style="max-width: 180px; height: auto; display: block; margin: 0 auto; border-radius: 8px; box-shadow: 0 4px 8px rgba(0,0,0,0.1);"
                 width="{$journal_logo_width}"
                 height="{$journal_logo_height}">
        </div>
        {/if}
        <h1 style="margin: 0; font-size: 32px; font-weight: 300;">Review Certificate</h1>
        <p style="margin: 15px 0 0 0; opacity: 0.9; font-size: 18px;">{$journal_name}</p>
    </div>

    <div style="padding: 40px; background: white; margin: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
        <div style="border-left: 4px solid #667eea; padding-left: 25px; margin-bottom: 30px;">
            <h2 style="color: #2c3e50; margin: 0 0 10px 0; font-size: 24px;">Dear {$reviewer_name},</h2>
            <p style="color: #7f8c8d; margin: 0; font-size: 16px;">Thank you for your expert review</p>
        </div>

        <p style="font-size: 16px; line-height: 1.6; margin-bottom: 25px;">We appreciate your valuable contribution to maintaining the quality and integrity of scholarly publishing. Your expertise helps advance knowledge in your field.</p>

        <div style="background: #f8f9fa; padding: 25px; border-radius: 8px; margin: 25px 0; border: 1px solid #e9ecef;">
            <div style="display: grid; gap: 15px;">
                <div style="padding: 10px 0; border-bottom: 1px solid #dee2e6;">
                    <strong style="color: #495057;">Article:</strong>
                    <span style="color: #6c757d;">{$article_title}</span>
                </div>
                <div style="padding: 10px 0; border-bottom: 1px solid #dee2e6;">
                    <strong style="color: #495057;">Authors:</strong>
                    <span style="color: #6c757d;">{$article_authors}</span>
                </div>
                <div style="padding: 10px 0; border-bottom: 1px solid #dee2e6;">
                    <strong style="color: #495057;">Issue:</strong>
                    <span style="color: #6c757d;">{$issue_title}</span>
                </div>
                <div style="padding: 10px 0;">
                    <strong style="color: #495057;">Completed:</strong>
                    <span style="color: #6c757d;">{$review_date}</span>
                </div>
            </div>
        </div>

        <p style="font-size: 16px; line-height: 1.6; margin-bottom: 20px;">Your certificate of review is attached to this email as recognition of your professional service.</p>

        <p style="font-size: 16px; line-height: 1.6; margin-bottom: 20px;">{$custom_message}</p>
    </div>

    <div style="background: #2c3e50; color: white; padding: 25px; text-align: center; margin: 20px; border-radius: 8px;">
        <p style="margin: 0 0 10px 0; font-size: 16px;">Best regards,</p>
        <p style="margin: 0 0 15px 0; font-size: 18px; font-weight: bold;">{$journal_name} Editorial Team</p>
        <p style="margin: 0; font-size: 14px; opacity: 0.8;">
            <a href="{$journal_url}" style="color: #3498db; text-decoration: none;">Visit our journal</a>
        </p>
    </div>
</div>'
        ],
        'minimal' => [
            'name' => 'Minimal',
            'subject' => 'Thank you for your review - {$journal_name}',
            'body' => '
<div style="font-family: Georgia, serif; max-width: 600px; margin: 0 auto; background: #ffffff; line-height: 1.8;">
    <div style="padding: 40px; border-bottom: 2px solid #e0e0e0;">
        {if $journal_logo_url}
        <div style="text-align: center; margin-bottom: 30px;">
            <img src="{$journal_logo_url}"
                 alt="{$journal_logo_alt}"
                 style="max-width: 150px; height: auto;"
                 width="{$journal_logo_width}"
                 height="{$journal_logo_height}">
        </div>
        {/if}
        <h1 style="text-align: center; margin: 0; font-size: 24px; color: #333; font-weight: normal;">{$journal_name}</h1>
        <p style="text-align: center; margin: 10px 0 0 0; font-size: 16px; color: #666; font-style: italic;">Certificate of Review</p>
    </div>

    <div style="padding: 40px;">
        <p style="font-size: 18px; margin-bottom: 25px;">Dear {$reviewer_name},</p>

        <p style="font-size: 16px; margin-bottom: 25px;">Thank you for completing your review of "<em>{$article_title}</em>" by {$article_authors}.</p>

        <p style="font-size: 16px; margin-bottom: 25px;">Your contribution to the peer review process is greatly appreciated and helps maintain the quality of scholarly publishing.</p>

        <div style="margin: 30px 0; padding: 20px; background: #f9f9f9; border-left: 4px solid #333;">
            <p style="margin: 0 0 8px 0; font-size: 14px;"><strong>Review completed:</strong> {$review_date}</p>
            <p style="margin: 0 0 8px 0; font-size: 14px;"><strong>Issue:</strong> {$issue_title}</p>
        </div>

        <p style="font-size: 16px; margin-bottom: 25px;">Please find your certificate attached to this email.</p>

        <p style="font-size: 16px; margin-bottom: 25px;">{$custom_message}</p>

        <p style="font-size: 16px; margin-top: 40px;">Sincerely,</p>
        <p style="font-size: 16px; margin-bottom: 5px;"><strong>{$journal_name} Editorial Team</strong></p>
        <p style="font-size: 14px; color: #666; margin: 0;">
            <a href="{$journal_url}" style="color: #333; text-decoration: none;">{$journal_url}</a>
        </p>
    </div>
</div>'
        ]
    ];
    
    /** @var object Plugin instance */
    private $plugin;
    
    /** @var object Context */
    private $context;
    
    public function __construct($plugin, $context) {
        $this->plugin = $plugin;
        $this->context = $context;
    }
    
    /**
     * Get all available email templates
     * @return array
     */
    public function getEmailTemplates() {
        $templates = [];
        
        // Get custom templates from database
        $customTemplates = $this->getCustomTemplates();
        
        // Merge with default templates
        foreach ($this->defaultTemplates as $key => $template) {
            $templates[$key] = $template;
        }
        
        foreach ($customTemplates as $template) {
            $templates['custom_' . $template['id']] = [
                'name' => $template['name'],
                'subject' => $template['subject'],
                'body' => $template['body'],
                'custom' => true
            ];
        }
        
        return $templates;
    }
    
    /**
     * Get custom templates from database
     * @return array
     */
    private function getCustomTemplates() {
        // For now, return empty array for custom templates
        // Custom templates can be implemented later with proper database access
        return [];
    }
    
    /**
     * Get specific email template
     * @param string $templateId
     * @return array|null
     */
    public function getEmailTemplate($templateId) {
        $templates = $this->getEmailTemplates();
        return $templates[$templateId] ?? null;
    }
    
    /**
     * Save custom email template
     * @param array $templateData
     * @return bool
     */
    public function saveEmailTemplate($templateData) {
        $templateId = $templateData['id'] ?? uniqid();
        
        $this->plugin->updateSetting(
            $this->context->getId(),
            'emailTemplate_' . $templateId,
            $templateData
        );
        
        return true;
    }
    
    /**
     * Delete custom email template
     * @param string $templateId
     * @return bool
     */
    public function deleteEmailTemplate($templateId) {
        $this->plugin->updateSetting(
            $this->context->getId(),
            'emailTemplate_' . $templateId,
            null
        );
        
        return true;
    }
    
    /**
     * Get journal logo URL for emails
     * @param object $request
     * @return string|null
     */
    public function getJournalLogoUrl($request) {
        $logoData = $this->context->getLocalizedData('pageHeaderLogoImage');
        $logoName = null;
        
        // 1. Try to get logo from DB setting
        if ($logoData && isset($logoData['uploadName'])) {
            $logoName = $logoData['uploadName'];
        }
        
        // 2. Fallback: Check for standard file names if DB setting is empty/broken
        if (!$logoName) {
            $journalId = $this->context->getId();
            $publicFileManager = new \APP\file\PublicFileManager();
            $locale = \PKP\facades\Locale::getLocale();
            
            // Try current locale png/jpg
            $candidates = [
                "pageHeaderLogoImage_{$locale}.png",
                "pageHeaderLogoImage_{$locale}.jpg",
                "pageHeaderLogoImage_en_US.png", // Fallback to en_US
                "pageHeaderLogoImage_en.png",    // Fallback to en
                "logoheader.png",                // Legacy candidates
                "logoheader.webp"
            ];
            
            foreach ($candidates as $candidate) {
                if (file_exists($publicFileManager->getContextFilesPath($journalId) . '/' . $candidate)) {
                    $logoName = $candidate;
                    break;
                }
            }
        }

        if (!$logoName) {
            return null;
        }
        
        // Verify file exists (sanity check for DB path too)
        $publicFilesDir = Config::getVar('files', 'public_files_dir');
        $logoPath = $publicFilesDir . '/journals/' . 
                   $this->context->getId() . '/' . $logoName;
        
        if (!file_exists($logoPath)) {
            return null;
        }
        
        // Return URL with cache busting timestamp
        $fileMtime = filemtime($logoPath);
        return $request->getBaseUrl() . '/public/journals/' . 
               $this->context->getId() . '/' . $logoName . '?t=' . $fileMtime;
    }
    
    /**
     * Get journal logo data
     * @return array|null
     */
    public function getJournalLogoData() {
        return $this->context->getLocalizedData('pageHeaderLogoImage');
    }
    
    /**
     * Replace placeholders in email template
     * @param string $template
     * @param array $data
     * @return string
     */
    public function replacePlaceholders($template, $data) {
        // Handle conditional blocks ({if $variable})
        $template = preg_replace_callback(
            '/\{if\s+\$(\w+)\}(.*?)\{\/if\}/s',
            function($matches) use ($data) {
                $variable = $matches[1];
                $content = $matches[2];
                return !empty($data[$variable]) ? $content : '';
            },
            $template
        );

        // Replace simple placeholders
        $placeholders = [
            '{$reviewer_name}' => $data['reviewer_name'] ?? '',
            '{$reviewer_email}' => $data['reviewer_email'] ?? '',
            '{$journal_name}' => $data['journal_name'] ?? '',
            '{$article_title}' => $data['article_title'] ?? '',
            '{$article_authors}' => $data['article_authors'] ?? '',
            '{$issue_title}' => $data['issue_title'] ?? '',
            '{$review_date}' => $data['review_date'] ?? '',
            '{$certificate_link}' => $data['certificate_link'] ?? '',
            '{$journal_logo_url}' => $data['journal_logo_url'] ?? '',
            '{$journal_logo_alt}' => $data['journal_logo_alt'] ?? 'Journal Logo',
            '{$journal_logo_width}' => $data['journal_logo_width'] ?? '180',
            '{$journal_logo_height}' => $data['journal_logo_height'] ?? '90',
            '{$journal_url}' => $data['journal_url'] ?? '',
            '{$contact_email}' => $data['contact_email'] ?? '',
            '{$custom_message}' => $data['custom_message'] ?? ''
        ];

        return str_replace(array_keys($placeholders), array_values($placeholders), $template);
    }
    
    /**
     * Get template data for email sending
     * @param object $reviewer
     * @param object $submission
     * @param object $request
     * @return array
     */
    public function getTemplateData($reviewer, $submission, $request) {
        $logoUrl = $this->getJournalLogoUrl($request);
        $logoData = $this->getJournalLogoData();
        
        // Get issue information
        $issueDao = DAORegistry::getDAO('IssueDAO');
        $issue = $issueDao->getBySubmissionId($submission->getId());
        $issueTitle = $issue ? $issue->getIssueIdentification() : 'Upcoming Issue';
        
        return [
            'reviewer_name' => $reviewer->getFullName(),
            'reviewer_email' => $reviewer->getEmail(),
            'journal_name' => $this->context->getLocalizedData('name'),
            'article_title' => $submission->getLocalizedTitle(),
            'article_authors' => $submission->getAuthorString(),
            'issue_title' => $issueTitle,
            'review_date' => date('F j, Y'),
            'journal_logo_url' => $logoUrl,
            'journal_logo_alt' => $logoData['altText'] ?? 'Journal Logo',
            'journal_logo_width' => $logoData['width'] ?? 180,
            'journal_logo_height' => $logoData['height'] ?? 90,
            'journal_url' => $request->getBaseUrl(),
            'contact_email' => $this->context->getData('contactEmail'),
            'custom_message' => $this->plugin->getSetting($this->context->getId(), 'customEmailMessage') ?? ''
        ];
    }
    
    /**
     * Send email using template
     * @param object $reviewer
     * @param object $submission
     * @param object $request
     * @param string $certificatePath
     * @return bool
     */
    public function sendCertificateEmail($reviewer, $submission, $request, $certificatePath) {
        $selectedTemplate = $this->plugin->getSetting($this->context->getId(), 'selectedEmailTemplate') ?? 'professional';
        $template = $this->getEmailTemplate($selectedTemplate);

        if (!$template) {
            error_log("GoValid Plugin: Email template '{$selectedTemplate}' not found");
            return false;
        }

        $templateData = $this->getTemplateData($reviewer, $submission, $request);

        $subject = $this->replacePlaceholders($template['subject'], $templateData);
        $body = $this->replacePlaceholders($template['body'], $templateData);

        $mail = new \PKP\mail\Mail();
        $mail->setFrom($this->context->getData('contactEmail'), $this->context->getData('contactName'));
        $mail->addRecipient($reviewer->getEmail(), $reviewer->getFullName());
        $mail->setSubject($subject);
        $mail->setBody($body);
        $mail->setContentType('text/html');

        if ($certificatePath && file_exists($certificatePath)) {
            $mail->addAttachment($certificatePath, 'ReviewCertificate.pdf');
        }

        $result = $mail->send();

        if ($result) {
            error_log("GoValid Plugin: Certificate email sent successfully to {$reviewer->getEmail()}");
        } else {
            error_log("GoValid Plugin: Failed to send certificate email to {$reviewer->getEmail()}");
        }

        return $result;
    }

    /**
     * Generate header content with logo positioning (PHP version)
     * @param array $config Template configuration
     * @param array $data Data for placeholders
     * @return string Header HTML content
     */
    private function generateHeaderContentPHP($config, $data) {
        $logoPosition = $config['header']['logoPosition'] ?? 'center';
        $showNameBesideLogo = $config['header']['showNameBesideLogo'] ?? false;
        $logoMaxWidth = $config['header']['logoMaxWidth'] ?? 200;

        // Generate logo HTML
        $logoHTML = '';
        if ($config['header']['showLogo'] && !empty($data['journal_logo_url'])) {
            $logoHTML = sprintf(
                '<img src="%s" alt="%s" style="max-width: %spx; height: auto; display: block;" width="%s" height="%s">',
                $data['journal_logo_url'],
                $data['journal_logo_alt'] ?? 'Logo',
                $logoMaxWidth,
                $data['journal_logo_width'] ?? '180',
                $data['journal_logo_height'] ?? '90'
            );
        }

        $title = $this->replacePlaceholders($config['header']['title'], $data);
        $subtitle = !empty($config['header']['subtitle']) ?
            $this->replacePlaceholders($config['header']['subtitle'], $data) : '';

        // Center position
        if ($logoPosition === 'center') {
            $html = '<div style="text-align: center;">';
            if (!empty($logoHTML)) {
                $html .= sprintf('<div style="margin-bottom: 20px; display: flex; justify-content: center;">%s</div>', $logoHTML);
            }
            $html .= sprintf('<h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>',
                $config['header']['textColor'], $title);
            if ($subtitle) {
                $html .= sprintf('<p style="margin: 10px 0 0 0; color: %s; font-size: 18px; opacity: 0.9;">%s</p>',
                    $config['header']['textColor'], $subtitle);
            }
            $html .= '</div>';
            return $html;
        }

        // Left or Right position
        if ($logoPosition === 'left' || $logoPosition === 'right') {
            if ($showNameBesideLogo && !empty($logoHTML)) {
                // Logo and name side by side
                $flexDirection = $logoPosition === 'left' ? 'row' : 'row-reverse';
                $textAlign = $logoPosition === 'left' ? 'left' : 'right';

                $html = sprintf('<div style="display: flex; align-items: center; gap: 20px; flex-direction: %s;">', $flexDirection);
                $html .= sprintf('<div style="flex-shrink: 0;">%s</div>', $logoHTML);
                $html .= sprintf('<div style="flex: 1; text-align: %s;">', $textAlign);
                $html .= sprintf('<h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>',
                    $config['header']['textColor'], $title);
                if ($subtitle) {
                    $html .= sprintf('<p style="margin: 10px 0 0 0; color: %s; font-size: 18px; opacity: 0.9;">%s</p>',
                        $config['header']['textColor'], $subtitle);
                }
                $html .= '</div></div>';
                return $html;
            } else {
                // Logo on top, then text
                $alignItems = $logoPosition === 'left' ? 'flex-start' : 'flex-end';
                $textAlign = $logoPosition === 'left' ? 'left' : 'right';

                $html = sprintf('<div style="display: flex; flex-direction: column; align-items: %s;">', $alignItems);
                if (!empty($logoHTML)) {
                    $html .= sprintf('<div style="margin-bottom: 20px;">%s</div>', $logoHTML);
                }
                $html .= sprintf('<div style="text-align: %s; width: 100%%;">', $textAlign);
                $html .= sprintf('<h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>',
                    $config['header']['textColor'], $title);
                if ($subtitle) {
                    $html .= sprintf('<p style="margin: 10px 0 0 0; color: %s; font-size: 18px; opacity: 0.9;">%s</p>',
                        $config['header']['textColor'], $subtitle);
                }
                $html .= '</div></div>';
                return $html;
            }
        }

        return '';
    }

    /**
     * Generate HTML from JSON template configuration
     * @param array $config Template configuration
     * @param array $data Data for placeholders
     * @return string HTML email content
     */
    public function generateHTMLFromConfig($config, $data) {
        // Header background style
        $headerBgStyle = '';
        if ($config['header']['bgType'] === 'gradient') {
            $headerBgStyle = sprintf(
                'background: linear-gradient(135deg, %s 0%%, %s 100%%);',
                $config['header']['gradientStart'],
                $config['header']['gradientEnd']
            );
        } else {
            $headerBgStyle = 'background: ' . $config['header']['bgColor'] . ';';
        }

        // Build HTML
        $html = sprintf('
<div style="font-family: %s; max-width: %spx; margin: 0 auto; background: %s;">
    <!-- Header -->
    <div style="%s color: %s; padding: %spx; text-align: center;">',
            $config['settings']['fontFamily'],
            $config['settings']['emailWidth'],
            $config['settings']['outerBgColor'],
            $headerBgStyle,
            $config['header']['textColor'],
            $config['header']['padding']
        );

        // Generate header content with logo positioning
        $html .= $this->generateHeaderContentPHP($config, $data);

        $html .= '
    </div>

    <!-- Content -->';

        $html .= sprintf('
    <div style="padding: %spx; background: %s; color: %s; line-height: %s; font-size: %spx;">',
            $config['content']['padding'],
            $config['content']['bgColor'],
            $config['content']['textColor'],
            $config['settings']['lineHeight'],
            $config['settings']['fontSize']
        );

        // Greeting
        if (!empty($config['content']['greeting'])) {
            $html .= sprintf('
        <p style="font-size: %spx; margin-bottom: 20px;">%s</p>',
                $config['settings']['fontSize'],
                $this->replacePlaceholders($config['content']['greeting'], $data)
            );
        }

        // Message
        if (!empty($config['content']['message'])) {
            $html .= sprintf('
        <p style="font-size: %spx; margin-bottom: 20px;">%s</p>',
                $config['settings']['fontSize'],
                nl2br($this->replacePlaceholders($config['content']['message'], $data))
            );
        }

        // Details box
        if ($config['content']['showDetailsBox']) {
            $html .= sprintf('
        <div style="background: %s; padding: 20px; border-radius: 8px; margin: 25px 0;">
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Article:</strong> %s</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Authors:</strong> %s</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Issue:</strong> %s</p>
            <p style="margin: 0; font-size: 14px;"><strong>Review completed:</strong> %s</p>
        </div>',
                $config['content']['detailsBgColor'],
                $data['article_title'] ?? '',
                $data['article_authors'] ?? '',
                $data['issue_title'] ?? '',
                $data['review_date'] ?? ''
            );
        }

        // Closing
        if (!empty($config['content']['closing'])) {
            $html .= sprintf('
        <p style="font-size: %spx; margin-bottom: 20px; white-space: pre-line;">%s</p>',
                $config['settings']['fontSize'],
                $this->replacePlaceholders($config['content']['closing'], $data)
            );
        }

        $html .= '
    </div>

    <!-- Footer -->';

        $html .= sprintf('
    <div style="background: %s; padding: %spx; text-align: center; border-top: %spx solid %s;">',
            $config['footer']['bgColor'],
            $config['footer']['padding'],
            $config['footer']['borderWidth'],
            $config['footer']['borderColor']
        );

        // Signature
        if (!empty($config['footer']['signature'])) {
            $html .= sprintf('
        <p style="margin: 0 0 10px 0; font-size: %spx; color: %s;">%s</p>',
                $config['settings']['fontSize'],
                $config['footer']['textColor'],
                $this->replacePlaceholders($config['footer']['signature'], $data)
            );
        }

        if (!empty($config['footer']['signatureName'])) {
            $html .= sprintf('
        <p style="margin: 0 0 15px 0; font-size: %spx; color: %s; font-weight: bold;">%s</p>',
                $config['settings']['fontSize'],
                $config['footer']['textColor'],
                $this->replacePlaceholders($config['footer']['signatureName'], $data)
            );
        }

        // URL
        if ($config['footer']['showUrl'] && !empty($data['journal_url'])) {
            $html .= sprintf('
        <p style="margin: 0; font-size: 14px; color: %s;">
            <a href="%s" style="color: %s; text-decoration: none;">%s</a>
        </p>',
                $config['footer']['textColor'],
                $data['journal_url'],
                $config['footer']['linkColor'],
                $data['journal_url']
            );
        }

        $html .= '
    </div>
</div>';

        return $html;
    }

    /**
     * Save JSON-based email template
     * @param array $templateData
     * @return array Result with status and message
     */
    public function saveJSONTemplate($templateData) {
        try {
            $templateId = $templateData['templateId'] ?? 'custom_' . uniqid();
            $config = json_decode($templateData['templateData'], true);

            if (!$config) {
                return ['status' => 'error', 'message' => 'Invalid template data'];
            }

            // Store template configuration
            $template = [
                'id' => $templateId,
                'name' => $config['name'] ?? 'Unnamed Template',
                'subject' => $config['subject'] ?? '',
                'config' => $config,
                'created' => date('Y-m-d H:i:s'),
                'modified' => date('Y-m-d H:i:s')
            ];

            $this->plugin->updateSetting(
                $this->context->getId(),
                'emailTemplate_' . $templateId,
                json_encode($template)
            );

            return ['status' => 'success', 'templateId' => $templateId, 'message' => 'Template saved successfully'];
        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error saving template: " . $e->getMessage());
            return ['status' => 'error', 'message' => 'Failed to save template'];
        }
    }

    /**
     * Get JSON template by ID
     * @param string $templateId
     * @return array|null
     */
    public function getJSONTemplate($templateId) {
        $templateJson = $this->plugin->getSetting($this->context->getId(), 'emailTemplate_' . $templateId);

        if ($templateJson) {
            return json_decode($templateJson, true);
        }

        return null;
    }

    /**
     * Send test email
     * @param string $toEmail
     * @param string $subject
     * @param string $htmlContent
     * @return array Result
     */
    public function sendTestEmail($toEmail, $subject, $htmlContent, $cc = null, $bcc = null, $attachments = []) {
        try {
            error_log("GoValid Plugin: sendTestEmail method called");

            // Create a simple mailable using Laravel's Mail facade
            \Illuminate\Support\Facades\Mail::send([], [], function ($message) use ($toEmail, $subject, $htmlContent, $cc, $bcc, $attachments) {
                $message->to($toEmail)
                        ->from($this->context->getData('contactEmail'), $this->context->getData('contactName'))
                        ->subject('[TEST] ' . $subject)
                        ->html($htmlContent);

                // Add CC if provided
                if (!empty($cc)) {
                    $ccEmails = array_map('trim', explode(',', $cc));
                    $ccEmails = array_filter($ccEmails); // Remove empty values
                    if (!empty($ccEmails)) {
                        $message->cc($ccEmails);
                        error_log("GoValid Plugin: Adding CC: " . implode(', ', $ccEmails));
                    }
                }

                // Add BCC if provided
                if (!empty($bcc)) {
                    $bccEmails = array_map('trim', explode(',', $bcc));
                    $bccEmails = array_filter($bccEmails); // Remove empty values
                    if (!empty($bccEmails)) {
                        $message->bcc($bccEmails);
                        error_log("GoValid Plugin: Adding BCC: " . implode(', ', $bccEmails));
                    }
                }

                // Add attachments if provided
                if (!empty($attachments) && is_array($attachments)) {
                    foreach ($attachments as $attachment) {
                        if (isset($attachment['data']) && isset($attachment['name']) && isset($attachment['type'])) {
                            $message->attachData(
                                $attachment['data'],
                                $attachment['name'],
                                ['mime' => $attachment['type']]
                            );
                            error_log("GoValid Plugin: Adding attachment: " . $attachment['name'] . " (" . $attachment['type'] . ")");
                        }
                    }
                }
            });

            error_log("GoValid Plugin: Test email sent successfully");
            $successMessage = 'Test email sent successfully';
            if (!empty($attachments)) {
                $successMessage .= ' with ' . count($attachments) . ' attachment(s)';
            }
            return ['status' => 'success', 'message' => $successMessage];
        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error sending test email: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return ['status' => 'error', 'message' => 'Error: ' . $e->getMessage()];
        }
    }

    /**
     * Send email with certificate attachment
     * @param string $toEmail Recipient email
     * @param string $subject Email subject
     * @param string $htmlContent Email HTML content
     * @param string $cc CC emails (comma-separated)
     * @param string $bcc BCC emails (comma-separated)
     * @param string $attachmentData Binary attachment data
     * @param string $attachmentFilename Attachment filename
     * @param string $mimeType Attachment MIME type
     * @return array Status array
     */
    public function sendEmailWithAttachment($toEmail, $subject, $htmlContent, $cc = null, $bcc = null, $attachmentData = null, $attachmentFilename = 'certificate.pdf', $mimeType = 'application/pdf', $additionalAttachments = [])
    {
        try {
            // Validate inputs
            if (empty($toEmail)) {
                throw new \Exception('Recipient email is required');
            }

            if (empty($subject)) {
                throw new \Exception('Email subject is required');
            }

            if (empty($htmlContent)) {
                throw new \Exception('Email content is required');
            }

            if (empty($attachmentData)) {
                throw new \Exception('Attachment data is required');
            }

            // Get sender email and name from context
            $fromEmail = $this->context->getData('contactEmail');
            $fromName = $this->context->getData('contactName');

            if (empty($fromEmail)) {
                throw new \Exception('Journal contact email not configured');
            }

            // Send email using Laravel Mail facade with attachment
            \Illuminate\Support\Facades\Mail::send([], [], function ($message) use (
                $toEmail,
                $subject,
                $htmlContent,
                $cc,
                $bcc,
                $fromEmail,
                $fromName,
                $attachmentData,
                $attachmentFilename,
                $mimeType,
                $additionalAttachments
            ) {
                $message->to($toEmail)
                        ->from($fromEmail, $fromName)
                        ->subject($subject)
                        ->html($htmlContent);

                // Add CC recipients if provided
                if (!empty($cc)) {
                    $ccEmails = array_map('trim', explode(',', $cc));
                    $ccEmails = array_filter($ccEmails);
                    if (!empty($ccEmails)) {
                        $message->cc($ccEmails);
                    }
                }

                // Add BCC recipients if provided
                if (!empty($bcc)) {
                    $bccEmails = array_map('trim', explode(',', $bcc));
                    $bccEmails = array_filter($bccEmails);
                    if (!empty($bccEmails)) {
                        $message->bcc($bccEmails);
                    }
                }

                // Attach certificate using attachData for raw binary data
                error_log("GoValid Plugin: Attaching certificate via attachData");
                error_log("GoValid Plugin: Filename: {$attachmentFilename}");
                error_log("GoValid Plugin: MIME type: {$mimeType}");
                error_log("GoValid Plugin: Data size: " . strlen($attachmentData) . " bytes");
                error_log("GoValid Plugin: Data first 20 bytes (hex): " . bin2hex(substr($attachmentData, 0, 20)));

                // Use attachData for raw binary data (proper method for binary attachments)
                $message->attachData($attachmentData, $attachmentFilename, [
                    'mime' => $mimeType
                ]);

                error_log("GoValid Plugin: Certificate attached successfully via attachData");

                // Attach additional files if provided
                if (!empty($additionalAttachments) && is_array($additionalAttachments)) {
                    foreach ($additionalAttachments as $attachment) {
                        if (isset($attachment['data']) && isset($attachment['name']) && isset($attachment['type'])) {
                            // Use attachData for additional files too
                            $message->attachData(
                                $attachment['data'],
                                $attachment['name'],
                                ['mime' => $attachment['type']]
                            );
                            error_log("GoValid Plugin: Attached additional file: " . $attachment['name'] . " (" . $attachment['type'] . ")");
                        }
                    }
                }
            });

            $successMessage = 'Email sent successfully with certificate attachment';
            if (!empty($additionalAttachments)) {
                $successMessage .= ' and ' . count($additionalAttachments) . ' additional attachment(s)';
            }
            return ['status' => 'success', 'message' => $successMessage];

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error sending email with attachment: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return ['status' => 'error', 'message' => 'Error: ' . $e->getMessage()];
        }
    }
}