# GoValidOJS Auto-Population Documentation

## Overview

The GoValidOJS plugin automatically populates certificate fields with real data from your OJS installation. This eliminates manual data entry and ensures accuracy by pulling information directly from your journal's database.

## Table of Contents

1. [Architecture Overview](#architecture-overview)
2. [Data Flow](#data-flow)
3. [API Endpoints](#api-endpoints)
4. [Frontend Implementation](#frontend-implementation)
5. [Backend Implementation](#backend-implementation)
6. [Data Fields](#data-fields)
7. [Troubleshooting](#troubleshooting)

## Architecture Overview

The auto-population feature uses a client-server architecture:

```
┌─────────────────┐     AJAX Request      ┌──────────────────┐
│                 │ ───────────────────► │                  │
│  Frontend (JS)  │                      │  Backend (PHP)   │
│  designer.tpl   │ ◄─────────────────── │  Plugin manage() │
└─────────────────┘     JSON Response     └──────────────────┘
                                                   │
                                                   ▼
                                          ┌──────────────────┐
                                          │   OJS Database   │
                                          │  (Issues, Articles)│
                                          └──────────────────┘
```

## Data Flow

### 1. Page Load
When the certificate designer page loads:
1. JavaScript calls `loadIssuesOnPageLoad()`
2. Sends GET request to `action=getIssues`
3. Backend fetches all published issues
4. Populates the Issues dropdown

### 2. Issue Selection
When user selects an issue:
1. JavaScript calls `loadArticlesForIssue(issueId)`
2. Sends GET request to `action=getSubmissions&issueId=X`
3. Backend fetches all articles in that issue
4. Populates the Articles dropdown
5. Auto-fills issue fields (volume, number, year, title)

### 3. Article Selection
When user selects an article:
1. JavaScript populates article fields from cached data
2. Fields include: title, DOI, submission ID, authors

## API Endpoints

### Get Issues Endpoint
```
GET /index.php/[journal]/$$call$$/grid/settings/plugins/settings-plugin-grid/manage
    ?verb=designer
    &plugin=goValidOJS
    &category=generic
    &action=getIssues
```

**Response Format:**
```json
{
  "status": true,
  "content": {
    "issues": [
      {
        "id": 1,
        "title": "November",
        "volume": 1,
        "number": "02",
        "year": 2024,
        "datePublished": "2025-07-11 09:36:42",
        "description": "",
        "published": true
      }
    ]
  }
}
```

### Get Submissions Endpoint
```
GET /index.php/[journal]/$$call$$/grid/settings/plugins/settings-plugin-grid/manage
    ?verb=designer
    &plugin=goValidOJS
    &category=generic
    &action=getSubmissions
    &issueId=1
```

**Response Format:**
```json
{
  "status": true,
  "content": {
    "submissions": [
      {
        "id": 123,
        "title": "Article Title",
        "doi": "10.1234/example.2024.001",
        "pages": "1-10",
        "dateSubmitted": "2024-01-01",
        "datePublished": "2024-01-15",
        "abstract": "Article abstract...",
        "issueId": 1,
        "sectionId": 1,
        "authors": "John Doe, Jane Smith"
      }
    ]
  }
}
```

## Frontend Implementation

### Key JavaScript Functions

#### 1. loadIssuesOnPageLoad()
Located in `templates/designer.tpl` (line ~5086)
```javascript
function loadIssuesOnPageLoad() {
    const url = getAjaxURL() + '&action=getIssues';
    
    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.content && data.content.issues) {
            populateIssuesDropdown(data.content.issues);
        }
    });
}
```

#### 2. loadArticlesForIssue(issueId)
Located in `templates/designer.tpl` (line ~5254)
```javascript
function loadArticlesForIssue(issueId) {
    const url = getAjaxURL() + '&action=getSubmissions&issueId=' + encodeURIComponent(issueId);
    
    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.content && data.content.submissions) {
            populateArticlesDropdown(data.content.submissions);
        }
    });
}
```

#### 3. populateIssuesDropdown(issues)
Populates the issue dropdown and stores issue data for later use:
```javascript
function populateIssuesDropdown(issues) {
    const issueSelect = document.getElementById('content_issue_select');
    
    // Clear existing options
    issueSelect.innerHTML = '<option value="">-- Select an Issue --</option>';
    
    // Store issues data for later use
    window.ojsIssuesData = {};
    
    issues.forEach(issue => {
        const option = document.createElement('option');
        option.value = issue.id;
        option.textContent = issue.title || `Vol. ${issue.volume} No. ${issue.number} (${issue.year})`;
        issueSelect.appendChild(option);
        
        // Store issue data
        window.ojsIssuesData[issue.id] = issue;
    });
}
```

## Backend Implementation

### Plugin Architecture

The GoValidOJSPlugin handles requests through its `manage()` method:

```php
public function manage($args, $request): ?\PKP\core\JSONMessage
{
    switch ($request->getUserVar('verb')) {
        case 'designer':
            $action = $request->getUserVar('action');
            if ($action && in_array($action, ['getIssues', 'getSubmissions', ...])) {
                return $this->_handleDesignerAjax($request);
            }
            // ... rest of code
    }
}
```

### Data Retrieval Methods

#### 1. _getIssues($context)
Located in `GoValidOJSPlugin.php` (line ~855)
```php
private function _getIssues($context)
{
    $issues = [];
    
    // Use the Repository pattern to get real published issues
    $issueCollector = Repo::issue()->getCollector()
        ->filterByContextIds([$context->getId()])
        ->filterByPublished(true)
        ->orderBy(Repo::issue()->getCollector()::ORDERBY_PUBLISHED_ISSUES);
    
    $publishedIssues = $issueCollector->getMany();
    
    foreach ($publishedIssues as $issue) {
        $issues[] = [
            'id' => $issue->getId(),
            'title' => $issue->getLocalizedTitle() ?: '',
            'volume' => $issue->getVolume() ?: '',
            'number' => $issue->getNumber() ?: '',
            'year' => $issue->getYear() ?: '',
            'datePublished' => $issue->getDatePublished() ?: '',
            'description' => $issue->getLocalizedDescription() ?: '',
            'published' => true
        ];
    }
    
    return $issues;
}
```

#### 2. _getSubmissions($context, $issueId)
Located in `GoValidOJSPlugin.php` (line ~931)
```php
private function _getSubmissions($context, $issueId = null)
{
    $submissions = [];
    
    // Get submissions using the repository pattern
    $submissionCollector = Repo::submission()->getCollector()
        ->filterByContextIds([$context->getId()])
        ->filterByStatus([PKPSubmission::STATUS_PUBLISHED]);
    
    $allSubmissions = $submissionCollector->getMany();
    
    // Filter by issue ID
    foreach ($allSubmissions as $submission) {
        $publication = $submission->getCurrentPublication();
        if ($publication && $publication->getData('issueId') == $issueId) {
            $submissions[] = [
                'id' => $submission->getId(),
                'title' => $publication->getLocalizedTitle() ?: 'Untitled',
                'doi' => $publication->getStoredPubId('doi') ?: '',
                'pages' => $publication->getData('pages') ?: '',
                'dateSubmitted' => $submission->getDateSubmitted(),
                'datePublished' => $publication->getData('datePublished') ?: '',
                'abstract' => strip_tags($publication->getLocalizedData('abstract') ?: ''),
                'issueId' => $issueId,
                'sectionId' => $publication->getData('sectionId') ?: 0,
                'authors' => $this->_getAuthorNames($publication)
            ];
        }
    }
    
    return $submissions;
}
```

## Data Fields

### Issue Fields
- **Volume**: Issue volume number
- **Number**: Issue number within the volume
- **Year**: Publication year
- **Title**: Localized issue title
- **Date Published**: Full publication date

### Article Fields
- **Title**: Article title in the current locale
- **DOI**: Digital Object Identifier if assigned
- **Pages**: Page range (e.g., "1-10")
- **Authors**: Comma-separated list of author names
- **Submission ID**: Internal OJS submission ID
- **Abstract**: Plain text abstract (HTML stripped)
- **Dates**: Submission and publication dates

## Troubleshooting

### Common Issues

1. **500 Error on getSubmissions**
   - Check PHP error logs
   - Ensure you're using OJS 3.4 compatible code
   - Verify database connection

2. **Empty Dropdowns**
   - Ensure issues are published
   - Check if articles are assigned to issues
   - Verify user has proper permissions

3. **Authentication Errors**
   - Ensure session cookies are included (`credentials: 'same-origin'`)
   - Check if user is logged in as admin/journal manager

### Debug Mode

Enable debugging in `_getSubmissions()`:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
error_log('GoValidOJS: Debug info...');
```

### Checking Data Flow

1. Open browser Developer Tools
2. Go to Network tab
3. Select an issue
4. Check the getSubmissions request/response
5. Look for console.log outputs

## Security Considerations

1. **Authentication**: All endpoints require user authentication
2. **Authorization**: Only journal managers/admins can access
3. **Context Isolation**: Data is filtered by journal context
4. **XSS Prevention**: All output is properly escaped
5. **CSRF Protection**: OJS handles CSRF tokens automatically

## Performance Optimization

1. **Caching**: Issue/article data is cached in JavaScript
2. **Lazy Loading**: Articles load only when issue is selected
3. **Pagination**: Consider adding pagination for large journals
4. **Indexing**: Ensure database indexes on issue_id and status fields

## Future Enhancements

1. **Search**: Add search functionality for articles
2. **Filters**: Filter by section, author, date range
3. **Preview**: Show article preview on hover
4. **Bulk Operations**: Select multiple articles at once
5. **Custom Fields**: Support for custom metadata fields