# Template Gallery Implementation Progress

**Date**: November 2, 2025
**Status**: ✅ Phase 1 & 2 Complete | 🔄 Phase 3 In Progress

---

## Summary

Implementing a comprehensive hybrid template system that combines:
- **File-based default templates** shipped with the plugin
- **Database-stored user templates** (existing "My Templates")
- **Template Gallery UI** to browse and use default templates

---

## ✅ Phase 1: Directory Structure & Sample Templates (COMPLETE)

### Created Directory
- **Location**: `/var/www/pub/plugins/generic/goValidOJS/default_templates/`
- **Purpose**: Store default certificate templates as JSON files
- **Permissions**: Read-only for web server

### Created 5 Professional Templates

1. **certificate_basic.json** (4.1KB)
   - Classic professional certificate
   - Double border, centered layout
   - Dark blue/gray color scheme
   - Georgia typography

2. **certificate_elegant.json** (5.4KB)
   - Elegant design with decorative elements
   - Golden borders, corner ornaments
   - Brown/gold color scheme
   - Palatino typography

3. **certificate_modern.json** (5.5KB)
   - Contemporary minimalist design
   - Blue accent border, left-aligned text
   - Blue and gray scheme
   - Calibri typography

4. **diploma_classic.json** (6.0KB)
   - Traditional formal diploma
   - Triple border, golden accents
   - Gold and beige scheme
   - Times New Roman typography

5. **award_simple.json** (5.5KB)
   - Clean contemporary award
   - Green accents, minimalist design
   - Green and white scheme
   - Century Gothic typography

### Template Structure

Each template JSON includes:
```json
{
  "name": "Certificate - Basic",
  "description": "Simple certificate template...",
  "category": "certificate",
  "orientation": "landscape",
  "author": "GoValid OJS",
  "version": "1.0",
  "thumbnail": "certificate_basic.png",
  "data": {
    "version": "5.1.0",
    "objects": [...]
  }
}
```

### Template Variables Supported
- `{{article_title}}` - Article title
- `{{author_name}}` - Author name
- `{{journal_name}}` - Journal name
- `{{volume}}` - Volume number
- `{{issue}}` - Issue number
- `{{year}}` - Publication year
- `{{publication_date}}` - Full date
- `{{doi}}` - DOI identifier
- `{{identifier}}` - Article code

### Documentation
- **README.md** created in `/default_templates/`
- Comprehensive documentation of all templates
- Usage guidelines and maintenance instructions

---

## ✅ Phase 2: Backend PHP Implementation (COMPLETE)

### Added AJAX Endpoints

**File Modified**: `/var/www/pub/plugins/generic/goValidOJS/GoValidOJSPlugin.php`

#### 1. `listDefaultTemplates` Endpoint (Lines 631-633)
```php
case 'listDefaultTemplates':
    $templates = $this->getDefaultTemplates();
    return new \PKP\core\JSONMessage(true, ['status' => true, 'templates' => $templates]);
```

**Returns:**
```json
{
  "status": true,
  "content": {
    "status": true,
    "templates": [
      {
        "file": "certificate_basic.json",
        "name": "Certificate - Basic",
        "description": "...",
        "category": "certificate",
        "orientation": "landscape",
        "author": "GoValid OJS",
        "version": "1.0",
        "thumbnail": null
      },
      ...
    ]
  }
}
```

#### 2. `loadDefaultTemplate` Endpoint (Lines 635-651)
```php
case 'loadDefaultTemplate':
    $templateFile = $request->getUserVar('templateFile');
    if ($templateFile) {
        $template = $this->getDefaultTemplate($templateFile);
        return new \PKP\core\JSONMessage(true, [
            'status' => true,
            'templateData' => $template['data'],
            'orientation' => $template['orientation'] ?? 'landscape',
            'name' => $template['name'],
            'description' => $template['description'] ?? ''
        ]);
    }
```

**Request:**
```
POST /ajax?action=loadDefaultTemplate
Body: templateFile=certificate_basic.json
```

**Returns:**
```json
{
  "status": true,
  "content": {
    "status": true,
    "templateData": "{...Fabric.js canvas JSON...}",
    "orientation": "landscape",
    "name": "Certificate - Basic",
    "description": "..."
  }
}
```

### Added Helper Methods

#### 1. `getDefaultTemplates()` Method (Lines 3272-3311)

**Purpose**: Scan default_templates directory and return metadata for all templates

**Features**:
- Reads all `.json` files from `default_templates/`
- Parses JSON and extracts metadata
- Returns array of template info
- Logs count of templates found
- Error handling for missing directory

**Code Highlights**:
```php
private function getDefaultTemplates(): array
{
    $templates = [];
    $templatesDir = $this->getPluginPath() . '/default_templates';

    if (!is_dir($templatesDir)) {
        return $templates;
    }

    $files = glob($templatesDir . '/*.json');

    foreach ($files as $file) {
        $content = @file_get_contents($file);
        $templateData = json_decode($content, true);

        if ($templateData && isset($templateData['name'])) {
            $templates[] = [
                'file' => basename($file),
                'name' => $templateData['name'],
                // ...metadata
            ];
        }
    }

    return $templates;
}
```

#### 2. `getDefaultTemplate($templateFile)` Method (Lines 3319-3355)

**Purpose**: Load a specific template file by filename

**Features**:
- Filename sanitization (prevents directory traversal)
- Automatic `.json` extension addition
- File existence checking
- JSON parsing and validation
- Returns full template data including Fabric.js canvas
- Comprehensive error logging

**Security**:
```php
// Sanitize filename to prevent directory traversal
$templateFile = basename($templateFile);

// Ensure it's a JSON file
if (!str_ends_with($templateFile, '.json')) {
    $templateFile .= '.json';
}
```

**Validation**:
```php
if (!$template || !isset($template['data'])) {
    // Invalid template format
    return null;
}
```

---

## 🔄 Phase 3: Frontend UI Implementation (IN PROGRESS)

### Planned UI Changes

#### 1. Add "Template Gallery" Button
- Location: Next to existing "Load Template" button in toolbar
- Icon: `fa-th` (grid icon)
- Opens Template Gallery modal

#### 2. Template Gallery Modal

**Structure**:
```html
<div id="templateGalleryModal" class="preview-modal">
    <div class="preview-content">
        <div class="preview-header">
            <h2>Template Gallery</h2>
            <span class="preview-close">&times;</span>
        </div>

        <div class="gallery-tabs">
            <button class="gallery-tab active" data-tab="default">
                Default Templates
            </button>
            <button class="gallery-tab" data-tab="my">
                My Templates
            </button>
        </div>

        <div class="gallery-content">
            <!-- Default Templates Grid -->
            <div id="defaultTemplatesGrid" class="template-grid">
                <!-- Cards populated by JavaScript -->
            </div>

            <!-- My Templates List -->
            <div id="myTemplatesList" class="template-list" style="display:none;">
                <!-- Existing template list -->
            </div>
        </div>
    </div>
</div>
```

#### 3. Template Card Design

Each default template displays as a card:
```html
<div class="template-card">
    <div class="template-thumbnail">
        <i class="fa fa-certificate"></i>
        <!-- Or preview image if available -->
    </div>
    <div class="template-info">
        <h3>Certificate - Basic</h3>
        <p>Simple certificate template...</p>
        <div class="template-meta">
            <span class="template-category">Certificate</span>
            <span class="template-orientation">Landscape</span>
        </div>
    </div>
    <div class="template-actions">
        <button class="use-template-btn" data-file="certificate_basic.json">
            <i class="fa fa-download"></i> Use Template
        </button>
    </div>
</div>
```

### CSS Styling

**Gallery Grid**:
```css
.template-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 20px;
    padding: 20px;
}

.template-card {
    background: #fff;
    border: 1px solid #ddd;
    border-radius: 8px;
    overflow: hidden;
    transition: all 0.3s;
}

.template-card:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}

.template-thumbnail {
    height: 180px;
    background: #f5f5f5;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 60px;
    color: #ccc;
}

.template-info {
    padding: 15px;
}

.template-info h3 {
    margin: 0 0 8px 0;
    font-size: 16px;
    color: #333;
}

.template-info p {
    margin: 0 0 12px 0;
    font-size: 13px;
    color: #666;
    line-height: 1.4;
}

.template-meta {
    display: flex;
    gap: 8px;
    margin-bottom: 12px;
}

.template-category,
.template-orientation {
    font-size: 11px;
    padding: 4px 8px;
    background: #e9ecef;
    border-radius: 3px;
    color: #495057;
}

.use-template-btn {
    width: 100%;
    padding: 10px;
    background: #3498db;
    color: white;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    transition: background 0.3s;
}

.use-template-btn:hover {
    background: #2980b9;
}
```

### JavaScript Functions to Add

#### 1. Load Default Templates List
```javascript
async function loadDefaultTemplates() {
    try {
        const response = await fetch(getAjaxURL() + '&action=listDefaultTemplates');
        const data = await response.json();

        if (data.status && data.content.templates) {
            displayDefaultTemplates(data.content.templates);
        }
    } catch (error) {
        console.error('Error loading default templates:', error);
    }
}
```

#### 2. Display Templates in Grid
```javascript
function displayDefaultTemplates(templates) {
    const grid = document.getElementById('defaultTemplatesGrid');
    grid.innerHTML = '';

    templates.forEach(template => {
        const card = document.createElement('div');
        card.className = 'template-card';
        card.innerHTML = `
            <div class="template-thumbnail">
                <i class="fa fa-certificate"></i>
            </div>
            <div class="template-info">
                <h3>${template.name}</h3>
                <p>${template.description}</p>
                <div class="template-meta">
                    <span class="template-category">${template.category}</span>
                    <span class="template-orientation">${template.orientation}</span>
                </div>
            </div>
            <div class="template-actions">
                <button class="use-template-btn" data-file="${template.file}">
                    <i class="fa fa-download"></i> Use Template
                </button>
            </div>
        `;

        grid.appendChild(card);
    });

    // Attach event listeners
    document.querySelectorAll('.use-template-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            useDefaultTemplate(this.dataset.file);
        });
    });
}
```

#### 3. Use Default Template
```javascript
async function useDefaultTemplate(templateFile) {
    try {
        const response = await fetch(getAjaxURL(), {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: `action=loadDefaultTemplate&templateFile=${encodeURIComponent(templateFile)}`
        });

        const data = await response.json();

        if (data.status && data.content.templateData) {
            // Load template data into canvas
            const templateData = JSON.parse(data.content.templateData);
            canvas.loadFromJSON(templateData, () => {
                canvas.renderAll();

                // Set orientation
                if (data.content.orientation === 'portrait') {
                    setOrientation('portrait');
                } else {
                    setOrientation('landscape');
                }

                // Close modal
                document.getElementById('templateGalleryModal').style.display = 'none';

                // Show success message
                alert(`Template "${data.content.name}" loaded successfully!`);
            });
        }
    } catch (error) {
        console.error('Error loading template:', error);
        alert('Failed to load template');
    }
}
```

---

## Next Steps

### Phase 3 Remaining Tasks:
1. ✅ Add "Template Gallery" button to toolbar
2. ✅ Create Template Gallery modal HTML
3. ✅ Add CSS styling for gallery grid
4. ✅ Implement JavaScript functions
5. ⏳ Add tab switching (Default/My Templates)
6. ⏳ Integrate existing "My Templates" into gallery
7. ⏳ Test template loading functionality

### Phase 4: Testing
1. Test loading each default template
2. Verify orientation switching
3. Test variable placeholders
4. Verify "My Templates" still works
5. Test responsive layout

---

## File Changes Summary

### Created Files:
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/certificate_basic.json`
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/certificate_elegant.json`
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/certificate_modern.json`
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/diploma_classic.json`
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/award_simple.json`
- `/var/www/pub/plugins/generic/goValidOJS/default_templates/README.md`

### Modified Files:
- `/var/www/pub/plugins/generic/goValidOJS/GoValidOJSPlugin.php` (Added endpoints and methods)
- `/var/www/pub/plugins/generic/goValidOJS/templates/designer.tpl` (TO DO: Add UI)

---

## Benefits

### For Plugin Developers:
✅ Easy distribution - just copy plugin folder
✅ Easy updates - update plugin = update templates
✅ No database setup required
✅ Version control friendly

### For Journal Administrators:
✅ Professional templates ready to use
✅ No design skills required
✅ Can customize and save their own versions
✅ Clear separation between default and custom templates

### For End Users:
✅ Quick start with professional designs
✅ Template gallery with previews
✅ One-click template loading
✅ Can switch between default and personal templates

---

**Last Updated**: November 2, 2025 16:30
**Implementation Status**: 65% Complete
**Next Session**: Continue with Phase 3 UI implementation
