/**
 * Email Template Designer JavaScript
 * Real-time preview and control handlers
 */

let currentFocusedField = null;
let templateConfig = {};
let isGoValidAuthenticated = false;
let goValidQuota = 0;
let goValidSubscription = null;

/**
 * Initialize the designer
 */
function initializeDesigner() {
    // Load existing template if editing
    loadExistingTemplate();

    // Setup range value displays
    setupRangeDisplays();

    // Initial preview
    updatePreview();

    // Initialize access control UI (will be updated by GoValid auth)
    updateAccessControlUI();

    console.log('Email Designer: Initialized');
}

/**
 * Switch between tabs (Settings tab requires authentication)
 */
function switchTab(tabName) {
    // Update tab buttons
    document.querySelectorAll('.etd-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    document.querySelector(`.etd-tab[data-tab="${tabName}"]`).classList.add('active');

    // Update tab panels
    document.querySelectorAll('.etd-tab-panel').forEach(panel => {
        panel.classList.remove('active');
    });
    document.getElementById(`tab-${tabName}`).classList.add('active');
}

/**
 * Toggle header background type
 */
function toggleHeaderBgType() {
    const bgType = document.getElementById('header_bgType').value;
    const solidBg = document.getElementById('header_solidBg');
    const gradientBg = document.getElementById('header_gradientBg');
    const gradientBg2 = document.getElementById('header_gradientBg2');

    if (bgType === 'solid') {
        solidBg.style.display = 'block';
        gradientBg.style.display = 'none';
        if (gradientBg2) gradientBg2.style.display = 'none';
    } else {
        solidBg.style.display = 'none';
        gradientBg.style.display = 'block';
        if (gradientBg2) gradientBg2.style.display = 'block';
    }

    updatePreview();
}

/**
 * Setup range input displays
 */
function setupRangeDisplays() {
    const ranges = [
        'header_padding',
        'content_padding',
        'footer_padding',
        'footer_borderWidth',
        'settings_fontSize',
        'settings_lineHeight'
    ];

    ranges.forEach(rangeId => {
        const range = document.getElementById(rangeId);
        if (range) {
            range.addEventListener('input', function() {
                const valueSpan = document.getElementById(rangeId + '_value');
                if (valueSpan) {
                    valueSpan.textContent = this.value;
                }
            });
        }
    });
}

/**
 * Get current template configuration
 */
function getTemplateConfig() {
    return {
        name: document.getElementById('templateName').value,
        subject: document.getElementById('templateSubject').value,
        header: {
            showLogo: document.getElementById('header_showLogo').checked,
            logoPosition: document.getElementById('header_logoPosition').value,
            showNameBesideLogo: document.getElementById('header_showNameBesideLogo').checked,
            logoMaxWidth: document.getElementById('header_logoMaxWidth').value,
            bgType: document.getElementById('header_bgType').value,
            bgColor: document.getElementById('header_bgColor').value,
            gradientStart: document.getElementById('header_gradientStart').value,
            gradientEnd: document.getElementById('header_gradientEnd').value,
            textColor: document.getElementById('header_textColor').value,
            title: document.getElementById('header_title').value,
            subtitle: document.getElementById('header_subtitle').value,
            padding: document.getElementById('header_padding').value
        },
        content: {
            greeting: document.getElementById('content_greeting').value,
            message: document.getElementById('content_message').value,
            showDetailsBox: document.getElementById('content_showDetailsBox').checked,
            articleDetails: document.getElementById('content_articleDetails') ? document.getElementById('content_articleDetails').value : '',
            detailsBgColor: document.getElementById('content_detailsBgColor').value,
            closing: document.getElementById('content_closing').value,
            bgColor: document.getElementById('content_bgColor').value,
            textColor: document.getElementById('content_textColor').value,
            padding: document.getElementById('content_padding').value
        },
        footer: {
            bgColor: document.getElementById('footer_bgColor').value,
            textColor: document.getElementById('footer_textColor').value,
            signature: document.getElementById('footer_signature').value,
            signatureName: document.getElementById('footer_signatureName').value,
            showUrl: document.getElementById('footer_showUrl').checked,
            linkColor: document.getElementById('footer_linkColor').value,
            padding: document.getElementById('footer_padding').value,
            borderWidth: document.getElementById('footer_borderWidth').value,
            borderColor: document.getElementById('footer_borderColor').value
        },
        settings: {
            emailWidth: document.getElementById('settings_emailWidth').value,
            fontFamily: document.getElementById('settings_fontFamily').value,
            fontSize: document.getElementById('settings_fontSize').value,
            lineHeight: document.getElementById('settings_lineHeight').value,
            outerBgColor: document.getElementById('settings_outerBgColor').value
        }
    };
}

/**
 * Update live preview
 */
function updatePreview() {
    console.log('updatePreview() called');
    const config = getTemplateConfig();
    templateConfig = config;

    const previewHTML = generateEmailHTML(config);
    const previewElement = document.getElementById('emailPreview');

    if (!previewElement) {
        console.error('Preview element not found!');
        return;
    }

    previewElement.innerHTML = previewHTML;
    console.log('Preview HTML updated');

    // Setup two-way sync for editable elements
    setupPreviewEditSync();
}

/**
 * Generate header content with logo positioning
 */
function generateHeaderContent(config, sampleData, replacePlaceholders) {
    const logoPosition = config.header.logoPosition || 'center';
    const showNameBesideLogo = config.header.showNameBesideLogo || false;
    const logoMaxWidth = config.header.logoMaxWidth || 200;

    // Generate logo HTML
    const logoHTML = config.header.showLogo ? (sampleData.journal_logo_url ? `
        <img src="${sampleData.journal_logo_url}" alt="${sampleData.journal_logo_alt}" style="max-width: ${logoMaxWidth}px; height: auto; display: block;" width="${sampleData.journal_logo_width}" height="${sampleData.journal_logo_height}">
    ` : `
        <div style="width: 180px; height: 90px; background: rgba(255,255,255,0.2); display: flex; align-items: center; justify-content: center; border-radius: 8px; border: 2px dashed rgba(255,255,255,0.5);">
            <span style="opacity: 0.7; font-size: 14px;">Logo</span>
        </div>
    `) : '';

    // Generate title/subtitle HTML (with contenteditable)
    const titleHTML = `
        <div style="flex: 1;">
            <h1 contenteditable="true" data-field="header_title" style="margin: 0; color: ${config.header.textColor}; font-size: 28px; font-weight: 600; outline: none;" spellcheck="false">${replacePlaceholders(config.header.title)}</h1>
            ${config.header.subtitle ? `
            <p contenteditable="true" data-field="header_subtitle" style="margin: 10px 0 0 0; color: ${config.header.textColor}; font-size: 18px; opacity: 0.9; outline: none;" spellcheck="false">${replacePlaceholders(config.header.subtitle)}</p>
            ` : ''}
        </div>
    `;

    // Center position
    if (logoPosition === 'center') {
        return `
            <div style="text-align: center;">
                ${config.header.showLogo ? `<div style="margin-bottom: 20px; display: flex; justify-content: center;">${logoHTML}</div>` : ''}
                ${titleHTML}
            </div>
        `;
    }

    // Left or Right position with option for name beside logo
    if (logoPosition === 'left' || logoPosition === 'right') {
        if (showNameBesideLogo && config.header.showLogo) {
            // Logo and name side by side
            const flexDirection = logoPosition === 'left' ? 'row' : 'row-reverse';
            const textAlign = logoPosition === 'left' ? 'left' : 'right';
            return `
                <div style="display: flex; align-items: center; gap: 20px; flex-direction: ${flexDirection};">
                    <div style="flex-shrink: 0;">
                        ${logoHTML}
                    </div>
                    <div style="flex: 1; text-align: ${textAlign};">
                        <h1 contenteditable="true" data-field="header_title" style="margin: 0; color: ${config.header.textColor}; font-size: 28px; font-weight: 600; outline: none;" spellcheck="false">${replacePlaceholders(config.header.title)}</h1>
                        ${config.header.subtitle ? `
                        <p contenteditable="true" data-field="header_subtitle" style="margin: 10px 0 0 0; color: ${config.header.textColor}; font-size: 18px; opacity: 0.9; outline: none;" spellcheck="false">${replacePlaceholders(config.header.subtitle)}</p>
                        ` : ''}
                    </div>
                </div>
            `;
        } else {
            // Logo on top, then text
            const alignItems = logoPosition === 'left' ? 'flex-start' : 'flex-end';
            const textAlign = logoPosition === 'left' ? 'left' : 'right';
            return `
                <div style="display: flex; flex-direction: column; align-items: ${alignItems};">
                    ${config.header.showLogo ? `<div style="margin-bottom: 20px;">${logoHTML}</div>` : ''}
                    <div style="text-align: ${textAlign}; width: 100%;">
                        <h1 contenteditable="true" data-field="header_title" style="margin: 0; color: ${config.header.textColor}; font-size: 28px; font-weight: 600; outline: none;" spellcheck="false">${replacePlaceholders(config.header.title)}</h1>
                        ${config.header.subtitle ? `
                        <p contenteditable="true" data-field="header_subtitle" style="margin: 10px 0 0 0; color: ${config.header.textColor}; font-size: 18px; opacity: 0.9; outline: none;" spellcheck="false">${replacePlaceholders(config.header.subtitle)}</p>
                        ` : ''}
                    </div>
                </div>
            `;
        }
    }

    return '';
}

/**
 * Generate email HTML from configuration
 */
function generateEmailHTML(config) {
    // Get journal info from page (set by PHP)
    const journalLogoUrl = window.journalLogoUrl || '';
    const journalName = window.journalName || 'Journal of Advanced Studies';
    const journalLogoAlt = window.journalLogoAlt || 'Journal Logo';
    const journalLogoWidth = window.journalLogoWidth || 180;
    const journalLogoHeight = window.journalLogoHeight || 90;

    // Sample data for preview
    const sampleData = {
        reviewer_name: 'Dr. Jane Smith',
        reviewer_email: 'jane.smith@university.edu',
        journal_name: journalName,
        article_title: 'Sample Article Title: A Comprehensive Study of Modern Research Methods',
        article_authors: 'John Doe, Mary Johnson, Robert Williams',
        issue_title: 'Vol. 1 No. 1 (2025)',
        review_date: new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }),
        journal_url: window.location.origin,
        journal_logo_url: journalLogoUrl,
        journal_logo_alt: journalLogoAlt,
        journal_logo_width: journalLogoWidth,
        journal_logo_height: journalLogoHeight,
        custom_message: 'Thank you for your continued support of our journal.'
    };

    // Replace placeholders in text
    const replacePlaceholders = (text) => {
        if (!text) return '';
        let result = text;
        Object.keys(sampleData).forEach(key => {
            const regex = new RegExp('\\{\\{' + key + '\\}\\}', 'g');
            result = result.replace(regex, sampleData[key]);
        });
        return result;
    };

    // Header background style
    let headerBgStyle = '';
    if (config.header.bgType === 'gradient') {
        headerBgStyle = `background: linear-gradient(135deg, ${config.header.gradientStart} 0%, ${config.header.gradientEnd} 100%);`;
    } else {
        headerBgStyle = `background: ${config.header.bgColor};`;
    }

    // Build HTML
    let html = `
<div style="font-family: ${config.settings.fontFamily}; max-width: ${config.settings.emailWidth}px; margin: 0 auto; background: ${config.settings.outerBgColor};">
    <!-- Header -->
    <div style="${headerBgStyle} color: ${config.header.textColor}; padding: ${config.header.padding}px;">
        ${generateHeaderContent(config, sampleData, replacePlaceholders)}
    </div>

    <!-- Content -->
    <div style="padding: ${config.content.padding}px; background: ${config.content.bgColor}; color: ${config.content.textColor}; line-height: ${config.settings.lineHeight}; font-size: ${config.settings.fontSize}px;">
        ${config.content.greeting ? `
        <p contenteditable="true" data-field="content_greeting" style="font-size: ${config.settings.fontSize}px; margin-bottom: 20px; outline: none;" spellcheck="false">${replacePlaceholders(config.content.greeting)}</p>
        ` : ''}

        ${config.content.message ? `
        <div contenteditable="true" data-field="content_message" style="font-size: ${config.settings.fontSize}px; margin-bottom: 20px; outline: none; min-height: 50px;" spellcheck="false">${replacePlaceholders(config.content.message)}</div>
        ` : ''}

        ${config.content.showDetailsBox ? `
        <div contenteditable="true" data-field="content_articleDetails" style="background: ${config.content.detailsBgColor}; padding: 20px; border-radius: 8px; margin: 25px 0; outline: none; min-height: 80px;" spellcheck="false">
            ${config.content.articleDetails ? replacePlaceholders(config.content.articleDetails).replace(/\n/g, '<br>') : `<p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Article:</strong> ${sampleData.article_title}</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Authors:</strong> ${sampleData.article_authors}</p>
            <p style="margin: 0 0 10px 0; font-size: 14px;"><strong>Issue:</strong> ${sampleData.issue_title}</p>
            <p style="margin: 0; font-size: 14px;"><strong>Review completed:</strong> ${sampleData.review_date}</p>`}
        </div>
        ` : ''}

        ${config.content.closing ? `
        <div contenteditable="true" data-field="content_closing" style="font-size: ${config.settings.fontSize}px; margin-bottom: 20px; white-space: pre-line; outline: none; min-height: 40px;" spellcheck="false">${replacePlaceholders(config.content.closing)}</div>
        ` : ''}
    </div>

    <!-- Footer -->
    <div style="background: ${config.footer.bgColor}; padding: ${config.footer.padding}px; text-align: center; border-top: ${config.footer.borderWidth}px solid ${config.footer.borderColor};">
        ${config.footer.signature ? `
        <p contenteditable="true" data-field="footer_signature" style="margin: 0 0 10px 0; font-size: ${config.settings.fontSize}px; color: ${config.footer.textColor}; outline: none;" spellcheck="false">${replacePlaceholders(config.footer.signature)}</p>
        ` : ''}
        ${config.footer.signatureName ? `
        <p contenteditable="true" data-field="footer_signatureName" style="margin: 0 0 15px 0; font-size: ${config.settings.fontSize}px; color: ${config.footer.textColor}; font-weight: bold; outline: none;" spellcheck="false">${replacePlaceholders(config.footer.signatureName)}</p>
        ` : ''}
        ${config.footer.showUrl ? `
        <p style="margin: 0; font-size: 14px; color: ${config.footer.textColor};">
            <a href="${sampleData.journal_url}" style="color: ${config.footer.linkColor}; text-decoration: none;">${sampleData.journal_url}</a>
        </p>
        ` : ''}
    </div>
</div>
    `;

    return html;
}

/**
 * Setup two-way sync between preview and form fields
 */
function setupPreviewEditSync() {
    // Find all editable elements in preview
    const editableElements = document.querySelectorAll('[contenteditable="true"][data-field]');

    console.log('Setting up preview edit sync for', editableElements.length, 'elements');

    editableElements.forEach(element => {
        const fieldId = element.getAttribute('data-field');

        // Add blur event to sync when user finishes editing
        element.addEventListener('blur', function() {
            syncPreviewToForm(fieldId, this);
        });

        // Add input event for real-time sync (optional, can be removed if too aggressive)
        element.addEventListener('input', function() {
            syncPreviewToForm(fieldId, this);
        });

        // Prevent Enter key from creating new lines in single-line fields (title, subtitle, etc.)
        if (fieldId.includes('title') || fieldId.includes('subtitle') || fieldId.includes('greeting') || fieldId.includes('signature')) {
            element.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    this.blur(); // Lose focus when Enter is pressed
                }
            });
        }
    });
}

/**
 * Sync edited content from preview back to form field
 */
function syncPreviewToForm(fieldId, element) {
    const formField = document.getElementById(fieldId);
    if (!formField) {
        console.warn('Form field not found:', fieldId);
        return;
    }

    // Get the edited content
    let content = element.innerHTML;

    // Clean up HTML for plain text fields
    if (fieldId === 'content_articleDetails') {
        // For article details, convert <br> to newlines and strip other HTML
        content = content.replace(/<br\s*\/?>/gi, '\n');
        content = content.replace(/<\/p>/gi, '\n');
        content = content.replace(/<[^>]+>/g, '');
        content = content.trim();
    } else if (fieldId === 'content_message' || fieldId === 'content_closing') {
        // For multi-line fields, preserve some formatting
        content = content.replace(/<br\s*\/?>/gi, '\n');
        content = content.replace(/<\/p>/gi, '\n');
        content = content.replace(/<p[^>]*>/gi, '');
        content = content.replace(/<[^>]+>/g, '');
        content = content.trim();
    } else {
        // For single-line fields (title, subtitle, greeting, signature), strip all HTML
        content = element.textContent.trim();
    }

    // Update the form field
    formField.value = content;

    console.log('Synced preview to form:', fieldId, '=', content);

    // Don't call updatePreview() here to avoid infinite loop
    // The change is already visible in the preview
}

/**
 * Insert variable at cursor position
 */
function insertVariable(varName) {
    const variable = '{{' + varName + '}}';

    // Try to insert in last focused field or default to message
    let targetField = currentFocusedField || document.getElementById('content_message');

    if (targetField && (targetField.tagName === 'TEXTAREA' || targetField.tagName === 'INPUT')) {
        const start = targetField.selectionStart;
        const end = targetField.selectionEnd;
        const text = targetField.value;

        targetField.value = text.substring(0, start) + variable + text.substring(end);
        targetField.focus();
        targetField.setSelectionRange(start + variable.length, start + variable.length);

        updatePreview();
    }
}

/**
 * Track focused field for variable insertion
 */
document.addEventListener('DOMContentLoaded', function() {
    const fields = document.querySelectorAll('input[type="text"], textarea');
    fields.forEach(field => {
        field.addEventListener('focus', function() {
            currentFocusedField = this;
        });
    });
});

/**
 * Toggle preview view (desktop/mobile)
 */
function togglePreviewView(view) {
    const previewContent = document.getElementById('emailPreview');
    const buttons = document.querySelectorAll('.etd-preview-btn');

    buttons.forEach(btn => btn.classList.remove('active'));
    document.querySelector(`.etd-preview-btn[data-view="${view}"]`).classList.add('active');

    if (view === 'mobile') {
        previewContent.style.maxWidth = '375px';
        previewContent.style.margin = '0 auto';
    } else {
        previewContent.style.maxWidth = '100%';
        previewContent.style.margin = '0';
    }
}

/**
 * Load preset configuration
 */
function loadPreset(presetName) {
    const presets = {
        professional: {
            header: {
                bgType: 'solid',
                bgColor: '#2c3e50',
                gradientStart: '#667eea',
                gradientEnd: '#764ba2',
                textColor: '#ffffff',
                title: 'Certificate of Review',
                subtitle: '{{journal_name}}',
                padding: 30,
                showLogo: true,
                logoPosition: 'center',
                showNameBesideLogo: false,
                logoMaxWidth: 200
            },
            content: {
                bgColor: '#ffffff',
                textColor: '#333333',
                padding: 40,
                detailsBgColor: '#f8f9fa',
                showDetailsBox: true
            },
            footer: {
                bgColor: '#ecf0f1',
                textColor: '#2c3e50',
                linkColor: '#3498db',
                padding: 25,
                borderWidth: 3,
                borderColor: '#2c3e50',
                showUrl: true
            },
            settings: {
                fontFamily: 'Arial, sans-serif',
                fontSize: 16,
                lineHeight: 1.6
            }
        },
        modern: {
            header: {
                bgType: 'gradient',
                bgColor: '#667eea',
                gradientStart: '#667eea',
                gradientEnd: '#764ba2',
                textColor: '#ffffff',
                title: 'Review Certificate',
                subtitle: '{{journal_name}}',
                padding: 40,
                showLogo: true,
                logoPosition: 'left',
                showNameBesideLogo: true,
                logoMaxWidth: 150
            },
            content: {
                bgColor: '#ffffff',
                textColor: '#333333',
                padding: 40,
                detailsBgColor: '#f8f9fa',
                showDetailsBox: true
            },
            footer: {
                bgColor: '#2c3e50',
                textColor: '#ffffff',
                linkColor: '#3498db',
                padding: 25,
                borderWidth: 0,
                borderColor: '#2c3e50',
                showUrl: true
            },
            settings: {
                fontFamily: "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif",
                fontSize: 16,
                lineHeight: 1.6
            }
        },
        minimal: {
            header: {
                bgType: 'solid',
                bgColor: '#ffffff',
                gradientStart: '#667eea',
                gradientEnd: '#764ba2',
                textColor: '#333333',
                title: '{{journal_name}}',
                subtitle: 'Certificate of Review',
                padding: 40,
                showLogo: true,
                logoPosition: 'center',
                showNameBesideLogo: false,
                logoMaxWidth: 180
            },
            content: {
                bgColor: '#ffffff',
                textColor: '#333333',
                padding: 40,
                detailsBgColor: '#f9f9f9',
                showDetailsBox: true
            },
            footer: {
                bgColor: '#ffffff',
                textColor: '#666666',
                linkColor: '#333333',
                padding: 40,
                borderWidth: 2,
                borderColor: '#e0e0e0',
                showUrl: true
            },
            settings: {
                fontFamily: 'Georgia, serif',
                fontSize: 16,
                lineHeight: 1.8
            }
        }
    };

    if (presets[presetName]) {
        applyPreset(presets[presetName]);
    }
}

/**
 * Apply preset to form
 */
function applyPreset(preset) {
    // Header
    if (preset.header) {
        document.getElementById('header_bgType').value = preset.header.bgType;
        document.getElementById('header_bgColor').value = preset.header.bgColor;
        document.getElementById('header_gradientStart').value = preset.header.gradientStart;
        document.getElementById('header_gradientEnd').value = preset.header.gradientEnd;
        document.getElementById('header_textColor').value = preset.header.textColor;
        document.getElementById('header_title').value = preset.header.title;
        document.getElementById('header_subtitle').value = preset.header.subtitle;
        document.getElementById('header_padding').value = preset.header.padding;
        document.getElementById('header_showLogo').checked = preset.header.showLogo;
        document.getElementById('header_logoPosition').value = preset.header.logoPosition || 'center';
        document.getElementById('header_showNameBesideLogo').checked = preset.header.showNameBesideLogo || false;
        document.getElementById('header_logoMaxWidth').value = preset.header.logoMaxWidth || 200;
        toggleHeaderBgType();
    }

    // Content
    if (preset.content) {
        document.getElementById('content_bgColor').value = preset.content.bgColor;
        document.getElementById('content_textColor').value = preset.content.textColor;
        document.getElementById('content_padding').value = preset.content.padding;
        document.getElementById('content_detailsBgColor').value = preset.content.detailsBgColor;
        document.getElementById('content_showDetailsBox').checked = preset.content.showDetailsBox;
    }

    // Footer
    if (preset.footer) {
        document.getElementById('footer_bgColor').value = preset.footer.bgColor;
        document.getElementById('footer_textColor').value = preset.footer.textColor;
        document.getElementById('footer_linkColor').value = preset.footer.linkColor;
        document.getElementById('footer_padding').value = preset.footer.padding;
        document.getElementById('footer_borderWidth').value = preset.footer.borderWidth;
        document.getElementById('footer_borderColor').value = preset.footer.borderColor;
        document.getElementById('footer_showUrl').checked = preset.footer.showUrl;
    }

    // Settings
    if (preset.settings) {
        document.getElementById('settings_fontFamily').value = preset.settings.fontFamily;
        document.getElementById('settings_fontSize').value = preset.settings.fontSize;
        document.getElementById('settings_lineHeight').value = preset.settings.lineHeight;
    }

    // Update displays and preview
    setupRangeDisplays();
    document.querySelectorAll('.etd-range').forEach(range => {
        const event = new Event('input');
        range.dispatchEvent(event);
    });

    updatePreview();
}

/**
 * Save template (requires GoValid authentication)
 */
function saveTemplate() {
    // Check authentication
    if (!isGoValidAuthenticated) {
        alert('⚠️ GoValid Login Required\n\nYou must login to GoValid to save templates.\n\nClick "Login to GoValid" button in the top right corner.');
        showLoginModal();
        return;
    }

    // Check if Free subscription
    if (isFreeSubscription()) {
        alert('⚠️ Upgrade Required\n\nSaving templates is only available for paid subscriptions (Starter, Business, Ultimate).\n\nPlease upgrade your GoValid subscription.');
        return;
    }

    const config = getTemplateConfig();

    if (!config.name || config.name.trim() === '') {
        alert('Please enter a template name.');
        return;
    }

    if (!config.subject || config.subject.trim() === '') {
        alert('Please enter an email subject.');
        return;
    }

    // Show confirmation dialog with template name and subject
    const confirmMessage = `📧 Save Email Template\n\n` +
                          `Template Name: ${config.name}\n` +
                          `Subject: ${config.subject}\n\n` +
                          `Do you want to save this template?`;

    if (!confirm(confirmMessage)) {
        return;
    }

    // Check if template with same name already exists
    checkTemplateExists(config.name).then(exists => {
        if (exists) {
            const overwriteMessage = `⚠️ Template Already Exists\n\n` +
                                    `A template with the name "${config.name}" already exists.\n\n` +
                                    `Do you want to overwrite it?`;

            if (!confirm(overwriteMessage)) {
                return;
            }
        }

        // Proceed with saving
        performSave(config);
    });
}

/**
 * Check if a template with the given name already exists in OJS
 */
async function checkTemplateExists(templateName) {
    try {
        // Generate the template key that would be used
        const templateKey = 'CUSTOM_' + templateName.toUpperCase().replace(/[^A-Z0-9]/g, '_');

        // Check if this key exists in the OJS templates
        const response = await fetch(buildActionURL('getOJSEmailTemplates'));
        const data = await response.json();

        if (data.status && data.content.templates) {
            // Check if any template has this key
            return data.content.templates.some(t => t.key === templateKey);
        }

        return false;
    } catch (error) {
        console.error('Error checking template:', error);
        return false;
    }
}

/**
 * Perform the actual save operation
 */
function performSave(config) {
    // Build proper AJAX URL
    const saveUrl = window.pluginAjaxUrl + '?action=saveTemplate';

    // Prepare data
    const saveData = {
        templateName: config.name,
        templateData: JSON.stringify(config)
    };

    // Convert to URL-encoded form data
    const formBody = Object.keys(saveData).map(key =>
        encodeURIComponent(key) + '=' + encodeURIComponent(saveData[key])
    ).join('&');

    fetch(saveUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: formBody
    })
    .then(response => response.json())
    .then(data => {
        // Check for success - handle both boolean true and string 'success'
        if (data.status === true || data.status === 'success' || data.success === true) {
            alert('✅ Template saved successfully!');
        } else {
            alert('Error saving template: ' + (data.message || 'Unknown error'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error saving template. Please try again.');
    });
}

/**
 * Load existing template
 */
function loadExistingTemplate() {
    // Check if we're editing an existing template
    const urlParams = new URLSearchParams(window.location.search);
    const templateId = urlParams.get('templateId');

    if (templateId) {
        // Load template data via AJAX
        fetch(window.location.href.split('?')[0] + '?op=getEmailTemplate&templateId=' + encodeURIComponent(templateId))
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success' && data.template) {
                applyTemplateData(data.template);
            }
        })
        .catch(error => {
            console.error('Error loading template:', error);
        });
    }
}

/**
 * Apply template data to form
 */
function applyTemplateData(template) {
    if (template.name) {
        document.getElementById('templateName').value = template.name;
    }

    if (template.config) {
        const config = typeof template.config === 'string' ? JSON.parse(template.config) : template.config;
        applyPreset(config);

        // Apply content fields
        if (config.content) {
            if (config.content.greeting) document.getElementById('content_greeting').value = config.content.greeting;
            if (config.content.message) document.getElementById('content_message').value = config.content.message;
            if (config.content.closing) document.getElementById('content_closing').value = config.content.closing;
        }

        if (config.subject) {
            document.getElementById('templateSubject').value = config.subject;
            document.getElementById('content_subject').value = config.subject;
        }

        // Apply footer fields
        if (config.footer) {
            if (config.footer.signature) document.getElementById('footer_signature').value = config.footer.signature;
            if (config.footer.signatureName) document.getElementById('footer_signatureName').value = config.footer.signatureName;
        }
    }
}

/**
 * Load template (requires GoValid authentication)
 */
function loadTemplate() {
    // Check authentication
    if (!isGoValidAuthenticated) {
        alert('⚠️ GoValid Login Required\n\nYou must login to GoValid to load saved templates.\n\nClick "Login to GoValid" button in the top right corner.');
        showLoginModal();
        return;
    }

    // Check if Free subscription
    if (isFreeSubscription()) {
        alert('⚠️ Upgrade Required\n\nLoading templates is only available for paid subscriptions (Starter, Business, Ultimate).\n\nPlease upgrade your GoValid subscription.');
        return;
    }

    // Redirect to template manager
    window.location.href = window.location.href.split('?')[0] + '?op=emailTemplateManager';
}

/**
 * Send test email - Show modal (auth check moved to submit button)
 */
function sendTestEmail() {
    // Always show modal, auth/quota check is in submitSendEmail
    showSendEmailModal();
}

/**
 * Show send email modal
 */
function showSendEmailModal() {
    const modal = document.getElementById('sendEmailModal');
    const emailInput = document.getElementById('testEmailInput');
    const ccInput = document.getElementById('ccEmailInput');
    const bccInput = document.getElementById('bccEmailInput');

    if (modal) {
        modal.style.display = 'flex';
        emailInput.value = '';
        ccInput.value = '';
        bccInput.value = '';

        // Initialize autocomplete for email fields
        initializeEmailAutocomplete();

        // Update submit button state based on auth/quota
        updateModalSubmitButton();

        // Focus after a small delay to ensure modal is visible
        setTimeout(() => {
            emailInput.focus();
        }, 100);

        // Allow Enter key to submit (but not if autocomplete is shown)
        emailInput.onkeypress = function(e) {
            const autocompleteDropdown = document.getElementById('testEmailAutocomplete');
            if (e.key === 'Enter' && !autocompleteDropdown.classList.contains('show')) {
                submitSendEmail();
            }
        };

        // Close modal when clicking overlay
        const overlay = modal.querySelector('.etd-modal-overlay');
        overlay.onclick = function() {
            closeSendEmailModal();
        };

        // Prevent closing when clicking modal content
        const modalContent = modal.querySelector('.etd-modal-content');
        modalContent.onclick = function(e) {
            e.stopPropagation();
        };
    }
}

/**
 * Close send email modal
 */
function closeSendEmailModal() {
    const modal = document.getElementById('sendEmailModal');
    if (modal) {
        modal.style.display = 'none';
    }

    // Clear attachments when closing modal
    selectedAttachments = [];
    const fileCountSpan = document.getElementById('attachmentFileCount');
    const attachmentsList = document.getElementById('attachmentsList');
    const errorDiv = document.getElementById('attachmentError');

    if (fileCountSpan) {
        fileCountSpan.textContent = 'No files selected';
        fileCountSpan.style.color = '#718096';
    }
    if (attachmentsList) {
        attachmentsList.innerHTML = '';
    }
    if (errorDiv) {
        errorDiv.style.display = 'none';
    }
}

/**
 * Update modal submit button state based on auth/subscription
 */
function updateModalSubmitButton() {
    const submitBtn = document.getElementById('modalSendEmailBtn');
    const warningText = document.getElementById('modalAuthWarning');

    if (!submitBtn) return;

    // Check authentication
    if (!isGoValidAuthenticated) {
        submitBtn.disabled = true;
        submitBtn.style.opacity = '0.5';
        submitBtn.style.cursor = 'not-allowed';
        submitBtn.title = 'Login required to send emails';

        if (warningText) {
            warningText.style.display = 'block';
            warningText.innerHTML = '⚠️ Please <a href="#" onclick="showLoginModal(); return false;" style="color: #667eea; text-decoration: underline;">login to GoValid</a> to send emails.';
        }
        return;
    }

    // Check if Free subscription
    if (isFreeSubscription()) {
        submitBtn.disabled = true;
        submitBtn.style.opacity = '0.5';
        submitBtn.style.cursor = 'not-allowed';
        submitBtn.title = 'Upgrade required to send emails';

        if (warningText) {
            warningText.style.display = 'block';
            warningText.innerHTML = '⚠️ Sending emails is only available for paid subscriptions (Starter, Business, Ultimate). Please upgrade your GoValid subscription.';
        }
        return;
    }

    // Check quota (for paid users who ran out)
    if (goValidQuota <= 0) {
        submitBtn.disabled = true;
        submitBtn.style.opacity = '0.5';
        submitBtn.style.cursor = 'not-allowed';
        submitBtn.title = 'No quota available';

        if (warningText) {
            warningText.style.display = 'block';
            warningText.innerHTML = '⚠️ You have used all your quota. Please wait for quota renewal or upgrade your plan.';
        }
        return;
    }

    // All checks passed - enable button
    submitBtn.disabled = false;
    submitBtn.style.opacity = '1';
    submitBtn.style.cursor = 'pointer';
    submitBtn.title = 'Send email (Remaining quota: ' + goValidQuota + ')';

    if (warningText) {
        warningText.style.display = 'none';
    }
}

/**
 * Store selected attachment files
 */
let selectedAttachments = [];

/**
 * Handle attachment file selection
 */
function handleAttachmentSelection(event) {
    const files = event.target.files;
    const errorDiv = document.getElementById('attachmentError');
    const fileCountSpan = document.getElementById('attachmentFileCount');
    const attachmentsList = document.getElementById('attachmentsList');

    // Reset error
    errorDiv.style.display = 'none';
    errorDiv.textContent = '';

    // Validate files
    const maxSize = 5 * 1024 * 1024; // 5MB in bytes
    const allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    const allowedExtensions = ['.pdf', '.jpg', '.jpeg', '.png', '.gif'];

    let validFiles = [];
    let errors = [];

    for (let i = 0; i < files.length; i++) {
        const file = files[i];

        // Check file size
        if (file.size > maxSize) {
            errors.push(`${file.name}: File size exceeds 5MB (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
            continue;
        }

        // Check file type
        const fileExtension = '.' + file.name.split('.').pop().toLowerCase();
        if (!allowedTypes.includes(file.type) && !allowedExtensions.includes(fileExtension)) {
            errors.push(`${file.name}: Invalid file type. Only PDF and images are allowed.`);
            continue;
        }

        validFiles.push(file);
    }

    // Show errors if any
    if (errors.length > 0) {
        errorDiv.style.display = 'block';
        errorDiv.innerHTML = errors.join('<br>');
    }

    // Update selected attachments
    selectedAttachments = validFiles;

    // Update UI
    if (validFiles.length === 0) {
        fileCountSpan.textContent = 'No files selected';
        fileCountSpan.style.color = '#718096';
        attachmentsList.innerHTML = '';
    } else {
        fileCountSpan.textContent = `${validFiles.length} file(s) selected`;
        fileCountSpan.style.color = '#10b981';

        // Display file list with remove buttons
        let html = '<div style="display: flex; flex-direction: column; gap: 6px;">';
        validFiles.forEach((file, index) => {
            const fileSize = (file.size / 1024).toFixed(1);
            const sizeUnit = fileSize < 1024 ? 'KB' : 'MB';
            const displaySize = fileSize < 1024 ? fileSize : (file.size / 1024 / 1024).toFixed(2);

            html += `
                <div style="display: flex; align-items: center; justify-content: space-between; padding: 8px; background: #f8f9fa; border-radius: 4px; font-size: 13px;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path>
                            <polyline points="13 2 13 9 20 9"></polyline>
                        </svg>
                        <span style="color: #2d3748;">${file.name}</span>
                        <span style="color: #718096; font-size: 12px;">(${displaySize} ${sizeUnit})</span>
                    </div>
                    <button type="button" onclick="removeAttachment(${index})" style="background: none; border: none; color: #e53e3e; cursor: pointer; padding: 4px 8px; font-size: 18px; line-height: 1;" title="Remove">
                        &times;
                    </button>
                </div>
            `;
        });
        html += '</div>';
        attachmentsList.innerHTML = html;
    }

    // Reset file input to allow selecting the same file again if removed
    event.target.value = '';
}

/**
 * Remove attachment by index
 */
function removeAttachment(index) {
    selectedAttachments.splice(index, 1);

    // Update UI
    const fileCountSpan = document.getElementById('attachmentFileCount');
    const attachmentsList = document.getElementById('attachmentsList');
    const errorDiv = document.getElementById('attachmentError');

    if (selectedAttachments.length === 0) {
        fileCountSpan.textContent = 'No files selected';
        fileCountSpan.style.color = '#718096';
        attachmentsList.innerHTML = '';
        errorDiv.style.display = 'none';
    } else {
        fileCountSpan.textContent = `${selectedAttachments.length} file(s) selected`;

        // Rebuild file list
        let html = '<div style="display: flex; flex-direction: column; gap: 6px;">';
        selectedAttachments.forEach((file, idx) => {
            const fileSize = (file.size / 1024).toFixed(1);
            const sizeUnit = fileSize < 1024 ? 'KB' : 'MB';
            const displaySize = fileSize < 1024 ? fileSize : (file.size / 1024 / 1024).toFixed(2);

            html += `
                <div style="display: flex; align-items: center; justify-content: space-between; padding: 8px; background: #f8f9fa; border-radius: 4px; font-size: 13px;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path>
                            <polyline points="13 2 13 9 20 9"></polyline>
                        </svg>
                        <span style="color: #2d3748;">${file.name}</span>
                        <span style="color: #718096; font-size: 12px;">(${displaySize} ${sizeUnit})</span>
                    </div>
                    <button type="button" onclick="removeAttachment(${idx})" style="background: none; border: none; color: #e53e3e; cursor: pointer; padding: 4px 8px; font-size: 18px; line-height: 1;" title="Remove">
                        &times;
                    </button>
                </div>
            `;
        });
        html += '</div>';
        attachmentsList.innerHTML = html;
    }
}

/**
 * Submit send email from modal
 */
function submitSendEmail() {
    // Check authentication first
    if (!isGoValidAuthenticated) {
        alert('⚠️ GoValid Login Required\n\nYou must login to GoValid to send emails.\n\nClick "Login to GoValid" button in the top right corner.');
        showLoginModal();
        return;
    }

    // Check if Free subscription
    if (isFreeSubscription()) {
        alert('⚠️ Upgrade Required\n\nSending emails is only available for paid subscriptions (Starter, Business, Ultimate).\n\nPlease upgrade your GoValid subscription.');
        return;
    }

    // Check quota (for paid users who ran out)
    if (goValidQuota <= 0) {
        alert('⚠️ No Quota Available\n\nYou have used all your quota. Please wait for quota renewal or upgrade your plan.');
        return;
    }

    const emailInput = document.getElementById('testEmailInput');
    const ccInput = document.getElementById('ccEmailInput');
    const bccInput = document.getElementById('bccEmailInput');

    const email = emailInput.value.trim();
    const cc = ccInput.value.trim();
    const bcc = bccInput.value.trim();

    if (!email) {
        alert('Please enter an email address.');
        emailInput.focus();
        return;
    }

    // Basic email validation function
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

    // Validate TO email
    if (!emailRegex.test(email)) {
        alert('Please enter a valid email address for TO field.');
        emailInput.focus();
        return;
    }

    // Validate CC emails if provided
    if (cc) {
        const ccEmails = cc.split(',').map(e => e.trim()).filter(e => e);
        for (let ccEmail of ccEmails) {
            if (!emailRegex.test(ccEmail)) {
                alert('Invalid email address in CC field: ' + ccEmail);
                ccInput.focus();
                return;
            }
        }
    }

    // Validate BCC emails if provided
    if (bcc) {
        const bccEmails = bcc.split(',').map(e => e.trim()).filter(e => e);
        for (let bccEmail of bccEmails) {
            if (!emailRegex.test(bccEmail)) {
                alert('Invalid email address in BCC field: ' + bccEmail);
                bccInput.focus();
                return;
            }
        }
    }

    // Get email config and HTML BEFORE closing modal
    const config = getTemplateConfig();
    const emailHTML = generateEmailHTML(config);

    // Store attachments reference BEFORE closing modal (which clears them)
    const attachmentsToSend = selectedAttachments ? [...selectedAttachments] : [];
    console.log('Email Designer: Storing attachments before closing modal:', attachmentsToSend.length);

    const formData = new FormData();
    formData.append('testEmail', email);
    formData.append('subject', config.subject);
    formData.append('htmlContent', emailHTML);

    // Add CC and BCC if provided
    if (cc) {
        formData.append('cc', cc);
    }
    if (bcc) {
        formData.append('bcc', bcc);
    }

    // Add attachments if any
    console.log('Email Designer: attachmentsToSend:', attachmentsToSend);
    console.log('Email Designer: attachmentsToSend length:', attachmentsToSend.length);
    if (attachmentsToSend.length > 0) {
        console.log('Email Designer: Adding ' + attachmentsToSend.length + ' attachments to FormData');
        for (let i = 0; i < attachmentsToSend.length; i++) {
            console.log('Email Designer: Adding attachment ' + i + ':', attachmentsToSend[i].name, attachmentsToSend[i].size, attachmentsToSend[i].type);
            formData.append('attachments[]', attachmentsToSend[i], attachmentsToSend[i].name);
        }
    } else {
        console.log('Email Designer: No attachments to add');
    }

    let recipientsList = email;
    if (cc) recipientsList += ', CC: ' + cc;
    if (bcc) recipientsList += ', BCC: ' + bcc;

    let statusMessage = 'Sending email to ' + recipientsList + '...';
    if (attachmentsToSend.length > 0) {
        statusMessage += '\nAttachments: ' + attachmentsToSend.length + ' file(s)';
    }

    // Close modal AFTER we've copied the data we need
    closeSendEmailModal();

    alert(statusMessage);

    fetch(window.location.href.split('?')[0] + '?action=sendTestEmail', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.status === 'success') {
            alert('✅ Email sent successfully!');
            // Decrease quota locally
            goValidQuota--;
            updateAccessControlUI();
            updateModalSubmitButton();
        } else {
            alert('Error sending email: ' + (data.message || 'Unknown error'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error sending email. Please try again.');
    });
}

/**
 * Update access control UI based on authentication status
 */
function updateAccessControlUI() {
    const authenticated = isGoValidAuthenticated;

    // Save button
    const saveBtn = document.getElementById('saveTemplateBtn');
    const saveLock = document.getElementById('saveTemplateLock');
    if (saveBtn && saveLock) {
        if (!authenticated) {
            saveBtn.style.opacity = '0.6';
            saveBtn.style.cursor = 'not-allowed';
            saveLock.style.display = 'inline';
            saveBtn.title = 'Login required to save templates';
        } else if (isFreeSubscription()) {
            saveBtn.style.opacity = '0.6';
            saveBtn.style.cursor = 'not-allowed';
            saveLock.style.display = 'inline';
            saveBtn.title = 'Upgrade to paid subscription (Starter, Business, Ultimate) to save templates';
        } else {
            saveBtn.style.opacity = '1';
            saveBtn.style.cursor = 'pointer';
            saveLock.style.display = 'none';
            saveBtn.title = '';
        }
    }

    // Load button
    const loadBtn = document.getElementById('loadTemplateBtn');
    const loadLock = document.getElementById('loadTemplateLock');
    if (loadBtn && loadLock) {
        if (!authenticated) {
            loadBtn.style.opacity = '0.6';
            loadBtn.style.cursor = 'not-allowed';
            loadLock.style.display = 'inline';
            loadBtn.title = 'Login required to load templates';
        } else if (isFreeSubscription()) {
            loadBtn.style.opacity = '0.6';
            loadBtn.style.cursor = 'not-allowed';
            loadLock.style.display = 'inline';
            loadBtn.title = 'Upgrade to paid subscription (Starter, Business, Ultimate) to load templates';
        } else {
            loadBtn.style.opacity = '1';
            loadBtn.style.cursor = 'pointer';
            loadLock.style.display = 'none';
            loadBtn.title = '';
        }
    }

    // Send Email button - Always enabled (restrictions moved to modal)
    const sendTestBtn = document.getElementById('sendTestBtn');
    const sendTestLock = document.getElementById('sendTestLock');
    if (sendTestBtn) {
        sendTestBtn.style.opacity = '1';
        sendTestBtn.style.cursor = 'pointer';
        if (sendTestLock) {
            sendTestLock.style.display = 'none';
        }

        if (!authenticated) {
            sendTestBtn.title = 'Open email modal (Login required to send)';
        } else if (isFreeSubscription()) {
            sendTestBtn.title = 'Open email modal (Upgrade to paid subscription required)';
        } else if (goValidQuota <= 0) {
            sendTestBtn.title = 'Open email modal (No quota available)';
        } else {
            sendTestBtn.title = 'Send email (Remaining quota: ' + goValidQuota + ')';
        }
    }

    // OJS Template - Load Template button
    const loadOJSTemplateBtn = document.getElementById('loadOJSTemplateBtn');
    if (loadOJSTemplateBtn) {
        loadOJSTemplateBtn.disabled = !authenticated;
        if (!authenticated) {
            loadOJSTemplateBtn.style.opacity = '0.6';
            loadOJSTemplateBtn.style.cursor = 'not-allowed';
            loadOJSTemplateBtn.title = 'Login required to load OJS templates';
        } else {
            loadOJSTemplateBtn.style.opacity = '1';
            loadOJSTemplateBtn.style.cursor = 'pointer';
            loadOJSTemplateBtn.title = '';
        }
    }

    // OJS Template - Save Changes button (in saveButtonsContainer)
    const saveButtonsContainer = document.getElementById('saveButtonsContainer');
    if (saveButtonsContainer) {
        const saveChangesBtn = saveButtonsContainer.querySelector('.etd-btn-primary');
        const saveAsBtn = document.getElementById('saveAsButton');

        if (!authenticated) {
            if (saveChangesBtn) {
                saveChangesBtn.disabled = true;
                saveChangesBtn.style.opacity = '0.6';
                saveChangesBtn.style.cursor = 'not-allowed';
                saveChangesBtn.title = 'Login required to save templates';
            }
            if (saveAsBtn) {
                saveAsBtn.disabled = true;
                saveAsBtn.style.opacity = '0.6';
                saveAsBtn.style.cursor = 'not-allowed';
                saveAsBtn.title = 'Login required to create templates';
            }
        } else {
            if (saveChangesBtn) {
                saveChangesBtn.disabled = false;
                saveChangesBtn.style.opacity = '1';
                saveChangesBtn.style.cursor = 'pointer';
                saveChangesBtn.title = '';
            }
            if (saveAsBtn) {
                saveAsBtn.disabled = false;
                saveAsBtn.style.opacity = '1';
                saveAsBtn.style.cursor = 'pointer';
                saveAsBtn.title = '';
            }
        }
    }

    console.log('Access control UI updated - Authenticated:', authenticated, 'Quota:', goValidQuota);
}

/**
 * Update authentication status from GoValid auth system
 */
function updateAuthenticationStatus(authData) {
    if (authData && authData.authenticated) {
        isGoValidAuthenticated = true;
        goValidQuota = authData.quota || 0;
        goValidSubscription = authData.subscription || null;
        console.log('Email Designer: GoValid authenticated - Quota:', goValidQuota);
        console.log('Email Designer: Subscription:', goValidSubscription);

        // Update user account section visibility (both in preview header and main header)
        const loginSection = document.getElementById('loginSection');
        const userAccountSection = document.getElementById('userAccountSection');
        const loginSectionHeader = document.getElementById('loginSectionHeader');
        const userAccountSectionHeader = document.getElementById('userAccountSectionHeader');

        if (loginSection) loginSection.style.display = 'none';
        if (userAccountSection) userAccountSection.style.display = 'block';
        if (loginSectionHeader) loginSectionHeader.style.display = 'none';
        if (userAccountSectionHeader) userAccountSectionHeader.style.display = 'block';

        // Populate user avatar/photo (both locations)
        if (goValidAuth && goValidAuth.userInfo) {
            const userAvatarImage = document.getElementById('userAvatarImage');
            const userAvatarIcon = document.getElementById('userAvatarIcon');
            const userAvatarImageHeader = document.getElementById('userAvatarImageHeader');
            const userAvatarIconHeader = document.getElementById('userAvatarIconHeader');

            console.log('Email Designer: User info:', goValidAuth.userInfo);
            console.log('Email Designer: Profile photo URL:', goValidAuth.userInfo.profile_photo_url);

            if (userAvatarImage && userAvatarIcon) {
                if (goValidAuth.userInfo.profile_photo_url) {
                    // Check if URL is already absolute
                    let photoUrl = goValidAuth.userInfo.profile_photo_url;
                    if (!photoUrl.startsWith('http://') && !photoUrl.startsWith('https://')) {
                        photoUrl = 'https://my.govalid.org' + photoUrl;
                    }
                    console.log('Email Designer: Setting avatar image to:', photoUrl);
                    userAvatarImage.src = photoUrl;
                    userAvatarImage.style.display = 'block';
                    userAvatarIcon.style.display = 'none';

                    // Update header avatar too
                    if (userAvatarImageHeader) {
                        userAvatarImageHeader.src = photoUrl;
                        userAvatarImageHeader.style.display = 'block';
                    }
                    if (userAvatarIconHeader) {
                        userAvatarIconHeader.style.display = 'none';
                    }
                } else {
                    // No photo, show icon
                    console.log('Email Designer: No profile photo, showing default icon');
                    userAvatarImage.style.display = 'none';
                    userAvatarIcon.style.display = 'block';

                    // Update header avatar too
                    if (userAvatarImageHeader) userAvatarImageHeader.style.display = 'none';
                    if (userAvatarIconHeader) userAvatarIconHeader.style.display = 'block';
                }
            }
        }
    } else {
        isGoValidAuthenticated = false;
        goValidQuota = 0;
        goValidSubscription = null;
        console.log('Email Designer: GoValid not authenticated');

        // Update user account section visibility (both in preview header and main header)
        const loginSection = document.getElementById('loginSection');
        const userAccountSection = document.getElementById('userAccountSection');
        const loginSectionHeader = document.getElementById('loginSectionHeader');
        const userAccountSectionHeader = document.getElementById('userAccountSectionHeader');

        if (loginSection) loginSection.style.display = 'block';
        if (userAccountSection) userAccountSection.style.display = 'none';
        if (loginSectionHeader) loginSectionHeader.style.display = 'block';
        if (userAccountSectionHeader) userAccountSectionHeader.style.display = 'none';
    }
    updateAccessControlUI();
    updateHeaderBranding();
}

// Make function available globally for designer-auth.js
window.updateEmailDesignerAuth = updateAuthenticationStatus;

/**
 * Check if user has a Free subscription
 * Free subscription is determined by:
 * 1. plan_tier is "FREE" or null
 * 2. OR plan_name contains "Free" (case-insensitive)
 * 3. OR quota is 0 (fallback)
 */
function isFreeSubscription() {
    if (!isGoValidAuthenticated) {
        return true; // Not logged in = treated as free
    }

    if (!goValidSubscription) {
        return goValidQuota <= 0; // Fallback to quota check
    }

    // Check plan_tier
    const planTier = goValidSubscription.plan_tier;
    if (planTier === 'FREE' || planTier === null || planTier === '') {
        return true;
    }

    // Check plan_name
    const planName = goValidSubscription.plan_name || '';
    if (planName.toLowerCase().includes('free')) {
        return true;
    }

    // Check if no active subscription
    if (goValidSubscription.is_active === false) {
        return true;
    }

    // Fallback: check quota
    return goValidQuota <= 0;
}

/**
 * Check if user has Ultimate or Business subscription
 * These premium plans hide GoValid branding
 */
function isPremiumPlan() {
    if (!isGoValidAuthenticated || !goValidSubscription) {
        return false;
    }

    const planTier = (goValidSubscription.plan_tier || '').toUpperCase();
    const planName = (goValidSubscription.plan_name || '').toLowerCase();

    // Check plan_tier for ULTI (Ultimate) or BUSI (Business)
    if (planTier === 'ULTI' || planTier === 'ULTIMATE' || planTier === 'BUSI' || planTier === 'BUSINESS') {
        return true;
    }

    // Check plan_name contains "ultimate" or "business"
    if (planName.includes('ultimate') || planName.includes('business')) {
        return true;
    }

    return false;
}

/**
 * Update header branding based on subscription level
 * Ultimate and Business plans hide GoValid branding
 */
function updateHeaderBranding() {
    const govalidSection = document.getElementById('etdGoValidSection');
    const journalSection = document.getElementById('etdJournalSection');
    const topHeader = document.getElementById('etdTopHeader');
    const journalName = document.getElementById('etdJournalName');

    if (!govalidSection || !journalSection) return;

    if (isPremiumPlan()) {
        // Hide GoValid branding
        govalidSection.style.display = 'none';

        // Stretch journal section to full width (left to right)
        if (topHeader) {
            topHeader.style.justifyContent = 'space-between';
        }

        if (journalSection) {
            journalSection.style.flex = '1';
            journalSection.querySelector('div').style.width = '100%';
            journalSection.querySelector('div').style.justifyContent = 'space-between';
        }

        // Adjust journal name styling for left alignment
        if (journalName) {
            journalName.style.textAlign = 'left';
            journalName.querySelector('div').style.fontSize = '24px';
        }

        console.log('Email Designer: Premium plan detected - GoValid branding hidden');
    } else {
        // Show GoValid branding (default)
        govalidSection.style.display = 'block';

        // Reset to flex layout
        if (topHeader) {
            topHeader.style.justifyContent = 'space-between';
        }

        if (journalSection) {
            journalSection.style.flex = '';
            journalSection.querySelector('div').style.width = '';
            journalSection.querySelector('div').style.justifyContent = '';
        }

        // Reset journal name styling
        if (journalName) {
            journalName.style.textAlign = 'right';
            journalName.querySelector('div').style.fontSize = '18px';
        }

        console.log('Email Designer: Standard plan - GoValid branding visible');
    }
}

/**
 * Email Autocomplete functionality
 */
let autocompleteTimeout = null;
let currentAutocompleteField = null;

/**
 * Initialize autocomplete for email fields
 */
function initializeEmailAutocomplete() {
    const fields = [
        { input: 'testEmailInput', dropdown: 'testEmailAutocomplete' },
        { input: 'ccEmailInput', dropdown: 'ccEmailAutocomplete' },
        { input: 'bccEmailInput', dropdown: 'bccEmailAutocomplete' }
    ];

    fields.forEach(field => {
        const inputElement = document.getElementById(field.input);
        const dropdownElement = document.getElementById(field.dropdown);

        if (inputElement && dropdownElement) {
            inputElement.addEventListener('input', function(e) {
                handleAutocompleteInput(e.target, dropdownElement);
            });

            inputElement.addEventListener('keydown', function(e) {
                handleAutocompleteKeyboard(e, dropdownElement);
            });

            inputElement.addEventListener('blur', function() {
                // Delay hiding to allow click on dropdown
                setTimeout(() => hideAutocomplete(dropdownElement), 200);
            });
        }
    });

    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.classList.contains('etd-autocomplete-input')) {
            document.querySelectorAll('.etd-autocomplete-dropdown').forEach(dropdown => {
                hideAutocomplete(dropdown);
            });
        }
    });
}

/**
 * Handle autocomplete input
 */
function handleAutocompleteInput(inputElement, dropdownElement) {
    clearTimeout(autocompleteTimeout);
    currentAutocompleteField = inputElement;

    const value = inputElement.value.trim();

    // For comma-separated fields (CC/BCC), get the last email being typed
    let searchTerm = value;
    if (inputElement.id === 'ccEmailInput' || inputElement.id === 'bccEmailInput') {
        const emails = value.split(',');
        searchTerm = emails[emails.length - 1].trim();
    }

    if (searchTerm.length < 2) {
        hideAutocomplete(dropdownElement);
        return;
    }

    // Show loading state
    dropdownElement.innerHTML = '<div class="etd-autocomplete-loading">Searching...</div>';
    dropdownElement.classList.add('show');

    // Debounce search
    autocompleteTimeout = setTimeout(() => {
        searchUsers(searchTerm, dropdownElement, inputElement);
    }, 300);
}

/**
 * Search users via API
 */
function searchUsers(searchTerm, dropdownElement, inputElement) {
    const url = window.location.href.split('?')[0] + '?action=searchUsers&q=' + encodeURIComponent(searchTerm);

    fetch(url)
        .then(response => response.json())
        .then(data => {
            if (data.results && data.results.length > 0) {
                displayAutocompleteResults(data.results, dropdownElement, inputElement);
            } else {
                dropdownElement.innerHTML = '<div class="etd-autocomplete-no-results">No users found</div>';
            }
        })
        .catch(error => {
            console.error('Autocomplete error:', error);
            dropdownElement.innerHTML = '<div class="etd-autocomplete-no-results">Error loading results</div>';
        });
}

/**
 * Display autocomplete results
 */
function displayAutocompleteResults(results, dropdownElement, inputElement) {
    let html = '';

    results.forEach((user, index) => {
        html += `
            <div class="etd-autocomplete-item" data-email="${user.email}" data-index="${index}">
                <div class="etd-autocomplete-item-name">${user.name}</div>
                <div class="etd-autocomplete-item-email">${user.email}</div>
            </div>
        `;
    });

    dropdownElement.innerHTML = html;

    // Add click handlers
    dropdownElement.querySelectorAll('.etd-autocomplete-item').forEach(item => {
        item.addEventListener('click', function() {
            selectAutocompleteItem(this.dataset.email, inputElement, dropdownElement);
        });
    });
}

/**
 * Select autocomplete item
 */
function selectAutocompleteItem(email, inputElement, dropdownElement) {
    if (inputElement.id === 'ccEmailInput' || inputElement.id === 'bccEmailInput') {
        // For CC/BCC, replace the last email being typed
        const currentValue = inputElement.value;
        const emails = currentValue.split(',').map(e => e.trim());
        emails[emails.length - 1] = email;
        inputElement.value = emails.join(', ');
    } else {
        // For TO field, replace entire value
        inputElement.value = email;
    }

    hideAutocomplete(dropdownElement);
    inputElement.focus();
}

/**
 * Handle keyboard navigation in autocomplete
 */
function handleAutocompleteKeyboard(e, dropdownElement) {
    if (!dropdownElement.classList.contains('show')) return;

    const items = dropdownElement.querySelectorAll('.etd-autocomplete-item');
    if (items.length === 0) return;

    let currentIndex = -1;
    items.forEach((item, index) => {
        if (item.classList.contains('active')) {
            currentIndex = index;
        }
    });

    if (e.key === 'ArrowDown') {
        e.preventDefault();
        currentIndex = (currentIndex + 1) % items.length;
        updateActiveItem(items, currentIndex);
    } else if (e.key === 'ArrowUp') {
        e.preventDefault();
        currentIndex = currentIndex <= 0 ? items.length - 1 : currentIndex - 1;
        updateActiveItem(items, currentIndex);
    } else if (e.key === 'Enter') {
        e.preventDefault();
        if (currentIndex >= 0 && items[currentIndex]) {
            selectAutocompleteItem(items[currentIndex].dataset.email, e.target, dropdownElement);
        }
    } else if (e.key === 'Escape') {
        hideAutocomplete(dropdownElement);
    }
}

/**
 * Update active item in dropdown
 */
function updateActiveItem(items, activeIndex) {
    items.forEach((item, index) => {
        if (index === activeIndex) {
            item.classList.add('active');
            item.scrollIntoView({ block: 'nearest' });
        } else {
            item.classList.remove('active');
        }
    });
}

/**
 * Hide autocomplete dropdown
 */
function hideAutocomplete(dropdownElement) {
    dropdownElement.classList.remove('show');
    dropdownElement.innerHTML = '';
}
