// Certificate Designer - Email Integration
// Handles sending certificates via email with template support

(function () {
    'use strict';

    let autocompleteTimeout;
    let emailTemplates = [];

    // Initialize email functionality
    function initializeCertificateEmail() {
        const sendEmailBtn = document.getElementById('sendCertificateEmail');
        if (sendEmailBtn) {
            sendEmailBtn.addEventListener('click', async function () {
                // Check if authenticated
                if (!goValidAuth.isAuthenticated) {
                    alert('⚠️ GoValid Login Required\n\nYou must login to GoValid to send certificate emails.\n\nClick the "Login to GoValid" button to continue.');
                    return;
                }

                // Check GoValid subscription before sending
                const canGenerate = await checkBeforeSave();
                if (!canGenerate) {
                    return;
                }

                showSendCertificateEmailModal();
            });
        }

        // Initialize autocomplete for email fields
        initializeEmailAutocomplete();

        // Initialize template selector change handler
        const templateSelect = document.getElementById('certEmailTemplate');
        if (templateSelect) {
            templateSelect.addEventListener('change', onCertTemplateSelected);
        }

        // Add focus/blur styling for contenteditable email body
        const emailBodyField = document.getElementById('certEmailBody');
        if (emailBodyField) {
            emailBodyField.addEventListener('focus', function () {
                this.style.background = '#fff9e6';
                this.style.borderColor = '#ffc107';
            });
            emailBodyField.addEventListener('blur', function () {
                this.style.background = '#fff';
                this.style.borderColor = '#ddd';
            });
        }
    }

    // Show the send certificate email modal
    window.showSendCertificateEmailModal = function () {
        const modal = document.getElementById('sendCertificateEmailModal');
        if (!modal) return;

        // Check content selection first
        const issueSelect = document.getElementById('content_issue_select');
        const articleSelect = document.getElementById('content_article_select');
        const reviewerSelect = document.getElementById('content_reviewer_name');

        // Check required fields - issue and article are required
        if (!issueSelect || !issueSelect.value) {
            alert('Please select an issue before sending email.');
            return;
        }

        if (!articleSelect || !articleSelect.value) {
            alert('Please select an article before sending email.');
            return;
        }

        // Reviewer is optional - show warning if not selected
        if (!reviewerSelect || !reviewerSelect.value) {
            const proceed = confirm('⚠️ No reviewer selected!\n\nYou can still send the email, but the {REVIEWER_NAME} placeholder will remain as-is on the certificate.\n\nDo you want to continue?');
            if (!proceed) {
                return;
            }
        }

        // Clear previous inputs
        document.getElementById('certToEmail').value = '';
        document.getElementById('certCcEmail').value = '';
        document.getElementById('certBccEmail').value = '';

        // Populate email chips from content selections
        populateEmailChips();

        // Load email templates
        loadCertificateEmailTemplates();

        // Update modal submit button state
        updateCertificateModalSubmitButton();

        // Show modal
        modal.style.display = 'flex';
    };

    // Populate email chips from selected reviewer, author, and editor
    function populateEmailChips() {
        const chipsContainer = document.getElementById('certEmailChipsContainer');
        const chipsDiv = document.getElementById('certEmailChips');

        if (!chipsContainer || !chipsDiv) return;

        // Clear existing chips
        chipsDiv.innerHTML = '';

        const emails = [];

        // Get reviewer email
        const reviewerSelect = document.getElementById('content_reviewer_name');
        if (reviewerSelect && reviewerSelect.selectedIndex > 0) {
            const selectedOption = reviewerSelect.options[reviewerSelect.selectedIndex];
            const email = selectedOption.dataset.email;
            if (email) {
                emails.push({ role: 'Reviewer', name: selectedOption.value, email: email });
            }
        }

        // Get author email
        const authorSelect = document.getElementById('content_author_name');
        if (authorSelect && authorSelect.selectedIndex > 0) {
            const selectedOption = authorSelect.options[authorSelect.selectedIndex];
            const email = selectedOption.dataset.email;
            if (email) {
                emails.push({ role: 'Author', name: selectedOption.value, email: email });
            }
        }

        // Get editor email
        const editorSelect = document.getElementById('content_editor_name');
        if (editorSelect && editorSelect.selectedIndex > 0) {
            const selectedOption = editorSelect.options[editorSelect.selectedIndex];
            const email = selectedOption.dataset.email;
            if (email) {
                emails.push({ role: 'Editor', name: selectedOption.value, email: email });
            }
        }

        // Create chips
        if (emails.length > 0) {
            emails.forEach(item => {
                const chip = document.createElement('span');
                chip.className = 'cert-email-chip';
                chip.innerHTML = `<span class="chip-role">${item.role}:</span> <span class="chip-email">${item.email}</span> <i class="fa fa-plus-circle"></i>`;
                chip.title = `Click to add ${item.name}'s email to recipient`;
                chip.addEventListener('click', function () {
                    addChipEmailToField(item.email);
                });
                chipsDiv.appendChild(chip);
            });
            chipsContainer.style.display = 'flex';
        } else {
            chipsContainer.style.display = 'none';
        }
    }

    // Add email from chip to the To field
    function addChipEmailToField(email) {
        const toField = document.getElementById('certToEmail');
        if (!toField) return;

        const currentValue = toField.value.trim();

        // Check if email already exists in the field
        if (currentValue) {
            const existingEmails = currentValue.split(',').map(e => e.trim().toLowerCase());
            if (existingEmails.includes(email.toLowerCase())) {
                // Email already in field, just focus
                toField.focus();
                return;
            }
            // Append with comma
            toField.value = currentValue + ', ' + email;
        } else {
            toField.value = email;
        }

        // Trigger input event for validation
        toField.dispatchEvent(new Event('input', { bubbles: true }));
        toField.focus();

        // Update submit button state
        updateCertificateModalSubmitButton();
    }

    // Close the send certificate email modal
    window.closeSendCertificateEmailModal = function () {
        const modal = document.getElementById('sendCertificateEmailModal');
        if (modal) {
            modal.style.display = 'none';
        }

        // Clear attachments when closing modal
        selectedCertAttachments = [];
        const fileCountSpan = document.getElementById('certAttachmentFileCount');
        const attachmentsList = document.getElementById('certAttachmentsList');
        const errorDiv = document.getElementById('certAttachmentError');

        if (fileCountSpan) {
            fileCountSpan.textContent = 'No files selected';
            fileCountSpan.style.color = '#718096';
        }
        if (attachmentsList) {
            attachmentsList.innerHTML = '';
        }
        if (errorDiv) {
            errorDiv.style.display = 'none';
        }
    };

    // Load email templates
    function loadCertificateEmailTemplates() {
        const select = document.getElementById('certEmailTemplate');
        if (!select) return;

        select.innerHTML = '<option value="">Loading templates...</option>';

        const cleanUrl = getCleanURL();
        fetch(cleanUrl.split('?')[0] + '?verb=emailTemplateDesigner&plugin=goValidOJS&category=generic&action=getOJSEmailTemplates', {
            method: 'GET'
        })
            .then(response => response.json())
            .then(data => {
                // OJS email templates are in data.content.templates
                const allTemplates = (data.content && data.content.templates) ? data.content.templates : [];

                // Filter only custom templates (those created by email designer)
                emailTemplates = allTemplates.filter(t => t.key && t.key.startsWith('CUSTOM_'));

                if (emailTemplates.length === 0) {
                    select.innerHTML = '<option value="">No email templates available</option>';
                    return;
                }

                select.innerHTML = '<option value="">-- Select Email Template --</option>';
                emailTemplates.forEach(template => {
                    const option = document.createElement('option');
                    option.value = template.key; // Use template key instead of name
                    option.textContent = template.name;
                    option.dataset.subject = template.subject || '';
                    select.appendChild(option);
                });
            })
            .catch(error => {
                console.error('Error loading email templates:', error);
                select.innerHTML = '<option value="">Error loading templates</option>';
            });
    }

    // Handle template selection - load content into preview
    async function onCertTemplateSelected() {
        const select = document.getElementById('certEmailTemplate');
        const templateKey = select.value;

        const subjectInput = document.getElementById('certEmailSubject');
        const bodyDiv = document.getElementById('certEmailBody');

        if (!templateKey) {
            subjectInput.value = '';
            subjectInput.placeholder = 'Select a template to load content...';
            bodyDiv.innerHTML = 'Select a template to load content...';
            return;
        }

        // Show loading state
        subjectInput.value = 'Loading template...';
        bodyDiv.innerHTML = 'Loading template...';

        try {
            // Fetch the template details
            const cleanUrl = getCleanURL();
            const response = await fetch(cleanUrl.split('?')[0] + '?verb=emailTemplateDesigner&plugin=goValidOJS&category=generic&action=getOJSEmailTemplates', {
                method: 'GET'
            });

            const data = await response.json();
            const allTemplates = (data.content && data.content.templates) ? data.content.templates : [];
            const template = allTemplates.find(t => t.key === templateKey);

            if (!template) {
                subjectInput.value = 'Template not found';
                bodyDiv.innerHTML = 'Template not found';
                return;
            }

            // Get content from issue/article/reviewer selection for replacement
            const replacements = getContentReplacements();

            // Replace placeholders in subject and body
            let subject = template.subject || 'No subject';
            let body = template.body || template.html || 'No content';

            // Replace {{placeholders}}
            Object.keys(replacements).forEach(key => {
                const placeholder = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
                subject = subject.replace(placeholder, replacements[key]);
                body = body.replace(placeholder, replacements[key]);
            });

            // Replace OJS/Smarty placeholders {$variable}
            const journalInfo = window.goValidJournalInfo || {};
            const journalLogoUrl = journalInfo.logo || '';
            const journalName = journalInfo.name || 'Journal';

            // Replace journal-related placeholders
            body = body.replace(/\{\$journalThumbnailUrl\}/g, journalLogoUrl);
            body = body.replace(/\{\$journalName\}/g, journalName);
            body = body.replace(/\{\$contextName\}/g, journalName);
            body = body.replace(/\{\$contextUrl\}/g, window.location.origin);

            // Clean up escaped newlines from database storage (e.g., \n becomes actual newlines)
            // These can cause extra whitespace when rendered
            body = body.replace(/\\n/g, '');

            // Trim any leading/trailing whitespace
            body = body.trim();

            // Set the content (allow editing)
            subjectInput.value = subject;
            bodyDiv.innerHTML = body;

        } catch (error) {
            console.error('Error loading template:', error);
            subjectInput.value = 'Error loading template';
            bodyDiv.innerHTML = 'Error loading template';
        }
    }

    // Get content replacements from selected issue/article/reviewer
    function getContentReplacements() {
        const issueSelect = document.getElementById('content_issue_select');
        const articleSelect = document.getElementById('content_article_select');
        const reviewerSelect = document.getElementById('content_reviewer_name');

        const replacements = {
            issue_name: issueSelect ? issueSelect.options[issueSelect.selectedIndex]?.text || '' : '',
            article_title: articleSelect ? articleSelect.options[articleSelect.selectedIndex]?.text || '' : '',
            reviewer_name: reviewerSelect ? reviewerSelect.options[reviewerSelect.selectedIndex]?.text || '' : '',
            journal_name: document.querySelector('.journal-info h2')?.textContent || 'Journal',
            date: new Date().toLocaleDateString(),
            year: new Date().getFullYear()
        };

        return replacements;
    }

    // Get edited email content from preview fields
    function getEditedEmailContent() {
        const subjectInput = document.getElementById('certEmailSubject');
        const bodyDiv = document.getElementById('certEmailBody');

        return {
            subject: subjectInput.value.trim(),
            body: bodyDiv.innerHTML
        };
    }

    // Check if user has Free subscription
    function isFreeSubscription() {
        if (!goValidAuth.isAuthenticated) {
            return true;
        }

        if (!goValidAuth.subscription) {
            return goValidAuth.quota <= 0;
        }

        const planTier = goValidAuth.subscription.plan_tier;
        if (planTier === 'FREE' || planTier === null || planTier === '') {
            return true;
        }

        const planName = goValidAuth.subscription.plan_name || '';
        if (planName.toLowerCase().includes('free')) {
            return true;
        }

        if (goValidAuth.subscription.is_active === false) {
            return true;
        }

        return goValidAuth.quota <= 0;
    }

    // Update modal submit button state
    function updateCertificateModalSubmitButton() {
        const submitBtn = document.getElementById('modalSendCertificateBtn');
        const warningDiv = document.getElementById('certModalAuthWarning');

        if (!submitBtn || !warningDiv) return;

        if (!goValidAuth.isAuthenticated) {
            submitBtn.disabled = true;
            submitBtn.style.opacity = '0.5';
            submitBtn.style.cursor = 'not-allowed';
            warningDiv.innerHTML = '⚠️ Please log in to GoValid to send emails.';
            warningDiv.style.display = 'block';
        } else if (isFreeSubscription()) {
            submitBtn.disabled = true;
            submitBtn.style.opacity = '0.5';
            submitBtn.style.cursor = 'not-allowed';
            warningDiv.innerHTML = '⚠️ Sending emails is only available for paid subscriptions (Starter, Business, Ultimate).<br>Please upgrade your GoValid subscription.';
            warningDiv.style.display = 'block';
        } else if (goValidAuth.quota <= 0) {
            submitBtn.disabled = true;
            submitBtn.style.opacity = '0.5';
            submitBtn.style.cursor = 'not-allowed';
            warningDiv.innerHTML = '⚠️ No quota available. Please wait for quota renewal or upgrade your plan.';
            warningDiv.style.display = 'block';
        } else {
            submitBtn.disabled = false;
            submitBtn.style.opacity = '1';
            submitBtn.style.cursor = 'pointer';
            warningDiv.style.display = 'none';
        }
    }

    /**
     * Store selected certificate attachments
     */
    let selectedCertAttachments = [];

    /**
     * Handle certificate attachment file selection
     */
    window.handleCertAttachmentSelection = function (event) {
        const files = event.target.files;
        const errorDiv = document.getElementById('certAttachmentError');
        const fileCountSpan = document.getElementById('certAttachmentFileCount');
        const attachmentsList = document.getElementById('certAttachmentsList');

        // Reset error
        errorDiv.style.display = 'none';
        errorDiv.textContent = '';

        // Validate files
        const maxSize = 5 * 1024 * 1024; // 5MB in bytes
        const allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        const allowedExtensions = ['.pdf', '.jpg', '.jpeg', '.png', '.gif'];

        let validFiles = [];
        let errors = [];

        for (let i = 0; i < files.length; i++) {
            const file = files[i];

            // Check file size
            if (file.size > maxSize) {
                errors.push(`${file.name}: File size exceeds 5MB (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
                continue;
            }

            // Check file type
            const fileExtension = '.' + file.name.split('.').pop().toLowerCase();
            if (!allowedTypes.includes(file.type) && !allowedExtensions.includes(fileExtension)) {
                errors.push(`${file.name}: Invalid file type. Only PDF and images are allowed.`);
                continue;
            }

            validFiles.push(file);
        }

        // Show errors if any
        if (errors.length > 0) {
            errorDiv.style.display = 'block';
            errorDiv.innerHTML = errors.join('<br>');
        }

        // Update selected attachments
        selectedCertAttachments = validFiles;

        // Update UI
        if (validFiles.length === 0) {
            fileCountSpan.textContent = 'No files selected';
            fileCountSpan.style.color = '#718096';
            attachmentsList.innerHTML = '';
        } else {
            fileCountSpan.textContent = `${validFiles.length} file(s) selected`;
            fileCountSpan.style.color = '#10b981';

            // Display file list with remove buttons
            let html = '<div style="display: flex; flex-direction: column; gap: 6px;">';
            validFiles.forEach((file, index) => {
                const fileSize = (file.size / 1024).toFixed(1);
                const sizeUnit = fileSize < 1024 ? 'KB' : 'MB';
                const displaySize = fileSize < 1024 ? fileSize : (file.size / 1024 / 1024).toFixed(2);

                html += `
                    <div style="display: flex; align-items: center; justify-content: space-between; padding: 8px; background: #f8f9fa; border-radius: 4px; font-size: 13px;">
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path>
                                <polyline points="13 2 13 9 20 9"></polyline>
                            </svg>
                            <span style="color: #2d3748;">${file.name}</span>
                            <span style="color: #718096; font-size: 12px;">(${displaySize} ${sizeUnit})</span>
                        </div>
                        <button type="button" onclick="removeCertAttachment(${index})" style="background: none; border: none; color: #e53e3e; cursor: pointer; padding: 4px 8px; font-size: 18px; line-height: 1;" title="Remove">
                            &times;
                        </button>
                    </div>
                `;
            });
            html += '</div>';
            attachmentsList.innerHTML = html;
        }

        // Reset file input
        event.target.value = '';
    };

    /**
     * Remove certificate attachment by index
     */
    window.removeCertAttachment = function (index) {
        selectedCertAttachments.splice(index, 1);

        // Update UI
        const fileCountSpan = document.getElementById('certAttachmentFileCount');
        const attachmentsList = document.getElementById('certAttachmentsList');
        const errorDiv = document.getElementById('certAttachmentError');

        if (selectedCertAttachments.length === 0) {
            fileCountSpan.textContent = 'No files selected';
            fileCountSpan.style.color = '#718096';
            attachmentsList.innerHTML = '';
            errorDiv.style.display = 'none';
        } else {
            fileCountSpan.textContent = `${selectedCertAttachments.length} file(s) selected`;

            // Rebuild file list
            let html = '<div style="display: flex; flex-direction: column; gap: 6px;">';
            selectedCertAttachments.forEach((file, idx) => {
                const fileSize = (file.size / 1024).toFixed(1);
                const sizeUnit = fileSize < 1024 ? 'KB' : 'MB';
                const displaySize = fileSize < 1024 ? fileSize : (file.size / 1024 / 1024).toFixed(2);

                html += `
                    <div style="display: flex; align-items: center; justify-content: space-between; padding: 8px; background: #f8f9fa; border-radius: 4px; font-size: 13px;">
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path>
                                <polyline points="13 2 13 9 20 9"></polyline>
                            </svg>
                            <span style="color: #2d3748;">${file.name}</span>
                            <span style="color: #718096; font-size: 12px;">(${displaySize} ${sizeUnit})</span>
                        </div>
                        <button type="button" onclick="removeCertAttachment(${idx})" style="background: none; border: none; color: #e53e3e; cursor: pointer; padding: 4px 8px; font-size: 18px; line-height: 1;" title="Remove">
                            &times;
                        </button>
                    </div>
                `;
            });
            html += '</div>';
            attachmentsList.innerHTML = html;
        }
    };

    // Submit send certificate email
    window.submitSendCertificateEmail = async function () {
        // Check authentication and subscription
        if (!goValidAuth.isAuthenticated) {
            alert('⚠️ Authentication Required\n\nPlease log in to GoValid to send emails.');
            return;
        }

        if (isFreeSubscription()) {
            alert('⚠️ Upgrade Required\n\nSending emails is only available for paid subscriptions (Starter, Business, Ultimate).\n\nPlease upgrade your GoValid subscription.');
            return;
        }

        if (goValidAuth.quota <= 0) {
            alert('⚠️ No Quota Available\n\nYou have used all your quota. Please wait for quota renewal or upgrade your plan.');
            return;
        }

        // Validate inputs
        const templateName = document.getElementById('certEmailTemplate').value;
        const toEmail = document.getElementById('certToEmail').value.trim();
        const ccEmail = document.getElementById('certCcEmail').value.trim();
        const bccEmail = document.getElementById('certBccEmail').value.trim();
        const attachmentFormat = 'jpg'; // Always use JPG format (direct canvas rendering)

        if (!templateName) {
            alert('Please select an email template.');
            return;
        }

        if (!toEmail) {
            alert('Please enter a recipient email address.');
            return;
        }

        // Email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(toEmail)) {
            alert('Please enter a valid email address.');
            return;
        }

        // Validate CC emails if provided
        if (ccEmail) {
            const ccEmails = ccEmail.split(',').map(e => e.trim()).filter(e => e);
            for (let email of ccEmails) {
                if (!emailRegex.test(email)) {
                    alert('Invalid CC email address: ' + email);
                    return;
                }
            }
        }

        // Validate BCC emails if provided
        if (bccEmail) {
            const bccEmails = bccEmail.split(',').map(e => e.trim()).filter(e => e);
            for (let email of bccEmails) {
                if (!emailRegex.test(email)) {
                    alert('Invalid BCC email address: ' + email);
                    return;
                }
            }
        }

        // Get certificate content data
        const issueSelect = document.getElementById('content_issue_select');
        const articleSelect = document.getElementById('content_article_select');
        const reviewerSelect = document.getElementById('content_reviewer_name');

        // Validate required inputs - issue and article are required, reviewer is optional
        if (!issueSelect.value) {
            alert('Please select an issue before sending email.');
            return;
        }

        if (!articleSelect.value) {
            alert('Please select an article before sending email.');
            return;
        }

        // No need to check reviewer again - already warned in showSendCertificateEmailModal

        // Show loading state
        const submitBtn = document.getElementById('modalSendCertificateBtn');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Sending...';
        submitBtn.disabled = true;

        try {
            // Get all replacement data
            const issueData = JSON.parse(issueSelect.options[issueSelect.selectedIndex].dataset.issue || '{}');
            const articleData = JSON.parse(articleSelect.options[articleSelect.selectedIndex].dataset.article || '{}');

            // Get reviewer data - content_reviewer_name is a select with options that have dataset
            const reviewerOption = reviewerSelect.options[reviewerSelect.selectedIndex];
            const reviewerData = reviewerOption ? {
                name: reviewerOption.text,
                email: reviewerOption.dataset.email || '',
                id: reviewerOption.dataset.reviewerId || ''
            } : {};

            // Get QR settings values
            const qrIdentifierValue = document.getElementById('qr_identifier_code')?.value;
            const qrIdentifier = qrIdentifierValue && qrIdentifierValue.trim() !== '' ? qrIdentifierValue.trim() : '{QR_IDENTIFIER}';
            const qrStartDateToggle = document.getElementById('qr_start_date_toggle')?.checked;
            const qrEndDateToggle = document.getElementById('qr_end_date_toggle')?.checked;
            const qrStartDate = qrStartDateToggle ? document.getElementById('qr_start_date')?.value : null;
            const qrEndDate = qrEndDateToggle ? document.getElementById('qr_end_date')?.value : null;

            const replacements = {
                '{ISSUE_TITLE}': issueData.title || 'Issue Title',
                '{ISSUE_VOLUME}': issueData.volume || 'Volume',
                '{ISSUE_NUMBER}': issueData.number || 'Number',
                '{ISSUE_YEAR}': issueData.year || new Date().getFullYear(),
                '{ARTICLE_TITLE}': articleData.title || 'Article Title',
                '{ARTICLE_DOI}': articleData.doi || 'DOI not available',
                '{ARTICLE_PAGES}': articleData.pages || 'Pages not specified',
                '{REVIEWER_NAME}': reviewerData.name || reviewerSelect.value || 'Reviewer Name',
                '{REVIEWER_EMAIL}': reviewerData.email || 'Email not available',
                '{CURRENT_DATE}': new Date().toLocaleDateString(),
                '{SUBMISSION_ID}': articleData.id || 'No ID',
                '{QR_ID}': (typeof getQRSettings === 'function' && getQRSettings().formatted_uuid) ? getQRSettings().formatted_uuid : '{QR_ID}',
                '{QR_IDENTIFIER}': qrIdentifier,
                '{QR_ISSUE_DATE}': qrStartDate ? new Date(qrStartDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
                '{QR_EXPIRE_DATE}': qrEndDate ? new Date(qrEndDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })
            };

            // Try to generate QR code, but don't fail if it doesn't work
            // User has already generated the certificate and sees the preview
            console.log('Generating QR code for certificate email...');
            try {
                const qrResult = await generateCertificateQR(replacements);

                // Store formatted_uuid (QR ID) if available
                if (qrResult.formatted_uuid) {
                    if (typeof getQRSettings === 'function') {
                        getQRSettings().formatted_uuid = qrResult.formatted_uuid;
                    }
                    replacements['{QR_ID}'] = qrResult.formatted_uuid;
                    console.log('Stored QR ID (formatted_uuid):', qrResult.formatted_uuid);
                }

                // Check various possible QR code locations in response
                let qrImageUrl = null;

                // Try different response structures
                if (qrResult.content?.data?.content?.qr_code?.qr_image_direct_url) {
                    qrImageUrl = qrResult.content.data.content.qr_code.qr_image_direct_url;
                } else if (qrResult.content?.data?.content?.qr_code?.qr_image_proxy_url) {
                    qrImageUrl = qrResult.content.data.content.qr_code.qr_image_proxy_url;
                } else if (qrResult.data?.content?.qr_code?.qr_image_direct_url) {
                    qrImageUrl = qrResult.data.content.qr_code.qr_image_direct_url;
                } else if (qrResult.qr_image_direct_url) {
                    qrImageUrl = qrResult.qr_image_direct_url;
                } else if (qrResult.qr_code) {
                    // Base64 encoded QR code
                    replacements['{QR_CODE}'] = 'data:image/png;base64,' + qrResult.qr_code;
                    qrImageUrl = 'base64';
                }

                if (qrImageUrl) {
                    console.log('QR code generated successfully for email, URL:', qrImageUrl);
                    if (qrImageUrl !== 'base64') {
                        replacements['{QR_CODE}'] = qrImageUrl;
                    }
                } else {
                    console.warn('QR code generation returned unexpected structure, continuing without it');
                    replacements['{QR_CODE}'] = ''; // Empty placeholder
                }
            } catch (qrError) {
                console.warn('QR code generation failed, continuing without it:', qrError);
                replacements['{QR_CODE}'] = ''; // Empty placeholder
            }

            // Render canvas directly to JPG for 100% accuracy with higher resolution
            console.log('Rendering canvas directly to JPG with 3x resolution multiplier...');

            // Set high quality JPG rendering with resolution multiplier
            const RESOLUTION_MULTIPLIER = 3; // 3x resolution for high quality output
            let certificateJpgData = null;
            let jpgQuality = 0.92; // Start with 92% quality
            let attempts = 0;
            const maxAttempts = 5;
            const maxSize = 5 * 1024 * 1024; // 5MB in bytes

            // Try rendering with decreasing quality until under 5MB
            while (attempts < maxAttempts) {
                // Use multiplier option to increase export resolution
                certificateJpgData = canvas.toDataURL({
                    format: 'jpeg',
                    quality: jpgQuality,
                    multiplier: RESOLUTION_MULTIPLIER // Export at 2x resolution
                });

                // Calculate size (base64 is ~33% larger than binary)
                const base64Length = certificateJpgData.split(',')[1].length;
                const sizeBytes = (base64Length * 3) / 4; // Approximate binary size
                const sizeMB = sizeBytes / 1024 / 1024;

                console.log(`Attempt ${attempts + 1}: JPG quality ${(jpgQuality * 100).toFixed(0)}%, size ~${sizeMB.toFixed(2)}MB`);

                if (sizeBytes <= maxSize) {
                    console.log(`✓ Certificate JPG generated successfully at ${(jpgQuality * 100).toFixed(0)}% quality, ~${sizeMB.toFixed(2)}MB`);
                    break;
                }

                // Reduce quality for next attempt
                jpgQuality -= 0.10;
                attempts++;

                if (attempts >= maxAttempts) {
                    throw new Error(`Certificate image too large (${sizeMB.toFixed(2)}MB). Please simplify your design, reduce image sizes, or remove some elements. Maximum allowed: 5MB`);
                }
            }

            // Get edited email content from preview
            const editedContent = getEditedEmailContent();

            // Store attachments reference BEFORE creating FormData
            const attachmentsToSend = selectedCertAttachments ? [...selectedCertAttachments] : [];
            console.log('Certificate email: Attachments to send:', attachmentsToSend.length);

            // Convert base64 data URL to Blob to avoid FormData encoding issues
            console.log('Converting base64 JPG to Blob...');
            const base64Data = certificateJpgData.split(',')[1]; // Remove data:image/jpeg;base64, prefix
            const binaryString = atob(base64Data); // Decode base64 to binary string
            const bytes = new Uint8Array(binaryString.length);
            for (let i = 0; i < binaryString.length; i++) {
                bytes[i] = binaryString.charCodeAt(i);
            }
            const certificateBlob = new Blob([bytes], { type: 'image/jpeg' });
            console.log('Certificate Blob created, size:', certificateBlob.size, 'bytes');

            // Send to backend - use FormData instead of URLSearchParams to support file uploads
            const formData = new FormData();
            formData.append('action', 'sendCertificateEmail');
            formData.append('templateName', templateName);
            formData.append('emailSubject', editedContent.subject);
            formData.append('emailBody', editedContent.body);
            formData.append('toEmail', toEmail);
            formData.append('ccEmail', ccEmail);
            formData.append('bccEmail', bccEmail);
            formData.append('attachmentFormat', attachmentFormat);
            formData.append('certificateImageFile', certificateBlob, 'certificate.jpg'); // Send as file blob
            formData.append('replacements', JSON.stringify(replacements));
            formData.append('orientation', currentOrientation);

            // Add additional attachments if any
            if (attachmentsToSend.length > 0) {
                console.log('Adding ' + attachmentsToSend.length + ' additional attachments');
                for (let i = 0; i < attachmentsToSend.length; i++) {
                    formData.append('additionalAttachments[]', attachmentsToSend[i], attachmentsToSend[i].name);
                    console.log('Added attachment:', attachmentsToSend[i].name, attachmentsToSend[i].size, attachmentsToSend[i].type);
                }
            }

            console.log('Sending certificate email with params:', {
                action: 'sendCertificateEmail',
                templateName: templateName,
                toEmail: toEmail,
                ccEmail: ccEmail,
                bccEmail: bccEmail,
                attachmentFormat: attachmentFormat,
                orientation: currentOrientation,
                hasCertificateImage: !!certificateJpgData,
                hasReplacements: !!replacements,
                hasEmailContent: !!(editedContent.subject && editedContent.body)
            });

            const response = await fetch(getAjaxURL(), {
                method: 'POST',
                // Don't set Content-Type header - FormData sets it automatically with boundary
                body: formData
            });

            console.log('Email send response status:', response.status, response.ok);

            const result = await response.json();
            console.log('Email send result:', result);

            if (result.status === 'success') {
                alert('✓ Certificate sent successfully!\n\nThe certificate has been sent to ' + toEmail);
                closeSendCertificateEmailModal();

                // Refresh auth status to update quota
                if (window.checkGoValidAuth) {
                    window.checkGoValidAuth();
                }
            } else {
                throw new Error(result.message || 'Failed to send email');
            }

        } catch (error) {
            console.error('Error sending certificate email:', error);
            alert('Error sending email: ' + error.message);
        } finally {
            // Restore button
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    };

    // Initialize email autocomplete
    function initializeEmailAutocomplete() {
        const fields = [
            { input: 'certToEmail', dropdown: 'certToEmailAutocomplete' },
            { input: 'certCcEmail', dropdown: 'certCcEmailAutocomplete' },
            { input: 'certBccEmail', dropdown: 'certBccEmailAutocomplete' }
        ];

        fields.forEach(field => {
            const inputElement = document.getElementById(field.input);
            const dropdownElement = document.getElementById(field.dropdown);

            if (inputElement && dropdownElement) {
                inputElement.addEventListener('input', function (e) {
                    handleAutocompleteInput(e.target, dropdownElement);
                });

                inputElement.addEventListener('keydown', function (e) {
                    handleAutocompleteKeyboard(e, dropdownElement);
                });

                // Close dropdown when clicking outside
                document.addEventListener('click', function (e) {
                    if (!inputElement.contains(e.target) && !dropdownElement.contains(e.target)) {
                        hideAutocomplete(dropdownElement);
                    }
                });
            }
        });
    }

    // Handle autocomplete input
    function handleAutocompleteInput(inputElement, dropdownElement) {
        clearTimeout(autocompleteTimeout);

        const value = inputElement.value.trim();
        let searchTerm = value;

        // For CC/BCC fields, only search the last email being typed
        if (inputElement.id === 'certCcEmail' || inputElement.id === 'certBccEmail') {
            const emails = value.split(',');
            searchTerm = emails[emails.length - 1].trim();
        }

        if (searchTerm.length < 2) {
            hideAutocomplete(dropdownElement);
            return;
        }

        dropdownElement.innerHTML = '<div class="etd-autocomplete-loading">Searching...</div>';
        dropdownElement.classList.add('show');

        autocompleteTimeout = setTimeout(() => {
            searchUsers(searchTerm, dropdownElement, inputElement);
        }, 300);
    }

    // Search users
    function searchUsers(searchTerm, dropdownElement, inputElement) {
        const cleanUrl = getCleanURL();
        fetch(cleanUrl.split('?')[0] + '?verb=emailTemplateDesigner&plugin=goValidOJS&category=generic&action=searchUsers&q=' + encodeURIComponent(searchTerm), {
            method: 'GET'
        })
            .then(response => response.json())
            .then(data => {
                if (data.results && data.results.length > 0) {
                    displayAutocompleteResults(data.results, dropdownElement, inputElement);
                } else {
                    dropdownElement.innerHTML = '<div class="etd-autocomplete-empty">No users found</div>';
                }
            })
            .catch(error => {
                console.error('Error searching users:', error);
                dropdownElement.innerHTML = '<div class="etd-autocomplete-empty">Error searching users</div>';
            });
    }

    // Display autocomplete results
    function displayAutocompleteResults(results, dropdownElement, inputElement) {
        dropdownElement.innerHTML = '';

        results.forEach((user, index) => {
            const item = document.createElement('div');
            item.className = 'etd-autocomplete-item';
            item.setAttribute('data-index', index);
            item.innerHTML = '<strong>' + escapeHtml(user.name) + '</strong><br><small style="color: #666;">' + escapeHtml(user.email) + '</small>';

            item.addEventListener('click', function () {
                selectAutocompleteItem(user.email, inputElement, dropdownElement);
            });

            dropdownElement.appendChild(item);
        });
    }

    // Select autocomplete item
    function selectAutocompleteItem(email, inputElement, dropdownElement) {
        if (inputElement.id === 'certCcEmail' || inputElement.id === 'certBccEmail') {
            // For CC/BCC, replace the last email being typed
            const currentValue = inputElement.value;
            const emails = currentValue.split(',').map(e => e.trim());
            emails[emails.length - 1] = email;
            inputElement.value = emails.join(', ');
        } else {
            inputElement.value = email;
        }

        hideAutocomplete(dropdownElement);
        inputElement.focus();
    }

    // Handle keyboard navigation in autocomplete
    function handleAutocompleteKeyboard(e, dropdownElement) {
        if (!dropdownElement.classList.contains('show')) return;

        const items = dropdownElement.querySelectorAll('.etd-autocomplete-item');
        if (items.length === 0) return;

        const currentIndex = Array.from(items).findIndex(item => item.classList.contains('active'));

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            const nextIndex = currentIndex < items.length - 1 ? currentIndex + 1 : 0;
            setActiveAutocompleteItem(items, nextIndex);
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            const prevIndex = currentIndex > 0 ? currentIndex - 1 : items.length - 1;
            setActiveAutocompleteItem(items, prevIndex);
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (currentIndex >= 0) {
                items[currentIndex].click();
            }
        } else if (e.key === 'Escape') {
            hideAutocomplete(dropdownElement);
        }
    }

    // Set active autocomplete item
    function setActiveAutocompleteItem(items, index) {
        items.forEach(item => item.classList.remove('active'));
        items[index].classList.add('active');
        items[index].scrollIntoView({ block: 'nearest' });
    }

    // Hide autocomplete dropdown
    function hideAutocomplete(dropdownElement) {
        dropdownElement.classList.remove('show');
        dropdownElement.innerHTML = '';
    }

    // Escape HTML
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initializeCertificateEmail);
    } else {
        initializeCertificateEmail();
    }

    // Export function to update button state (called from auth module)
    window.updateCertificateEmailButtonState = updateCertificateModalSubmitButton;

    // Test function to download JPG directly from canvas (for debugging)
    window.testDownloadJPG = async function () {
        // Check if authenticated
        if (!goValidAuth.isAuthenticated) {
            alert('⚠️ GoValid Login Required\n\nYou must login to GoValid to download certificates.\n\nClick the "Login to GoValid" button to continue.');
            return;
        }

        if (typeof canvas === 'undefined' || !canvas) {
            alert('Canvas not found. Please make sure the designer is loaded.');
            return;
        }

        // Check GoValid subscription before downloading
        const canGenerate = await checkBeforeSave();
        if (!canGenerate) {
            return;
        }

        try {
            console.log('Testing direct canvas-to-JPG rendering with 3x resolution...');

            const RESOLUTION_MULTIPLIER = 3; // 3x resolution for high quality output
            const maxSize = 5 * 1024 * 1024; // 5MB in bytes
            let jpgQuality = 0.92; // Start with 92% quality
            let attempts = 0;
            const maxAttempts = 5;
            let jpgDataUrl = null;
            let finalSize = 0;
            let finalQuality = 0;

            // Try rendering with decreasing quality until under 5MB
            while (attempts < maxAttempts) {
                jpgDataUrl = canvas.toDataURL({
                    format: 'jpeg',
                    quality: jpgQuality,
                    multiplier: RESOLUTION_MULTIPLIER // Export at 2x resolution
                });

                // Calculate size (base64 is ~33% larger than binary)
                const base64Length = jpgDataUrl.split(',')[1].length;
                const sizeBytes = (base64Length * 3) / 4;
                const sizeMB = sizeBytes / 1024 / 1024;

                console.log(`Attempt ${attempts + 1}: JPG quality ${(jpgQuality * 100).toFixed(0)}%, size ~${sizeMB.toFixed(2)}MB`);

                if (sizeBytes <= maxSize) {
                    finalSize = sizeMB;
                    finalQuality = jpgQuality;
                    console.log(`✓ Certificate JPG generated successfully at ${(jpgQuality * 100).toFixed(0)}% quality, ~${sizeMB.toFixed(2)}MB`);
                    break;
                }

                jpgQuality -= 0.10;
                attempts++;

                if (attempts >= maxAttempts) {
                    alert(`Certificate image too large after ${maxAttempts} attempts. Please simplify your design.`);
                    return;
                }
            }

            if (!jpgDataUrl) {
                alert('Failed to generate JPG image');
                return;
            }

            // Create download link
            const link = document.createElement('a');
            link.download = `certificate_test_${Date.now()}.jpg`;
            link.href = jpgDataUrl;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            alert(`JPG downloaded successfully!\n\nQuality: ${(finalQuality * 100).toFixed(0)}%\nSize: ~${finalSize.toFixed(2)}MB\n\nCheck your downloads folder.`);
            console.log('Test download completed successfully');

        } catch (error) {
            console.error('Error testing JPG download:', error);
            alert('Error testing JPG download: ' + error.message);
        }
    };

})();
