<?php

/**
 * @file plugins/generic/goValidOJS/classes/OJSEmailTemplateIntegration.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class OJSEmailTemplateIntegration
 * @ingroup plugins_generic_goValidOJS
 *
 * @brief Integrates goValid email designer with OJS native email templates
 */

namespace APP\plugins\generic\goValidOJS\classes;

use PKP\core\Core;
use PKP\config\Config;
use PKP\db\DAORegistry;
use APP\facades\Repo;

class OJSEmailTemplateIntegration {

    /** @var object Plugin instance */
    private $plugin;

    /** @var object Context */
    private $context;

    public function __construct($plugin, $context) {
        $this->plugin = $plugin;
        $this->context = $context;
    }

    /**
     * Get all OJS email templates for the current context
     * @return array List of email templates
     */
    public function getOJSEmailTemplates() {
        $templates = [];

        try {
            // Get all email templates using the Collector
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $collector = $emailTemplateRepo->getCollector($this->context->getId());

            $locale = $this->context->getPrimaryLocale();

            // Get templates
            $emailTemplates = $collector->getMany();

            foreach ($emailTemplates as $template) {
                $alternateTo = $template->getData('alternateTo');
                $mailableKey = $alternateTo ?? $template->getData('key'); // Use alternateTo if present, else template key

                // Check if this Mailable supports templates (alternates)
                $supportsTemplates = false;
                $mailRepo = app(\APP\mail\Repository::class);
                $mailable = $mailRepo->get($mailableKey, $this->context);
                if ($mailable) {
                    $supportsTemplates = $mailable::getSupportsTemplates();
                }

                $templates[] = [
                    'key' => $template->getData('key'),
                    'name' => $template->getLocalizedData('name', $locale) ?? $template->getData('key'),
                    'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                    'body' => $template->getLocalizedData('body', $locale) ?? '',
                    'isCustom' => !empty($template->getData('id')),
                    'description' => $template->getLocalizedData('description', $locale) ?? '',
                    'alternateTo' => $alternateTo,
                    'mailableKey' => $mailableKey,
                    'supportsTemplates' => $supportsTemplates
                ];
            }

            error_log("GoValid Plugin: Loaded " . count($templates) . " OJS email templates");

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error loading OJS email templates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return $templates;
    }

    /**
     * Get a specific OJS email template by key
     * @param string $templateKey The email template key
     * @return array|null Template data or null if not found
     */
    public function getOJSEmailTemplate($templateKey) {
        try {
            // Get email template from repository
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $template = $emailTemplateRepo->getByKey($this->context->getId(), $templateKey);

            if ($template) {
                $locale = $this->context->getPrimaryLocale();
                $alternateTo = $template->getData('alternateTo');
                $mailableKey = $alternateTo ?? $template->getData('key');

                // Check if this Mailable supports templates (alternates)
                $supportsTemplates = false;
                $mailRepo = app(\APP\mail\Repository::class);
                $mailable = $mailRepo->get($mailableKey, $this->context);
                if ($mailable) {
                    $supportsTemplates = $mailable::getSupportsTemplates();
                }

                return [
                    'key' => $template->getData('key'),
                    'name' => $template->getLocalizedData('name', $locale) ?? $templateKey,
                    'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                    'body' => $template->getLocalizedData('body', $locale) ?? '',
                    'isCustom' => !empty($template->getData('id')),
                    'description' => $template->getLocalizedData('description', $locale) ?? '',
                    'alternateTo' => $alternateTo,
                    'mailableKey' => $mailableKey,
                    'supportsTemplates' => $supportsTemplates
                ];
            }

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error loading OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return null;
    }

    /**
     * Save/update an OJS email template
     * @param string $templateKey The email template key
     * @param string $subject The email subject
     * @param string $body The email body (HTML)
     * @param string $templateName Optional custom name for new templates
     * @return bool Success status
     */
    public function saveOJSEmailTemplate($templateKey, $subject, $body, $templateName = null, $alternateTo = null) {
        try {
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $locale = $this->context->getPrimaryLocale();

            // Get existing template or create new
            $template = $emailTemplateRepo->getByKey($this->context->getId(), $templateKey);

            if (!$template) {
                // Create new custom template
                $template = app(\PKP\emailTemplate\EmailTemplate::class);
                $template->setData('key', $templateKey);
                $template->setData('contextId', $this->context->getId());

                // Set custom name if provided
                if ($templateName) {
                    $template->setData('name', [$locale => $templateName]);
                }

                // Set alternateTo if provided (makes it appear in native OJS UI)
                if ($alternateTo) {
                    $template->setData('alternateTo', $alternateTo);
                }
            }

            // Update subject and body
            $template->setData('subject', [$locale => $subject]);
            $template->setData('body', [$locale => $body]);

            // Save using repository
            if ($template->getData('id')) {
                $emailTemplateRepo->edit($template, []);
            } else {
                $emailTemplateRepo->add($template);
            }

            error_log("GoValid Plugin: Successfully saved OJS email template '$templateKey'");
            return true;

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error saving OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Reset OJS email template to default
     * @param string $templateKey The email template key
     * @return bool Success status
     */
    public function resetOJSEmailTemplate($templateKey) {
        try {
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $template = $emailTemplateRepo->getByKey($this->context->getId(), $templateKey);

            if ($template && $template->getData('id')) {
                $emailTemplateRepo->delete($template);
                error_log("GoValid Plugin: Successfully reset OJS email template '$templateKey' to default");
                return true;
            }

            error_log("GoValid Plugin: Template '$templateKey' is already default or not found");
            return false;

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error resetting OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Get list of available OJS template variables
     * @param string $templateKey Optional template key to get specific variables
     * @return array List of variables with descriptions
     */
    public function getAvailableVariables($templateKey = null) {
        // Common variables available in most templates
        $commonVars = [
            '{$recipientName}' => 'Recipient\'s full name',
            '{$recipientEmail}' => 'Recipient\'s email address',
            '{$recipientUsername}' => 'Recipient\'s username',
            '{$senderName}' => 'Sender\'s full name',
            '{$senderEmail}' => 'Sender\'s email address',
            '{$contextName}' => 'Journal/Press name',
            '{$contextUrl}' => 'Journal/Press URL',
            '{$contextAcronym}' => 'Journal/Press acronym',
            '{$signature}' => 'Email signature'
        ];

        // Submission-related variables
        $submissionVars = [
            '{$submissionTitle}' => 'Submission title',
            '{$submissionAbstract}' => 'Submission abstract',
            '{$authorName}' => 'Author name',
            '{$authorEmail}' => 'Author email',
            '{$authors}' => 'All authors',
            '{$submissionUrl}' => 'Submission URL'
        ];

        // Review-related variables
        $reviewVars = [
            '{$reviewerName}' => 'Reviewer name',
            '{$reviewDueDate}' => 'Review due date',
            '{$reviewAssignmentUrl}' => 'Review assignment URL'
        ];

        // Editorial decision variables
        $editorialVars = [
            '{$editorName}' => 'Editor name',
            '{$editorContactName}' => 'Editor contact name',
            '{$decisionDescription}' => 'Editorial decision description'
        ];

        // Combine based on template type
        $allVars = array_merge($commonVars, $submissionVars, $reviewVars, $editorialVars);

        return $allVars;
    }

    /**
     * Apply goValid design template to an OJS email template
     * @param string $ojsTemplateKey OJS email template key
     * @param array $designConfig goValid design configuration
     * @return array Result with status and message
     */
    public function applyDesignToOJSTemplate($ojsTemplateKey, $designConfig) {
        try {
            // Get the OJS template
            $ojsTemplate = $this->getOJSEmailTemplate($ojsTemplateKey);

            if (!$ojsTemplate) {
                return ['status' => 'error', 'message' => 'OJS template not found'];
            }

            // Extract the body content (preserve the text)
            $originalBody = $ojsTemplate['body'];

            // Wrap the original content with the design
            $wrappedBody = $this->wrapContentWithDesign($originalBody, $designConfig);

            // Save back to OJS
            $success = $this->saveOJSEmailTemplate(
                $ojsTemplateKey,
                $ojsTemplate['subject'],
                $wrappedBody
            );

            if ($success) {
                return ['status' => 'success', 'message' => 'Design applied successfully'];
            } else {
                return ['status' => 'error', 'message' => 'Failed to save template'];
            }

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error applying design to OJS template: " . $e->getMessage());
            return ['status' => 'error', 'message' => 'Error: ' . $e->getMessage()];
        }
    }

    /**
     * Wrap original content with design elements
     * @param string $originalContent Original email body
     * @param array $designConfig Design configuration
     * @return string Wrapped HTML content
     */
    private function wrapContentWithDesign($originalContent, $designConfig) {
        $headerBgStyle = '';
        if ($designConfig['header']['bgType'] === 'gradient') {
            $headerBgStyle = sprintf(
                'background: linear-gradient(135deg, %s 0%%, %s 100%%);',
                $designConfig['header']['gradientStart'],
                $designConfig['header']['gradientEnd']
            );
        } else {
            $headerBgStyle = 'background: ' . $designConfig['header']['bgColor'] . ';';
        }

        // Generate logo HTML if enabled
        $logoHTML = '';
        if (!empty($designConfig['header']['showLogo'])) {
            $logoPosition = $designConfig['header']['logoPosition'] ?? 'center';
            $showNameBesideLogo = !empty($designConfig['header']['showNameBesideLogo']);
            $logoMaxWidth = $designConfig['header']['logoMaxWidth'] ?? 200;

            // Logo placeholder - match email designer exactly (no margin-bottom, display: block)
            $logoHTML = '<img src="{$journalThumbnailUrl}" alt="{$journalName} Logo" style="max-width: ' . $logoMaxWidth . 'px; height: auto; display: block;">';

            if ($showNameBesideLogo && $logoPosition !== 'center') {
                // Side-by-side layout - match email designer exactly
                $flexDirection = $logoPosition === 'left' ? 'row' : 'row-reverse';
                $headerContent = sprintf(
                    '<div style="display: flex; align-items: center; gap: 20px; flex-direction: %s;">
                        <div style="flex-shrink: 0;">%s</div>
                        <div style="flex: 1; text-align: left;">
                            <h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>
                            %s
                        </div>
                    </div>',
                    $flexDirection,
                    $logoHTML,
                    $designConfig['header']['textColor'],
                    $designConfig['header']['title'] ?? '{$contextName}',
                    !empty($designConfig['header']['subtitle']) ? '<p style="margin: 10px 0 0 0; color: ' . $designConfig['header']['textColor'] . '; font-size: 18px; opacity: 0.9;">' . $designConfig['header']['subtitle'] . '</p>' : ''
                );
            } else {
                // Stacked layout (logo on top) - match email designer
                $headerContent = sprintf(
                    '<div style="text-align: center;">
                        %s
                        <h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>
                        %s
                    </div>',
                    $logoHTML,
                    $designConfig['header']['textColor'],
                    $designConfig['header']['title'] ?? '{$contextName}',
                    !empty($designConfig['header']['subtitle']) ? '<p style="margin: 10px 0 0 0; color: ' . $designConfig['header']['textColor'] . '; font-size: 18px; opacity: 0.9;">' . $designConfig['header']['subtitle'] . '</p>' : ''
                );
            }
        } else {
            // No logo - just title and subtitle
            $headerContent = sprintf(
                '<h1 style="margin: 0; color: %s; font-size: 28px; font-weight: 600;">%s</h1>
                %s',
                $designConfig['header']['textColor'],
                $designConfig['header']['title'] ?? '{$contextName}',
                !empty($designConfig['header']['subtitle']) ? '<p style="margin: 10px 0 0 0; font-size: 18px; opacity: 0.9;">' . $designConfig['header']['subtitle'] . '</p>' : ''
            );
        }

        $html = sprintf('
<div style="font-family: %s; max-width: %spx; margin: 0 auto; background: %s;">
    <!-- Header -->
    <div style="%s color: %s; padding: %spx;">
        %s
    </div>

    <!-- Original Content -->
    <div style="padding: %spx; background: %s; color: %s; line-height: %s; font-size: %spx;">
        %s
    </div>

    <!-- Footer -->
    <div style="background: %s; padding: %spx; text-align: center; border-top: %spx solid %s;">
        <p style="margin: 0 0 10px 0; font-size: %spx; color: %s;">%s</p>
        <p style="margin: 0 0 15px 0; font-size: %spx; color: %s; font-weight: bold;">%s</p>
        %s
    </div>
</div>',
            $designConfig['settings']['fontFamily'],
            $designConfig['settings']['emailWidth'],
            $designConfig['settings']['outerBgColor'],
            $headerBgStyle,
            $designConfig['header']['textColor'],
            $designConfig['header']['padding'],
            $headerContent,
            $designConfig['content']['padding'],
            $designConfig['content']['bgColor'],
            $designConfig['content']['textColor'],
            $designConfig['settings']['lineHeight'],
            $designConfig['settings']['fontSize'],
            $originalContent,
            $designConfig['footer']['bgColor'],
            $designConfig['footer']['padding'],
            $designConfig['footer']['borderWidth'],
            $designConfig['footer']['borderColor'],
            $designConfig['settings']['fontSize'],
            $designConfig['footer']['textColor'],
            $designConfig['footer']['signature'] ?? 'Best regards,',
            $designConfig['settings']['fontSize'],
            $designConfig['footer']['textColor'],
            $designConfig['footer']['signatureName'] ?? '{$contextName} Editorial Team',
            $designConfig['footer']['showUrl'] ? '<p style="margin: 0; font-size: 14px; color: ' . $designConfig['footer']['textColor'] . ';"><a href="{$contextUrl}" style="color: ' . $designConfig['footer']['linkColor'] . '; text-decoration: none;">{$contextUrl}</a></p>' : ''
        );

        return $html;
    }

    /**
     * Get all custom email templates (those created via "Save As" or custom templates with IDs)
     * @return array List of custom email templates
     */
    public function getCustomEmailTemplates() {
        $templates = [];

        try {
            // Get all email templates using the Collector
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $collector = $emailTemplateRepo->getCollector($this->context->getId());

            $locale = $this->context->getPrimaryLocale();

            // Get templates
            $emailTemplates = $collector->getMany();

            foreach ($emailTemplates as $template) {
                // Only include templates that have an ID (custom/saved templates)
                // Templates without IDs are default system templates
                if (!empty($template->getData('id'))) {
                    $key = $template->getData('key');
                    $alternateTo = $template->getData('alternateTo');
                    $mailableKey = $alternateTo ?? $key;

                    // Get created/modified timestamps
                    $createdAt = $template->getData('dateCreated');
                    $modifiedAt = $template->getData('dateModified');

                    $templates[] = [
                        'id' => $template->getData('id'),
                        'key' => $key,
                        'name' => $template->getLocalizedData('name', $locale) ?? $key,
                        'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                        'body' => $template->getLocalizedData('body', $locale) ?? '',
                        'description' => $template->getLocalizedData('description', $locale) ?? '',
                        'alternateTo' => $alternateTo,
                        'mailableKey' => $mailableKey,
                        'created_at' => $createdAt ? date('Y-m-d H:i:s', strtotime($createdAt)) : date('Y-m-d H:i:s'),
                        'modified_at' => $modifiedAt ? date('Y-m-d H:i:s', strtotime($modifiedAt)) : null
                    ];
                }
            }

            // Sort by created date, newest first
            usort($templates, function($a, $b) {
                return strtotime($b['created_at']) - strtotime($a['created_at']);
            });

            error_log("GoValid Plugin: Loaded " . count($templates) . " custom email templates");

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error loading custom email templates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return $templates;
    }

    /**
     * Get a specific custom email template by ID
     * @param int $templateId The email template database ID
     * @return array|null Template data or null if not found
     */
    public function getCustomEmailTemplate($templateId) {
        try {
            // Get email template from repository by ID
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);

            // Get all templates for this context and find by ID
            $collector = $emailTemplateRepo->getCollector($this->context->getId());
            $emailTemplates = $collector->getMany();

            foreach ($emailTemplates as $template) {
                if ($template->getData('id') == $templateId) {
                    $locale = $this->context->getPrimaryLocale();
                    $key = $template->getData('key');
                    $alternateTo = $template->getData('alternateTo');
                    $mailableKey = $alternateTo ?? $key;

                    // Parse the template body to extract design configuration
                    $body = $template->getLocalizedData('body', $locale) ?? '';
                    $design = $this->extractDesignFromBody($body);

                    return [
                        'id' => $template->getData('id'),
                        'key' => $key,
                        'name' => $template->getLocalizedData('name', $locale) ?? $key,
                        'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                        'body' => $body,
                        'description' => $template->getLocalizedData('description', $locale) ?? '',
                        'alternateTo' => $alternateTo,
                        'mailableKey' => $mailableKey,
                        'design' => $design,
                        'greeting' => $design['greeting'] ?? '',
                        'message' => $design['message'] ?? '',
                        'article_details' => $design['article_details'] ?? '',
                        'closing' => $design['closing'] ?? ''
                    ];
                }
            }

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error loading custom email template ID '$templateId': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return null;
    }

    /**
     * Extract design configuration from email body HTML
     * This attempts to parse the designed email to extract settings
     * @param string $body Email body HTML
     * @return array Design configuration
     */
    private function extractDesignFromBody($body) {
        $design = [
            'greeting' => '',
            'message' => '',
            'article_details' => '',
            'closing' => ''
        ];

        // Check if this body was created by wrapContentWithDesign (has our wrapper structure)
        // Look for the "Original Content" section comment
        if (preg_match('/<!-- Original Content -->(.*?)<!-- Footer -->/s', $body, $contentMatch)) {
            // Extract the content section HTML
            $contentSection = $contentMatch[1];

            // Remove the wrapping div with styles
            if (preg_match('/<div[^>]*>(.*?)<\/div>\s*$/s', $contentSection, $innerMatch)) {
                $innerContent = $innerMatch[1];

                // Now extract individual paragraphs as greeting, message, closing
                $paragraphs = [];
                if (preg_match_all('/<p[^>]*>(.*?)<\/p>/s', $innerContent, $pMatches)) {
                    foreach ($pMatches[1] as $p) {
                        $text = strip_tags(str_replace('<br />', "\n", str_replace('<br>', "\n", $p)));
                        if (trim($text)) {
                            $paragraphs[] = $text;
                        }
                    }
                }

                // Extract article details div if exists
                if (preg_match('/<div[^>]*background[^>]*>(.*?)<\/div>/s', $innerContent, $detailsMatch)) {
                    $design['article_details'] = strip_tags(str_replace('<br />', "\n", str_replace('<br>', "\n", $detailsMatch[1])));
                }

                // Assign paragraphs to greeting, message, closing
                if (count($paragraphs) > 0) {
                    $design['greeting'] = $paragraphs[0];
                }
                if (count($paragraphs) > 1) {
                    // Last paragraph is closing, middle ones are message
                    $design['closing'] = array_pop($paragraphs);
                    if (count($paragraphs) > 1) {
                        array_shift($paragraphs); // Remove greeting
                        $design['message'] = implode("\n\n", $paragraphs);
                    }
                } elseif (count($paragraphs) == 1) {
                    // Only one paragraph, treat as message
                    $design['message'] = $paragraphs[0];
                    $design['greeting'] = '';
                }
            }
        } else {
            // Not a wrapped template, treat entire body as message
            $design['message'] = strip_tags(str_replace('<br />', "\n", str_replace('<br>', "\n", $body)));
        }

        return $design;
    }

    /**
     * Save a custom email template to OJS
     * @param array $templateData Template data with keys: key, name, subject, body, description, designConfig
     * @return array Result with success status and message
     */
    public function saveCustomTemplate($templateData) {
        try {
            $emailTemplateRepo = app(\PKP\emailTemplate\Repository::class);
            $locale = $this->context->getPrimaryLocale();

            // Check if template with this key already exists
            $collector = $emailTemplateRepo->getCollector($this->context->getId());
            $existingTemplates = $collector->getMany();

            $existingTemplate = null;
            foreach ($existingTemplates as $template) {
                if ($template->getData('key') === $templateData['key']) {
                    $existingTemplate = $template;
                    break;
                }
            }

            if ($existingTemplate) {
                // Update existing template
                $emailTemplateRepo->edit($existingTemplate, [
                    'name' => [$locale => $templateData['name']],
                    'subject' => [$locale => $templateData['subject']],
                    'body' => [$locale => $templateData['body']],
                    'description' => [$locale => $templateData['description']]
                ]);

                error_log("GoValid Plugin: Updated existing template with key: {$templateData['key']}");
                return ['success' => true, 'message' => 'Template updated successfully'];

            } else {
                // Create new template
                $newTemplate = $emailTemplateRepo->newDataObject([
                    'key' => $templateData['key'],
                    'contextId' => $this->context->getId(),
                    'name' => [$locale => $templateData['name']],
                    'subject' => [$locale => $templateData['subject']],
                    'body' => [$locale => $templateData['body']],
                    'description' => [$locale => $templateData['description']],
                    'enabled' => true
                ]);

                $emailTemplateRepo->add($newTemplate);

                error_log("GoValid Plugin: Created new template with key: {$templateData['key']}");
                return ['success' => true, 'message' => 'Template created successfully'];
            }

        } catch (\Exception $e) {
            error_log("GoValid Plugin: Error saving custom template: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}
