    function loadTemplateList() {
        console.log('Loading template list...');
        // Use clean AJAX URL
        const url = buildActionURL('list');
        
        fetch(url, {
            method: 'GET'
        })
        .then(response => {
            console.log('Response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Received data:', data);
            const templateList = document.getElementById('templateList');
            const templateCount = document.getElementById('templateCount');
            
            if (!templateList) {
                console.warn('templateList element not found - templates feature disabled');
                return;
            }
            
            // Handle OJS JSONMessage format
            const result = data.content || data;
            const templates = result.templates || [];
            
            // Update template count display
            if (templateCount) {
                templateCount.textContent = templates.length + '/10';
                templateCount.style.color = templates.length >= 10 ? '#dc3545' : '#333';
            }
            
            if (templates.length > 0) {
                let html = '';
                templates.forEach(template => {
                    html += '<div class="template-item">';
                    html += '<span>' + template.name + '</span>';
                    html += '<div class="template-actions">';
                    html += '<button class="template-btn load" onclick="loadTemplate(\'' + template.name + '\')">Load</button>';
                    html += '<button class="template-btn delete" onclick="deleteTemplate(\'' + template.name + '\')">Delete</button>';
                    html += '</div>';
                    html += '</div>';
                });
                templateList.innerHTML = html;
                console.log('Templates loaded:', templates.length);
            } else {
                templateList.innerHTML = '<p style="padding: 10px; margin: 0; font-size: 11px; color: #666;">No saved templates</p>';
                console.log('No templates found');
            }
        })
        .catch(error => {
            console.error('Error loading templates:', error);
            const templateList = document.getElementById('templateList');
            templateList.innerHTML = '<p style="padding: 10px; margin: 0; font-size: 11px; color: #f00;">Error loading templates</p>';
        });
    }
    
    function loadTemplate(templateName) {
        if (confirm('Load template "' + templateName + '"? This will replace the current design.')) {
            fetch(getAjaxURL(), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=load&templateName=' + encodeURIComponent(templateName)
            })
            .then(response => response.json())
            .then(data => {
                // Handle OJS JSONMessage format
                const result = data.content || data;
                if (result.templateData) {
                    // Set orientation first
                    if (result.orientation && result.orientation !== currentOrientation) {
                        document.getElementById(result.orientation).checked = true;
                        changeOrientation(result.orientation);
                    }
                    
                    // Then load the template with error handling
                    try {
                        const cleanedData = cleanTemplateData(result.templateData);
                        canvas.loadFromJSON(cleanedData, function() {
                            // Fix any invalid textBaseline values for all text objects
                            canvas.getObjects('text').forEach(function(obj) {
                                fixTextBaseline(obj);
                            });
                            
                            // Restore background image controls if background image exists
                            restoreBackgroundControls();
                            
                            canvas.renderAll();
                            document.getElementById('templateName').value = templateName;
                        });
                    } catch (error) {
                        console.error('Error loading template:', error);
                        alert('Error loading template: ' + error.message);
                    }
                }
            })
            .catch(error => {
                alert('Error loading template');
            });
        }
    }
    
    function deleteTemplate(templateName) {
        if (confirm('Delete template "' + templateName + '"? This cannot be undone.')) {
            fetch(getAjaxURL(), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=delete&templateName=' + encodeURIComponent(templateName)
            })
            .then(response => response.json())
            .then(data => {
                loadTemplateList();
            })
            .catch(error => {
                alert('Error deleting template');
            });
        }
    }
    
    // Template buttons removed - functionality moved to Load Template button in toolbar
    
    // Load journal logo automatically using PHP-provided data
    function loadJournalLogo() {
        console.log('Loading journal logo...');

        // Use journal information provided by global variable set in template
        var journalInfo = window.goValidJournalInfo || {
            name: 'Open Journal Systems',
            logo: null
        };

        console.log('Journal info:', journalInfo);
        
        if (journalInfo.logo) {
            console.log('Found logo URL:', journalInfo.logo);
            displayJournalLogo(journalInfo.logo, journalInfo.name);
        } else {
            console.log('No logo found, displaying journal name:', journalInfo.name);
            displayJournalName(journalInfo.name);
        }
    }
    
    function displayJournalLogo(logoUrl, journalName) {
        const journalLogoDiv = document.getElementById('journalLogo');  // Left side in HTML
        const journalNameDiv = document.getElementById('journalName');  // Right side in HTML
        console.log('displayJournalLogo called with:', journalName);
        console.log('journalNameDiv element:', journalNameDiv);

        // Check if user has Premium plan (Ultimate or Business)
        const isPremium = isPremiumPlan();
        console.log('isPremium:', isPremium);

        if (isPremium) {
            // For Premium users (Ultimate/Business): Swap positions
            // Put NAME on LEFT (journalLogo div), LOGO on RIGHT (journalName div)
            journalLogoDiv.innerHTML = '<div style="font-size: 16px; font-weight: 600; color: #333;">' + journalName + '</div>';
            journalNameDiv.innerHTML = '<img src="' + logoUrl + '" alt="' + journalName + '" style="max-width: 180px; max-height: 100px; object-fit: contain;" title="' + journalName + '">';
            console.log('Premium user: Journal name on left, logo on right (swapped)');
        } else {
            // For Free/Starter users: Keep original - LOGO on LEFT, NAME on RIGHT
            journalLogoDiv.innerHTML = '<img src="' + logoUrl + '" alt="' + journalName + '" style="max-width: 180px; max-height: 100px; object-fit: contain;" title="' + journalName + '">';
            if (journalNameDiv) {
                journalNameDiv.textContent = journalName;
                console.log('Free/Starter user: Logo on left, name on right (original)');
            } else {
                console.error('journalNameDiv not found!');
            }
        }
    }

    function displayJournalName(journalName) {
        const journalLogoDiv = document.getElementById('journalLogo');
        const journalNameDiv = document.getElementById('journalName');
        console.log('displayJournalName called with:', journalName);
        console.log('journalNameDiv element:', journalNameDiv);
        // Show placeholder when no logo
        journalLogoDiv.innerHTML = '<div style="width: 180px; height: 80px; background: #f0f4ff; border: 2px dashed #667eea; border-radius: 8px; display: flex; align-items: center; justify-content: center; color: #667eea; font-size: 12px;"><i class="fa fa-image"></i></div>';
        if (journalNameDiv) {
            journalNameDiv.textContent = journalName;
            console.log('Journal name set to:', journalNameDiv.textContent);
        } else {
            console.error('journalNameDiv not found!');
        }
    }
    
    // Handle orientation change
    function handleOrientationChange() {
        const portraitRadio = document.getElementById('portrait');
        const landscapeRadio = document.getElementById('landscape');
        
        portraitRadio.addEventListener('change', function() {
            if (this.checked) {
                changeOrientation('portrait');
            }
        });
        
        landscapeRadio.addEventListener('change', function() {
            if (this.checked) {
                changeOrientation('landscape');
            }
        });
    }
    
    function changeOrientation(orientation) {
        currentOrientation = orientation;
        
        // Save current canvas state
        const canvasData = canvas.toJSON();
        
        // Update canvas dimensions with zoom
        const width = orientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
        const height = orientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
        
        canvas.setWidth(width * currentZoom);
        canvas.setHeight(height * currentZoom);
        canvas.setZoom(currentZoom);
        
        // Reload canvas content
        canvas.loadFromJSON(canvasData, function() {
            canvas.renderAll();
            // Auto-fit after orientation change
            setTimeout(autoFitCanvas, 100);
        });
    }
    
    // Zoom functionality
    function handleZoom() {
        const zoomInBtn = document.getElementById('zoomIn');
        const zoomOutBtn = document.getElementById('zoomOut');
        const zoomFitBtn = document.getElementById('zoomFit');
        const zoomLevelSpan = document.getElementById('zoomLevel');
        
        function updateZoomDisplay() {
            zoomLevelSpan.textContent = Math.round(currentZoom * 100) + '%';
            
            // Apply zoom to canvas
            canvas.setZoom(currentZoom);
            const width = currentOrientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
            const height = currentOrientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
            canvas.setWidth(width * currentZoom);
            canvas.setHeight(height * currentZoom);
            canvas.renderAll();
            
            // Enable/disable zoom buttons based on limits
            zoomInBtn.disabled = currentZoom >= maxZoom;
            zoomOutBtn.disabled = currentZoom <= minZoom;
        }
        
        zoomInBtn.addEventListener('click', function() {
            if (currentZoom < maxZoom) {
                currentZoom = Math.min(currentZoom + zoomStep, maxZoom);
                updateZoomDisplay();
            }
        });
        
        zoomOutBtn.addEventListener('click', function() {
            if (currentZoom > minZoom) {
                currentZoom = Math.max(currentZoom - zoomStep, minZoom);
                updateZoomDisplay();
            }
        });
        
        zoomFitBtn.addEventListener('click', function() {
            autoFitCanvas();
        });
        
        // Keyboard shortcuts for zoom
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey || e.metaKey) {
                if (e.key === '=' || e.key === '+') {
                    e.preventDefault();
                    zoomInBtn.click();
                } else if (e.key === '-' || e.key === '_') {
                    e.preventDefault();
                    zoomOutBtn.click();
                } else if (e.key === '0') {
                    e.preventDefault();
                    currentZoom = 1.0;
                    updateZoomDisplay();
                }
            }
        });
        
        // Mouse wheel zoom
        const canvasContainer = document.querySelector('.canvas-area');
        canvasContainer.addEventListener('wheel', function(e) {
            if (e.ctrlKey || e.metaKey) {
                e.preventDefault();
                const delta = e.deltaY < 0 ? 1 : -1;
                const newZoom = currentZoom + (delta * 0.1);
                
                if (newZoom >= minZoom && newZoom <= maxZoom) {
                    currentZoom = Math.round(newZoom * 10) / 10;
                    updateZoomDisplay();
                }
            }
        });
    }
    
    // Reviewer autocomplete functionality
    function setupReviewerAutocomplete() {
        const input = document.getElementById('preview_reviewer_name');
        const dropdown = document.getElementById('reviewerDropdown');
        const loadBtn = document.getElementById('loadReviewers');
        
        // Check if elements exist
        if (!input || !dropdown || !loadBtn) {
            console.warn('Reviewer autocomplete elements not found - feature disabled');
            return;
        }
        
        let reviewerData = [];
        
        // Load reviewers button
        loadBtn.addEventListener('click', function() {
            loadReviewers();
        });
        
        // Search as you type
        let searchTimeout;
        input.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            const searchTerm = this.value.trim();
            
            if (searchTerm.length < 2) {
                dropdown.style.display = 'none';
                return;
            }
            
            searchTimeout = setTimeout(() => {
                searchReviewers(searchTerm);
            }, 300);
        });
        
        // Hide dropdown on click outside
        document.addEventListener('click', function(e) {
            if (!input.contains(e.target) && !dropdown.contains(e.target)) {
                dropdown.style.display = 'none';
            }
        });
        
        function loadReviewers(search = '') {
            const url = buildActionURL('getReviewers') + (search ? '&search=' + encodeURIComponent(search) : '');
            
            fetch(url)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.json();
                })
                .then(data => {
                    const result = data.content || data;
                    if (result.reviewers && Array.isArray(result.reviewers)) {
                        reviewerData = result.reviewers;
                        displayReviewers(result.reviewers);
                    } else if (result.success === false) {
                        console.error('Server error:', result.message);
                        displayReviewers([]); // Show empty list
                    } else {
                        console.warn('No reviewers found in response');
                        displayReviewers([]);
                    }
                })
                .catch(error => {
                    console.error('Error loading reviewers:', error);
                    displayReviewers([]);
                });
        }
        
        function searchReviewers(searchTerm) {
            // First search in loaded data
            if (reviewerData.length > 0) {
                const filtered = reviewerData.filter(r => 
                    r.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                    r.email.toLowerCase().includes(searchTerm.toLowerCase())
                );
                displayReviewers(filtered);
            } else {
                // Load from server with search
                loadReviewers(searchTerm);
            }
        }
        
        function displayReviewers(reviewers) {
            if (reviewers.length === 0) {
                dropdown.innerHTML = '<div style="padding: 10px; color: #666;">No reviewers found</div>';
                dropdown.style.display = 'block';
                return;
            }
            
            dropdown.innerHTML = reviewers.map(reviewer => `
                <div class="reviewer-item" style="padding: 10px; cursor: pointer; border-bottom: 1px solid #f0f0f0;" 
                     data-name="${reviewer.name}" 
                     data-email="${reviewer.email}">
                    <div style="font-weight: 600; color: #333;">${reviewer.name}</div>
                    <div style="font-size: 12px; color: #666;">${reviewer.email}</div>
                    ${reviewer.affiliation ? `<div style="font-size: 11px; color: #999;">${reviewer.affiliation}</div>` : ''}
                </div>
            `).join('');
            
            // Add click handlers
            dropdown.querySelectorAll('.reviewer-item').forEach(item => {
                item.addEventListener('click', function() {
                    input.value = this.dataset.name;
                    dropdown.style.display = 'none';
                });
                
                item.addEventListener('mouseenter', function() {
                    this.style.background = '#f5f5f5';
                });
                
                item.addEventListener('mouseleave', function() {
                    this.style.background = 'white';
                });
            });
            
            dropdown.style.display = 'block';
        }
    }
    
    // Preview functionality
    function setupPreview() {
        const previewBtn = document.getElementById('previewCertificate');
        const modal = document.getElementById('previewModal');

        // Setup reviewer autocomplete
        setupReviewerAutocomplete();

        // Setup Generate Certificate button (removed from UI but may still exist in modal)
        const generateCertBtn = document.getElementById('generateCertificate');
        if (generateCertBtn) {
            generateCertBtn.addEventListener('click', function() {
                generateFinalCertificate();
            });
        }

        // Skip if preview button doesn't exist (it was removed from UI)
        if (!previewBtn || !modal) {
            return;
        }

        const closeBtn = modal.querySelector('.preview-close');
        const cancelBtn = document.getElementById('closePreview');
        const generateBtn = document.getElementById('generatePreview');
        const statusDiv = document.getElementById('previewStatus');
        const previewSetup = document.getElementById('previewSetup');
        const previewDisplay = document.getElementById('previewDisplay');
        const previewFrame = document.getElementById('previewFrame');
        const downloadBtn = document.getElementById('downloadPreview');
        const backBtn = document.getElementById('backToSetup');

        // Open modal
        previewBtn.addEventListener('click', function() {
            // Update preview info with current selections
            const issueSelect = document.getElementById('content_issue_select');
            const articleSelect = document.getElementById('content_article_select');
            const reviewerInput = document.getElementById('content_reviewer_name');
            
            // Update issue info
            const issueInfo = document.getElementById('previewIssueInfo');
            if (issueSelect && issueSelect.value && window.ojsIssuesData) {
                const selectedIssue = window.ojsIssuesData[issueSelect.value];
                if (selectedIssue) {
                    issueInfo.textContent = selectedIssue.title || `Vol. ${selectedIssue.volume} No. ${selectedIssue.number} (${selectedIssue.year})`;
                } else {
                    issueInfo.innerHTML = '<em style="color: #e53e3e;">No issue selected</em>';
                }
            } else {
                issueInfo.innerHTML = '<em style="color: #e53e3e;">No issue selected</em>';
            }
            
            // Update article info
            const articleInfo = document.getElementById('previewArticleInfo');
            if (articleSelect && articleSelect.value && window.ojsArticlesData) {
                const selectedArticle = window.ojsArticlesData[articleSelect.value];
                if (selectedArticle) {
                    articleInfo.textContent = selectedArticle.title || 'Untitled Article';
                } else {
                    articleInfo.innerHTML = '<em style="color: #e53e3e;">No article selected</em>';
                }
            } else {
                articleInfo.innerHTML = '<em style="color: #e53e3e;">No article selected</em>';
            }
            
            // Update reviewer info
            const reviewerInfo = document.getElementById('previewReviewerInfo');
            if (reviewerInput && reviewerInput.value) {
                reviewerInfo.textContent = reviewerInput.value;
            } else {
                reviewerInfo.innerHTML = '<em style="color: #e53e3e;">No reviewer name entered</em>';
            }
            
            modal.style.display = 'block';
        });
        
        // Close modal
        function closeModal() {
            modal.style.display = 'none';
            statusDiv.style.display = 'none';
            previewSetup.style.display = 'block';
            previewDisplay.style.display = 'none';
            
            // Clean up preview URL
            if (window.currentPreviewUrl) {
                URL.revokeObjectURL(window.currentPreviewUrl);
                window.currentPreviewUrl = null;
            }
            previewFrame.src = '';
        }
        
        // Show setup section
        function showSetup() {
            previewSetup.style.display = 'block';
            previewDisplay.style.display = 'none';
            statusDiv.style.display = 'none';
        }
        
        // Show preview section
        function showPreview() {
            previewSetup.style.display = 'none';
            previewDisplay.style.display = 'block';
        }
        
        closeBtn.addEventListener('click', closeModal);
        cancelBtn.addEventListener('click', closeModal);
        backBtn.addEventListener('click', showSetup);
        
        // Download preview button
        downloadBtn.addEventListener('click', function() {
            if (window.currentPreviewUrl) {
                const a = document.createElement('a');
                a.href = window.currentPreviewUrl;
                a.download = 'certificate_preview.pdf';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
            }
        });
        
        // Close on outside click
        window.addEventListener('click', function(e) {
            if (e.target === modal) {
                closeModal();
            }
        });
        
        // Generate preview
        generateBtn.addEventListener('click', async function() {
            // Validate required data is selected
            const issueSelect = document.getElementById('content_issue_select');
            const articleSelect = document.getElementById('content_article_select');
            
            if (!issueSelect?.value) {
                statusDiv.style.display = 'block';
                statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Please select an issue in the Content tab first.</p>';
                return;
            }
            
            if (!articleSelect?.value) {
                statusDiv.style.display = 'block';
                statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Please select an article in the Content tab first.</p>';
                return;
            }
            
            // Check GoValid subscription before generating preview
            const canGenerate = await checkBeforeSave();
            if (!canGenerate) {
                statusDiv.style.display = 'block';
                statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Unable to generate preview. Please check your subscription status.</p>';
                return;
            }
            
            statusDiv.style.display = 'block';
            statusDiv.innerHTML = '<p style="color: #666;"><i class="fa fa-spinner fa-spin"></i> Generating preview...</p>';
            
            // Get real data from Content tab selections
            const issueVolume = document.getElementById('content_issue_volume')?.value || '';
            const issueNumber = document.getElementById('content_issue_number')?.value || '';
            const issueYear = document.getElementById('content_issue_year')?.value || '';
            const issueTitle = document.getElementById('content_issue_title')?.value || '';
            const articleTitle = document.getElementById('content_article_title')?.value || '';
            const articleDOI = document.getElementById('content_article_doi')?.value || '';
            const submissionId = document.getElementById('content_submission_id')?.value || '';
            
            // Get reviewer name from input field if available
            const reviewerNameInput = document.getElementById('content_reviewer_name');
            const reviewerName = reviewerNameInput?.value || 'John Doe';
            
            // Get journal name from the page
            let journalName = 'Journal Name'; // fallback

            // Get from window.goValidJournalInfo which is set by PHP template
            if (window.goValidJournalInfo && window.goValidJournalInfo.name && window.goValidJournalInfo.name !== 'Journal Name') {
                journalName = window.goValidJournalInfo.name;
            } else {
                // Try to get from content field as backup
                const journalInput = document.getElementById('content_journal_name');
                if (journalInput && journalInput.value && journalInput.value.trim() !== '') {
                    journalName = journalInput.value.trim();
                }
            }
            
            console.log('Journal name for preview:', journalName);

            // Get QR settings from Tab 2
            const qrIdentifierValue = document.getElementById('qr_identifier_code')?.value;
            const qrIdentifier = qrIdentifierValue && qrIdentifierValue.trim() !== '' ? qrIdentifierValue.trim() : '{QR_IDENTIFIER}';

            // Check if date toggles are enabled before getting values
            const qrStartDateToggle = document.getElementById('qr_start_date_toggle')?.checked;
            const qrEndDateToggle = document.getElementById('qr_end_date_toggle')?.checked;
            const qrStartDate = qrStartDateToggle ? document.getElementById('qr_start_date')?.value : null;
            const qrEndDate = qrEndDateToggle ? document.getElementById('qr_end_date')?.value : null;

            // Build replacements with real data or fallbacks
            const replacements = {
                '{REVIEWER_NAME}': reviewerName,
                '{JOURNAL_NAME}': journalName,
                '{REVIEW_DATE}': new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }),
                '{ARTICLE_TITLE}': articleTitle || 'Please select an article',
                '{ARTICLE_DOI}': articleDOI || 'No DOI available',
                '{ISSUE_TITLE}': issueTitle || 'Please select an issue',
                '{ISSUE_VOLUME}': issueVolume || 'Vol',
                '{ISSUE_NUMBER}': issueNumber || 'No',
                '{ISSUE_YEAR}': issueYear || new Date().getFullYear().toString(),
                '{SUBMISSION_ID}': submissionId || 'No ID',
                '{QR_IDENTIFIER}': qrIdentifier,
                '{QR_ISSUE_DATE}': qrStartDate ? new Date(qrStartDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
                '{QR_EXPIRE_DATE}': qrEndDate ? new Date(qrEndDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })
            };
            
            console.log('All replacements for preview:', replacements);
            
            // Generate QR code if authentication is available and QR code is in template
            const canvasJson = canvas.toJSON();
            const templateString = JSON.stringify(canvasJson);
            console.log('=== QR CODE DEBUG ===');
            
            // Check for QR_CODE in different ways
            let hasQRCode = false;
            if (canvasJson.objects) {
                canvasJson.objects.forEach((obj, i) => {
                    // Check text objects
                    if (obj.type === 'text' && obj.text) {
                        console.log(`Text object ${i}: "${obj.text}"`);
                        if (obj.text.includes('{QR_CODE}') || obj.text.includes('QR_CODE') || 
                            obj.text.trim() === '{QR_CODE}' || obj.text.trim() === 'QR_CODE') {
                            hasQRCode = true;
                            console.log('Found QR_CODE placeholder in text object', i);
                        }
                    }
                });
            }
            
            console.log('Template contains {QR_CODE}?', hasQRCode);
            console.log('Template string search:', templateString.includes('{QR_CODE}'));
            console.log('GoValid authenticated?', goValidAuth.isAuthenticated);
            
            // For preview, always show a QR placeholder instead of generating real QR code
            if (hasQRCode) {
                if (goValidAuth.isAuthenticated) {
                    console.log('Adding QR placeholder for preview...');
                    // Create a simple QR placeholder image data URL
                    replacements['{QR_CODE}'] = createQRPlaceholder();
                    console.log('QR placeholder added to replacements');
                } else {
                    console.log('Adding authentication message for QR...');
                    replacements['{QR_CODE}'] = '[QR Code: Please authenticate with GoValid]';
                }
            } else {
                console.log('No QR_CODE placeholder found in template');
            }
            
            console.log('Final replacements object:', replacements);
            console.log('=== END QR DEBUG ===');
            
            statusDiv.innerHTML = '<p style="color: #666;"><i class="fa fa-spinner fa-spin"></i> Generating PDF preview...</p>';
            
            // Prepare canvas data (reuse existing canvasJson variable)
            console.log('=== CANVAS DATA DEBUG ===');
            console.log('Canvas JSON keys:', Object.keys(canvasJson));
            console.log('Has backgroundImage?', 'backgroundImage' in canvasJson);
            console.log('Has objects?', 'objects' in canvasJson);
            if (canvasJson.objects) {
                console.log('Number of objects:', canvasJson.objects.length);
                canvasJson.objects.forEach((obj, i) => {
                    if (obj.type === 'image') {
                        console.log(`Image object ${i}:`, {
                            type: obj.type,
                            left: obj.left,
                            top: obj.top,
                            width: obj.width,
                            height: obj.height,
                            scaleX: obj.scaleX,
                            scaleY: obj.scaleY,
                            src_preview: obj.src ? obj.src.substring(0, 50) + '...' : 'no src',
                            isBackground: obj.isBackground
                        });
                    }
                });
            }
            if (canvasJson.backgroundImage) {
                console.log('Background image found:', {
                    src_preview: canvasJson.backgroundImage.src ? canvasJson.backgroundImage.src.substring(0, 50) + '...' : 'no src'
                });
            }
            console.log('=== END CANVAS DEBUG ===');
            
            const canvasData = JSON.stringify(canvasJson);
            
            // Get current security level for dynamic QR sizing
            const currentSecurityLevel = getQRSettings().security_level || 'secure';
            
            // Send preview request
            fetch(getAjaxURL(), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=preview' +
                    '&canvasData=' + encodeURIComponent(canvasData) +
                    '&replacements=' + encodeURIComponent(JSON.stringify(replacements)) +
                    '&orientation=' + currentOrientation +
                    '&security_level=' + encodeURIComponent(currentSecurityLevel)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Preview generation failed');
                }
                return response.blob();
            })
            .then(blob => {
                // Create object URL for the PDF
                const url = URL.createObjectURL(blob);
                
                // Store the blob URL for download button
                window.currentPreviewUrl = url;
                
                // Show PDF in iframe
                previewFrame.src = url;
                
                // Switch to preview display
                showPreview();
                
                statusDiv.innerHTML = '<p style="color: #48bb78;"><i class="fa fa-check"></i> Preview generated successfully!</p>';
            })
            .catch(error => {
                console.error('Preview error:', error);
                statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Error generating preview. Please try again.</p>';
            });
        });
    }
    
    // Auto-fit canvas to container
    function autoFitCanvas() {
        const container = document.querySelector('.canvas-area');
        const containerWidth = container.clientWidth - 80; // Subtract padding
        const containerHeight = container.clientHeight - 80;
        
        const canvasWidth = currentOrientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
        const canvasHeight = currentOrientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
        
        const scaleX = containerWidth / canvasWidth;
        const scaleY = containerHeight / canvasHeight;
        
        currentZoom = Math.min(scaleX, scaleY, maxZoom);
        currentZoom = Math.max(currentZoom, minZoom);
        currentZoom = Math.round(currentZoom * 4) / 4; // Round to nearest 0.25
        
        // Apply zoom
        const zoomLevelSpan = document.getElementById('zoomLevel');
        if (zoomLevelSpan) {
            zoomLevelSpan.textContent = Math.round(currentZoom * 100) + '%';
        }
        
        canvas.setZoom(currentZoom);
        canvas.setWidth(canvasWidth * currentZoom);
        canvas.setHeight(canvasHeight * currentZoom);
        canvas.renderAll();
        
        // Update zoom button states
        const zoomInBtn = document.getElementById('zoomIn');
        const zoomOutBtn = document.getElementById('zoomOut');
        if (zoomInBtn && zoomOutBtn) {
            zoomInBtn.disabled = currentZoom >= maxZoom;
            zoomOutBtn.disabled = currentZoom <= minZoom;
        }
    }
    
    // Setup authentication event listeners
    // Tab functionality
    function setupTabs() {
        const tabButtons = document.querySelectorAll('.tab-btn');
        const tabContents = document.querySelectorAll('.tab-content');
        
        if (tabButtons.length === 0) {
            console.warn('No tab buttons found - tab functionality disabled');
            return;
        }
        
        // Fix tab content IDs to match the new tab order
        function fixTabContentIds() {
            // Get all tab content sections
            const allTabContents = document.querySelectorAll('.tab-content');
            
            // Find the Properties tab (already correct)
            const propertiesTab = document.getElementById('properties-tab');
            
            // Find the Content tab (should be the one with reviewer name field)
            let contentTab = null;
            allTabContents.forEach(tab => {
                if (tab.querySelector('#content_reviewer_name')) {
                    contentTab = tab;
                    tab.id = 'content-tab';
                    
                    // Make sure the heading is correct
                    const heading = tab.querySelector('h4');
                    if (heading) {
                        heading.textContent = 'Certificate Content';
                    }
                }
            });
            
            // Find the QR Settings tab (should be the one with qr_security_level)
            let qrSettingsTab = null;
            allTabContents.forEach(tab => {
                if (tab.querySelector('#qr_security_level')) {
                    qrSettingsTab = tab;
                    tab.id = 'qr-settings-tab';

                    // Make sure the heading is correct
                    const heading = tab.querySelector('h4');
                    if (heading) {
                        heading.textContent = 'QR Content';
                    }
                }
            });
            
            console.log('Tab content IDs and headings fixed:', {
                properties: propertiesTab?.id,
                content: contentTab?.id,
                qrSettings: qrSettingsTab?.id
            });
        }
        
        // Fix tab content IDs on page load
        fixTabContentIds();
        
        tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');
                const targetContent = document.getElementById(targetTab + '-tab');
                
                if (!targetContent) {
                    console.warn(`Tab content ${targetTab}-tab not found`);
                    return;
                }
                
                // Update active states
                tabButtons.forEach(btn => btn.classList.remove('active'));
                tabContents.forEach(content => {
                    content.classList.remove('active');
                });
                
                this.classList.add('active');
                targetContent.classList.add('active');
            });
        });

        // Function to programmatically switch tabs
        window.switchToTab = function(targetTab) {
            const tabButtons = document.querySelectorAll('.tab-btn');
            const tabContents = document.querySelectorAll('.tab-content');
            const targetContent = document.getElementById(targetTab + '-tab');

            if (!targetContent) {
                console.warn(`Tab content ${targetTab}-tab not found`);
                return;
            }

            // Update active states
            tabButtons.forEach(btn => btn.classList.remove('active'));
            tabContents.forEach(content => content.classList.remove('active'));

            // Find and activate the button
            const targetButton = document.querySelector(`[data-tab="${targetTab}"]`);
            if (targetButton) {
                targetButton.classList.add('active');
            }
            targetContent.classList.add('active');

            // When switching to QR Content tab, trigger selectAwardedName to populate the field
            if (targetTab === 'qr-settings' && typeof selectAwardedName === 'function') {
                selectAwardedName();
            }
        };

        // Also add click handler to tab buttons to trigger selectAwardedName
        tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');
                if (targetTab === 'qr-settings' && typeof selectAwardedName === 'function') {
                    // Small delay to ensure tab content is visible first
                    setTimeout(selectAwardedName, 50);
                }
            });
        });
    }

    // QR Settings functionality
    let qrSettings = {
        security_level: 'SMART',
        qr_name: '',
        generated_by: '',
        signed_by: '',
        signed_by_enabled: false,
        identifier: '',
        awarded_to: '{REVIEWER_NAME}',
        mask_name: false,
        purpose: 'Peer Review Recognition',
        purpose_enabled: true,
        description: '',
        description_enabled: true,
        start_date: '',
        end_date: '',
        password_protection: false,
        password_type: 'master',
        password: '',
        formatted_uuid: '',  // QR ID from API response (e.g., 6ACE-CE-477e51db)
        anti_counterfeit_tags: '',
        anti_counterfeit_enabled: false
    };
    
    function setupQRSettings() {
        // Load saved settings from local storage
        const savedSettings = localStorage.getItem('govalidQRSettings');
        if (savedSettings) {
            qrSettings = JSON.parse(savedSettings);
            // Always force signed_by_enabled to false on page load for security
            qrSettings.signed_by_enabled = false;
            // Always reset security_level to SMART as default
            qrSettings.security_level = 'SMART';
            applyQRSettings();
        }
        
        // Update generated_by and signed_by with user's name when authenticated
        updateUserNameFields();
        
        // Setup toggle handlers (only for fields that can be toggled)
        const toggleHandlers = [
            {
                toggle: 'qr_purpose_toggle',
                field: 'qr_purpose',
                enabledKey: 'purpose_enabled'
            },
            {
                toggle: 'qr_description_toggle',
                field: 'qr_description',
                enabledKey: 'description_enabled'
            },
            {
                toggle: 'qr_start_date_toggle',
                field: 'qr_start_date',
                enabledKey: 'start_date_enabled'
            },
            {
                toggle: 'qr_end_date_toggle',
                field: 'qr_end_date',
                enabledKey: 'end_date_enabled'
            }
        ];
        
        toggleHandlers.forEach(handler => {
            const toggle = document.getElementById(handler.toggle);
            const field = document.getElementById(handler.field);
            
            toggle.addEventListener('change', function() {
                if (this.checked) {
                    field.disabled = false;
                    field.style.opacity = '1';
                    field.style.display = '';
                    field.style.marginTop = '5px';
                    field.style.marginBottom = '0';
                } else {
                    field.disabled = true;
                    field.style.display = 'none';
                    field.style.marginTop = '0';
                    field.style.marginBottom = '0';
                }
            });
            
            // Set initial state
            if (toggle.checked) {
                field.disabled = false;
                field.style.opacity = '1';
                field.style.display = '';
                field.style.marginTop = '5px';
                field.style.marginBottom = '0';
            } else {
                field.disabled = true;
                field.style.display = 'none';
                field.style.marginTop = '0';
                field.style.marginBottom = '0';
            }
            
            // Force trigger the change event to ensure proper initialization
            const event = new Event('change');
            toggle.dispatchEvent(event);
        });
        
        // Special handler for signed_by toggle (doesn't affect disabled state since it's readonly)
        const signedByToggle = document.getElementById('qr_signed_by_toggle');
        const signedByField = document.getElementById('qr_signed_by');
        const signedByContainer = signedByField.parentElement;
        
        signedByToggle.addEventListener('change', function() {
            // Auto-set security level to Enterprise when Signed By is enabled
            const securityLevel = document.getElementById('qr_security_level');
            const signedBySection = document.getElementById('signedBySection');
            
            if (this.checked) {
                signedBySection.style.display = 'block';
                
                securityLevel.value = 'ENTERPRISE';
                // Don't disable the select, just make it readonly visually
                securityLevel.style.background = '#e8f4f8';
                securityLevel.style.pointerEvents = 'none';
                securityLevel.style.color = '#2563eb';
                securityLevel.style.fontWeight = '600';
            } else {
                signedBySection.style.display = 'none';
                
                securityLevel.style.background = '';
                securityLevel.style.pointerEvents = '';
                securityLevel.style.color = '';
                securityLevel.style.fontWeight = '';
            }
        });
        
        // Set initial state
        if (signedByToggle.checked) {
            signedBySection.style.display = 'block';
        } else {
            signedBySection.style.display = 'none';
        }
        
        // Force trigger the change event to ensure proper initialization
        const signedByEvent = new Event('change');
        signedByToggle.dispatchEvent(signedByEvent);
        
        // Password protection toggle
        const passwordToggle = document.getElementById('qr_password_protection_toggle');
        const passwordOptions = document.getElementById('password_options');
        const passwordTypeSelect = document.getElementById('qr_password_type');
        const passwordField = document.getElementById('qr_password');
        
        passwordToggle.addEventListener('change', function() {
            passwordOptions.style.display = this.checked ? 'block' : 'none';
            if (!this.checked) {
                passwordField.value = '';
                passwordTypeSelect.value = 'master';
                passwordField.style.display = 'none';
            } else {
                // Trigger password type change to show/hide password field
                passwordTypeSelect.dispatchEvent(new Event('change'));
            }
        });
        
        // Password type selection handler
        passwordTypeSelect.addEventListener('change', function() {
            if (this.value === 'unique') {
                passwordField.style.display = 'block';
                passwordField.required = true;
            } else {
                passwordField.style.display = 'none';
                passwordField.required = false;
                passwordField.value = ''; // Clear password when switching to master
            }
        });

        // Anti-counterfeit tags toggle
        const antiCounterfeitToggle = document.getElementById('qr_anti_counterfeit_toggle');
        if (antiCounterfeitToggle) {
            antiCounterfeitToggle.addEventListener('change', function() {
                const section = document.getElementById('antiCounterfeitSection');
                if (section) {
                    section.style.display = this.checked ? 'block' : 'none';
                    if (this.checked) {
                        loadExistingAntiCounterfeitTags();
                    }
                }
            });
        }

        // Security level change handler - update QR sizes in canvas
        const securityLevelSelect = document.getElementById('qr_security_level');
        securityLevelSelect.addEventListener('change', function() {
            console.log('Security level changed to:', this.value);
            // Update all existing QR objects with new size
            updateQRObjectsSizes();
        });

        // Global variable to store current QR verification URL
        let currentQRVerificationUrl = '';
        let currentQRImageUrl = '';

        // Helper function to place QR image at placeholder position and update identifier
        function placeQRImageAtPlaceholder(imageSource, fallbackUrl, identifierCode, verificationUrl, qrIdCode) {
            // Store verification URL globally
            if (verificationUrl) {
                currentQRVerificationUrl = verificationUrl;
                currentQRImageUrl = imageSource;
            }
            // Find the QR placeholder, identifier, and QR ID on canvas
            const objects = canvas.getObjects();
            let qrPlaceholder = null;
            let qrIdentifier = null;
            let qrIdField = null;

            for (let obj of objects) {
                // Check for QR code placeholder (supports both old 'qr_code' and new '{QR_CODE}' fieldType)
                if (obj.fieldType === 'qr_code' || obj.fieldType === '{QR_CODE}') {
                    qrPlaceholder = obj;
                } else if (obj.fieldType === 'qr_identifier' || obj.fieldType === '{QR_IDENTIFIER}') {
                    qrIdentifier = obj;
                } else if (obj.fieldType === '{QR_ID}') {
                    qrIdField = obj;
                }
            }

            // If no placeholder found, use default position
            const defaultLeft = 100;
            const defaultTop = 100;
            const defaultSize = 150;

            let targetLeft, targetTop, targetWidth, targetHeight;

            if (qrPlaceholder) {
                // Get placeholder position and size
                targetLeft = qrPlaceholder.left;
                targetTop = qrPlaceholder.top;
                targetWidth = qrPlaceholder.width * (qrPlaceholder.scaleX || 1);
                targetHeight = qrPlaceholder.height * (qrPlaceholder.scaleY || 1);
            } else {
                targetLeft = defaultLeft;
                targetTop = defaultTop;
                targetWidth = defaultSize;
                targetHeight = defaultSize;
            }

            // Function to actually place the image on canvas
            function placeImage(imageUrl, sourceDescription) {
                fabric.Image.fromURL(imageUrl, function(qrObject, isError) {
                    if (!isError && qrObject && qrObject.getElement() && qrObject.width > 0) {
                        // Calculate scale to fit target size
                        const scaleX = targetWidth / qrObject.width;
                        const scaleY = targetHeight / qrObject.height;

                        qrObject.set({
                            left: targetLeft,
                            top: targetTop,
                            scaleX: scaleX,
                            scaleY: scaleY,
                            fieldType: 'qr_code_image',
                            id: 'qr_' + Date.now()
                        });

                        // Remove the placeholder ONLY after successful load
                        if (qrPlaceholder) {
                            canvas.remove(qrPlaceholder);
                        }

                        canvas.add(qrObject);
                        canvas.setActiveObject(qrObject);
                        canvas.renderAll();

                        // Update preview
                        const previewSection = document.getElementById('qrPreviewSection');
                        const previewImage = document.getElementById('qrPreviewImage');
                        if (previewSection && previewImage) {
                            previewImage.src = imageUrl;
                            previewSection.style.display = 'block';
                        }

                        // Update QR Identifier text if provided and identifier placeholder exists
                        if (identifierCode && qrIdentifier) {
                            qrIdentifier.set({
                                text: identifierCode,
                                backgroundColor: '',  // Remove yellow background
                                fill: '#333',  // Change to normal text color
                                fontWeight: 'normal',  // Remove bold
                                padding: 0  // Remove padding
                            });
                            console.log('✅ QR Identifier updated to:', identifierCode);
                        }

                        // Update QR ID text if provided and QR ID placeholder exists
                        if (qrIdCode && qrIdField) {
                            qrIdField.set({
                                text: qrIdCode,
                                backgroundColor: '',  // Remove yellow background
                                fill: '#333',  // Change to normal text color
                                fontWeight: 'normal',  // Remove bold
                                padding: 0  // Remove padding
                            });
                            console.log('✅ QR ID updated to:', qrIdCode);
                        }

                        canvas.renderAll();

                        console.log('✅ QR code placed at placeholder position from', sourceDescription, ':', targetLeft, targetTop, 'size:', targetWidth, 'x', targetHeight);
                    } else {
                        console.error('❌ Failed to place image from', sourceDescription);
                        // Try proxy fallback if available
                        if (fallbackUrl && sourceDescription !== 'proxy') {
                            tryProxyFallback();
                        } else {
                            alert('Failed to load QR image. The image has been generated but cannot be displayed due to CORS restrictions.');
                        }
                    }
                }, { crossOrigin: 'anonymous' });
            }

            // Function to try proxy fallback (fetches JSON with base64)
            function tryProxyFallback() {
                console.log('Trying proxy fallback:', fallbackUrl);
                fetch(fallbackUrl)
                    .then(response => response.json())
                    .then(data => {
                        if (data.content && data.content.image_data) {
                            // Create data URL from base64
                            const dataUrl = `data:image/png;base64,${data.content.image_data}`;
                            console.log('Extracted base64 from proxy, placing image...');
                            placeImage(dataUrl, 'proxy');
                        } else {
                            throw new Error('No image data in proxy response');
                        }
                    })
                    .catch(error => {
                        console.error('Proxy also failed:', error);
                        alert('Failed to load QR image. The image has been generated but cannot be displayed due to CORS restrictions.');
                    });
            }

            // Start with primary URL
            placeImage(imageSource, 'direct URL');
        }

        // QR Preview button handlers
        document.getElementById('downloadQRBtn').addEventListener('click', function() {
            const img = document.getElementById('qrPreviewImage');
            if (!img.src) {
                alert('No QR code to download');
                return;
            }

            // Create a download link
            const link = document.createElement('a');
            link.href = img.src;
            link.download = 'qr-code-' + Date.now() + '.png';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            console.log('QR code downloaded');
        });

        document.getElementById('verifyQRBtn').addEventListener('click', function() {
            if (!currentQRVerificationUrl) {
                alert('No verification URL available');
                return;
            }

            window.open(currentQRVerificationUrl, '_blank');
            console.log('Opening verification URL:', currentQRVerificationUrl);
        });

        // Generate QR button
        document.getElementById('saveQRSettings').addEventListener('click', async function() {
            // Mark this as user-initiated QR generation
            // Always get identifier from the main identifier code field
            const identifierValue = document.getElementById('qr_identifier_code').value;

            // Collect settings
            qrSettings = {
                security_level: document.getElementById('qr_security_level').value,
                qr_name: document.getElementById('qr_name').value,
                generated_by: document.getElementById('qr_generated_by').value,
                signed_by: document.getElementById('qr_signed_by').value,
                signed_by_enabled: document.getElementById('qr_signed_by_toggle').checked,
                identifier: identifierValue,
                awarded_to: document.getElementById('qr_awarded_to').value,
                mask_name: document.getElementById('qr_mask_name').checked,
                purpose: document.getElementById('qr_purpose').value,
                purpose_enabled: document.getElementById('qr_purpose_toggle').checked,
                description: document.getElementById('qr_description').value,
                description_enabled: document.getElementById('qr_description_toggle').checked,
                start_date: document.getElementById('qr_start_date').value,
                start_date_enabled: document.getElementById('qr_start_date_toggle').checked,
                end_date: document.getElementById('qr_end_date').value,
                end_date_enabled: document.getElementById('qr_end_date_toggle').checked,
                password_protection: document.getElementById('qr_password_protection_toggle').checked,
                password_type: document.getElementById('qr_password_type').value,
                password: document.getElementById('qr_password').value,
                anti_counterfeit_enabled: document.getElementById('qr_anti_counterfeit_toggle')?.checked || false,
                anti_counterfeit_tags: (document.getElementById('qr_anti_counterfeit_toggle')?.checked && document.getElementById('qr_anti_counterfeit_tags')?.value.trim()) || ''
            };

            // Save to local storage
            localStorage.setItem('govalidQRSettings', JSON.stringify(qrSettings));
            
            // Check authentication
            if (!goValidAuth.isAuthenticated) {
                alert('Please authenticate with GoValid first to generate QR codes.');
                return;
            }

            // Check PIN verification if Signed By is enabled
            if (qrSettings.signed_by_enabled) {
                if (!window.pinVerificationStatus || !window.pinVerificationStatus.isPinVerified()) {
                    alert('Please verify your PIN first before generating a signed QR code.\n\nEnter your 6-digit signing PIN in the "Signed By" section and click "Verify PIN".');
                    return;
                }
            }

            // Disable button and show loading
            const genBtn = this;
            const originalHTML = genBtn.innerHTML;
            genBtn.disabled = true;
            genBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Generating QR...';
            
            try {
                // Generate QR code
                console.log('Generating QR with settings:', qrSettings);
                const qrResult = await generateCertificateQR();
                
                if (qrResult.success && (qrResult.qr_code || qrResult.qr_image_url)) {
                    // Store the formatted_uuid (QR ID) from API response
                    if (qrResult.formatted_uuid) {
                        qrSettings.formatted_uuid = qrResult.formatted_uuid;
                        console.log('Stored QR ID (formatted_uuid):', qrSettings.formatted_uuid);
                    }

                    // Get the identifier code - use formatted_uuid if identifier is blank
                    const identifierCode = qrSettings.identifier ||
                                         qrResult.formatted_uuid ||
                                         qrResult.certificate_id ||
                                         '';

                    // Log all available fields
                    console.log('QR Result fields:', Object.keys(qrResult));
                    console.log('Full QR Result:', qrResult);
                    console.log('Using identifier code:', identifierCode);

                    // Use verify_url (short format: https://govalid.org/v/{token})
                    const verificationUrl = qrResult.verify_url ||
                                          qrResult.verification_url ||
                                          '';

                    console.log('Using verification URL:', verificationUrl);

                    // Get the QR ID (formatted_uuid) for the {QR_ID} field
                    const qrIdCode = qrResult.formatted_uuid || '';

                    // Create a QR code image on the canvas
                    if (qrResult.qr_code) {
                        // Use base64 data if available (direct method)
                        const imageSource = 'data:image/png;base64,' + qrResult.qr_code;
                        console.log('Creating QR image from base64 data');

                        // Place QR at placeholder position with identifier, verification URL, and QR ID
                        placeQRImageAtPlaceholder(imageSource, null, identifierCode, verificationUrl, qrIdCode);

                    } else if (qrResult.qr_image_url || qrResult.qr_image_direct_url) {
                        // Use the direct URL if available (no CORS issues)
                        const imageUrlToUse = qrResult.qr_image_direct_url || qrResult.qr_image_url;
                        const fallbackUrl = qrResult.qr_image_proxy_url;

                        console.log('Loading QR image from URL:', imageUrlToUse);
                        console.log('Fallback proxy URL:', fallbackUrl);
                        console.log('URL type:', qrResult.qr_image_direct_url ? 'Direct (CORS-free)' : 'Original');

                        // Place QR image at placeholder position using helper function with fallback, identifier, verification URL, and QR ID
                        console.log('Placing QR from URL at placeholder position with identifier:', identifierCode);
                        placeQRImageAtPlaceholder(imageUrlToUse, fallbackUrl, identifierCode, verificationUrl, qrIdCode);

                        // Switch to Design tab to show the QR
                        setTimeout(function() {
                            const designTab = document.getElementById('tab-design');
                            if (designTab) {
                                designTab.click();
                            }
                        }, 1000);  // Increased timeout to allow image loading
                    }
                    
                    // Show success
                    genBtn.innerHTML = '<i class="fa fa-check"></i> QR Generated!';
                    genBtn.style.background = '#48bb78';
                    
                    // If fallback was used, show a notification
                    if (qrResult.fallback_used) {
                        console.log('QR generated using custom type fallback');
                        // You could add a visual indicator here if desired
                        // For now, just log it
                    }
                    
                    // Switch to Design tab to show the QR
                    const designTab = document.querySelector('[data-tab="design"]');
                    if (designTab) {
                        designTab.click();
                    }

                    // Reset PIN verification state if Signed By is enabled
                    // User must verify PIN again for next QR generation
                    if (qrSettings.signed_by_enabled && window.resetPINVerificationState) {
                        window.resetPINVerificationState();
                    }

                    // Start 10-second countdown before allowing next generation
                    let countdown = 10;
                    genBtn.innerHTML = '<i class="fa fa-check"></i> QR Generated! (' + countdown + 's)';
                    genBtn.style.background = '#48bb78';

                    const countdownInterval = setInterval(() => {
                        countdown--;
                        if (countdown > 0) {
                            genBtn.innerHTML = '<i class="fa fa-hourglass-half"></i> Please wait... (' + countdown + 's)';
                            genBtn.style.background = '#718096';
                        } else {
                            clearInterval(countdownInterval);
                            genBtn.innerHTML = originalHTML;
                            genBtn.style.background = '';
                            genBtn.disabled = false;
                        }
                    }, 1000);
                } else {
                    throw new Error(qrResult.message || 'Failed to generate QR code');
                }
            } catch (error) {
                console.error('QR generation error:', error);
                alert('Failed to generate QR code: ' + error.message);
                // On error, use shorter 3-second countdown
                let errorCountdown = 3;
                genBtn.innerHTML = '<i class="fa fa-exclamation-triangle"></i> Error (' + errorCountdown + 's)';
                genBtn.style.background = '#e53e3e';

                const errorInterval = setInterval(() => {
                    errorCountdown--;
                    if (errorCountdown > 0) {
                        genBtn.innerHTML = '<i class="fa fa-exclamation-triangle"></i> Error (' + errorCountdown + 's)';
                    } else {
                        clearInterval(errorInterval);
                        genBtn.innerHTML = originalHTML;
                        genBtn.style.background = '';
                        genBtn.disabled = false;
                    }
                }, 1000);
            }
        });
        
        
        
        // Set default dates
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('qr_start_date').value = today;
        
        // Set end date to 1 year from now
        const oneYearLater = new Date();
        oneYearLater.setFullYear(oneYearLater.getFullYear() + 1);
        document.getElementById('qr_end_date').value = oneYearLater.toISOString().split('T')[0];
        
        // Setup enhanced identifier generator
        setupIdentifierGenerator();
    }

    // ============================================================
    // ENHANCED IDENTIFIER GENERATOR SYSTEM
    // ============================================================

    let institutionPatterns = [];
    let selectedPatternId = null;
    let lastUsedSequence = 1;

    // Format labels for display
    const FORMAT_LABELS = {
        'alpha5': '5-digit Alphanumeric',
        'numeric6': '6-digit Numeric',
        'invoice': 'Invoice Format',
        'serial': 'Serial Number',
        'dateseq': 'Date Sequential',
        'cert': 'Certificate',
        'institution_pattern': 'Institution Pattern',
        'custom': 'Custom Pattern'
    };

    function setupIdentifierGenerator() {
        console.log('Setting up enhanced identifier generator...');

        // Restore saved sequence from localStorage
        const savedSeq = localStorage.getItem('govalidLastSequence');
        if (savedSeq) {
            lastUsedSequence = parseInt(savedSeq, 10) || 1;
            const seqInput = document.getElementById('custom_sequence_start');
            if (seqInput) seqInput.value = lastUsedSequence;
        }

        // Format selector change handler
        const formatSelect = document.getElementById('identifier_format_select');
        if (formatSelect) {
            formatSelect.addEventListener('change', handleFormatChange);
            // Initialize with current selection
            handleFormatChange();
        }

        // Custom pattern input handler
        const customPatternInput = document.getElementById('custom_pattern_input');
        if (customPatternInput) {
            customPatternInput.addEventListener('input', updateIdentifierPreview);
        }

        // Sequence start input handler
        const seqStartInput = document.getElementById('custom_sequence_start');
        if (seqStartInput) {
            seqStartInput.addEventListener('input', function() {
                let val = parseInt(this.value, 10) || 1;
                if (val < 1) val = 1;
                if (val > 999) val = 999;
                this.value = val;
                updateIdentifierPreview();
            });
        }

        // Remember sequence checkbox handler
        const rememberCheckbox = document.getElementById('remember_sequence');
        if (rememberCheckbox) {
            rememberCheckbox.addEventListener('change', function() {
                if (!this.checked) {
                    localStorage.removeItem('govalidLastSequence');
                }
            });
        }

        // Generate button - directly generates single identifier
        const generateBtn = document.getElementById('generate_identifier_btn');
        if (generateBtn) {
            generateBtn.addEventListener('click', function(e) {
                e.preventDefault();
                generateIdentifiers(1);
            });
        }

        // Identifier field input handler for manual entry
        const identifierField = document.getElementById('qr_identifier_code');
        if (identifierField) {
            identifierField.addEventListener('input', function() {
                updateIdentifierPreview();
                qrSettings.identifier = this.value;
            });
        }

        // Institution pattern selector handler
        const patternSelector = document.getElementById('pattern_selector');
        if (patternSelector) {
            patternSelector.addEventListener('change', handlePatternSelection);
        }

        // Setup institution pattern visibility based on auth
        setupInstitutionPatternVisibility();

        // Initialize preview
        updateIdentifierPreview();

        console.log('Identifier generator setup complete');
    }

    function handleFormatChange() {
        const formatSelect = document.getElementById('identifier_format_select');
        const format = formatSelect ? formatSelect.value : 'cert';

        // Show/hide custom pattern container
        const customContainer = document.getElementById('custom_pattern_container');
        const customHelp = document.getElementById('custom_pattern_help');
        if (customContainer) customContainer.style.display = format === 'custom' ? 'block' : 'none';
        if (customHelp) customHelp.style.display = format === 'custom' ? 'block' : 'none';

        // Show/hide institution pattern section
        const institutionSection = document.getElementById('identifier_pattern_section');
        if (institutionSection) {
            if (format === 'institution_pattern') {
                institutionSection.style.display = 'block';
                // Check if user has institution access
                checkAndLoadInstitutionPatterns();
            } else {
                institutionSection.style.display = 'none';
            }
        }

        // Update format label in preview
        const formatLabel = document.getElementById('preview_format_label');
        if (formatLabel) {
            formatLabel.textContent = FORMAT_LABELS[format] || format;
        }

        // Show/hide sequence info for sequential formats
        const seqInfo = document.getElementById('preview_sequence_info');
        if (seqInfo) {
            const showSeq = ['dateseq', 'cert', 'custom'].includes(format);
            seqInfo.style.display = showSeq ? 'block' : 'none';
        }

        updateIdentifierPreview();
    }

    function generateIdentifiers(quantity) {
        const formatSelect = document.getElementById('identifier_format_select');
        const format = formatSelect ? formatSelect.value : 'cert';
        const identifierField = document.getElementById('qr_identifier_code');

        if (!identifierField) return;

        // Handle institution pattern generation
        if (format === 'institution_pattern') {
            generateInstitutionPatternNumbers(quantity);
            return;
        }

        let identifiers = [];
        const seqStartInput = document.getElementById('custom_sequence_start');
        let startSeq = seqStartInput ? (parseInt(seqStartInput.value, 10) || 1) : lastUsedSequence;

        for (let i = 0; i < quantity; i++) {
            const currentSeq = startSeq + i;
            identifiers.push(generateSingleIdentifier(format, currentSeq));
        }

        // Update field with generated identifiers
        identifierField.value = identifiers.join('\n');

        // Update last used sequence
        lastUsedSequence = startSeq + quantity;

        // Save sequence if remember is checked
        const rememberCheckbox = document.getElementById('remember_sequence');
        if (rememberCheckbox && rememberCheckbox.checked) {
            localStorage.setItem('govalidLastSequence', lastUsedSequence);
            if (seqStartInput) seqStartInput.value = lastUsedSequence;
        }

        // Update preview
        updateIdentifierPreview();

        // Update qrSettings
        qrSettings.identifier = identifiers[0]; // Use first identifier for QR

        // Show notification
        showIdentifierNotification(quantity, format);

        console.log('Generated ' + quantity + ' identifier(s):', identifiers);
    }

    function generateSingleIdentifier(format, seqNumber) {
        const seq = seqNumber || 1;
        const seqStr = seq.toString().padStart(3, '0');
        const date = new Date();
        const yyyy = date.getFullYear();
        const yy = yyyy.toString().slice(-2);
        const mm = (date.getMonth() + 1).toString().padStart(2, '0');
        const dd = date.getDate().toString().padStart(2, '0');

        switch (format) {
            case 'alpha5':
                return generateAlphanumeric(5);
            case 'numeric6':
                return generateNumeric(6);
            case 'invoice':
                return `INV-${yy}${mm}${dd}-${seqStr}`;
            case 'serial':
                return `SN-${generateNumeric(5)}-${seqStr}`;
            case 'dateseq':
                return `${dd}${mm}${yy}-${seqStr}`;
            case 'cert':
                return `CERT-${yyyy}${mm}${dd}-${seqStr}`;
            case 'custom':
                return generateCustomPattern(seqNumber);
            default:
                return `CERT-${yyyy}${mm}${dd}-${seqStr}`;
        }
    }

    function generateAlphanumeric(length) {
        const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        let result = '';
        for (let i = 0; i < length; i++) {
            result += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        return result;
    }

    function generateNumeric(length) {
        let result = '';
        for (let i = 0; i < length; i++) {
            result += Math.floor(Math.random() * 10);
        }
        return result;
    }

    function generateCustomPattern(seqNumber) {
        const patternInput = document.getElementById('custom_pattern_input');
        let pattern = patternInput ? patternInput.value : 'CERT-#SEQ#';

        if (!pattern) pattern = 'CERT-#SEQ#';

        const seq = seqNumber || 1;
        const seqStr = seq.toString().padStart(3, '0');

        // Replace #SEQ# placeholder
        let result = pattern.replace(/#SEQ#/g, seqStr);

        // Replace other placeholders
        result = result.replace(/[#?*]/g, function(c) {
            if (c === '#') return Math.floor(Math.random() * 10);
            if (c === '?') {
                const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
                return letters.charAt(Math.floor(Math.random() * letters.length));
            }
            if (c === '*') {
                const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
                return chars.charAt(Math.floor(Math.random() * chars.length));
            }
            return c;
        });

        return result;
    }

    function updateIdentifierPreview() {
        const identifierField = document.getElementById('qr_identifier_code');
        const previewContent = document.getElementById('preview_content');
        const seqInfoText = document.getElementById('sequence_info_text');
        const seqInfoDiv = document.getElementById('preview_sequence_info');
        const formatSelect = document.getElementById('identifier_format_select');

        if (!previewContent) return;

        const currentValue = identifierField ? identifierField.value.trim() : '';
        const format = formatSelect ? formatSelect.value : 'cert';

        let displayValue = '';

        if (currentValue) {
            // Show actual value (handle multiple lines)
            const lines = currentValue.split('\n').filter(l => l.trim());
            if (lines.length > 1) {
                displayValue = lines[0] + '\n... and ' + (lines.length - 1) + ' more';
            } else {
                displayValue = currentValue;
            }
        } else {
            // Show example
            const seqInput = document.getElementById('custom_sequence_start');
            const seq = seqInput ? (parseInt(seqInput.value, 10) || 1) : 1;
            displayValue = 'Example: ' + generateSingleIdentifier(format, seq);
        }

        previewContent.textContent = displayValue;

        // Update sequence info
        if (seqInfoDiv && seqInfoText) {
            const showSeq = ['dateseq', 'cert', 'custom'].includes(format);
            if (showSeq) {
                const seqInput = document.getElementById('custom_sequence_start');
                const seq = seqInput ? (parseInt(seqInput.value, 10) || 1) : lastUsedSequence;
                seqInfoText.textContent = 'Next sequence: ' + seq.toString().padStart(3, '0');
                seqInfoDiv.style.display = 'block';
            } else {
                seqInfoDiv.style.display = 'none';
            }
        }
    }

    function showIdentifierNotification(quantity, format) {
        // Create notification element
        let notification = document.getElementById('identifier-notification');
        if (!notification) {
            notification = document.createElement('div');
            notification.id = 'identifier-notification';
            notification.style.cssText = 'position: fixed; top: 20px; right: 20px; background: #059669; color: white; padding: 12px 20px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.2); z-index: 10000; font-size: 13px; animation: slideIn 0.3s ease;';
            document.body.appendChild(notification);
        }

        const formatName = FORMAT_LABELS[format] || format;
        notification.innerHTML = '<i class="fa fa-check-circle" style="margin-right: 8px;"></i>' +
            quantity + ' ' + formatName + ' identifier' + (quantity > 1 ? 's' : '') + ' generated';
        notification.style.display = 'block';

        // Auto-hide after 3 seconds
        setTimeout(function() {
            notification.style.display = 'none';
        }, 3000);
    }

    // ============================================================
    // INSTITUTION PATTERN FUNCTIONALITY
    // ============================================================

    function setupInstitutionPatternVisibility() {
        // Check if institution pattern option should be visible
        function updateVisibility() {
            const formatSelect = document.getElementById('identifier_format_select');
            const institutionOption = formatSelect ? formatSelect.querySelector('option[value="institution_pattern"]') : null;

            if (!institutionOption) return;

            if (goValidAuth && goValidAuth.isAuthenticated) {
                const hasAccess = (
                    (goValidAuth.subscription && goValidAuth.subscription.subscription_source === 'institution') ||
                    (goValidAuth.userInfo && goValidAuth.userInfo.institution_info) ||
                    (goValidAuth.userInfo && goValidAuth.userInfo.institution_name) ||
                    (goValidAuth.subscription && goValidAuth.subscription.institution_name)
                );

                institutionOption.style.display = hasAccess ? '' : 'none';
                institutionOption.disabled = !hasAccess;

                if (hasAccess && institutionPatterns.length === 0) {
                    loadInstitutionPatterns();
                }
            } else {
                institutionOption.style.display = 'none';
                institutionOption.disabled = true;
            }
        }

        window.updateIdentifierFormatVisibility = updateVisibility;
        updateVisibility();
    }

    function checkAndLoadInstitutionPatterns() {
        if (goValidAuth && goValidAuth.isAuthenticated && institutionPatterns.length === 0) {
            loadInstitutionPatterns();
        }
    }

    async function loadInstitutionPatterns() {
        if (!goValidAuth || !goValidAuth.isAuthenticated) return;

        try {
            const response = await fetch(buildActionURL('getPatterns'), {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });

            const data = await response.json();
            console.log('Institution patterns response:', data);

            let patterns = [];
            if (data.status && data.content) {
                const content = data.content;
                if (content.success && content.patterns) {
                    patterns = content.patterns;
                }
            }

            const selector = document.getElementById('pattern_selector');
            if (!selector) return;

            if (patterns && patterns.length > 0) {
                institutionPatterns = patterns;
                selector.innerHTML = '<option value="">Select a numbering pattern...</option>';
                patterns.forEach(function(pattern) {
                    const option = document.createElement('option');
                    option.value = pattern.id;
                    option.textContent = pattern.name || pattern.pattern_name || 'Pattern ' + pattern.id;
                    option.dataset.pattern = JSON.stringify(pattern);
                    selector.appendChild(option);
                });
                console.log('Loaded ' + patterns.length + ' institution patterns');
            } else {
                selector.innerHTML = '<option value="">No patterns available</option>';
                const infoDiv = document.getElementById('pattern_info');
                if (infoDiv) {
                    infoDiv.innerHTML = '<small style="color: #666;">Contact your institution administrator to set up numbering patterns.</small>';
                }
            }
        } catch (error) {
            console.error('Failed to load institution patterns:', error);
            const selector = document.getElementById('pattern_selector');
            if (selector) selector.innerHTML = '<option value="">Error loading patterns</option>';
        }
    }

    function handlePatternSelection() {
        const selector = document.getElementById('pattern_selector');
        selectedPatternId = selector ? selector.value : null;

        if (!selectedPatternId) {
            document.getElementById('pattern_info').innerHTML = '';
            return;
        }

        // Find pattern data
        const option = selector.options[selector.selectedIndex];
        let pattern = null;

        if (option && option.dataset.pattern) {
            try {
                pattern = JSON.parse(option.dataset.pattern);
            } catch (e) {
                console.error('Failed to parse pattern data:', e);
            }
        }

        if (!pattern) {
            pattern = institutionPatterns.find(p => p.id == selectedPatternId);
        }

        if (pattern) {
            displayPatternInfo(pattern);
            // Auto-generate a number
            generateInstitutionPatternNumbers(1);
        }
    }

    function displayPatternInfo(pattern) {
        const infoDiv = document.getElementById('pattern_info');
        if (!infoDiv) return;

        infoDiv.innerHTML =
            '<strong>Pattern:</strong> ' + (pattern.name || pattern.code) + '<br>' +
            '<strong>Template:</strong> ' + pattern.pattern_template + '<br>' +
            '<strong>Current:</strong> ' + pattern.current_counter + '<br>' +
            '<strong>Next:</strong> #' + ((pattern.current_counter || 0) + (pattern.increment || 1));
    }

    async function generateInstitutionPatternNumbers(quantity) {
        if (!goValidAuth || !goValidAuth.isAuthenticated || !selectedPatternId) return;

        try {
            const response = await fetch(buildActionURL('generatePattern'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'Accept': 'application/json'
                },
                body: 'pattern_id=' + parseInt(selectedPatternId) + '&quantity=' + quantity + '&document_type=certificate'
            });

            const data = await response.json();
            const result = data.content || data;

            if (result.success && result.data && result.data.generated) {
                const generated = result.data.generated;
                const numbers = Array.isArray(generated)
                    ? generated.map(item => typeof item === 'string' ? item : item.number)
                    : [];

                if (numbers.length > 0) {
                    const identifierField = document.getElementById('qr_identifier_code');
                    if (identifierField) {
                        identifierField.value = numbers.join('\n');
                    }
                    qrSettings.identifier = numbers[0];
                    updateIdentifierPreview();
                    showIdentifierNotification(numbers.length, 'institution_pattern');
                    console.log('Generated institution pattern numbers:', numbers);
                }
            }
        } catch (error) {
            console.error('Failed to generate institution pattern numbers:', error);
        }
    }

    // Keep old function name for backward compatibility
    async function generatePatternNumbers(patternId, quantity) {
        selectedPatternId = patternId;
        await generateInstitutionPatternNumbers(quantity);
        return [];
    }
    
    function applyQRSettings() {
        document.getElementById('qr_security_level').value = qrSettings.security_level;
        document.getElementById('qr_name').value = qrSettings.qr_name;
        // Don't set generated_by and signed_by from settings - they come from authenticated user
        const signedByToggle = document.getElementById('qr_signed_by_toggle');
        signedByToggle.checked = qrSettings.signed_by_enabled === true;
        
        // Check if signed by is enabled and set security level accordingly
        const securityLevel = document.getElementById('qr_security_level');
        if (signedByToggle.checked) {
            securityLevel.value = 'enterprise';
            // Don't disable the select, just make it readonly visually
            securityLevel.style.background = '#e8f4f8';
            securityLevel.style.pointerEvents = 'none';
            securityLevel.style.color = '#2563eb';
            securityLevel.style.fontWeight = '600';
        } else {
            // Reset security level styling when signed by is disabled
            securityLevel.style.background = '';
            securityLevel.style.pointerEvents = '';
            securityLevel.style.color = '';
            securityLevel.style.fontWeight = '';
        }
        document.getElementById('qr_identifier_code').value = qrSettings.identifier;
        document.getElementById('qr_awarded_to').value = qrSettings.awarded_to;
        document.getElementById('qr_mask_name').checked = qrSettings.mask_name || false;
        document.getElementById('qr_purpose').value = qrSettings.purpose;
        document.getElementById('qr_purpose_toggle').checked = qrSettings.purpose_enabled !== false;
        document.getElementById('qr_description').value = qrSettings.description;
        document.getElementById('qr_description_toggle').checked = qrSettings.description_enabled !== false;
        document.getElementById('qr_start_date').value = qrSettings.start_date;
        document.getElementById('qr_end_date').value = qrSettings.end_date;
        document.getElementById('qr_password_protection_toggle').checked = qrSettings.password_protection || false;
        document.getElementById('qr_password_type').value = qrSettings.password_type || 'master';
        document.getElementById('qr_password').value = qrSettings.password;
        
        // Set visibility of password options
        const passwordOptions = document.getElementById('password_options');
        passwordOptions.style.display = qrSettings.password_protection ? 'block' : 'none';
        
        // Set password field visibility based on type
        const passwordField = document.getElementById('qr_password');
        if (qrSettings.password_protection && qrSettings.password_type === 'unique') {
            passwordField.style.display = 'block';
            passwordField.required = true;
        } else {
            passwordField.style.display = 'none';
            passwordField.required = false;
        }
        
        // Apply disabled states (only for editable fields)
        const signedBySection = document.getElementById('signedBySection');
        if (qrSettings.signed_by_enabled) {
            signedBySection.style.display = 'block';
        } else {
            signedBySection.style.display = 'none';
        }
        document.getElementById('qr_purpose').disabled = !qrSettings.purpose_enabled;
        document.getElementById('qr_purpose').style.opacity = qrSettings.purpose_enabled ? '1' : '0.5';
        document.getElementById('qr_description').disabled = !qrSettings.description_enabled;
        document.getElementById('qr_description').style.opacity = qrSettings.description_enabled ? '1' : '0.5';
        
        // Update user name fields based on current auth status
        updateUserNameFields();
    }
    
    // Get QR settings for API call
    function getQRSettings() {
        return qrSettings;
    }
    
    // Update user name fields based on authentication
    function updateUserNameFields() {
        const generatedByField = document.getElementById('qr_generated_by');
        const signedByField = document.getElementById('qr_signed_by');
        
        if (goValidAuth.isAuthenticated && goValidAuth.userInfo) {
            const fullName = goValidAuth.userInfo.first_name + ' ' + goValidAuth.userInfo.last_name;
            
            // Clear and populate the dropdown
            generatedByField.innerHTML = '';
            generatedByField.disabled = false;
            generatedByField.style.background = 'white';
            generatedByField.style.cursor = 'pointer';
            
            // Add full name option
            const fullNameOption = document.createElement('option');
            fullNameOption.value = fullName;
            fullNameOption.textContent = fullName;
            generatedByField.appendChild(fullNameOption);
            
            // Add institution name option if available
            if (goValidAuth.subscription && goValidAuth.subscription.institution_name) {
                const institutionOption = document.createElement('option');
                institutionOption.value = goValidAuth.subscription.institution_name;
                institutionOption.textContent = goValidAuth.subscription.institution_name;
                generatedByField.appendChild(institutionOption);
            }
            
            // Set signed by field
            signedByField.value = fullName;
            
            // Update the settings object - check if current value is still valid
            if (!qrSettings.generated_by || (qrSettings.generated_by !== fullName && 
                (!goValidAuth.subscription || qrSettings.generated_by !== goValidAuth.subscription.institution_name))) {
                qrSettings.generated_by = fullName;
            }
            qrSettings.signed_by = fullName;
            
            // Select the current value
            generatedByField.value = qrSettings.generated_by;
        } else {
            generatedByField.innerHTML = '<option value="">Login to see options</option>';
            generatedByField.disabled = true;
            generatedByField.style.background = '#f5f5f5';
            generatedByField.style.cursor = 'not-allowed';
            
            signedByField.value = '';
            signedByField.placeholder = 'Login to see your name';
        }
    }
    
    // Generate Certificate QR Code with all fields
    async function generateCertificateQR(replacements = {}) {
        if (!goValidAuth.isAuthenticated) {
            throw new Error('Authentication required for QR generation');
        }
        
        // Prepare metadata object first
        const metadataObj = {
            generated_by: qrSettings.generated_by || '',
            identifier: qrSettings.identifier || '',
            name: qrSettings.awarded_to || '',  // This is the recipient/awarded to
            created_date: qrSettings.start_date || new Date().toISOString().split('T')[0],
            expire_date: qrSettings.end_date || new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toISOString().split('T')[0], // Default 1 year expiry
            purpose_title: qrSettings.purpose_enabled ? qrSettings.purpose : 'Certificate',
            // Optional fields
            ...(qrSettings.description_enabled && qrSettings.description ? {
                description: qrSettings.description
            } : {}),
            // Include signing toggle using the correct backend field name
            ...(qrSettings.signed_by_enabled && qrSettings.signed_by ? {
                sign_by: qrSettings.signed_by
            } : {})
        };
        
        // Get user email for the request
        const userEmail = goValidAuth.userInfo ? goValidAuth.userInfo.email : '';
        
        // Use the simplified format for /api/v1/ojs/generate-qr/ endpoint
        const requestData = {
            email: userEmail,
            toggle_sign_by_cert: qrSettings.signed_by_enabled ? 'on' : 'off',
            // Only include session_token for signed certificates (OTP session token)
            // For regular certificates, the JWT token in Authorization header is sufficient
            ...(qrSettings.signed_by_enabled && window.ojsSessionToken ? {
                session_token: window.ojsSessionToken
            } : {}),
            qr_data: {
                qr_type: 'certificate',
                qr_name: qrSettings.qr_name || 'Publication Certificate',
                security_level: qrSettings.signed_by_enabled ? 'ENTERPRISE' : qrSettings.security_level,
                mask_name: qrSettings.mask_name ? "1" : "0",  // Send as string "1" or "0"
                // Password protection fields (inside qr_data)
                is_password_protected: qrSettings.password_protection,
                password_type: qrSettings.password_type || 'master',  // 'master', 'unique', or 'disable'
                // Include unique password if password_type is 'unique' and password is provided
                ...(qrSettings.password_type === 'unique' && qrSettings.password ? {
                    field_password: qrSettings.password
                } : {}),
                // Include anti-counterfeit tags if enabled
                ...(qrSettings.anti_counterfeit_enabled && qrSettings.anti_counterfeit_tags ? {
                    anti_counterfeit_tags: qrSettings.anti_counterfeit_tags
                } : {}),
                form_data: {
                    title: metadataObj.purpose_title,  // Research Publication title
                    ...(metadataObj.identifier ? { identifier: metadataObj.identifier } : {}),
                    recipient: metadataObj.name,       // Author/recipient name
                    issuer: metadataObj.generated_by || qrSettings.generated_by || '', // Journal/issuer name
                    issue_date: metadataObj.created_date,    // Publication date
                    ...(metadataObj.description ? { description: metadataObj.description } : {}),
                    ...(metadataObj.expire_date ? { expire_date: metadataObj.expire_date } : {}),
                    ...(qrSettings.signed_by_enabled && metadataObj.sign_by ? { 
                        sign_by: metadataObj.sign_by 
                    } : {}),
                    // Add OTP verification field for ENTERPRISE security level (required for OTP verification metadata)
                    ...(qrSettings.signed_by_enabled && window.ojsSessionToken ? {
                        otp_verified: true
                    } : {})
                }
            }
        };
        
        console.log('Sending QR generation request to /api/v1/ojs/generate-qr/ with simplified format:', JSON.stringify(requestData, null, 2));
        console.log('Signing enabled:', qrSettings.signed_by_enabled);
        console.log('Session token included:', window.ojsSessionToken ? 'YES' : 'NO');
        console.log('Security level:', qrSettings.security_level);
        console.log('Password protection enabled:', qrSettings.password_protection);
        console.log('Password type:', qrSettings.password_type);
        console.log('OTP verified field included:', qrSettings.signed_by_enabled && window.ojsSessionToken ? 'YES' : 'NO');
        if (window.ojsSessionToken) {
            console.log('Session token value:', window.ojsSessionToken);
        }
        console.log('Expected behavior: OTP required ONLY for signed certificates (toggle_sign_by_cert = "on")');
        console.log('Current timestamp:', new Date().toISOString());
        
        try {
            // Make API request through the plugin's proxy
            const response = await fetch(buildActionURL('generateCertificateQR'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(requestData)
            });
            
            console.log('QR generation response status:', response.status);
            console.log('QR generation response ok:', response.ok);
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('QR generation failed with status:', response.status);
                console.error('Error response:', errorText);
                throw new Error(`HTTP ${response.status}: ${errorText}`);
            }
            
            const result = await response.json();
            console.log('QR generation response data:', result);
            console.log('Response content:', result.content);
            console.log('Response success:', result.success);
            console.log('Content success:', result.content?.success);
            
            // Log the actual data structure we're working with
            if (result.content) {
                const data = result.content.data || result.content;
                console.log('Certificate type response data:', data);
                console.log('Certificate response - Looking for QR code in fields:', {
                    qr_code: data.qr_code,
                    qr_code_base64: data.qr_code_base64,
                    qr_image: data.qr_image,
                    qr_code_data: data.qr_code_data,
                    qr_image_url: data.qr_image_url,
                    data_qr_image_url: data.data?.qr_image_url,
                    success: data.success,
                    httpCode: data.httpCode,
                    message: data.message
                });
            }
            
            if (result.status && result.content && result.content.success) {
                // Handle the OJS API response format - check multiple possible locations
                const content = result.content;
                // The response structure is: result.content.qr_code (direct from API v1 OJS endpoint)
                const qrData = content.qr_code ||           // Primary location for OJS API v1
                              content.data?.qr_code ||      // Alternative location
                              content.content?.qr_code ||   // Nested location
                              content.data?.content?.qr_code; // Deep nested location
                
                console.log('Processing successful OJS API response');
                console.log('Full result structure:', JSON.stringify(result, null, 2));
                console.log('Content:', content);
                console.log('Content.data:', content.data);
                console.log('Content.data.content:', content.data?.content);
                
                // Try to find the QR data in all possible locations
                console.log('Looking for QR data in different locations:');
                console.log('- content.data?.content?.qr_code:', content.data?.content?.qr_code);
                console.log('- content.content?.qr_code:', content.content?.qr_code);
                console.log('- content.qr_code:', content.qr_code);
                console.log('- content.data:', content.data);
                
                console.log('QR Data found:', qrData);
                
                // Extract QR code information from the new response structure
                let qrCodeBase64 = null;
                let qrImageUrl = null;
                
                if (qrData) {
                    // Check for the new direct URL first (no CORS issues)
                    if (qrData.qr_image_direct_url) {
                        qrImageUrl = qrData.qr_image_direct_url;
                        console.log('Found QR image direct URL (CORS-free):', qrImageUrl);
                    } else if (qrData.qr_image_url) {
                        // Fallback to original URL
                        qrImageUrl = qrData.qr_image_url;
                        console.log('Found QR image URL:', qrImageUrl);
                    }
                    
                    // Store alternative URLs for fallback
                    const qrImageProxyUrl = qrData.qr_image_proxy_url;
                    if (qrImageProxyUrl) {
                        console.log('Proxy URL available:', qrImageProxyUrl);
                    }
                    
                    // Also check for any base64 data
                    if (qrData.qr_image_base64) {
                        qrCodeBase64 = qrData.qr_image_base64;
                    } else if (qrData.image_data) {
                        qrCodeBase64 = qrData.image_data;
                    }
                }
                
                // If we have an image URL, check if it's a direct URL (no CORS issues)
                if (!qrCodeBase64 && qrImageUrl) {
                    // Check if it's a direct URL from the new backend solution
                    const isDirectUrl = qrImageUrl.includes('/qr-codes/secure-qr-image/') || 
                                       qrImageUrl.includes('/api/v1/ojs/proxy-image/');
                    
                    if (isDirectUrl) {
                        console.log('✅ Using direct URL (CORS-free):', qrImageUrl);
                        // For direct URLs, we don't need base64 conversion
                        // The URL will be used directly in fabric.Image.fromURL
                    } else {
                        // For R2 URLs, try to convert to base64 (might fail due to CORS)
                        try {
                            console.log('Loading QR image from URL:', qrImageUrl);
                            qrCodeBase64 = await new Promise((resolve, reject) => {
                                const img = new Image();
                                img.crossOrigin = 'anonymous';
                                img.onload = function() {
                                    try {
                                        const canvas = document.createElement('canvas');
                                        const ctx = canvas.getContext('2d');
                                        canvas.width = img.width;
                                        canvas.height = img.height;
                                        ctx.drawImage(img, 0, 0);
                                        const base64 = canvas.toDataURL('image/png').split(',')[1];
                                        resolve(base64);
                                    } catch (canvasError) {
                                        console.log('Canvas conversion failed, will use URL directly');
                                        resolve(null);
                                    }
                                };
                                img.onerror = function() {
                                    console.log('Direct image load failed, will use URL directly');
                                    resolve(null);
                                };
                                img.src = qrImageUrl;
                            });
                            if (qrCodeBase64) {
                                console.log('Successfully converted QR image URL to base64');
                            }
                        } catch (error) {
                            console.error('Failed to load QR image from URL:', error);
                        }
                    }
                }
                
                console.log('Final QR processing result:', {
                    hasQrCodeBase64: !!qrCodeBase64,
                    hasQrImageUrl: !!qrImageUrl,
                    qrImageUrl: qrImageUrl,
                    verificationUrl: qrData?.verification_url,
                    qrId: qrData?.id
                });
                
                // Return success if we have either base64 data or image URL
                if (qrCodeBase64 || qrImageUrl) {
                    return {
                        success: true,
                        qr_code: qrCodeBase64,
                        qr_image_url: qrImageUrl,
                        qr_image_direct_url: qrData?.qr_image_direct_url, // New CORS-free URL
                        qr_image_proxy_url: qrData?.qr_image_proxy_url,   // Alternative proxy URL
                        verification_url: qrData?.verification_url,
                        verify_url: qrData?.verify_url,  // Short format verification URL
                        formatted_uuid: qrData?.formatted_uuid,  // Human-readable QR ID (6ACE-CE-74967fe7)
                        certificate_id: qrData?.id || qrData?.uuid,
                        // Add direct access to the QR data for image creation
                        qr_data: qrData
                    };
                } else {
                    console.error('No QR code found in OJS API response');
                    console.error('QR Data structure:', qrData);
                    throw new Error('No QR code generated in response');
                }
            } else {
                console.error('QR generation failed - Full response:', result);
                if (result.content && result.content.data) {
                    console.error('Error details from server:', result.content.data);
                }
                const errorMsg = result.message || result.content?.message || result.content?.error || 'QR generation failed';
                throw new Error(errorMsg);
            }
        } catch (error) {
            console.error('QR generation error:', error);
            console.error('Error details:', error.message);
            
            // Check if OTP is required
            if (error.message && error.message.includes('OTP verification required')) {
                console.log('OTP verification required for digital signature');
                console.log('Current QR settings:', qrSettings);
                console.log('Toggle setting sent:', qrSettings.signed_by_enabled ? 'on' : 'off');
                
                // Only show OTP modal if user actually enabled digital signatures
                if (qrSettings.signed_by_enabled) {
                    console.log('Digital signatures enabled, OTP verification required');
                    throw new Error('OTP verification required. Please check your email and enter the OTP code.');
                } else {
                    // This shouldn't happen with toggle_sign_by_cert: 'off', but log it for debugging
                    console.error('Unexpected: OTP required even with toggle_sign_by_cert: off');
                    console.error('Request data:', qrData);
                    throw new Error('Unexpected OTP requirement. Digital signatures are disabled but server still requires OTP.');
                }
            }
            
            throw error;
        }
    }
    
    // Function to load QR image via proxy when CORS fails
    async function tryProxyImageLoad(imageUrl) {
        try {
            console.log('Trying to load QR image via proxy:', imageUrl);
            
            // Build the URL with OJS plugin parameters like other successful requests
            let baseURL = getAjaxURL();
            // Check if URL already has query parameters
            const separator = baseURL.includes('?') ? '&' : '?';
            const proxyUrl = baseURL + separator + 'action=proxyQRImage';
            
            console.log('Proxy URL:', proxyUrl);
            
            const response = await fetch(proxyUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'url=' + encodeURIComponent(imageUrl)
            });
            
            console.log('Proxy response status:', response.status);
            console.log('Proxy response ok:', response.ok);
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('Proxy request failed with status:', response.status);
                console.error('Proxy error response:', errorText);
                throw new Error('Proxy request failed: ' + response.status + ' - ' + errorText);
            }
            
            const result = await response.json();
            console.log('Proxy response:', result);
            
            if (result.status && result.content && result.content.success) {
                const base64Data = result.content.image_base64;
                const imageSource = 'data:image/png;base64,' + base64Data;
                
                console.log('Successfully got image via proxy, creating Fabric object...');
                
                // Create Fabric image from base64 data
                fabric.Image.fromURL(imageSource, function(qrObject) {
                    qrObject.set({
                        left: 100,
                        top: 100,
                        scaleX: 0.5,
                        scaleY: 0.5,
                        id: 'qr_' + Date.now()
                    });
                    canvas.add(qrObject);
                    canvas.setActiveObject(qrObject);
                    canvas.renderAll();
                    
                    console.log('QR code successfully added to canvas via proxy!');
                });
            } else {
                console.error('Proxy failed:', result);
                if (result.content) {
                    console.error('Proxy error details:', result.content);
                    if (result.content.message) {
                        alert('Failed to load QR code image: ' + result.content.message);
                    } else {
                        alert('Failed to load QR code image. Please try again.');
                    }
                } else {
                    alert('Failed to load QR code image. Please try again.');
                }
            }
        } catch (error) {
            console.error('Proxy image load failed:', error);
            console.error('Error stack:', error.stack);
            
            // Fallback: Show the QR URL for manual access
            if (imageUrl) {
                console.log('Fallback: Providing QR URL for manual access');
                const fallbackMessage = 'The QR code was generated successfully!\n\n' +
                    'However, the image cannot be displayed due to CORS restrictions from Cloudflare R2.\n\n' +
                    'Options:\n' +
                    '1. Copy this URL and open in a new tab to view/download the QR code:\n' +
                    imageUrl + '\n\n' +
                    '2. Contact the backend team to enable CORS on Cloudflare R2\n' +
                    '3. Request the API to return QR images as base64 data';
                
                if (confirm(fallbackMessage + '\n\nCopy QR URL to clipboard?')) {
                    // Copy the URL to clipboard if possible
                    if (navigator.clipboard) {
                        navigator.clipboard.writeText(imageUrl).then(function() {
                            console.log('QR URL copied to clipboard');
                            alert('QR URL copied to clipboard! You can paste it in a new tab.');
                        }).catch(function(err) {
                            console.error('Failed to copy URL to clipboard:', err);
                            prompt('Copy this URL manually:', imageUrl);
                        });
                    } else {
                        prompt('Copy this URL manually:', imageUrl);
                    }
                }
            } else {
                console.log('QR code generation succeeded, but image display failed. The QR was created successfully.');
            }
        }
    }

    // Load existing anti-counterfeit tags from user's GoValid account
    async function loadExistingAntiCounterfeitTags() {
        try {
            const response = await fetch(buildActionURL('getAntiCounterfeitTags'), {
                method: 'GET',
                headers: {
                    'Accept': 'application/json'
                }
            });

            const result = await response.json();
            console.log('Anti-counterfeit tags response:', result);

            // Handle OJS JSONMessage wrapper - data is inside 'content'
            const data = result.content || result;
            console.log('Anti-counterfeit tags data:', data);

            if (data.success && data.tags && data.tags.length > 0) {
                const container = document.getElementById('existingAntiCounterfeitTagsContainer');
                const list = document.getElementById('existingAntiCounterfeitTagsList');

                if (container && list) {
                    list.innerHTML = '';
                    data.tags.forEach(tag => {
                        const badge = document.createElement('span');
                        badge.className = 'tag-badge';
                        badge.textContent = tag;
                        badge.style.cssText = 'display: inline-block; padding: 2px 8px; margin: 2px; background: #f3e8ff; border: 1px solid #c084fc; border-radius: 12px; font-size: 11px; color: #7c3aed; cursor: pointer; transition: background-color 0.2s;';
                        badge.addEventListener('mouseenter', function() {
                            this.style.backgroundColor = '#7c3aed';
                            this.style.color = '#fff';
                            this.style.borderColor = '#7c3aed';
                        });
                        badge.addEventListener('mouseleave', function() {
                            this.style.backgroundColor = '#f3e8ff';
                            this.style.color = '#7c3aed';
                            this.style.borderColor = '#c084fc';
                        });
                        badge.addEventListener('click', function() {
                            addAntiCounterfeitTag(tag);
                        });
                        list.appendChild(badge);
                    });
                    container.style.display = 'block';
                }
            }
        } catch (error) {
            console.error('Error loading anti-counterfeit tags:', error);
        }
    }

    // Add anti-counterfeit tag to input field
    function addAntiCounterfeitTag(tag) {
        const input = document.getElementById('qr_anti_counterfeit_tags');
        if (input) {
            const currentTags = input.value.split(',').map(t => t.trim()).filter(t => t);
            if (!currentTags.includes(tag)) {
                currentTags.push(tag);
                input.value = currentTags.join(', ');
                // Trigger change event to update qrSettings
                input.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }
    }

