/**
 * GoValid Certificate Designer - Shapes Module
 * Handles adding and managing shapes on the canvas
 */

(function() {
    'use strict';

    // Shape dropdown event handler
    const shapeDropdown = document.getElementById('addShapeDropdown');
    if (shapeDropdown) {
        shapeDropdown.addEventListener('change', function() {
            const shapeType = this.value;
            if (shapeType) {
                requireAuth('addShape', function() {
                    addShape(shapeType);
                    // Reset dropdown to default option
                    shapeDropdown.value = '';
                });
            }
        });
    }

    /**
     * Add a shape to the canvas
     * @param {string} shapeType - Type of shape to add (rectangle, square, circle, ellipse, triangle, line)
     */
    function addShape(shapeType) {
        let shape;
        const defaultOptions = {
            left: 100,
            top: 100,
            fill: 'rgba(102, 126, 234, 0.5)', // Semi-transparent blue
            stroke: '#667eea',
            strokeWidth: 2,
            objectType: 'shape', // Custom property to identify shapes
            shapeType: shapeType  // Store the specific shape type
        };

        switch(shapeType) {
            case 'rectangle':
                shape = new fabric.Rect({
                    ...defaultOptions,
                    width: 200,
                    height: 100,
                    rx: 0,  // Corner radius
                    ry: 0
                });
                break;

            case 'square':
                shape = new fabric.Rect({
                    ...defaultOptions,
                    width: 150,
                    height: 150,
                    rx: 0,
                    ry: 0
                });
                break;

            case 'circle':
                shape = new fabric.Circle({
                    ...defaultOptions,
                    radius: 50
                });
                break;

            case 'ellipse':
                shape = new fabric.Ellipse({
                    ...defaultOptions,
                    rx: 80,  // Horizontal radius
                    ry: 50   // Vertical radius
                });
                break;

            case 'triangle':
                shape = new fabric.Triangle({
                    ...defaultOptions,
                    width: 100,
                    height: 100
                });
                break;

            case 'line':
                shape = new fabric.Line([50, 100, 250, 100], {
                    left: 100,
                    top: 100,
                    stroke: '#667eea',
                    strokeWidth: 3,
                    objectType: 'shape',
                    shapeType: 'line'
                });
                break;

            default:
                console.error('Unknown shape type:', shapeType);
                return;
        }

        // Add the shape to canvas
        canvas.add(shape);
        canvas.setActiveObject(shape);
        updateProperties(shape);
        updateLayerList();

        console.log('Added shape:', shapeType);
    }

    /**
     * Update shape properties in the properties panel
     * This function extends the updateProperties function for shapes
     */
    window.updateShapeProperties = function(shape) {
        if (!shape || !shape.objectType || shape.objectType !== 'shape') {
            return '';
        }

        let html = '<div class="properties-fields">';

        // Shape Type Display
        html += '<div class="property-field">';
        html += '<label>Shape Type:</label>';
        html += '<input type="text" value="' + capitalizeFirst(shape.shapeType) + '" disabled style="background: #f5f5f5; cursor: not-allowed;">';
        html += '</div>';

        // Fill Color and Opacity in one row
        html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
        html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0;">';
        html += '<label for="shapeFill" style="margin-bottom: 5px;">Fill:</label>';

        // Extract color from rgba or use default
        let fillColor = '#667eea';
        if (shape.fill) {
            if (typeof shape.fill === 'string') {
                if (shape.fill.startsWith('rgba') || shape.fill.startsWith('rgb')) {
                    // Extract RGB values and convert to hex
                    const rgbMatch = shape.fill.match(/\d+/g);
                    if (rgbMatch && rgbMatch.length >= 3) {
                        fillColor = rgbToHex(parseInt(rgbMatch[0]), parseInt(rgbMatch[1]), parseInt(rgbMatch[2]));
                    }
                } else {
                    fillColor = shape.fill;
                }
            }
        }

        html += '<input type="color" id="shapeFill" value="' + fillColor + '" onchange="updateShapeFill()" style="width: 100%; height: 32px;">';
        html += '</div>';
        html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0;">';
        html += '<div style="display: flex; justify-content: space-between; align-items: baseline; margin-bottom: 2px;">';
        html += '<label for="fillOpacity">Opacity:</label>';
        html += '<span id="opacityValue" style="font-size: 11px; color: #666;">' + (shape.opacity ? Math.round(shape.opacity * 100) : 100) + '%</span>';
        html += '</div>';
        html += '<input type="range" id="fillOpacity" min="0" max="100" value="' + (shape.opacity ? shape.opacity * 100 : 100) + '" onchange="updateShapeOpacity()" style="width: calc(100% - 35px); margin-top: 0;">';
        html += '</div>';
        html += '</div>';

        // Stroke Color and Width in one row
        html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
        html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0;">';
        html += '<label for="shapeStroke" style="margin-bottom: 5px;">Stroke:</label>';
        html += '<input type="color" id="shapeStroke" value="' + (shape.stroke || '#000000') + '" onchange="updateShapeStroke()" style="width: 100%; height: 32px;">';
        html += '</div>';
        html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0;">';
        html += '<label for="strokeWidth" style="margin-bottom: 5px;">Width:</label>';
        html += '<input type="number" id="strokeWidth" value="' + (shape.strokeWidth || 2) + '" onchange="updateStrokeWidth()" min="0" max="20">';
        html += '</div>';
        html += '</div>';

        // Shape-specific properties - Corner Radius (for rectangles/squares only)
        let hasCornerRadius = (shape.shapeType === 'rectangle' || shape.shapeType === 'square');

        if (shape.shapeType === 'circle') {
            // Radius
            html += '<div class="property-field">';
            html += '<label for="circleRadius">Radius:</label>';
            html += '<input type="number" id="circleRadius" value="' + (shape.radius || 50) + '" onchange="updateCircleRadius()" min="10" max="500">';
            html += '</div>';
        }

        if (shape.shapeType === 'ellipse') {
            // Horizontal and Vertical Radius
            html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="ellipseRx">H-Radius:</label>';
            html += '<input type="number" id="ellipseRx" value="' + (shape.rx || 50) + '" onchange="updateEllipseRx()" min="10" max="500">';
            html += '</div>';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="ellipseRy">V-Radius:</label>';
            html += '<input type="number" id="ellipseRy" value="' + (shape.ry || 50) + '" onchange="updateEllipseRy()" min="10" max="500">';
            html += '</div>';
            html += '</div>';
        }

        // Common size properties for non-circle shapes
        if (shape.shapeType !== 'circle' && shape.shapeType !== 'line') {
            html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="shapeWidth">Width:</label>';
            html += '<input type="number" id="shapeWidth" value="' + Math.round(shape.width * (shape.scaleX || 1)) + '" onchange="updateShapeWidth()" min="10" max="1000">';
            html += '</div>';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="shapeHeight">Height:</label>';
            html += '<input type="number" id="shapeHeight" value="' + Math.round(shape.height * (shape.scaleY || 1)) + '" onchange="updateShapeHeight()" min="10" max="1000">';
            html += '</div>';
            html += '</div>';
        }

        // Position
        html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
        html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
        html += '<label for="shapeLeft">X Position:</label>';
        html += '<input type="number" id="shapeLeft" value="' + Math.round(shape.left) + '" onchange="updateShapePosition()">';
        html += '</div>';
        html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
        html += '<label for="shapeTop">Y Position:</label>';
        html += '<input type="number" id="shapeTop" value="' + Math.round(shape.top) + '" onchange="updateShapePosition()">';
        html += '</div>';
        html += '</div>';

        // Corner Radius and Rotation in one row (or just Rotation if no corner radius)
        if (hasCornerRadius) {
            html += '<div style="display: flex; gap: 20px; margin-bottom: 12px;">';
            // Corner column
            html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0; padding-right: 10px; box-sizing: border-box;">';
            html += '<div style="display: flex; justify-content: space-between; align-items: baseline; margin-bottom: 2px;">';
            html += '<label for="cornerRadius">Corner:</label>';
            html += '<span id="radiusValue" style="font-size: 11px; color: #666;">' + (shape.rx || 0) + 'px</span>';
            html += '</div>';
            html += '<input type="range" id="cornerRadius" min="0" max="50" value="' + (shape.rx || 0) + '" onchange="updateCornerRadius()" style="width: calc(100% - 25px); margin-top: 0;">';
            html += '</div>';
            // Rotation column
            html += '<div style="flex: 1; display: flex; flex-direction: column; margin-bottom: 0; padding-left: 10px; box-sizing: border-box;">';
            html += '<div style="display: flex; justify-content: space-between; align-items: baseline; margin-bottom: 2px;">';
            html += '<label for="shapeRotation">Rotation:</label>';
            html += '<span id="rotationValue" style="font-size: 11px; color: #666;">' + (shape.angle || 0) + '°</span>';
            html += '</div>';
            html += '<input type="range" id="shapeRotation" min="0" max="360" value="' + (shape.angle || 0) + '" onchange="updateShapeRotation()" style="width: calc(100% - 25px); margin-left: 0px; margin-top: 0;">';
            html += '</div>';
            html += '</div>';
        } else {
            // Just rotation for shapes without corner radius
            html += '<div class="property-field">';
            html += '<div style="display: flex; justify-content: space-between; align-items: baseline; margin-bottom: 2px;">';
            html += '<label for="shapeRotation">Rotation:</label>';
            html += '<span id="rotationValue" style="font-size: 11px; color: #666;">' + (shape.angle || 0) + '°</span>';
            html += '</div>';
            html += '<input type="range" id="shapeRotation" min="0" max="360" value="' + (shape.angle || 0) + '" onchange="updateShapeRotation()" style="width: calc(100% - 25px); margin-left: 0px; margin-top: 0;">';
            html += '</div>';
        }

        // Delete button
        html += '<button type="button" onclick="deleteObject()" class="delete-btn">🗑️ Delete</button>';

        html += '</div>'; // Close properties-fields

        return html;
    };

    // Helper function to convert RGB to Hex
    function rgbToHex(r, g, b) {
        return "#" + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
    }

    // Helper function to capitalize first letter
    function capitalizeFirst(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    // Property update functions for shapes
    window.updateShapeFill = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const color = document.getElementById('shapeFill').value;
            const opacity = selectedObject.opacity || 1;
            selectedObject.set('fill', color);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateShapeOpacity = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const opacity = document.getElementById('fillOpacity').value / 100;
            selectedObject.set('opacity', opacity);
            document.getElementById('opacityValue').textContent = Math.round(opacity * 100) + '%';
            canvas.renderAll();
        }
    };

    window.updateShapeStroke = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const color = document.getElementById('shapeStroke').value;
            selectedObject.set('stroke', color);
            canvas.renderAll();
        }
    };

    window.updateStrokeWidth = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const width = parseInt(document.getElementById('strokeWidth').value);
            selectedObject.set('strokeWidth', width);
            canvas.renderAll();
        }
    };

    window.updateCornerRadius = function() {
        if (selectedObject && (selectedObject.shapeType === 'rectangle' || selectedObject.shapeType === 'square')) {
            const radius = parseInt(document.getElementById('cornerRadius').value);
            selectedObject.set({ rx: radius, ry: radius });
            document.getElementById('radiusValue').textContent = radius + 'px';
            canvas.renderAll();
        }
    };

    window.updateCircleRadius = function() {
        if (selectedObject && selectedObject.shapeType === 'circle') {
            const radius = parseInt(document.getElementById('circleRadius').value);
            selectedObject.set('radius', radius);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateEllipseRx = function() {
        if (selectedObject && selectedObject.shapeType === 'ellipse') {
            const rx = parseInt(document.getElementById('ellipseRx').value);
            selectedObject.set('rx', rx);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateEllipseRy = function() {
        if (selectedObject && selectedObject.shapeType === 'ellipse') {
            const ry = parseInt(document.getElementById('ellipseRy').value);
            selectedObject.set('ry', ry);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateShapeWidth = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const newWidth = parseInt(document.getElementById('shapeWidth').value);
            const scaleX = newWidth / selectedObject.width;
            selectedObject.set('scaleX', scaleX);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateShapeHeight = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const newHeight = parseInt(document.getElementById('shapeHeight').value);
            const scaleY = newHeight / selectedObject.height;
            selectedObject.set('scaleY', scaleY);
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateShapePosition = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const left = parseInt(document.getElementById('shapeLeft').value);
            const top = parseInt(document.getElementById('shapeTop').value);
            selectedObject.set({ left: left, top: top });
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    window.updateShapeRotation = function() {
        if (selectedObject && selectedObject.objectType === 'shape') {
            const angle = parseInt(document.getElementById('shapeRotation').value);
            selectedObject.set('angle', angle);
            document.getElementById('rotationValue').textContent = angle + '°';
            selectedObject.setCoords();
            canvas.renderAll();
        }
    };

    console.log('Shapes module loaded');
})();
