# GoValidOJS Developer Guide

## Plugin Structure

```
goValidOJS/
├── GoValidOJSPlugin.php          # Main plugin class
├── classes/
│   ├── GoValidAuthManager.php    # GoValid API authentication
│   ├── GoValidOJSSettingsForm.php # Settings form handler
│   └── GoValidOJSHandler.php     # Page request handler
├── templates/
│   ├── designer.tpl              # Certificate designer interface
│   └── settingsForm.tpl          # Plugin settings form
├── locale/                       # Language files
├── docs/                         # Documentation
└── version.xml                   # Plugin metadata
```

## Key Classes and Methods

### GoValidOJSPlugin

Main plugin class that extends GenericPlugin.

#### Important Methods

```php
// Handle plugin requests
public function manage($args, $request): ?\PKP\core\JSONMessage

// Handle AJAX requests from designer
private function _handleDesignerAjax($request): \PKP\core\JSONMessage

// Get published issues
private function _getIssues($context)

// Get submissions for an issue
private function _getSubmissions($context, $issueId = null)
```

### Data Access in OJS 3.4

#### Using Repositories (Recommended)

```php
// Get issues
$issues = Repo::issue()->getCollector()
    ->filterByContextIds([$contextId])
    ->filterByPublished(true)
    ->getMany();

// Get submissions
$submissions = Repo::submission()->getCollector()
    ->filterByContextIds([$contextId])
    ->filterByStatus([PKPSubmission::STATUS_PUBLISHED])
    ->getMany();
```

#### Accessing Publication Data

```php
$publication = $submission->getCurrentPublication();

// Get data fields
$title = $publication->getLocalizedTitle();
$doi = $publication->getStoredPubId('doi');
$pages = $publication->getData('pages');
$issueId = $publication->getData('issueId');
$abstract = $publication->getLocalizedData('abstract');
```

## AJAX Communication Pattern

### Frontend Request
```javascript
// Build URL with action parameter
const url = getAjaxURL() + '&action=getSubmissions&issueId=' + issueId;

// Make request with credentials
fetch(url, {
    method: 'GET',
    headers: {'Accept': 'application/json'},
    credentials: 'same-origin'
})
.then(response => response.json())
.then(data => {
    // Handle response
});
```

### Backend Response
```php
// Return JSON response
return new \PKP\core\JSONMessage(true, [
    'submissions' => $submissions
]);
```

## Adding New Auto-Population Fields

### 1. Backend: Add to Data Array

In `_getSubmissions()` method:
```php
$submissions[] = [
    'id' => $submission->getId(),
    'title' => $publication->getLocalizedTitle(),
    'yourNewField' => $publication->getData('yourFieldName'),
    // ... other fields
];
```

### 2. Frontend: Add Input Field

In `designer.tpl`:
```html
<div class="content-field">
    <label for="content_your_field">Your Field Label</label>
    <input type="text" id="content_your_field" class="content-input" readonly>
</div>
```

### 3. Frontend: Populate Field

In `handleArticleSelectionChange()`:
```javascript
document.getElementById('content_your_field').value = article.yourNewField || '';
```

## Common OJS 3.4 Patterns

### Context (Journal) Access
```php
$context = $request->getContext();
$contextId = $context ? $context->getId() : null;
```

### User Authorization
```php
$user = $request->getUser();
if (!$user || !$user->hasRole([Role::ROLE_ID_MANAGER], $contextId)) {
    return new JSONMessage(false, __('user.authorization.accessDenied'));
}
```

### Database Queries (Avoid if possible)
```php
// Use repositories instead, but if needed:
$result = Repo::submission()->dao->retrieve($sql, $params);
```

## Error Handling

### PHP Error Handling
```php
try {
    // Your code
} catch (\Exception $e) {
    error_log('GoValidOJS: Error - ' . $e->getMessage());
    return new JSONMessage(false, ['error' => $e->getMessage()]);
}
```

### JavaScript Error Handling
```javascript
fetch(url)
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .catch(error => {
        console.error('Error:', error);
    });
```

## Testing

### Manual Testing
1. Enable PHP error reporting
2. Check browser console for JavaScript errors
3. Monitor Network tab for AJAX requests
4. Check server logs for PHP errors

### Debug Logging
```php
error_log('GoValidOJS: ' . json_encode($data));
```

```javascript
console.log('Debug:', data);
```

## Performance Considerations

1. **Use Repository collectors** with proper filters
2. **Limit results** when possible
3. **Cache data** in JavaScript when appropriate
4. **Avoid N+1 queries** - fetch related data efficiently

## Security Best Practices

1. **Always validate input**
```php
$issueId = (int) $request->getUserVar('issueId');
```

2. **Check permissions**
```php
if (!$this->getEnabled($contextId)) {
    return new JSONMessage(false, __('plugins.generic.goValidOJS.error.disabled'));
}
```

3. **Escape output**
```javascript
element.textContent = data.title; // Safe
element.innerHTML = escapeHtml(data.content); // If HTML needed
```

4. **Use CSRF tokens** (OJS handles automatically)

## Useful OJS Functions

```php
// Get plugin URL
$this->getPluginPath()

// Get template resource
$this->getTemplateResource('designer.tpl')

// Get/set plugin settings
$this->getSetting($contextId, 'settingName')
$this->updateSetting($contextId, 'settingName', $value)

// Localization
__('plugins.generic.goValidOJS.displayName')
```

## Debugging Tips

1. **Enable error display temporarily**
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

2. **Log SQL queries**
```php
Config::setVar('debug', 'show_sql', true);
```

3. **Check OJS version compatibility**
```php
$ojsVersion = Application::get()->getCurrentVersion();
```

4. **Inspect object structure**
```php
error_log(print_r($object, true));
```