<?php

/**
 * @file plugins/generic/goValidOJS/classes/GoValidAuthManager.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * GoValid Authentication Manager
 */

namespace APP\plugins\generic\goValidOJS\classes;

use PKP\session\Session;

class GoValidAuthManager
{
    private $session;
    private $apiBaseUrl = 'https://my.govalid.org/api/v1';
    private $apiBaseUrlNoVersion = 'https://my.govalid.org/api';

    public function __construct()
    {
        $request = \Application::get()->getRequest();
        $this->session = $request->getSession();
    }

    /**
     * Set a session variable (OJS 3.4 compatible)
     */
    private function setSessionVar($key, $value)
    {
        $this->session->setSessionVar($key, $value);
    }

    /**
     * Get a session variable (OJS 3.4 compatible)
     */
    private function getSessionVar($key)
    {
        return $this->session->getSessionVar($key);
    }

    /**
     * Remove a session variable (OJS 3.4 compatible)
     */
    private function unsetSessionVar($key)
    {
        $this->session->unsetSessionVar($key);
    }
    
    /**
     * Authenticate user with GoValid
     * @param string $username Username or email
     * @param string $password
     * @return array ['success' => bool, 'access' => string, 'refresh' => string, 'user' => array]
     */
    public function authenticate($username, $password)
    {
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - authenticate() START for user: $username\n", FILE_APPEND);
        error_log('GoValidOJS: Starting authentication for user: ' . $username);
        error_log('GoValidOJS: API URL: ' . $this->apiBaseUrl . '/auth/login/');
        
        $ch = curl_init($this->apiBaseUrl . '/auth/login/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'username' => $username,
            'password' => $password
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // For debugging
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        error_log('GoValidOJS: HTTP Code: ' . $httpCode);
        error_log('GoValidOJS: cURL Error: ' . $curlError);
        error_log('GoValidOJS: Raw Response: ' . $response);
        
        if ($curlError) {
            error_log('GoValidOJS: cURL Error occurred: ' . $curlError);
            return ['success' => false, 'message' => 'Network error: ' . $curlError];
        }

        // Handle successful login (HTTP 200)
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            error_log('GoValidOJS: Parsed response: ' . json_encode($data));
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Login API response: " . json_encode($data) . "\n", FILE_APPEND);

            // Handle the actual API response format
            if (isset($data['success']) && $data['success'] === true && isset($data['data'])) {
                $responseData = $data['data'];
                error_log('GoValidOJS: Success response with data structure');
                if (isset($responseData['access_token']) && isset($responseData['refresh_token'])) {
                    // Store tokens in session and get enriched user data
                    $enrichedUserData = $this->storeTokens(
                        $responseData['access_token'],
                        $responseData['refresh_token'],
                        $responseData['user'] ?? []
                    );
                    return [
                        'success' => true,
                        'access' => $responseData['access_token'],
                        'refresh' => $responseData['refresh_token'],
                        'user' => $enrichedUserData  // Return enriched data with profile photo
                    ];
                }
            }

            // Fallback for direct token format (if API changes)
            if (isset($data['access']) && isset($data['refresh'])) {
                error_log('GoValidOJS: Success response with direct token format');
                $enrichedUserData = $this->storeTokens($data['access'], $data['refresh'], $data['user'] ?? []);
                return [
                    'success' => true,
                    'access' => $data['access'],
                    'refresh' => $data['refresh'],
                    'user' => $enrichedUserData  // Return enriched data with profile photo
                ];
            }

            // Check for error message in response
            if (isset($data['message'])) {
                error_log('GoValidOJS: API returned error message: ' . $data['message']);
                return ['success' => false, 'message' => $data['message']];
            }
        }

        // Handle 2FA requirement (HTTP 202)
        if ($httpCode === 202) {
            $data = json_decode($response, true);
            error_log('GoValidOJS: HTTP 202 - Additional verification required');
            error_log('GoValidOJS: 2FA response: ' . json_encode($data));

            // Check for 2FA requirement
            if (isset($data['requires_2fa']) && $data['requires_2fa'] === true) {
                error_log('GoValidOJS: 2FA verification required');
                return [
                    'success' => false,
                    'requires_2fa' => true,
                    'message' => $data['message'] ?? '2FA verification required',
                    '2fa_session_token' => $data['error']['2fa_session_token'] ?? null,
                    'backup_tokens_available' => $data['error']['backup_tokens_available'] ?? false,
                    'error_code' => $data['error']['code'] ?? '2FA_REQUIRED',
                    'error_details' => $data['error']['details'] ?? 'Please provide your 6-digit authentication code'
                ];
            }

            // Check for Email OTP requirement
            if (isset($data['otp_required']) && $data['otp_required'] === true) {
                error_log('GoValidOJS: Email OTP verification required');
                return [
                    'success' => false,
                    'otp_required' => true,
                    'message' => $data['message'] ?? 'Email OTP verification required',
                    'otp_session_token' => $data['error']['otp_session_token'] ?? null,
                    'risk_level' => $data['error']['risk_level'] ?? 'high',
                    'masked_email' => $data['error']['masked_email'] ?? null,
                    'error_code' => $data['error']['code'] ?? 'EMAIL_OTP_REQUIRED',
                    'error_details' => $data['error']['details'] ?? 'Please check your email for verification code'
                ];
            }
        }

        error_log('GoValidOJS: Login failed - HTTP ' . $httpCode . ' - Response: ' . $response);
        
        // Try to parse error message from response
        if ($response) {
            $errorData = json_decode($response, true);
            if (isset($errorData['message'])) {
                return ['success' => false, 'message' => $errorData['message']];
            }
            if (isset($errorData['error'])) {
                return ['success' => false, 'message' => $errorData['error']];
            }
        }
        
        return ['success' => false, 'message' => 'Authentication failed (HTTP ' . $httpCode . ')'];
    }

    /**
     * Verify 2FA/OTP code
     * @param string $sessionToken 2FA session token from login response
     * @param string $otpCode 6-digit OTP code
     * @return array ['success' => bool, 'access' => string, 'refresh' => string, 'user' => array] or error
     */
    public function verify2FA($sessionToken, $otpCode)
    {
        error_log('GoValidOJS: Starting 2FA verification');
        error_log('GoValidOJS: Session token: ' . substr($sessionToken, 0, 20) . '...');
        error_log('GoValidOJS: OTP code length: ' . strlen($otpCode));

        $ch = curl_init($this->apiBaseUrl . '/auth/2fa/verify/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            '2fa_session_token' => $sessionToken,
            'otp_token' => $otpCode
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        error_log('GoValidOJS: 2FA verification HTTP Code: ' . $httpCode);
        error_log('GoValidOJS: 2FA verification Response: ' . $response);

        if ($curlError) {
            error_log('GoValidOJS: 2FA verification cURL Error: ' . $curlError);
            return ['success' => false, 'message' => 'Network error: ' . $curlError];
        }

        if ($httpCode === 200) {
            $data = json_decode($response, true);
            error_log('GoValidOJS: 2FA verification parsed response: ' . json_encode($data));

            // Handle successful verification
            if (isset($data['success']) && $data['success'] === true && isset($data['data'])) {
                $responseData = $data['data'];
                if (isset($responseData['access_token']) && isset($responseData['refresh_token'])) {
                    // Store tokens in session
                    $this->storeTokens(
                        $responseData['access_token'],
                        $responseData['refresh_token'],
                        $responseData['user'] ?? []
                    );
                    return [
                        'success' => true,
                        'access' => $responseData['access_token'],
                        'refresh' => $responseData['refresh_token'],
                        'user' => $responseData['user'] ?? []
                    ];
                }
            }
        }

        // Handle errors
        error_log('GoValidOJS: 2FA verification failed - HTTP ' . $httpCode);

        if ($response) {
            $errorData = json_decode($response, true);
            if (isset($errorData['message'])) {
                return ['success' => false, 'message' => $errorData['message']];
            }
            if (isset($errorData['error'])) {
                $errorMsg = is_array($errorData['error']) ?
                    ($errorData['error']['details'] ?? $errorData['error']['message'] ?? 'Verification failed') :
                    $errorData['error'];
                return ['success' => false, 'message' => $errorMsg];
            }
        }

        return ['success' => false, 'message' => 'Invalid or expired code'];
    }

    /**
     * Verify email OTP for risk-based authentication
     * @param string $otpSessionToken Session token from OTP send
     * @param string $otpCode The 6-digit OTP code
     * @return array ['success' => bool, 'access' => string, 'refresh' => string, 'user' => array] or error
     */
    public function verifyEmailOTP($otpSessionToken, $otpCode)
    {
        error_log('GoValidOJS: Starting Email OTP verification (risk-based)');
        error_log('GoValidOJS: OTP Session token: ' . substr($otpSessionToken, 0, 20) . '...');
        error_log('GoValidOJS: OTP code length: ' . strlen($otpCode));

        // Use the unified OTP verification endpoint
        $ch = curl_init($this->apiBaseUrl . '/otp/ojs/verify/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'otp_session_token' => $otpSessionToken,
            'otp_code' => $otpCode
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        error_log('GoValidOJS: Email OTP verification HTTP Code: ' . $httpCode);
        error_log('GoValidOJS: Email OTP verification Response: ' . $response);

        if ($curlError) {
            error_log('GoValidOJS: Email OTP verification cURL Error: ' . $curlError);
            return ['success' => false, 'message' => 'Network error: ' . $curlError];
        }

        if ($httpCode === 200) {
            $data = json_decode($response, true);
            error_log('GoValidOJS: Email OTP verification parsed response: ' . json_encode($data));

            // Handle OJS-specific response format
            if (isset($data['status']) && $data['status'] === true && isset($data['content'])) {
                $content = $data['content'];

                // Check if verification was successful
                if (isset($content['success']) && $content['success'] === true) {
                    error_log('GoValidOJS: Email OTP verified successfully');

                    // Check if we have JWT tokens directly in the response
                    if (isset($content['access_token']) && isset($content['refresh_token'])) {
                        error_log('GoValidOJS: Using JWT tokens from OTP verification response');

                        // Store tokens and return
                        $enrichedUserData = $this->storeTokens(
                            $content['access_token'],
                            $content['refresh_token'],
                            $content['user'] ?? []
                        );

                        return [
                            'success' => true,
                            'access' => $content['access_token'],
                            'refresh' => $content['refresh_token'],
                            'user' => $enrichedUserData
                        ];
                    }

                    // Fallback: Use session_token method (legacy)
                    if (isset($content['session_token'])) {
                        error_log('GoValidOJS: Email OTP verified, getting JWT tokens with session: ' . $content['session_token']);

                        // Get user email from content if available
                        $userEmail = $content['user']['email'] ?? null;

                        if ($userEmail) {
                            // Use the session token to complete authentication
                            $authResult = $this->authenticateWithSessionToken($userEmail, $content['session_token']);

                            if ($authResult['success']) {
                                return [
                                    'success' => true,
                                    'access' => $authResult['access'],
                                    'refresh' => $authResult['refresh'],
                                    'user' => $authResult['user'] ?? []
                                ];
                            }
                        }
                    }
                }
            }
        }

        // Handle errors
        error_log('GoValidOJS: Email OTP verification failed - HTTP ' . $httpCode);

        if ($response) {
            $errorData = json_decode($response, true);

            // Handle OJS response format
            if (isset($errorData['content']) && isset($errorData['content']['error'])) {
                return ['success' => false, 'message' => $errorData['content']['error']];
            }

            if (isset($errorData['message'])) {
                return ['success' => false, 'message' => $errorData['message']];
            }
            if (isset($errorData['error'])) {
                $errorMsg = is_array($errorData['error']) ?
                    ($errorData['error']['details'] ?? $errorData['error']['message'] ?? 'Verification failed') :
                    $errorData['error'];
                return ['success' => false, 'message' => $errorMsg];
            }
        }

        return ['success' => false, 'message' => 'Invalid or expired code'];
    }

    /**
     * Authenticate with OJS session token to get JWT tokens
     */
    private function authenticateWithSessionToken($email, $sessionToken)
    {
        error_log('GoValidOJS: Authenticating with session token for: ' . $email);

        // Call the OJS QR generation endpoint to exchange session token for JWT
        $ch = curl_init($this->apiBaseUrl . '/ojs/generate-qr/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'email' => $email,
            'session_token' => $sessionToken,
            'qr_data' => [
                'qr_type' => 'url',
                'qr_name' => 'Test',
                'security_level' => 'OPEN',
                'form_data' => ['url' => 'https://test.com']
            ]
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        error_log('GoValidOJS: Session token exchange HTTP Code: ' . $httpCode);
        error_log('GoValidOJS: Session token exchange Response: ' . substr($response, 0, 500));

        // For now, return a simple auth response
        // TODO: Implement proper JWT token exchange
        return [
            'success' => true,
            'access' => $sessionToken,  // Temporary - use session token as access token
            'refresh' => $sessionToken,
            'user' => ['email' => $email]
        ];
    }

    /**
     * Store JWT tokens in session
     * @return array The enriched user data with profile photo
     */
    private function storeTokens($accessToken, $refreshToken, $userData)
    {
        $this->setSessionVar('govalid_access_token', $accessToken);
        $this->setSessionVar('govalid_refresh_token', $refreshToken);
        $this->setSessionVar('govalid_token_expiry', time() + 3600); // 1 hour
        $this->setSessionVar('govalid_token_timestamp', time()); // Store when token was saved

        // Write debug file
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - storeTokens called\n", FILE_APPEND);

        // After storing the token, fetch the profile to get the profile photo
        $profileData = $this->getUserProfile();
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Profile data: " . json_encode($profileData) . "\n", FILE_APPEND);

        if ($profileData && isset($profileData['profile']['profile_photo_url'])) {
            $userData['profile_photo_url'] = $profileData['profile']['profile_photo_url'];
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Added profile_photo_url: " . $userData['profile_photo_url'] . "\n", FILE_APPEND);
        } else {
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - No profile_photo_url found\n", FILE_APPEND);
        }

        // Fetch badge from the dedicated badge API endpoint (community/badges/avatar/)
        // This returns the user's PRIMARY badge set in GoValid, which is more accurate than identity_badge
        $badgeData = $this->getUserBadge();
        if ($badgeData && isset($badgeData['has_badge']) && $badgeData['has_badge'] === true) {
            $userData['has_badge'] = true;
            $userData['badge_slug'] = $badgeData['badge_slug'] ?? null;
            $userData['badge_name'] = $badgeData['badge_name'] ?? null;
            // Build full icon URL
            $iconUrl = $badgeData['avatar_icon_url'] ?? null;
            if ($iconUrl && strpos($iconUrl, 'http') !== 0) {
                $iconUrl = 'https://my.govalid.org' . $iconUrl;
            }
            $userData['avatar_icon_url'] = $iconUrl;
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Added badge from API: " . ($userData['badge_slug'] ?? 'none') . " - " . ($userData['badge_name'] ?? '') . " - " . ($userData['avatar_icon_url'] ?? '') . "\n", FILE_APPEND);
        } else {
            $userData['has_badge'] = false;
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - No badge found from badge API\n", FILE_APPEND);
        }

        // Also extract institution_info if available
        if ($profileData && isset($profileData['institution_info'])) {
            $userData['institution_info'] = $profileData['institution_info'];
            $userData['institution_name'] = $profileData['institution_info']['institution_name'] ?? null;
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Added institution_info: " . ($userData['institution_name'] ?? 'none') . "\n", FILE_APPEND);
        }

        // Store the enriched user data to session
        $this->setSessionVar('govalid_user', json_encode($userData));
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - Final user data saved to session: " . json_encode($userData) . "\n", FILE_APPEND);

        // Return the enriched user data
        return $userData;
    }
    
    /**
     * Get current token
     */
    public function getToken()
    {
        $token = $this->getSessionVar('govalid_access_token');
        $expires = $this->getSessionVar('govalid_token_expiry');
        $timestamp = $this->getSessionVar('govalid_token_timestamp');
        
        // If we have a token and it hasn't been more than 1 hour since login
        if ($token && $timestamp && (time() - $timestamp) < 3600) {
            return $token;
        }
        
        // Check expiry time
        if ($expires && $expires > time()) {
            return $token;
        }
        
        // Try to refresh if expired
        if ($this->refreshToken()) {
            return $this->getSessionVar('govalid_access_token');
        }
        
        return null;
    }
    
    /**
     * Get current user data
     */
    public function getCurrentUser()
    {
        $userData = $this->getSessionVar('govalid_user');
        return $userData ? json_decode($userData, true) : null;
    }

    /**
     * Get user profile data from GoValid API including profile photo
     */
    public function getUserProfile()
    {
        $token = $this->getToken();
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - getUserProfile: token = " . ($token ? substr($token, 0, 20) . '...' : 'NULL') . "\n", FILE_APPEND);

        if (!$token) {
            error_log('GoValidAuthManager: getUserProfile - No token available');
            @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - getUserProfile: No token, returning null\n", FILE_APPEND);
            return null;
        }

        error_log('GoValidAuthManager: Fetching user profile from API');
        $ch = curl_init($this->apiBaseUrl . '/users/profile/');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - getUserProfile API: HTTP $httpCode, Response: " . substr($response, 0, 200) . "\n", FILE_APPEND);

        error_log('GoValidAuthManager: Profile API HTTP Code: ' . $httpCode);
        if ($curlError) {
            error_log('GoValidAuthManager: Profile API cURL Error: ' . $curlError);
        }
        error_log('GoValidAuthManager: Profile API Response: ' . substr($response, 0, 500));

        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if (isset($data['success']) && $data['success'] === true) {
                error_log('GoValidAuthManager: Profile data retrieved successfully');
                if (isset($data['profile']['profile_photo_url'])) {
                    error_log('GoValidAuthManager: Profile photo URL: ' . $data['profile']['profile_photo_url']);
                } else {
                    error_log('GoValidAuthManager: No profile_photo_url in response');
                }
                return $data;
            }
        }

        error_log('GoValidAuthManager: Failed to get profile data');
        return null;
    }

    /**
     * Get user badge data from GoValid API
     * Returns badge information for displaying on profile avatar
     * Uses the community badges avatar endpoint: GET /api/v1/community/badges/avatar/
     */
    public function getUserBadge()
    {
        $token = $this->getToken();
        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - getUserBadge: token = " . ($token ? substr($token, 0, 20) . '...' : 'NULL') . "\n", FILE_APPEND);

        if (!$token) {
            error_log('GoValidAuthManager: getUserBadge - No token available');
            return null;
        }

        error_log('GoValidAuthManager: Fetching user badge from community badges API');
        $ch = curl_init($this->apiBaseUrl . '/community/badges/avatar/');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Accept: application/json',
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        @file_put_contents(__DIR__ . '/../debug_auth.log', date('Y-m-d H:i:s') . " - getUserBadge API: HTTP $httpCode, Response: " . substr($response, 0, 500) . "\n", FILE_APPEND);

        error_log('GoValidAuthManager: Badge API HTTP Code: ' . $httpCode);
        if ($curlError) {
            error_log('GoValidAuthManager: Badge API cURL Error: ' . $curlError);
        }
        error_log('GoValidAuthManager: Badge API Response: ' . substr($response, 0, 500));

        if ($httpCode === 200) {
            $data = json_decode($response, true);
            // Check for has_badge field from the community badges API response
            if ($data && isset($data['has_badge']) && $data['has_badge'] === true) {
                error_log('GoValidAuthManager: Badge data retrieved successfully - Type: ' . ($data['badge_type'] ?? 'unknown') . ', Slug: ' . ($data['badge_slug'] ?? 'unknown'));
                return $data;
            } else {
                error_log('GoValidAuthManager: User has no badge (has_badge = false or missing)');
            }
        }

        error_log('GoValidAuthManager: Failed to get badge data or no badge assigned');
        return null;
    }

    /**
     * Get user full name
     */
    public function getUserFullName()
    {
        $userJson = $this->getSessionVar('govalid_user');
        if (!$userJson) {
            return null;
        }
        
        $user = json_decode($userJson, true);
        
        // Build full name from first_name and last_name
        $fullName = trim(($user['first_name'] ?? '') . ' ' . ($user['last_name'] ?? ''));
        
        // Fall back to username if no name is set
        if (empty($fullName)) {
            $fullName = $user['username'] ?? '';
        }
        
        return $fullName;
    }
    
    /**
     * Get user info
     */
    public function getUserInfo()
    {
        $userJson = $this->getSessionVar('govalid_user');
        if (!$userJson) {
            return null;
        }
        
        return json_decode($userJson, true);
    }
    
    /**
     * Get user email from stored session data
     * @return string|null
     */
    public function getUserEmail()
    {
        $user = $this->getUserInfo();
        return $user['email'] ?? $user['username'] ?? null;
    }
    
    /**
     * Check if user is authenticated
     */
    public function isAuthenticated()
    {
        // First check if we have a token in session
        $token = $this->getSessionVar('govalid_access_token');
        $timestamp = $this->getSessionVar('govalid_token_timestamp');
        
        // If we have a token and it's less than 1 hour old, consider authenticated
        if ($token && $timestamp && (time() - $timestamp) < 3600) {
            return true;
        }
        
        // Otherwise check normally
        return $this->getToken() !== null;
    }
    
    /**
     * Logout
     */
    public function logout()
    {
        $this->unsetSessionVar('govalid_access_token');
        $this->unsetSessionVar('govalid_refresh_token');
        $this->unsetSessionVar('govalid_user');
        $this->unsetSessionVar('govalid_token_expiry');
        $this->unsetSessionVar('govalid_token_timestamp');
    }
    
    /**
     * Refresh access token
     */
    public function refreshToken()
    {
        $refreshToken = $this->getSessionVar('govalid_refresh_token');
        
        if (!$refreshToken) {
            return false;
        }
        
        $ch = curl_init($this->apiBaseUrl . '/auth/refresh/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'refresh' => $refreshToken
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            
            // Handle the actual API response format
            if (isset($data['success']) && $data['success'] === true && isset($data['data']['access_token'])) {
                $this->setSessionVar('govalid_access_token', $data['data']['access_token']);
                $this->setSessionVar('govalid_token_expiry', time() + 3600);
                $this->setSessionVar('govalid_token_timestamp', time()); // Update timestamp
                return true;
            }
            
            // Fallback for direct token format
            if (isset($data['access'])) {
                $this->setSessionVar('govalid_access_token', $data['access']);
                $this->setSessionVar('govalid_token_expiry', time() + 3600);
                $this->setSessionVar('govalid_token_timestamp', time()); // Update timestamp
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Refresh access token using a provided refresh token
     */
    public function refreshTokenWithToken($refreshToken)
    {
        if (!$refreshToken) {
            return false;
        }
        
        $ch = curl_init($this->apiBaseUrl . '/auth/refresh/');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'refresh' => $refreshToken
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            
            // Handle the actual API response format
            if (isset($data['success']) && $data['success'] === true && isset($data['data']['access_token'])) {
                // Store the new tokens
                $this->storeTokens(
                    $data['data']['access_token'],
                    $refreshToken,
                    $data['data']['user'] ?? []
                );
                return true;
            }
            
            // Fallback for direct token format
            if (isset($data['access'])) {
                $this->storeTokens($data['access'], $refreshToken, $data['user'] ?? []);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Make authenticated API request
     */
    public function apiRequest($endpoint, $data = [], $method = 'POST', $useVersionedApi = true)
    {
        $token = $this->getToken();
        if (!$token) {
            // Try to refresh token
            if ($this->refreshToken()) {
                $token = $this->getToken();
            } else {
                return ['success' => false, 'message' => 'Not authenticated'];
            }
        }

        // Use versioned or non-versioned API based on endpoint
        $baseUrl = $useVersionedApi ? $this->apiBaseUrl : $this->apiBaseUrlNoVersion;
        $fullUrl = $baseUrl . '/' . ltrim($endpoint, '/');

        // Detailed logging for debugging
        $this->debugLog('========== API REQUEST ==========');
        $this->debugLog('Endpoint: ' . $fullUrl);
        $this->debugLog('Method: ' . $method);
        $this->debugLog('JWT Token: ' . substr($token, 0, 80) . '...');
        $this->debugLog('Request Data: ' . json_encode($data));
        
        $ch = curl_init($fullUrl);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        
        if ($method === 'POST' || $method === 'PUT') {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json',
            'Authorization: Bearer ' . $token
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        // If token expired, try to refresh and retry
        if ($httpCode === 401) {
            if ($this->refreshToken()) {
                return $this->apiRequest($endpoint, $data, $method, $useVersionedApi);
            }
        }
        
        $decodedResponse = json_decode($response, true);

        $this->debugLog('API Response HTTP Code: ' . $httpCode);
        $this->debugLog('API Response Body: ' . substr($response, 0, 1000));
        $this->debugLog('========== API REQUEST END ==========');
        
        // If it's a 500 error, log the full response for debugging
        if ($httpCode === 500) {
            error_log('GoValidOJS API 500 Error - Full Response: ' . $response);
        }
        
        // Log any cURL errors
        $curlError = curl_error($ch);
        if ($curlError) {
            error_log('GoValidOJS API cURL Error: ' . $curlError);
        }
        
        // For successful responses, return the actual API response
        if ($httpCode >= 200 && $httpCode < 300) {
            // If the API returns a standard format with success/data structure
            if (isset($decodedResponse['success']) && isset($decodedResponse['data'])) {
                return $decodedResponse;
            }
            // Otherwise wrap it in our standard format
            return [
                'success' => true,
                'data' => $decodedResponse,
                'httpCode' => $httpCode
            ];
        } else {
            // For error responses, maintain our error format
            return [
                'success' => false,
                'message' => $decodedResponse['message'] ?? $decodedResponse['error'] ?? 'Request failed',
                'data' => $decodedResponse,
                'httpCode' => $httpCode
            ];
        }
    }
    
    
    /**
     * Get user subscription and quota information
     * @return array
     */
    public function getUserSubscription()
    {
        try {
            // Try multiple subscription endpoints to find the correct one
            $endpoints = [
                '/ojs/check-subscription/?email=' . urlencode($this->getUserEmail()),
                '/users/subscriptions/',
                '/financial/subscription/',
                '/subscription/',
                '/user/subscription/',
                '/credits/subscription/'
            ];
            
            $finalResponse = null;
            $workingEndpoint = null;
            
            foreach ($endpoints as $endpoint) {
                error_log('GoValidAuthManager: Trying subscription endpoint: ' . $endpoint);
                $response = $this->apiRequest($endpoint, [], 'GET');
                
                error_log('GoValidAuthManager: Response from ' . $endpoint . ': ' . json_encode($response));
                
                if ($response['success'] && isset($response['data'])) {
                    $finalResponse = $response;
                    $workingEndpoint = $endpoint;
                    error_log('GoValidAuthManager: Found working endpoint: ' . $endpoint);
                    break;
                }
            }
            
            if (!$finalResponse) {
                error_log('GoValidAuthManager: No working subscription endpoint found');
                return [
                    'success' => false,
                    'message' => 'No subscription endpoints available'
                ];
            }
            
            $data = $finalResponse['data'];
            
            // Log the raw data for debugging
            error_log('GoValidAuthManager: Raw subscription data from ' . $workingEndpoint . ': ' . json_encode($data));
            error_log('GoValidAuthManager: Data keys: ' . implode(', ', array_keys($data)));

            // Debug: Check for profile.business_model in different locations
            if (isset($data['profile'])) {
                error_log('GoValidAuthManager: Found profile at data level: ' . json_encode($data['profile']));
            }
            if (isset($data['content']['profile'])) {
                error_log('GoValidAuthManager: Found profile at data.content level: ' . json_encode($data['content']['profile']));
            }
            if (isset($data['content']['credit_balance'])) {
                error_log('GoValidAuthManager: Found credit_balance at data.content level: ' . json_encode($data['content']['credit_balance']));
            }

            // Check for new multi-currency API format (2024+)
            // This format has profile.business_model and credit_balance
            // Check both at data level and data.content level
            $apiData = null;
            if (isset($data['profile']['business_model'])) {
                $apiData = $data;
                error_log('GoValidAuthManager: Found new API format at data level');
            } elseif (isset($data['content']['profile']['business_model'])) {
                $apiData = $data['content'];
                error_log('GoValidAuthManager: Found new API format at data.content level');
            }

            if ($apiData !== null) {
                $businessModel = $apiData['profile']['business_model'];
                error_log('GoValidAuthManager: Found new API format with profile.business_model: ' . $businessModel);

                if ($businessModel === 'credit' && isset($apiData['credit_balance'])) {
                    // Credit-based user with multi-currency support
                    $creditBalance = $apiData['credit_balance'];
                    error_log('GoValidAuthManager: Processing credit user with multi-currency. Credit balance: ' . json_encode($creditBalance));

                    $balance = $creditBalance['balance'] ?? 0;
                    $currency = $creditBalance['currency'] ?? 'IDR';
                    $currencySymbol = $creditBalance['currency_symbol'] ?? 'Rp';
                    $formattedBalance = $creditBalance['formatted_balance'] ?? ($currencySymbol . ' ' . number_format($balance, 2));
                    $bonusQr = $creditBalance['bonus_qr_quota'] ?? 0;
                    $canGenerate = $creditBalance['can_generate_qr'] ?? ($balance > 0 || $bonusQr > 0);
                    $estimatedCapacity = $creditBalance['estimated_qr_capacity'] ?? $bonusQr;

                    // Also keep balance_idr for backward compatibility (convert if needed)
                    $balanceIdr = $creditBalance['balance_idr'] ?? 0;
                    if ($balanceIdr == 0 && $balance > 0 && $currency === 'USD') {
                        // Rough conversion for backward compatibility (1 USD ~ 15,500 IDR)
                        $balanceIdr = $balance * 15500;
                    }

                    return [
                        'success' => true,
                        'business_model' => 'credit',
                        // New multi-currency fields
                        'balance' => $balance,
                        'currency' => $currency,
                        'currency_symbol' => $currencySymbol,
                        'formatted_balance' => $formattedBalance,
                        // Legacy field for backward compatibility
                        'balance_idr' => $balanceIdr,
                        // QR capacity
                        'bonus_qr_quota' => $bonusQr,
                        'estimated_qr_capacity' => $estimatedCapacity,
                        'can_generate_qr' => $canGenerate,
                        'qr_capacity' => $estimatedCapacity,
                        // Statistics
                        'qr_codes_generated' => $creditBalance['qr_codes_generated'] ?? 0,
                        'total_topped_up' => $creditBalance['total_topped_up'] ?? 0,
                        'total_spent' => $creditBalance['total_spent'] ?? 0,
                        'endpoint_used' => $workingEndpoint
                    ];
                } elseif ($businessModel === 'plan' && isset($apiData['subscriptions'])) {
                    // Plan-based user - use existing subscription logic
                    $subscriptions = $apiData['subscriptions'];
                    error_log('GoValidAuthManager: Processing plan user from new API format');

                    // Priority: institution > personal
                    if (isset($subscriptions['institution']) && $subscriptions['institution']['exists']) {
                        $subInfo = $subscriptions['institution'];
                        $source = 'institution';
                    } elseif (isset($subscriptions['personal']) && $subscriptions['personal']['exists']) {
                        $subInfo = $subscriptions['personal'];
                        $source = 'personal';
                    } else {
                        // No active subscription, but user might have credit balance as fallback
                        if (isset($apiData['credit_balance']) && $apiData['credit_balance']['exists']) {
                            $creditBalance = $apiData['credit_balance'];
                            return [
                                'success' => true,
                                'business_model' => 'credit',
                                'balance' => $creditBalance['balance'] ?? 0,
                                'currency' => $creditBalance['currency'] ?? 'IDR',
                                'currency_symbol' => $creditBalance['currency_symbol'] ?? 'Rp',
                                'formatted_balance' => $creditBalance['formatted_balance'] ?? 'Rp 0',
                                'balance_idr' => $creditBalance['balance_idr'] ?? 0,
                                'bonus_qr_quota' => $creditBalance['bonus_qr_quota'] ?? 0,
                                'can_generate_qr' => $creditBalance['can_generate_qr'] ?? false,
                                'qr_capacity' => $creditBalance['estimated_qr_capacity'] ?? 0,
                                'endpoint_used' => $workingEndpoint
                            ];
                        }
                        return [
                            'success' => false,
                            'message' => 'No active subscriptions found'
                        ];
                    }

                    $institutionName = null;
                    if ($source === 'institution') {
                        $institutionName = $subInfo['institution_name'] ?? $apiData['profile']['institution_name'] ?? null;
                    }

                    return [
                        'success' => true,
                        'business_model' => 'plan',
                        'plan_name' => $subInfo['plan_name'] ?? 'Business',
                        'plan_tier' => $subInfo['plan_tier'] ?? 'BUSI',
                        'monthly_qr_limit' => $subInfo['monthly_limit'] ?? 0,
                        'qr_codes_used' => $subInfo['used_this_month'] ?? 0,
                        'remaining_qr_codes' => $subInfo['remaining'] ?? 0,
                        'can_generate_qr' => ($subInfo['remaining'] ?? 0) > 0,
                        'qr_capacity' => $subInfo['remaining'] ?? 0,
                        'is_active' => $subInfo['is_active'] ?? true,
                        'subscription_source' => $source,
                        'institution_name' => $institutionName,
                        'endpoint_used' => $workingEndpoint
                    ];
                }
            }

            // Check for OJS-specific subscription_info structure first
            if (isset($data['subscription_info'])) {
                error_log('GoValidAuthManager: Found OJS subscription_info structure');
                $subscriptionInfo = $data['subscription_info'];
                
                return [
                    'success' => true,
                    'business_model' => 'plan',
                    'plan_name' => $subscriptionInfo['plan_name'] ?? 'Business',
                    'plan_tier' => $subscriptionInfo['plan_tier'] ?? 'BUSI',
                    'monthly_qr_limit' => $subscriptionInfo['monthly_limit'] ?? 0,
                    'qr_codes_used' => $subscriptionInfo['used_this_month'] ?? 0,
                    'remaining_qr_codes' => $subscriptionInfo['remaining'] ?? 0,
                    'can_generate_qr' => ($subscriptionInfo['remaining'] ?? 0) > 0,
                    'qr_capacity' => $subscriptionInfo['remaining'] ?? 0,
                    'is_active' => true,
                    'subscription_source' => $subscriptionInfo['quota_source'] ?? 'personal',
                    'endpoint_used' => $workingEndpoint
                ];
            }
            
            // Check for new OJS check-subscription format
            if (isset($data['content']) && isset($data['content']['subscriptions'])) {
                error_log('GoValidAuthManager: Found OJS check-subscription format');
                @file_put_contents(__DIR__ . '/../debug.log', date('Y-m-d H:i:s') . ' - Found OJS check-subscription format' . "\n", FILE_APPEND);

                $subscriptions = $data['content']['subscriptions'];

                // DEBUG: Log the full subscriptions data
                error_log('GoValidAuthManager: Full subscriptions data: ' . json_encode($subscriptions));
                @file_put_contents(__DIR__ . '/../debug.log', date('Y-m-d H:i:s') . ' - Full subscriptions data: ' . json_encode($subscriptions) . "\n", FILE_APPEND);

                // Get profile data if available
                $profile = isset($data['content']['profile']) ? $data['content']['profile'] : [];

                // DEBUG: Log institution and personal subscription status
                $instExists = isset($subscriptions['institution']['exists']) ? ($subscriptions['institution']['exists'] ? 'YES' : 'NO') : 'NOT SET';
                $persExists = isset($subscriptions['personal']['exists']) ? ($subscriptions['personal']['exists'] ? 'YES' : 'NO') : 'NOT SET';
                error_log('GoValidAuthManager: Institution exists? ' . $instExists);
                error_log('GoValidAuthManager: Personal exists? ' . $persExists);
                @file_put_contents(__DIR__ . '/../debug.log', date('Y-m-d H:i:s') . ' - Institution exists? ' . $instExists . "\n", FILE_APPEND);
                @file_put_contents(__DIR__ . '/../debug.log', date('Y-m-d H:i:s') . ' - Personal exists? ' . $persExists . "\n", FILE_APPEND);

                // Priority: institution > personal
                if (isset($subscriptions['institution']) && $subscriptions['institution']['exists']) {
                    $subInfo = $subscriptions['institution'];
                    $source = 'institution';
                } elseif (isset($subscriptions['personal']) && $subscriptions['personal']['exists']) {
                    $subInfo = $subscriptions['personal'];
                    $source = 'personal';
                } else {
                    error_log('GoValidAuthManager: No active subscriptions found in OJS format');
                    return [
                        'success' => false,
                        'message' => 'No active subscriptions found'
                    ];
                }

                error_log('GoValidAuthManager: Using ' . $source . ' subscription: ' . json_encode($subInfo));
                
                // Get institution name from either subscription data or profile
                $institutionName = null;
                if ($source === 'institution') {
                    $institutionName = $subInfo['institution_name'] ?? $profile['institution_name'] ?? null;
                }
                
                return [
                    'success' => true,
                    'business_model' => 'plan',
                    'plan_name' => $subInfo['plan_name'] ?? 'Business',
                    'plan_tier' => $subInfo['plan_tier'] ?? 'BUSI',
                    'monthly_qr_limit' => $subInfo['monthly_limit'] ?? 0,
                    'qr_codes_used' => $subInfo['used_this_month'] ?? 0,
                    'remaining_qr_codes' => $subInfo['remaining'] ?? 0,
                    'can_generate_qr' => ($subInfo['remaining'] ?? 0) > 0,
                    'qr_capacity' => $subInfo['remaining'] ?? 0,
                    'is_active' => $subInfo['is_active'] ?? true,
                    'subscription_source' => $source,
                    'institution_name' => $institutionName,
                    'endpoint_used' => $workingEndpoint
                ];
            } else {
            
            // Check all possible business model field names
            $businessModel = null;
            $possibleBusinessFields = ['business_model', 'model', 'subscription_type', 'type', 'plan_type'];
            foreach ($possibleBusinessFields as $field) {
                if (isset($data[$field])) {
                    $businessModel = $data[$field];
                    error_log('GoValidAuthManager: Found business model in field "' . $field . '": ' . $businessModel);
                    break;
                }
            }
            
            // Check for different business model values
            $isPlanBased = in_array($businessModel, ['plan-based', 'plan_based', 'plan', 'subscription', 'monthly']);
            $isCreditBased = in_array($businessModel, ['credit-based', 'credit_based', 'credit', 'prepaid']);
            
            error_log('GoValidAuthManager: Business model analysis - Value: ' . $businessModel . ', isPlan: ' . ($isPlanBased ? 'YES' : 'NO') . ', isCredit: ' . ($isCreditBased ? 'YES' : 'NO'));
                
                // WORKAROUND: Handle API bug where institution members show as credit-based
                if ($isCreditBased && isset($data['credit_info']) && 
                    isset($data['credit_info']['balance_idr']) && 
                    $data['credit_info']['balance_idr'] === 0) {
                    
                    error_log('GoValidAuthManager: Detected possible institution member showing as credit user');
                    
                    // Try alternative endpoint if we haven't already
                    if ($workingEndpoint !== '/financial/subscription/') {
                        $altResponse = $this->apiRequest('/financial/subscription/', [], 'GET');
                        if ($altResponse['success'] && isset($altResponse['data'])) {
                            $altData = $altResponse['data'];
                            $altBusinessModel = $altData['business_model'] ?? $altData['model'] ?? null;
                            if (in_array($altBusinessModel, ['plan-based', 'plan_based', 'plan'])) {
                                error_log('GoValidAuthManager: User is actually plan-based via alternative endpoint');
                                $data = $altData;
                                $businessModel = $altBusinessModel;
                                $isPlanBased = true;
                                $isCreditBased = false;
                            }
                        }
                    }
                }
                
                // Handle plan-based users (including institution members)
                if ($isPlanBased) {
                    // Extract plan information with flexible field names
                    $planData = $data['plan'] ?? $data['subscription_plan'] ?? $data['current_plan'] ?? [];
                    $planName = $planData['name'] ?? $planData['plan_name'] ?? $data['plan_name'] ?? 'Free';
                    $planTier = $planData['tier'] ?? $planData['level'] ?? $data['tier'] ?? 'FREE';
                    
                    // Extract quota information with flexible field names
                    $monthlyLimit = $planData['monthly_qr_limit'] ?? $planData['qr_limit'] ?? $data['monthly_qr_limit'] ?? $data['qr_limit'] ?? 10;
                    $qrUsed = $data['qr_codes_generated_this_month'] ?? $data['qr_codes_used'] ?? $data['used_this_month'] ?? 0;
                    $remaining = $data['remaining_qr_codes'] ?? $data['remaining'] ?? ($monthlyLimit - $qrUsed);
                    
                    error_log('GoValidAuthManager: Plan parsing - Name: ' . $planName . ', Tier: ' . $planTier . ', Limit: ' . $monthlyLimit . ', Used: ' . $qrUsed . ', Remaining: ' . $remaining);
                    
                    $result = [
                        'success' => true,
                        'business_model' => 'plan',
                        'plan_name' => $planName,
                        'plan_tier' => $planTier,
                        'monthly_qr_limit' => $monthlyLimit,
                        'qr_codes_used' => $qrUsed,
                        'remaining_qr_codes' => $remaining,
                        'can_generate_qr' => $remaining > 0,
                        'qr_capacity' => $remaining,
                        'is_active' => $data['is_active'] ?? $data['active'] ?? true,
                        'subscription_source' => $data['subscription_source'] ?? $data['source'] ?? 'personal'
                    ];
                    
                    // Add institution information if available
                    if (isset($data['subscription_source']) && $data['subscription_source'] === 'institution') {
                        $result['membership_type'] = 'institution';
                        $result['institution_name'] = $data['institution_name'] ?? null;
                        $result['member_quota'] = $data['member_quota'] ?? $data['plan']['monthly_qr_limit'];
                        $result['department'] = $data['department'] ?? null;
                    }
                    
                    return $result;
                } else if ($isCreditBased || (!$isPlanBased && !$isCreditBased)) {
                    // Credit-based user OR fallback when business model is unclear
                    error_log('GoValidAuthManager: Processing as credit-based user (detected: ' . ($isCreditBased ? 'YES' : 'FALLBACK') . ')');
                    
                    // Credit-based user - updated to match new API structure with flexible field names
                    $creditInfo = $data['credit_info'] ?? $data['credits'] ?? $data['balance_info'] ?? [];
                    
                    // Extract balance with flexible field names
                    $balanceIdr = $creditInfo['balance_idr'] ?? $creditInfo['balance'] ?? $data['balance_idr'] ?? $data['balance'] ?? 0;
                    $bonusQr = $creditInfo['bonus_qr_quota'] ?? $creditInfo['bonus_qr'] ?? $data['bonus_qr_quota'] ?? $data['bonus_qr'] ?? 0;
                    $canGenerate = $data['can_generate_qr'] ?? $data['can_generate'] ?? ($balanceIdr > 0 || $bonusQr > 0);
                    $capacity = $data['qr_capacity'] ?? $data['capacity'] ?? ($bonusQr + ($balanceIdr > 0 ? 1 : 0));
                    
                    error_log('GoValidAuthManager: Credit parsing - Balance: ' . $balanceIdr . ', Bonus QR: ' . $bonusQr . ', Can Generate: ' . ($canGenerate ? 'YES' : 'NO') . ', Capacity: ' . $capacity);
                    
                    return [
                        'success' => true,
                        'business_model' => 'credit',
                        'balance_idr' => $balanceIdr,
                        'bonus_qr_quota' => $bonusQr,
                        'can_generate_qr' => $canGenerate,
                        'qr_capacity' => $capacity,
                        'total_topped_up' => $creditInfo['total_topped_up'] ?? $creditInfo['topped_up'] ?? 0,
                        'total_spent' => $creditInfo['total_spent'] ?? $creditInfo['spent'] ?? 0,
                        'qr_codes_generated' => $creditInfo['qr_codes_generated'] ?? $creditInfo['generated'] ?? 0
                    ];
                } else {
                    // Unknown business model - return raw data for debugging
                    error_log('GoValidAuthManager: Unknown business model, returning raw data for debugging');
                    return [
                        'success' => true,
                        'business_model' => 'unknown',
                        'raw_data' => $data,
                        'detected_model' => $businessModel,
                        'endpoint_used' => $workingEndpoint,
                        'message' => 'Unknown business model: ' . $businessModel
                    ];
                }
            }
            
            return [
                'success' => false,
                'message' => 'Failed to fetch subscription data'
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error fetching subscription: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Check if user can generate QR code
     * @param string $qrType Type of QR code to generate
     * @param int $quantity Number of QR codes to generate
     * @return array
     */
    public function checkQRGenerationCapability($qrType = 'url', $quantity = 1)
    {
        $subscription = $this->getUserSubscription();
        
        if (!$subscription['success']) {
            return [
                'can_generate' => false,
                'reason' => 'subscription_check_failed',
                'message' => $subscription['message']
            ];
        }
        
        // Check if user is on Free plan
        if ($subscription['business_model'] === 'plan' && $subscription['plan_tier'] === 'FREE') {
            return [
                'can_generate' => false,
                'reason' => 'free_plan',
                'message' => 'Certificate generation requires at least Starter plan. Please upgrade your subscription.',
                'subscription_info' => [
                    'plan' => $subscription['plan_name'],
                    'tier' => $subscription['plan_tier']
                ]
            ];
        }
        
        if (!$subscription['can_generate_qr']) {
            if ($subscription['business_model'] === 'plan') {
                return [
                    'can_generate' => false,
                    'reason' => 'quota_exceeded',
                    'message' => 'Monthly QR code limit reached',
                    'subscription_info' => [
                        'plan' => $subscription['plan_name'],
                        'used' => $subscription['qr_codes_used'],
                        'limit' => $subscription['monthly_qr_limit']
                    ]
                ];
            } else {
                return [
                    'can_generate' => false,
                    'reason' => 'insufficient_balance',
                    'message' => 'Insufficient credit balance',
                    'balance_info' => [
                        'current_balance' => $subscription['balance_idr'],
                        'required' => $this->getQRPrice($qrType, $quantity)
                    ]
                ];
            }
        }
        
        // Additional check for credit users - verify balance against price
        if ($subscription['business_model'] === 'credit') {
            $price = $this->getQRPrice($qrType, $quantity);
            if ($subscription['balance_idr'] < $price && $subscription['bonus_qr_quota'] < $quantity) {
                return [
                    'can_generate' => false,
                    'reason' => 'insufficient_balance',
                    'message' => 'Insufficient balance for this QR type',
                    'balance_info' => [
                        'current_balance' => $subscription['balance_idr'],
                        'bonus_qr' => $subscription['bonus_qr_quota'],
                        'required' => $price
                    ]
                ];
            }
        }
        
        return [
            'can_generate' => true,
            'subscription_info' => $subscription
        ];
    }
    
    /**
     * Test API endpoints to find the correct one
     * @return array
     */
    public function testQREndpoints()
    {
        $endpoints = [
            '/qr/',
            '/qr',
            '/qr_codes/',
            '/qr_codes',
            '/qr/generate/',
            '/qr/generate',
            '/qr_codes/generate/',
            '/qr_codes/generate',
            '/api/v1/qr/',
            '/api/v1/qr',
            '/api/v1/qr_codes/',
            '/api/v1/qr_codes',
            '/api/v1/qr/generate/',
            '/api/v1/qr_codes/generate/'
        ];
        
        $results = [];
        
        foreach ($endpoints as $endpoint) {
            try {
                $response = $this->apiRequest($endpoint, [], 'GET');
                $results[$endpoint] = [
                    'success' => $response['success'] ?? false,
                    'httpCode' => $response['httpCode'] ?? 0,
                    'message' => $response['message'] ?? 'No message'
                ];
            } catch (\Exception $e) {
                $results[$endpoint] = [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * Get QR code pricing for credit users
     * @param string $qrType
     * @param int $quantity
     * @return int Price in IDR
     */
    public function getQRPrice($qrType = 'url', $quantity = 1)
    {
        try {
            $response = $this->apiRequest(
                '/credits/api/pricing/?qr_type=' . $qrType . '&quantity=' . $quantity,
                [],
                'GET'
            );
            
            if (isset($response['data']['total_price'])) {
                return $response['data']['total_price'];
            }
        } catch (\Exception $e) {
            // Default pricing if API fails
        }
        
        // Default pricing structure
        $basePrices = [
            'url' => 5000,
            'text' => 5000,
            'email' => 5000,
            'phone' => 5000,
            'sms' => 5000,
            'wifi' => 7500,
            'vcard' => 7500,
            'location' => 10000,
            'pdf' => 15000,
            'excel' => 15000,
            'document' => 15000
        ];
        
        $basePrice = $basePrices[$qrType] ?? 5000;
        return $basePrice * $quantity;
    }
    
    /**
     * Send OTP for OJS QR generation
     * @param string $email User email
     * @return array
     */
    public function sendOJSOtp($email)
    {
        try {
            error_log('GoValidOJS: Sending OTP to email: ' . $email);
            error_log('GoValidOJS: Using UNIFIED OTP endpoint for certificate signing');

            // Use unified OTP - force send for certificate signing (not risk-based login)
            $response = $this->apiRequest('/otp/unified/create-and-send/', [
                'platform' => 'ojs',
                'purpose' => 'certificate_signing',
                'delivery_method' => 'email',
                'destination' => $email
            ], 'POST', true);

            error_log('GoValidOJS: Unified OTP send response: ' . json_encode($response));

            // Check if we got a successful response
            if (isset($response['httpCode'])) {
                error_log('GoValidOJS: Unified OTP API HTTP Code: ' . $response['httpCode']);
            }

            // Unified OTP returns session_id instead of otp_id
            if (isset($response['data']['session_id'])) {
                error_log('GoValidOJS: OTP session ID received: ' . $response['data']['session_id']);
                // Map session_id to otp_id for backward compatibility
                $response['data']['otp_id'] = $response['data']['session_id'];
            }

            return $response;
        } catch (\Exception $e) {
            error_log('GoValidOJS: Exception in sendOJSOtp: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Send OTP with fast-fail timeout for queue system
     * @param string $email User email
     * @param int $timeout Timeout in seconds (default 3)
     * @return array ['success' => bool, 'session_id' => string] or error
     */
    public function sendOJSOtpFastFail($email, $timeout = 3)
    {
        try {
            error_log('GoValidOJS: Fast-fail OTP send to: ' . $email . ' with timeout: ' . $timeout . 's');

            $token = $this->getToken();
            if (!$token) {
                error_log('GoValidOJS: Fast-fail - no auth token');
                return ['success' => false, 'message' => 'Not authenticated'];
            }

            $url = $this->apiBaseUrl . '/otp/unified/create-and-send/';
            $payload = [
                'platform' => 'ojs',
                'purpose' => 'certificate_signing',
                'delivery_method' => 'email',
                'destination' => $email
            ];

            error_log('GoValidOJS: Fast-fail URL: ' . $url);

            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json',
                'Authorization: Bearer ' . $token
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);           // Total timeout
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);    // Connection timeout
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            error_log('GoValidOJS: Fast-fail HTTP Code: ' . $httpCode);

            if ($curlError) {
                error_log('GoValidOJS: Fast-fail cURL error: ' . $curlError);
                return ['success' => false, 'message' => 'Network timeout: ' . $curlError];
            }

            if ($httpCode >= 200 && $httpCode < 300) {
                $data = json_decode($response, true);
                error_log('GoValidOJS: Fast-fail success response: ' . json_encode($data));

                // Extract session_id from response
                $sessionId = null;
                if (isset($data['data']['session_id'])) {
                    $sessionId = $data['data']['session_id'];
                } elseif (isset($data['session_id'])) {
                    $sessionId = $data['session_id'];
                }

                if ($sessionId) {
                    return [
                        'success' => true,
                        'session_id' => $sessionId
                    ];
                }

                return ['success' => false, 'message' => 'No session_id in response'];
            }

            error_log('GoValidOJS: Fast-fail failed - HTTP ' . $httpCode);
            return ['success' => false, 'message' => 'Request failed with HTTP ' . $httpCode];

        } catch (\Exception $e) {
            error_log('GoValidOJS: Fast-fail exception: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Exception: ' . $e->getMessage()];
        }
    }

    /**
     * Verify OTP for OJS QR generation
     * @param string $email User email
     * @param string $otp OTP code
     * @return array
     */
    public function verifyOJSOtp($email, $otp, $otpId = null)
    {
        try {
            error_log('GoValidOJS: Verifying OTP with UNIFIED OTP endpoint');

            // Prepare verification data for Unified OTP endpoint
            // Map parameters: otp_id -> session_id, otp -> otp_code
            $verifyData = [
                'otp_code' => $otp,
                'session_id' => $otpId
            ];

            // Validate required parameters
            if (!$otpId) {
                error_log('GoValidOJS: ERROR - No session_id provided for verification');
                return [
                    'success' => false,
                    'message' => 'Session ID required for verification'
                ];
            }

            error_log('GoValidOJS: Verifying with session_id: ' . $otpId);
            error_log('GoValidOJS: Verification data: ' . json_encode($verifyData));

            // Use Unified OTP verify endpoint - NO authentication required (session-based)
            // Must call API directly since apiRequest() requires auth token
            $fullUrl = $this->apiBaseUrl . '/otp/unified/verify/';
            error_log('GoValidOJS: Calling Unified OTP verify: ' . $fullUrl);

            $ch = curl_init($fullUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($verifyData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

            $apiResponse = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            error_log('GoValidOJS: Unified OTP verify HTTP Code: ' . $httpCode);
            error_log('GoValidOJS: Unified OTP verify raw response: ' . $apiResponse);

            if ($curlError) {
                error_log('GoValidOJS: cURL error: ' . $curlError);
                return [
                    'success' => false,
                    'message' => 'Network error: ' . $curlError
                ];
            }

            $response = json_decode($apiResponse, true);
            if (!$response) {
                error_log('GoValidOJS: Failed to parse JSON response');
                return [
                    'success' => false,
                    'message' => 'Invalid API response'
                ];
            }

            $response['httpCode'] = $httpCode;

            error_log('GoValidOJS: Unified OTP verify response: ' . json_encode($response));

            // Unified OTP verification is tracked server-side
            // For certificate signing, we need to store the session_id as the "token"
            // The session_id from send step is used as the session_token for QR generation
            if ($response['success']) {
                error_log('GoValidOJS: ✓ UNIFIED OTP: Verification successful');

                // CRITICAL: Use the QR session token from the backend response
                // The unified OTP verify endpoint already creates and stores the token in database
                // We just need to extract it from the response and use it
                $qrSessionToken = null;

                // Try to get qr_session_token from response (multiple possible locations)
                if (isset($response['qr_session_token'])) {
                    $qrSessionToken = $response['qr_session_token'];
                } elseif (isset($response['data']['qr_session_token'])) {
                    $qrSessionToken = $response['data']['qr_session_token'];
                }

                if ($qrSessionToken) {
                    error_log('GoValidOJS: Using QR session token from backend: ' . substr($qrSessionToken, 0, 20) . '...');
                } else {
                    error_log('GoValidOJS: WARNING - No qr_session_token in response, using session_id as fallback');
                    $qrSessionToken = $otpId;  // Fallback to session ID
                }

                // NOTE: No longer need to call create-qr-session endpoint
                // The backend unified OTP verify already stores the token in database + cache
                error_log('GoValidOJS: ✓ QR session token ready (stored by backend during OTP verify)');

                // Store verification status in OJS session
                $this->setSessionVar('govalid_otp_verified', true);
                $this->setSessionVar('govalid_otp_email', $email);
                $this->setSessionVar('govalid_otp_expiry', time() + 600); // 10 minutes

                // Store the QR session token (for reference)
                $this->setSessionVar('govalid_otp_session_token', $qrSessionToken);
                error_log('GoValidOJS: Session token stored in OJS session: ' . substr($qrSessionToken, 0, 20) . '...');

                // Store user data if present
                if (isset($response['data']['user'])) {
                    $this->setSessionVar('govalid_otp_user', json_encode($response['data']['user']));
                    error_log('GoValidOJS: User data stored from unified OTP response');
                }

                // Add the QR session token to the response for JavaScript to use
                $response['qr_session_token'] = $qrSessionToken;
                $response['data']['qr_session_token'] = $qrSessionToken;
                error_log('GoValidOJS: Added QR session token to response: ' . substr($qrSessionToken, 0, 20) . '...');
            } else {
                error_log('GoValidOJS: ✗ UNIFIED OTP: Verification failed');
            }

            return $response;
        } catch (\Exception $e) {
            error_log('GoValidOJS: Exception in verifyOJSOtp: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get stored OTP session token
     * @return string|null
     */
    public function getOTPSessionToken()
    {
        $token = $this->getSessionVar('govalid_otp_session_token');
        $expiry = $this->getSessionVar('govalid_otp_expiry');

        if ($token && $expiry && time() < $expiry) {
            return $token;
        }

        return null;
    }

    /**
     * Custom debug logger that writes to a specific file
     */
    private function debugLog($message)
    {
        $logFile = dirname(__FILE__) . '/../debug_pin.log';
        $timestamp = date('Y-m-d H:i:s');
        @file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND);
    }

    /**
     * Verify PIN for certificate signing
     * @param string $pin 6-digit PIN code
     * @param string|null $frontendToken Optional JWT token passed from frontend
     * @return array ['success' => bool, 'qr_session_token' => string] or error
     */
    public function verifyPIN($pin, $frontendToken = null)
    {
        try {
            $this->debugLog('========== PIN VERIFICATION START ==========');
            $this->debugLog('PIN length: ' . strlen($pin));

            // Try frontend token first, then fall back to session token
            $token = $frontendToken ?: $this->getToken();
            $this->debugLog('Using token from: ' . ($frontendToken ? 'frontend' : 'session'));
            $this->debugLog('Token available: ' . ($token ? 'yes (length=' . strlen($token) . ')' : 'NO'));

            if (!$token) {
                $this->debugLog('PIN verify FAILED - no token');
                return ['success' => false, 'message' => 'Not authenticated - no token available'];
            }

            // Call GoValid API to verify PIN
            $apiUrl = $this->apiBaseUrl . '/ojs/verify-pin/';
            $this->debugLog('Calling API: ' . $apiUrl);

            $ch = curl_init($apiUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['pin' => $pin]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json',
                'Authorization: Bearer ' . $token
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            $this->debugLog('PIN verify HTTP Code: ' . $httpCode);
            $this->debugLog('PIN verify Raw Response: ' . $response);
            $this->debugLog('PIN verify cURL Error: ' . ($curlError ?: 'none'));

            if ($curlError) {
                $this->debugLog('PIN verify FAILED - cURL error');
                return ['success' => false, 'message' => 'Network error: ' . $curlError];
            }

            $data = json_decode($response, true);
            $this->debugLog('PIN verify parsed data: ' . json_encode($data));

            // Handle nested response structure (like OJS 3.3 does)
            $content = $data['content'] ?? $data;
            $isSuccess = ($content['success'] ?? false) || ($data['success'] ?? false);

            $this->debugLog('PIN verify content: ' . json_encode($content));
            $this->debugLog('PIN verify isSuccess: ' . ($isSuccess ? 'true' : 'false'));

            if ($httpCode >= 200 && $httpCode < 300 && $isSuccess) {
                // Get qr_session_token from either location
                $qrSessionToken = $content['qr_session_token'] ?? $data['qr_session_token'] ?? null;

                $this->debugLog('PIN verify SUCCESS! qr_session_token: ' . ($qrSessionToken ? $qrSessionToken : 'NOT PROVIDED'));

                // Store the qr_session_token in OJS session for later use
                if ($qrSessionToken) {
                    $this->setSessionVar('govalid_otp_session_token', $qrSessionToken);
                    $this->setSessionVar('govalid_otp_expiry', time() + 600); // 10 minutes
                    $this->setSessionVar('govalid_otp_verified', true);
                    // Also store which JWT token was used during PIN verification
                    $this->setSessionVar('govalid_pin_jwt_token', $token);
                    // CRITICAL: Also ensure the access token is stored in session
                    // This ensures QR generation uses the same token that was used for PIN verification
                    $currentSessionToken = $this->getSessionVar('govalid_access_token');
                    $this->debugLog('Current session access_token: ' . ($currentSessionToken ? substr($currentSessionToken, 0, 50) . '...' : 'NONE'));
                    if (!$currentSessionToken || $currentSessionToken !== $token) {
                        $this->setSessionVar('govalid_access_token', $token);
                        $this->setSessionVar('govalid_token_timestamp', time());
                        $this->debugLog('UPDATED govalid_access_token in session to match PIN verify token');
                    }
                    $this->debugLog('Stored qr_session_token: ' . $qrSessionToken);
                    $this->debugLog('Stored PIN JWT token: ' . substr($token, 0, 50) . '...');
                    $this->debugLog('========== PIN VERIFICATION END (SUCCESS) ==========');
                }

                return [
                    'success' => true,
                    'message' => $content['message'] ?? $data['message'] ?? 'PIN verified successfully',
                    'verified' => true,
                    'qr_session_token' => $qrSessionToken
                ];
            }

            // Handle error
            $errorMsg = $content['error'] ?? $data['error'] ?? $content['message'] ?? $data['message'] ?? 'PIN verification failed';
            $this->debugLog('PIN verify FAILED - ' . $errorMsg);
            $this->debugLog('========== PIN VERIFICATION END (FAILED) ==========');

            return [
                'success' => false,
                'message' => $errorMsg,
                'is_locked' => $content['is_locked'] ?? $data['is_locked'] ?? false
            ];

        } catch (\Exception $e) {
            error_log('GoValidOJS: Exception in verifyPIN: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Generate QR code with automatic quota/balance checking
     * @param array $qrData QR code data
     * @return array
     */
    public function generateQRCode($qrData)
    {
        // First check if user can generate
        $capability = $this->checkQRGenerationCapability(
            $qrData['qr_type'] ?? 'url',
            $qrData['quantity'] ?? 1
        );
        
        if (!$capability['can_generate']) {
            return [
                'success' => false,
                'error' => $capability['reason'],
                'message' => $capability['message'],
                'details' => $capability
            ];
        }
        
        // Proceed with generation
        try {
            // Get current user info for email
            $userInfo = $this->getCurrentUser();
            $userEmail = $userInfo['email'] ?? '';
            
            // Check if signing is enabled to determine if OTP is required
            $isSigningEnabled = isset($qrData['toggle_sign_by_cert']) && $qrData['toggle_sign_by_cert'] === 'on';
            
            // Log the endpoint we're trying  
            error_log('GoValidOJS: Attempting QR generation at OJS endpoint: /api/v1/ojs/generate-qr/');
            error_log('GoValidOJS: QR data: ' . json_encode($qrData));
            error_log('GoValidOJS: Signing enabled: ' . ($isSigningEnabled ? 'yes' : 'no'));
            
            // Prepare request data for the simplified /ojs/generate-qr/ endpoint
            $requestData = [
                'email' => $userEmail,
                'qr_data' => $qrData['qr_data'] ?? $qrData
            ];

            // Add toggle_sign_by_cert field if present
            if (isset($qrData['toggle_sign_by_cert'])) {
                $requestData['toggle_sign_by_cert'] = $qrData['toggle_sign_by_cert'];
            }

            // Handle session_token based on certificate type
            // For signed certificates: OTP session token is required
            // For regular certificates: NO session_token in body (JWT in Authorization header is used)
            if ($isSigningEnabled) {
                $this->debugLog('========== QR GENERATION - SIGNED CERT ==========');

                // First check if session_token was sent from frontend (preferred)
                $frontendToken = $qrData['session_token'] ?? null;
                $ojsSessionToken = $this->getOTPSessionToken();
                $otpVerified = $this->getSessionVar('govalid_otp_verified');

                $this->debugLog('Frontend session_token: ' . ($frontendToken ? $frontendToken : 'NOT PROVIDED'));
                $this->debugLog('OJS session token: ' . ($ojsSessionToken ? $ojsSessionToken : 'NOT IN SESSION'));
                $this->debugLog('OTP verified flag: ' . ($otpVerified ? 'true' : 'false'));

                $otpSessionToken = $frontendToken ?: $ojsSessionToken;
                $this->debugLog('Using qr_session_token: ' . ($otpSessionToken ? $otpSessionToken : 'NONE'));

                if (!$otpSessionToken && !$otpVerified) {
                    $this->debugLog('NO session token available - OTP required');
                    return [
                        'success' => false,
                        'error' => 'otp_required',
                        'message' => 'OTP verification required for signed certificates',
                        'requires_otp' => true
                    ];
                }

                // If we have a session token, include it
                if ($otpSessionToken) {
                    $requestData['session_token'] = $otpSessionToken;
                    $this->debugLog('Including session_token in request: ' . $otpSessionToken);
                }

                // Also include the OTP verification email to help backend validate
                $otpEmail = $this->getSessionVar('govalid_otp_email');
                if ($otpEmail) {
                    $requestData['otp_verified_email'] = $otpEmail;
                    $this->debugLog('Including OTP verified email: ' . $otpEmail);
                }

                // OTP verification status is now included in qr_data.form_data by the frontend
                // This ensures the "otp_verified: true" field is in the correct location for ENTERPRISE QR metadata

            } else {
                // For unsigned certificates, DO NOT include session_token in body
                // The JWT token in Authorization header (added by apiRequest) is sufficient
                $this->debugLog('Regular certificate - using JWT token from Authorization header (no session_token in body)');
            }

            // Log the JWT token being used for API request
            $jwtToken = $this->getToken();
            $this->debugLog('JWT token for QR generation: ' . ($jwtToken ? substr($jwtToken, 0, 50) . '...' : 'NO TOKEN'));

            // IMPORTANT: For signed certificates, use the SAME JWT token that was used during PIN verification
            // The qr_session_token is bound to the specific JWT token
            $pinJwtToken = $isSigningEnabled ? $this->getSessionVar('govalid_pin_jwt_token') : null;
            $this->debugLog('PIN JWT token from session: ' . ($pinJwtToken ? substr($pinJwtToken, 0, 50) . '...' : 'NOT IN SESSION'));

            if ($pinJwtToken) {
                if ($pinJwtToken !== $jwtToken) {
                    $this->debugLog('WARNING - JWT tokens differ! Using PIN JWT token instead');
                    // Use the PIN verification JWT for the API call
                    $this->setSessionVar('govalid_access_token', $pinJwtToken);
                } else {
                    $this->debugLog('JWT tokens match - good!');
                }
            }

            $this->debugLog('Final request data: ' . json_encode($requestData));
            $this->debugLog('================================================');

            // Use the simplified OJS endpoint
            $response = $this->apiRequest('/ojs/generate-qr/', $requestData, 'POST', true);

            // Log the response
            $this->debugLog('QR generation response: ' . json_encode($response));
            
            // If there's an error, log more details
            if (!$response['success']) {
                error_log('GoValidOJS: QR generation failed with HTTP ' . ($response['httpCode'] ?? 'unknown'));
                error_log('GoValidOJS: Error message: ' . ($response['message'] ?? 'No message'));
                if (isset($response['data'])) {
                    error_log('GoValidOJS: Error data: ' . json_encode($response['data']));
                }
                
                // If OTP session is expired (only relevant for signed certificates), clear it
                if ($isSigningEnabled && isset($response['data']['error']) && 
                    strpos($response['data']['error'], 'session token') !== false) {
                    $this->unsetSessionVar('govalid_otp_session_token');
                    $this->unsetSessionVar('govalid_otp_email');
                    $this->unsetSessionVar('govalid_otp_expiry');
                    
                    return [
                        'success' => false,
                        'error' => 'otp_expired',
                        'message' => 'OTP session expired. Please verify OTP again.',
                        'requires_otp' => true
                    ];
                }
            }
            
            // Update cached subscription info after successful generation
            $this->unsetSessionVar('govalid_subscription_cache');
            
            return $response;
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => 'generation_failed',
                'message' => $e->getMessage()
            ];
        }
    }
}