<?php

/**
 * @file plugins/generic/goValidOJS/GoValidOJSPlugin.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class GoValidOJSPlugin
 *
 * Plugin to automatically generate and send PDF certificates to reviewers after completing reviews.
 */

namespace APP\plugins\generic\goValidOJS;

use APP\core\Application;
use APP\facades\Repo;
use PKP\plugins\GenericPlugin;
use PKP\plugins\Hook;
use PKP\security\Role;
use PKP\submission\PKPSubmission;
use APP\plugins\generic\goValidOJS\classes\EmailTemplateManager;

class GoValidOJSPlugin extends GenericPlugin
{
    /**
     * @copydoc Plugin::register()
     *
     * @param null|mixed $mainContextId
     */
    public function register($category, $path, $mainContextId = null): bool
    {
        $success = parent::register($category, $path, $mainContextId);

        $requestUri = $_SERVER['REQUEST_URI'] ?? 'N/A';
        $logMsg = 'GoValidOJS: register called. mainContextId: ' . var_export($mainContextId, true) . ', enabled: ' . var_export($this->getEnabled($mainContextId), true) . ', success: ' . var_export($success, true) . ', REQUEST_URI: ' . $requestUri;
        error_log($logMsg);
        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - ' . $logMsg . "\n", FILE_APPEND);

        if (Application::isUnderMaintenance()) {
            return $success;
        }

        if ($success) {
            $this->addLocaleData();
        }

        // IMPORTANT: Register page handler hook EARLY
        // Register it regardless of enabled state to ensure it's always available
        $callable = [$this, 'setupPageHandler'];
        $isCallable = is_callable($callable);
        Hook::add('LoadHandler', $callable, Hook::SEQUENCE_CORE);
        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - LoadHandler hook registered with SEQUENCE_CORE (callable: ' . ($isCallable ? 'YES' : 'NO') . ')' . "\n",
            FILE_APPEND);

        // Test the hook immediately
        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - Testing hook with dummy call...' . "\n",
            FILE_APPEND);
        $testPage = 'test';
        $testOp = 'index';
        $testSource = '';
        $testHandler = null;
        $this->setupPageHandler('LoadHandler', [&$testPage, &$testOp, &$testSource, &$testHandler]);
        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - Hook test complete' . "\n",
            FILE_APPEND);

        if ($success) {
            Hook::add('ReviewerAction::confirmReview', [$this, 'handleReviewComplete']);
        }

        // Add floating toolbar in backend for editors (Section Editors, Copyeditors, etc.)
        // Use TemplateManager::display hook which fires for all template renders
        if ($success) {
            Hook::add('TemplateManager::display', [$this, 'addBackendToolbar']);
            @file_put_contents($this->getPluginPath() . '/debug.log',
                date('Y-m-d H:i:s') . ' - TemplateManager::display hook REGISTERED' . "\n",
                FILE_APPEND);
        }

        // Smart Verify widget on frontend - use simple script injection
        if ($success) {
            Hook::add('TemplateManager::display', [$this, 'injectSmartVerifyScript']);
        }

        return $success;
    }

    /**
     * Add floating toolbar to backend pages for editors
     * This allows Section Editors, Editorial Assistants, etc. to access designers
     */
    public function addBackendToolbar(string $hookName, array $params): bool
    {
        $templateMgr = $params[0];
        $template = &$params[1];

        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - addBackendToolbar() CALLED - template: ' . $template . "\n",
            FILE_APPEND);

        // Only show on backend pages (dashboard, submissions, etc.)
        if (strpos($template, 'layouts/backend.tpl') === false &&
            strpos($template, 'dashboard/') === false &&
            strpos($template, 'submission/') === false &&
            strpos($template, 'workflow/') === false) {
            return false;
        }

        $request = Application::get()->getRequest();
        $user = $request->getUser();
        $context = $request->getContext();

        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - User: ' . ($user ? $user->getId() : 'NULL') . ', Context: ' . ($context ? $context->getId() : 'NULL') . "\n",
            FILE_APPEND);

        // Only show if user is logged in and has context
        if (!$user || !$context) {
            return false;
        }

        // Check if plugin is enabled for this context
        $contextId = $context->getId();
        if (!$this->getEnabled($contextId)) {
            return false;
        }

        // Get FAB visibility settings (default to true if not set)
        $showFabForManagers = $this->getSetting($contextId, 'showFabForManagers') ?? true;
        $showFabForEditors = $this->getSetting($contextId, 'showFabForEditors') ?? true;
        $showFabForCopyeditors = $this->getSetting($contextId, 'showFabForCopyeditors') ?? true;
        $showFabForAssistants = $this->getSetting($contextId, 'showFabForAssistants') ?? true;

        // Build allowed roles based on settings
        $allowedRoleIds = [Role::ROLE_ID_SITE_ADMIN]; // Site admin always has access

        if ($showFabForManagers) {
            $allowedRoleIds[] = Role::ROLE_ID_MANAGER;
        }
        if ($showFabForEditors) {
            $allowedRoleIds[] = Role::ROLE_ID_SUB_EDITOR;
        }
        if ($showFabForCopyeditors) {
            // Copyeditors and Layout Editors typically use SUB_EDITOR role in OJS
            // but we should also check for specific production roles if available
        }
        if ($showFabForAssistants) {
            $allowedRoleIds[] = Role::ROLE_ID_ASSISTANT;
        }

        $hasAccess = false;
        $userRoles = $user->getRoles($contextId);
        foreach ($userRoles as $role) {
            if (in_array($role->getId(), $allowedRoleIds)) {
                $hasAccess = true;
                break;
            }
        }

        if (!$hasAccess) {
            return false;
        }

        // Get URLs
        $urls = $this->getDesignerUrls($request);
        // Note: $templateMgr is already available from $params[0]

        // Inject CSS for floating toolbar
        $templateMgr->addStyleSheet(
            'govalidojs-backend-toolbar',
            '
            .govalidojs-toolbar {
                position: fixed;
                bottom: 20px;
                right: 20px;
                z-index: 9999;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            }
            .govalidojs-toolbar-toggle {
                width: 50px;
                height: 50px;
                border-radius: 50%;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                border: none;
                cursor: pointer;
                box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
                display: flex;
                align-items: center;
                justify-content: center;
                transition: transform 0.3s, box-shadow 0.3s;
            }
            .govalidojs-toolbar-toggle:hover {
                transform: scale(1.1);
                box-shadow: 0 6px 20px rgba(102, 126, 234, 0.6);
            }
            .govalidojs-toolbar-toggle svg {
                width: 24px;
                height: 24px;
                fill: white;
            }
            .govalidojs-toolbar-menu {
                position: absolute;
                bottom: 60px;
                right: 0;
                background: white;
                border-radius: 12px;
                box-shadow: 0 5px 30px rgba(0,0,0,0.2);
                padding: 8px;
                display: none;
                min-width: 200px;
            }
            .govalidojs-toolbar-menu.show {
                display: block;
                animation: slideUp 0.2s ease;
            }
            @keyframes slideUp {
                from { opacity: 0; transform: translateY(10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            .govalidojs-toolbar-menu a {
                display: flex;
                align-items: center;
                gap: 10px;
                padding: 12px 16px;
                color: #333;
                text-decoration: none;
                border-radius: 8px;
                font-size: 14px;
                transition: background 0.2s;
            }
            .govalidojs-toolbar-menu a:hover {
                background: #f0f0f0;
            }
            .govalidojs-toolbar-menu a svg {
                width: 20px;
                height: 20px;
            }
            .govalidojs-toolbar-header {
                padding: 8px 16px;
                font-size: 11px;
                color: #888;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                border-bottom: 1px solid #eee;
                margin-bottom: 4px;
            }
            ',
            ['inline' => true, 'contexts' => 'backend']
        );

        // Inject JavaScript for floating toolbar
        $templateMgr->addJavaScript(
            'govalidojs-backend-toolbar',
            '
            (function() {
                // Wait for DOM to be ready
                function initToolbar() {
                    if (document.getElementById("govalidojs-toolbar")) return;

                    var toolbar = document.createElement("div");
                    toolbar.id = "govalidojs-toolbar";
                    toolbar.className = "govalidojs-toolbar";
                    toolbar.innerHTML = \'<button class="govalidojs-toolbar-toggle" onclick="toggleGoValidMenu()" title="GoValid Tools">\' +
                        \'<svg viewBox="0 0 24 24"><path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"/></svg>\' +
                        \'</button>\' +
                        \'<div class="govalidojs-toolbar-menu" id="govalidojs-menu">\' +
                        \'<div class="govalidojs-toolbar-header">GoValid Tools</div>\' +
                        \'<a href="' . $urls['certificateDesigner'] . '" target="_blank">\' +
                        \'<svg viewBox="0 0 24 24"><path fill="#667eea" d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20M9,13V19H7V13H9M15,15V19H17V15H15M11,11V19H13V11H11Z"/></svg>\' +
                        \'Certificate Designer</a>\' +
                        \'<a href="' . $urls['emailDesigner'] . '" target="_blank">\' +
                        \'<svg viewBox="0 0 24 24"><path fill="#764ba2" d="M20,8L12,13L4,8V6L12,11L20,6M20,4H4C2.89,4 2,4.89 2,6V18A2,2 0 0,0 4,20H20A2,2 0 0,0 22,18V6C22,4.89 21.1,4 20,4Z"/></svg>\' +
                        \'Email Designer</a>\' +
                        \'</div>\';

                    document.body.appendChild(toolbar);
                }

                // Initialize when DOM is ready
                if (document.readyState === "loading") {
                    document.addEventListener("DOMContentLoaded", initToolbar);
                } else {
                    initToolbar();
                }

                // Also try after a short delay for Vue apps
                setTimeout(initToolbar, 500);
                setTimeout(initToolbar, 1500);
            })();

            function toggleGoValidMenu() {
                var menu = document.getElementById("govalidojs-menu");
                if (menu) {
                    menu.classList.toggle("show");
                }
            }

            // Close menu when clicking outside
            document.addEventListener("click", function(e) {
                var toolbar = document.getElementById("govalidojs-toolbar");
                var menu = document.getElementById("govalidojs-menu");
                if (toolbar && menu && !toolbar.contains(e.target)) {
                    menu.classList.remove("show");
                }
            });
            ',
            ['inline' => true, 'contexts' => 'backend']
        );

        return false;
    }

    /**
     * Add Smart Verify widget to frontend pages
     * This displays a floating verification widget for visitors
     */
    public function addSmartVerifyWidget(string $hookName, array $params): bool
    {
        // EARLY debug - log immediately to prove function is called
        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - addSmartVerifyWidget() CALLED\n", FILE_APPEND);

        // Prevent multiple injections
        static $alreadyInjected = false;
        if ($alreadyInjected) {
            @file_put_contents('/tmp/govalid_widget_debug.log',
                date('Y-m-d H:i:s') . " - Already injected, skipping\n", FILE_APPEND);
            return false;
        }

        $templateMgr = $params[0];
        $template = &$params[1];

        // Debug logging
        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - Template: $template\n", FILE_APPEND);

        // Only show on frontend pages (not backend/dashboard)
        if (strpos($template, 'layouts/backend.tpl') !== false ||
            strpos($template, 'dashboard') !== false ||
            strpos($template, 'management') !== false ||
            strpos($template, 'workflow') !== false ||
            strpos($template, 'goValidOJS') !== false ||
            strpos($template, 'submission') !== false) {
            @file_put_contents('/tmp/govalid_widget_debug.log',
                date('Y-m-d H:i:s') . " - SKIP: backend template\n", FILE_APPEND);
            return false;
        }

        // Check if this is a frontend template - be more permissive
        $isFrontend = (strpos($template, 'frontend/') !== false);

        if (!$isFrontend) {
            @file_put_contents('/tmp/govalid_widget_debug.log',
                date('Y-m-d H:i:s') . " - SKIP: not frontend template\n", FILE_APPEND);
            return false;
        }

        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - IS FRONTEND!\n", FILE_APPEND);

        // Get context
        $request = Application::get()->getRequest();
        $context = $request->getContext();
        if (!$context) {
            @file_put_contents('/tmp/govalid_widget_debug.log',
                date('Y-m-d H:i:s') . " - SKIP: no context\n", FILE_APPEND);
            return false;
        }

        $contextId = $context->getId();
        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - Context ID: $contextId\n", FILE_APPEND);

        // Check if Smart Verify widget is enabled
        $enableSmartVerifyWidget = $this->getSetting($contextId, 'enableSmartVerifyWidget');
        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - enableSmartVerifyWidget: " . var_export($enableSmartVerifyWidget, true) . "\n", FILE_APPEND);

        if (!$enableSmartVerifyWidget) {
            @file_put_contents('/tmp/govalid_widget_debug.log',
                date('Y-m-d H:i:s') . " - SKIP: widget not enabled\n", FILE_APPEND);
            return false;
        }

        // Mark as injected to prevent duplicates
        $alreadyInjected = true;
        @file_put_contents('/tmp/govalid_widget_debug.log',
            date('Y-m-d H:i:s') . " - INJECTING WIDGET!\n", FILE_APPEND);

        // Get widget position
        $widgetPosition = $this->getSetting($contextId, 'smartVerifyWidgetPosition') ?: 'bottom-right';

        // Assign variables to template
        $templateMgr->assign('widgetPosition', $widgetPosition);
        $templateMgr->assign('baseUrl', $request->getBaseUrl());

        // Get the widget template content
        $widgetTemplate = $this->getTemplateResource('smartVerifyWidget.tpl');

        // Register the output filter to inject the widget before </body>
        $templateMgr->registerFilter('output', function($output) use ($templateMgr, $widgetTemplate) {
            // Only inject once and if </body> is present
            if (strpos($output, '</body>') !== false && strpos($output, 'govalid-smart-verify-widget') === false) {
                $widgetHtml = $templateMgr->fetch($widgetTemplate);
                $output = str_replace('</body>', $widgetHtml . '</body>', $output);
            }
            return $output;
        });

        return false;
    }

    /**
     * Inject Smart Verify widget script on frontend pages
     * Uses simple JavaScript injection - minimal and safe
     */
    public function injectSmartVerifyScript(string $hookName, array $params): bool
    {
        static $alreadyInjected = false;
        if ($alreadyInjected) {
            return false;
        }

        $templateMgr = $params[0];
        $template = &$params[1];

        // Only inject on frontend pages
        if (strpos($template, 'frontend/') === false) {
            return false;
        }

        // Skip backend templates
        if (strpos($template, 'backend') !== false || strpos($template, 'dashboard') !== false) {
            return false;
        }

        $request = Application::get()->getRequest();
        $context = $request->getContext();
        if (!$context) {
            return false;
        }

        $contextId = $context->getId();

        // Check if widget is enabled
        if (!$this->getSetting($contextId, 'enableSmartVerifyWidget')) {
            return false;
        }

        $alreadyInjected = true;

        // Get settings
        $widgetPosition = $this->getSetting($contextId, 'smartVerifyWidgetPosition') ?: 'bottom-right';

        // Get proper plugin file URL using OJS method
        $pluginPath = $this->getPluginPath();

        // Add CSS using plugin's getPluginPath (with cache busting)
        $templateMgr->addStyleSheet(
            'smartVerifyWidget',
            $request->getBaseUrl() . '/' . $pluginPath . '/css/smart-verify-widget.css?v=20251210c',
            ['contexts' => 'frontend']
        );

        // Add jsQR library from CDN
        $templateMgr->addJavaScript(
            'jsQR',
            'https://cdn.jsdelivr.net/npm/jsqr@1.4.0/dist/jsQR.min.js',
            ['contexts' => 'frontend']
        );

        // Add widget loader script (with cache busting)
        $templateMgr->addJavaScript(
            'smartVerifyWidgetLoader',
            $request->getBaseUrl() . '/' . $pluginPath . '/js/smart-verify-widget-loader.js?v=20251210d',
            ['contexts' => 'frontend', 'priority' => 100]
        );

        // Pass config to JavaScript
        $templateMgr->addHeader(
            'smartVerifyConfig',
            '<script>window.GoValidSmartVerifyConfig = {position: "' . htmlspecialchars($widgetPosition) . '"};</script>',
            ['contexts' => 'frontend']
        );

        return false;
    }

    /**
     * Get the direct URLs for the designers
     * These URLs can be accessed by Section Editors, Editorial Assistants, etc.
     * without needing Journal Manager access
     *
     * @param PKPRequest $request
     * @return array ['certificateDesigner' => url, 'emailDesigner' => url]
     */
    public function getDesignerUrls($request): array
    {
        $context = $request->getContext();
        $contextPath = $context ? $context->getPath() : 'index';
        $baseUrl = $request->getBaseUrl();

        return [
            'certificateDesigner' => $baseUrl . '/index.php/' . $contextPath . '/govalidojs/designer',
            'emailDesigner' => $baseUrl . '/index.php/' . $contextPath . '/govalidojs/emailTemplateDesigner'
        ];
    }

    /**
     * @copydoc Plugin::getDisplayName()
     */
    public function getDisplayName(): string
    {
        return __('plugins.generic.goValidOJS.displayName');
    }

    /**
     * @copydoc Plugin::getDescription()
     */
    public function getDescription(): string
    {
        return __('plugins.generic.goValidOJS.description');
    }

    /**
     * @copydoc Plugin::getActions()
     */
    public function getActions($request, $actionArgs): array
    {
        $actions = parent::getActions($request, $actionArgs);
        
        if ($this->getEnabled()) {
            $router = $request->getRouter();
            
            // Add settings action
            $settingsUrl = $router->url($request, null, null, 'manage', null, [
                'verb' => 'settings',
                'plugin' => $this->getName(),
                'category' => 'generic'
            ]);
            
            array_unshift($actions, new \PKP\linkAction\LinkAction(
                'settings',
                new \PKP\linkAction\request\AjaxModal($settingsUrl, __('manager.plugins.settings')),
                __('manager.plugins.settings'),
                null
            ));
            
            // Certificate Designer link removed - functionality not working
            // // Add designer action using direct URL to page handler
            // // This bypasses the grid handler authorization issues
            // $contextPath = $context ? $context->getPath() : 'index';
            // $designerUrl = $request->getBaseUrl() . '/' . $contextPath . '/govalidojs/designer';
            // 
            // array_unshift($actions, new \PKP\linkAction\LinkAction(
            //     'designer',
            //     new \PKP\linkAction\request\RedirectAction($designerUrl),
            //     __('plugins.generic.goValidOJS.certificateDesigner'),
            //     null
            // ));
        }
        
        return $actions;
    }

    /**
     * @copydoc Plugin::getManagementVerbs()
     */
    public function getManagementVerbs(): array
    {
        $verbs = parent::getManagementVerbs();
        if ($this->getEnabled()) {
            $verbs[] = ['settings', __('manager.plugins.settings')];
        }
        return $verbs;
    }

    /**
     * @copydoc Plugin::getManagementVerbLinkAction()
     */
    public function getManagementVerbLinkAction($request, $verb): ?\PKP\linkAction\LinkAction
    {
        $router = $request->getRouter();
        
        list($verbName, $verbLocalized) = $verb;
        
        if ($verbName === 'settings') {
            $settingsUrl = $router->url($request, null, null, 'manage', null, [
                'verb' => 'settings',
                'plugin' => $this->getName(),
                'category' => 'generic'
            ]);
            
            return new \PKP\linkAction\LinkAction(
                $verbName,
                new \PKP\linkAction\request\AjaxModal($settingsUrl, $verbLocalized),
                $verbLocalized,
                null
            );
        }
        
        return null;
    }


    /**
     * @copydoc Plugin::manage()
     */
    public function manage($args, $request): ?\PKP\core\JSONMessage
    {
        try {
            // Debug logging
            error_log('========== GoValidOJS manage() called ==========');
            error_log('Request method: ' . $request->getRequestMethod());
            error_log('Verb: ' . $request->getUserVar('verb'));
            error_log('Plugin name from request: ' . $request->getUserVar('plugin'));
            error_log('Category from request: ' . $request->getUserVar('category'));
            
            // Check permissions
            $user = $request->getUser();
            $context = $request->getContext();
            
            if (!$user) {
                error_log('GoValidOJS: No user found');
                return new \PKP\core\JSONMessage(false, __('manager.plugins.access.denied'));
            }
            
            $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;
            
            if (!$user->hasRole([\PKP\security\Role::ROLE_ID_SITE_ADMIN, \PKP\security\Role::ROLE_ID_MANAGER], $contextId)) {
                error_log('GoValidOJS: Permission denied for user: ' . $user->getId());
                error_log('GoValidOJS: User roles: ' . json_encode($user->getRoles($contextId)));
                return new \PKP\core\JSONMessage(false, __('manager.plugins.access.denied'));
            }
            
            // Check if this is coming from our plugin
            if ($request->getUserVar('plugin') === 'goValidOJS' || $this->getName() === 'govalidojsplugin') {
                // For designer verb, we don't need the standard manage response
                if ($request->getUserVar('verb') === 'designer') {
                    $this->_showDesigner($request);
                    // This should never be reached since _showDesigner calls exit
                    return new \PKP\core\JSONMessage(true);
                }
            }
            
            switch ($request->getUserVar('verb')) {
            case 'settings':
                // Log the request details for debugging
                error_log('GoValidOJS Debug: Request method: ' . $request->getRequestMethod());
                error_log('GoValidOJS Debug: Action: ' . $request->getUserVar('action'));
                error_log('GoValidOJS Debug: Save: ' . $request->getUserVar('save'));
                error_log('GoValidOJS Debug: POST data: ' . json_encode($_POST));
                
                // Handle AJAX requests for authentication and other actions
                $action = $request->getUserVar('action');
                if ($action && in_array($action, ['login', 'logout', 'checkAuth', 'getSubscription'])) {
                    return $this->_handleAjaxActions($request);
                }
                
                // Handle form submission
                if ($request->getUserVar('save')) {
                    return $this->_handleFormSubmission($request);
                }
                
                // Show settings form
                return $this->_showSettings($request);
            case 'designer':
                // Handle AJAX requests for designer
                $action = $request->getUserVar('action');

                // Enhanced debugging - write to file
                $debugLog = $this->getPluginPath() . '/debug.log';
                @file_put_contents($debugLog, "\n==================== DESIGNER VERB ====================\n", FILE_APPEND);
                @file_put_contents($debugLog, 'Action from getUserVar: ' . var_export($action, true) . "\n", FILE_APPEND);
                @file_put_contents($debugLog, '$_POST: ' . json_encode($_POST) . "\n", FILE_APPEND);
                @file_put_contents($debugLog, '$_GET: ' . json_encode($_GET) . "\n", FILE_APPEND);
                @file_put_contents($debugLog, '$_REQUEST: ' . json_encode($_REQUEST) . "\n", FILE_APPEND);
                @file_put_contents($debugLog, 'Request Method: ' . $_SERVER['REQUEST_METHOD'] . "\n", FILE_APPEND);
                @file_put_contents($debugLog, "====================================================\n", FILE_APPEND);

                // Try to get action from multiple sources
                if (!$action) {
                    $action = $_POST['action'] ?? $_GET['action'] ?? $_REQUEST['action'] ?? null;
                    error_log('GoValidOJS Designer - Action from fallback: ' . var_export($action, true));
                }

                if ($action && in_array($action, ['list', 'login', 'logout', 'checkAuth', 'save', 'load', 'delete', 'preview', 'getSubscription', 'checkCapability', 'getUserInfo', 'getReviewers', 'getReviewersForSubmission', 'getEditorsForSubmission', 'getPatterns', 'previewPattern', 'generatePattern', 'testNumberingAPI', 'generateCertificateQR', 'proxyQRImage', 'searchReviewers', 'searchAuthors', 'searchEditors', 'refreshAuth', 'getIssues', 'getSubmissions', 'getAntiCounterfeitTags'])) {
                    error_log('GoValidOJS Designer - Routing to _handleDesignerAjax for action: ' . $action);
                    return $this->_handleDesignerAjax($request);
                }

                error_log('GoValidOJS Designer - No valid action found, showing designer page');
                // Show designer page only if no action is specified
                $this->_showDesigner($request);
                // This should never be reached since _showDesigner calls exit
                return new \PKP\core\JSONMessage(true);
        }
        
        // If no verb matches, return an error
        return new \PKP\core\JSONMessage(false, ['error' => 'Invalid verb']);
        
        } catch (\Exception $e) {
            error_log('GoValidOJS manage() Exception: ' . $e->getMessage());
            error_log('GoValidOJS manage() Stack trace: ' . $e->getTraceAsString());
            
            // Get the specific line where error occurred
            $trace = $e->getTrace();
            if (!empty($trace)) {
                error_log('GoValidOJS: Error at ' . ($trace[0]['file'] ?? 'unknown') . ':' . ($trace[0]['line'] ?? 'unknown'));
            }
            
            return new \PKP\core\JSONMessage(false, ['error' => 'Plugin error: ' . $e->getMessage()]);
        }
    }

    /**
     * Show plugin settings using proper Form class
     */
    private function _showSettings($request): \PKP\core\JSONMessage
    {
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;

        try {
            require_once($this->getPluginPath() . '/classes/GoValidOJSSettingsForm.php');
            $form = new \APP\plugins\generic\goValidOJS\classes\GoValidOJSSettingsForm($this, $contextId);
            $form->initData();

            $content = $form->fetch($request);

            return new \PKP\core\JSONMessage(true, $content);
        } catch (\Exception $e) {
            error_log('GoValidOJS Form Error: ' . $e->getMessage());
            error_log('GoValidOJS Form Stack Trace: ' . $e->getTraceAsString());
            return new \PKP\core\JSONMessage(false, 'Error loading settings form: ' . $e->getMessage());
        }
    }

    /**
     * Handle form submission
     */
    private function _handleFormSubmission($request): \PKP\core\JSONMessage
    {
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;
        
        require_once($this->getPluginPath() . '/classes/GoValidOJSSettingsForm.php');
        $form = new \APP\plugins\generic\goValidOJS\classes\GoValidOJSSettingsForm($this, $contextId);
        $form->readInputData();
        
        if ($form->validate()) {
            $form->execute();
            return new \PKP\core\JSONMessage(true);
        }
        
        return new \PKP\core\JSONMessage(false, $form->fetch($request));
    }

    /**
     * Handle AJAX actions (authentication and other non-form actions)
     */
    private function _handleAjaxActions($request): \PKP\core\JSONMessage
    {
        $action = $request->getUserVar('action');
        $context = $request->getContext();
        
        switch ($action) {
            case 'login':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                $username = $request->getUserVar('username');
                $password = $request->getUserVar('password');
                
                if (!$username || !$password) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Username and password are required']);
                }
                
                $result = $authManager->authenticate($username, $password);
                return new \PKP\core\JSONMessage(true, $result);
                
            case 'logout':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                $authManager->logout();
                return new \PKP\core\JSONMessage(true, ['success' => true]);
                
            case 'checkAuth':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                $isAuthenticated = $authManager->isAuthenticated();
                $result = ['authenticated' => $isAuthenticated];

                if ($isAuthenticated) {
                    $userData = $authManager->getCurrentUser();
                    $subscription = $authManager->getUserSubscription();

                    error_log('GoValidOJS checkAuth: User data before profile fetch: ' . json_encode($userData));

                    // Get profile data including profile photo
                    $profileData = $authManager->getUserProfile();
                    error_log('GoValidOJS checkAuth: Profile data returned: ' . json_encode($profileData));

                    if ($profileData && isset($profileData['profile']['profile_photo_url'])) {
                        // Add profile photo URL to user data
                        $userData['profile_photo_url'] = $profileData['profile']['profile_photo_url'];
                        error_log('GoValidOJS checkAuth: Added profile_photo_url: ' . $userData['profile_photo_url']);
                    } else {
                        error_log('GoValidOJS checkAuth: No profile_photo_url in profileData');
                    }

                    // Add institution_info if user is part of an institution
                    if ($profileData && isset($profileData['institution_info'])) {
                        $userData['institution_info'] = $profileData['institution_info'];
                        error_log('GoValidOJS checkAuth: Added institution_info: ' . json_encode($userData['institution_info']));
                    }

                    // Add institution_name from profile if available (for institution operators)
                    if ($profileData && isset($profileData['profile']['institution_name'])) {
                        $userData['institution_name'] = $profileData['profile']['institution_name'];
                        error_log('GoValidOJS checkAuth: Added institution_name from profile: ' . $userData['institution_name']);
                    }

                    error_log('GoValidOJS checkAuth: Final user data: ' . json_encode($userData));

                    $result['user'] = $userData;
                    $result['userInfo'] = $userData;  // For compatibility
                    $result['subscription'] = $subscription;
                }

                return new \PKP\core\JSONMessage(true, $result);
                
            case 'getSubscription':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $subscription = $authManager->getUserSubscription();
                return new \PKP\core\JSONMessage(true, $subscription);
                
            default:
                return new \PKP\core\JSONMessage(false, ['error' => 'Invalid action: ' . $action]);
        }
    }

    /**
     * Handle AJAX requests for designer
     */
    private function _handleDesignerAjax($request): \PKP\core\JSONMessage
    {
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;
        $action = $request->getUserVar('action');

        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: _handleDesignerAjax START - action: ' . $action . "\n", FILE_APPEND);

        // Actions that don't require authentication
        $publicActions = ['login', 'verify2FA', 'verifyEmailOTP', 'checkAuth', 'list', 'sendOtp', 'verifyOtp', 'getAntiCounterfeitTags', 'loadDefaultTemplate', 'listDefaultTemplates'];

        // Check authentication for protected actions
        if (!in_array($action, $publicActions)) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Action requires authentication check' . "\n", FILE_APPEND);
            require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
            $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

            if (!$authManager->isAuthenticated()) {
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Authentication FAILED - returning NOT_AUTHENTICATED' . "\n", FILE_APPEND);
                return new \PKP\core\JSONMessage(false, [
                    'success' => false,
                    'message' => 'Authentication required',
                    'error' => 'NOT_AUTHENTICATED'
                ]);
            }
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Authentication PASSED' . "\n", FILE_APPEND);
        }

        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: About to enter switch statement for action: ' . $action . "\n", FILE_APPEND);
        
        switch ($action) {
            case 'list':
                $savedTemplates = $this->getSetting($contextId, 'savedTemplates') ?: [];
                $templateList = [];
                foreach ($savedTemplates as $name => $data) {
                    $templateList[] = [
                        'name' => $name,
                        'timestamp' => $data['timestamp'] ?? ''
                    ];
                }
                
                return new \PKP\core\JSONMessage(true, ['templates' => $templateList]);

            case 'login':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                $username = $request->getUserVar('username');
                $password = $request->getUserVar('password');

                if (!$username || !$password) {
                    // Send JSON directly to avoid OJS wrapping
                    header('Content-Type: application/json');
                    echo json_encode(['status' => false, 'content' => ['success' => false, 'message' => 'Username and password are required']]);
                    exit;
                }

                $result = $authManager->authenticate($username, $password);

                // Send JSON directly to avoid OJS wrapping
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'content' => $result]);
                exit;

            case 'logout':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                $authManager->logout();
                return new \PKP\core\JSONMessage(true, ['success' => true]);

            case 'verify2FA':
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA case entered' . "\n", FILE_APPEND);
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                $sessionToken = $request->getUserVar('2fa_session_token');
                $otpCode = $request->getUserVar('otp_code');

                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA - sessionToken: ' . ($sessionToken ? substr($sessionToken, 0, 20) . '...' : 'NULL') . "\n", FILE_APPEND);
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA - otpCode: ' . ($otpCode ? 'provided (length=' . strlen($otpCode) . ')' : 'NULL') . "\n", FILE_APPEND);

                if (!$sessionToken || !$otpCode) {
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA - Missing parameters!' . "\n", FILE_APPEND);
                    header('Content-Type: application/json');
                    echo json_encode(['status' => false, 'content' => ['success' => false, 'message' => '2FA session token and OTP code are required']]);
                    exit;
                }

                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA - Calling authManager->verify2FA()' . "\n", FILE_APPEND);
                $result = $authManager->verify2FA($sessionToken, $otpCode);
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verify2FA - Result: ' . json_encode($result) . "\n", FILE_APPEND);

                // Send JSON directly
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'content' => $result]);
                exit;

            case 'verifyEmailOTP':
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP case entered' . "\n", FILE_APPEND);
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                $otpSessionToken = $request->getUserVar('otp_session_token');
                $otpCode = $request->getUserVar('otp_code');

                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP - otpSessionToken: ' . ($otpSessionToken ? substr($otpSessionToken, 0, 20) . '...' : 'NULL') . "\n", FILE_APPEND);
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP - otpCode: ' . ($otpCode ? 'provided (length=' . strlen($otpCode) . ')' : 'NULL') . "\n", FILE_APPEND);

                if (!$otpSessionToken || !$otpCode) {
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP - Missing parameters!' . "\n", FILE_APPEND);
                    header('Content-Type: application/json');
                    echo json_encode(['status' => false, 'content' => ['success' => false, 'message' => 'OTP session token and OTP code are required']]);
                    exit;
                }

                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP - Calling authManager->verifyEmailOTP()' . "\n", FILE_APPEND);
                $result = $authManager->verifyEmailOTP($otpSessionToken, $otpCode);
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: verifyEmailOTP - Result: ' . json_encode($result) . "\n", FILE_APPEND);

                // Send JSON directly
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'content' => $result]);
                exit;

            case 'refreshAuth':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                $refreshToken = $request->getUserVar('refresh_token');
                if (!$refreshToken) {
                    return new \PKP\core\JSONMessage(true, ['authenticated' => false]);
                }
                
                // Try to refresh the token
                $success = $authManager->refreshTokenWithToken($refreshToken);
                if ($success) {
                    $result = [
                        'authenticated' => true,
                        'user' => $authManager->getCurrentUser(),
                        'subscription' => $authManager->getUserSubscription()
                    ];
                } else {
                    $result = ['authenticated' => false];
                }
                
                return new \PKP\core\JSONMessage(true, $result);
                
            case 'checkAuth':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                $isAuthenticated = $authManager->isAuthenticated();
                $result = ['authenticated' => $isAuthenticated];

                if ($isAuthenticated) {
                    $userData = $authManager->getCurrentUser();
                    $subscription = $authManager->getUserSubscription();

                    error_log('GoValidOJS checkAuth: User data before profile fetch: ' . json_encode($userData));

                    // Get profile data including profile photo
                    $profileData = $authManager->getUserProfile();
                    error_log('GoValidOJS checkAuth: Profile data returned: ' . json_encode($profileData));

                    if ($profileData && isset($profileData['profile']['profile_photo_url'])) {
                        // Add profile photo URL to user data
                        $userData['profile_photo_url'] = $profileData['profile']['profile_photo_url'];
                        error_log('GoValidOJS checkAuth: Added profile_photo_url: ' . $userData['profile_photo_url']);
                    } else {
                        error_log('GoValidOJS checkAuth: No profile_photo_url in profileData');
                    }

                    // Add institution_info if user is part of an institution
                    if ($profileData && isset($profileData['institution_info'])) {
                        $userData['institution_info'] = $profileData['institution_info'];
                        error_log('GoValidOJS checkAuth: Added institution_info: ' . json_encode($userData['institution_info']));
                    }

                    // Add institution_name from profile if available (for institution operators)
                    if ($profileData && isset($profileData['profile']['institution_name'])) {
                        $userData['institution_name'] = $profileData['profile']['institution_name'];
                        error_log('GoValidOJS checkAuth: Added institution_name from profile: ' . $userData['institution_name']);
                    }

                    error_log('GoValidOJS checkAuth: Final user data: ' . json_encode($userData));

                    $result['user'] = $userData;
                    $result['userInfo'] = $userData;  // For compatibility
                    $result['subscription'] = $subscription;
                }

                return new \PKP\core\JSONMessage(true, $result);
                
            case 'preview':
                $this->_generatePreview($request);
                exit;
                
            case 'save':
                $templateData = $request->getUserVar('templateData');
                if ($templateData) {
                    $data = json_decode($templateData, true);
                    $savedTemplates = $this->getSetting($contextId, 'savedTemplates') ?: [];
                    
                    // Check if this is an update to existing template
                    $isUpdate = isset($savedTemplates[$data['name']]);
                    
                    // Enforce 10 template limit for new templates
                    if (!$isUpdate && count($savedTemplates) >= 10) {
                        return new \PKP\core\JSONMessage(false, [
                            'status' => false,
                            'message' => 'Template limit reached! You have saved 10/10 templates. Please delete an existing template before saving a new one.'
                        ]);
                    }
                    
                    $savedTemplates[$data['name']] = $data;
                    $this->updateSetting($contextId, 'savedTemplates', $savedTemplates);
                    
                    return new \PKP\core\JSONMessage(true, ['status' => true, 'message' => 'Template saved successfully!']);
                }
                break;
                
            case 'load':
                $templateName = $request->getUserVar('templateName');
                if ($templateName) {
                    $savedTemplates = $this->getSetting($contextId, 'savedTemplates') ?: [];
                    if (isset($savedTemplates[$templateName])) {
                        return new \PKP\core\JSONMessage(true, [
                            'status' => true, 
                            'templateData' => $savedTemplates[$templateName]['data'],
                            'orientation' => $savedTemplates[$templateName]['orientation'] ?? 'portrait'
                        ]);
                    }
                }
                break;
                
            case 'delete':
                $templateName = $request->getUserVar('templateName');
                if ($templateName) {
                    $savedTemplates = $this->getSetting($contextId, 'savedTemplates') ?: [];
                    if (isset($savedTemplates[$templateName])) {
                        unset($savedTemplates[$templateName]);
                        $this->updateSetting($contextId, 'savedTemplates', $savedTemplates);
                    }
                    return new \PKP\core\JSONMessage(true, ['status' => true, 'message' => 'Template deleted successfully!']);
                }
                break;

            case 'listDefaultTemplates':
                $templates = $this->getDefaultTemplates();
                return new \PKP\core\JSONMessage(true, ['status' => true, 'templates' => $templates]);

            case 'loadDefaultTemplate':
                $templateFile = $request->getUserVar('templateFile');
                if ($templateFile) {
                    $template = $this->getDefaultTemplate($templateFile);
                    if ($template) {
                        return new \PKP\core\JSONMessage(true, [
                            'status' => true,
                            'templateData' => $template['data'],
                            'orientation' => $template['orientation'] ?? 'landscape',
                            'name' => $template['name'],
                            'description' => $template['description'] ?? ''
                        ]);
                    } else {
                        return new \PKP\core\JSONMessage(false, ['status' => false, 'message' => 'Template not found']);
                    }
                }
                return new \PKP\core\JSONMessage(false, ['status' => false, 'message' => 'Template file not specified']);

            case 'getSubscription':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $subscription = $authManager->getUserSubscription();
                return new \PKP\core\JSONMessage(true, $subscription);
                
            case 'checkCapability':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $qrType = $request->getUserVar('qr_type') ?: 'url';
                $quantity = (int)$request->getUserVar('quantity') ?: 1;
                
                $capability = $authManager->checkQRGenerationCapability($qrType, $quantity);
                return new \PKP\core\JSONMessage(true, $capability);
                
            case 'getUserInfo':
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['authenticated' => false]);
                }
                
                $userInfo = $authManager->getUserInfo();
                $fullName = $authManager->getUserFullName();
                
                return new \PKP\core\JSONMessage(true, [
                    'authenticated' => true,
                    'user' => $userInfo,
                    'fullName' => $fullName
                ]);
                
            case 'getReviewers':
                // Get reviewers for the current journal
                try {
                    $search = $request->getUserVar('search');
                    $reviewers = $this->_getReviewers($context, $search);
                    return new \PKP\core\JSONMessage(true, ['reviewers' => $reviewers]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error in getReviewers - ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'Error loading reviewers: ' . $e->getMessage(),
                        'reviewers' => []
                    ]);
                }
                break;
                
            case 'searchReviewers':
                // Search reviewers for the current journal
                try {
                    $query = $request->getUserVar('query');
                    error_log('GoValidOJS: searchReviewers called with query: ' . $query);
                    
                    if (!$query || strlen($query) < 2) {
                        error_log('GoValidOJS: searchReviewers - query too short or empty');
                        return new \PKP\core\JSONMessage(true, ['reviewers' => []]);
                    }
                    
                    if (!$context) {
                        error_log('GoValidOJS: searchReviewers - no context available');
                        return new \PKP\core\JSONMessage(false, [
                            'success' => false,
                            'message' => 'No journal context available',
                            'reviewers' => []
                        ]);
                    }
                    
                    $reviewers = $this->_getReviewers($context, $query);
                    error_log('GoValidOJS: searchReviewers - found ' . count($reviewers) . ' reviewers');
                    return new \PKP\core\JSONMessage(true, ['reviewers' => $reviewers]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error in searchReviewers - ' . $e->getMessage());
                    error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'Error searching reviewers: ' . $e->getMessage(),
                        'reviewers' => []
                    ]);
                }
                break;
                
            case 'searchAuthors':
                // Search authors for the current journal
                try {
                    $query = $request->getUserVar('query');
                    error_log('GoValidOJS: searchAuthors called with query: ' . $query);
                    
                    if (!$query || strlen($query) < 2) {
                        error_log('GoValidOJS: searchAuthors - query too short or empty');
                        return new \PKP\core\JSONMessage(true, ['authors' => []]);
                    }
                    
                    if (!$context) {
                        error_log('GoValidOJS: searchAuthors - no context available');
                        return new \PKP\core\JSONMessage(false, [
                            'success' => false,
                            'message' => 'No journal context available',
                            'authors' => []
                        ]);
                    }
                    
                    $authors = $this->_getAuthors($context, $query);
                    error_log('GoValidOJS: searchAuthors - found ' . count($authors) . ' authors');
                    error_log('GoValidOJS: searchAuthors - response: ' . json_encode(['authors' => $authors]));
                    
                    // Return direct format to match frontend expectations
                    return new \PKP\core\JSONMessage(true, ['authors' => $authors]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error in searchAuthors - ' . $e->getMessage());
                    error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'Error searching authors: ' . $e->getMessage(),
                        'authors' => []
                    ]);
                }
                break;
                
            case 'searchEditors':
                // Search editors for the current journal
                try {
                    $query = $request->getUserVar('query');
                    error_log('GoValidOJS: searchEditors called with query: ' . $query);
                    
                    if (!$query || strlen($query) < 2) {
                        error_log('GoValidOJS: searchEditors - query too short or empty');
                        return new \PKP\core\JSONMessage(true, ['editors' => []]);
                    }
                    
                    if (!$context) {
                        error_log('GoValidOJS: searchEditors - no context available');
                        return new \PKP\core\JSONMessage(false, [
                            'success' => false,
                            'message' => 'No journal context available',
                            'editors' => []
                        ]);
                    }
                    
                    $editors = $this->_getEditors($context, $query);
                    error_log('GoValidOJS: searchEditors - found ' . count($editors) . ' editors');
                    error_log('GoValidOJS: searchEditors - response: ' . json_encode(['editors' => $editors]));
                    
                    // Return direct format to match frontend expectations
                    return new \PKP\core\JSONMessage(true, ['editors' => $editors]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error in searchEditors - ' . $e->getMessage());
                    error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'Error searching editors: ' . $e->getMessage(),
                        'editors' => []
                    ]);
                }
                break;
                
            case 'getIssues':
                try {
                    error_log('GoValidOJS: getIssues called');
                    error_log('GoValidOJS: Context: ' . ($context ? 'ID=' . $context->getId() . ', Name=' . $context->getName() : 'NULL'));
                    
                    $issues = $this->_getIssues($context);
                    
                    error_log('GoValidOJS: Returning ' . count($issues) . ' issues');
                    
                    return new \PKP\core\JSONMessage(true, ['issues' => $issues]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error in getIssues - ' . $e->getMessage());
                    error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'Error loading issues: ' . $e->getMessage(),
                        'issues' => []
                    ]);
                }
                
                // CRITICAL: Add break to prevent fall-through
                break;
                
            case 'getSubmissions':
                error_log('GoValidOJS: ===== getSubmissions action called =====');
                error_log('GoValidOJS: Request method: ' . $request->getRequestMethod());
                error_log('GoValidOJS: All request vars: ' . json_encode($request->getUserVars()));
                
                try {
                    $issueId = $request->getUserVar('issueId');
                    error_log('GoValidOJS: getSubmissions called for issue ID: ' . $issueId);
                    error_log('GoValidOJS: Context: ' . ($context ? 'ID=' . $context->getId() : 'NULL'));
                    
                    if (!$context) {
                        error_log('GoValidOJS: No context available for getSubmissions');
                        return new \PKP\core\JSONMessage(true, [
                            'success' => true, 
                            'message' => 'No context available',
                            'submissions' => []
                        ]);
                    }
                    
                    if (!$issueId) {
                        error_log('GoValidOJS: No issue ID provided for getSubmissions');
                        return new \PKP\core\JSONMessage(true, [
                            'success' => true, 
                            'message' => 'No issue ID provided',
                            'submissions' => []
                        ]);
                    }
                    
                    // Call the method with error handling
                    error_log('GoValidOJS: About to call _getSubmissions...');
                    $submissions = $this->_getSubmissions($context, $issueId);
                    error_log('GoValidOJS: _getSubmissions returned successfully');
                    
                    error_log('GoValidOJS: Returning ' . count($submissions) . ' submissions');
                    
                    // If no submissions found, return a helpful message
                    if (empty($submissions)) {
                        return new \PKP\core\JSONMessage(true, [
                            'success' => true,
                            'message' => 'No published articles found for this issue',
                            'submissions' => []
                        ]);
                    }
                    
                    return new \PKP\core\JSONMessage(true, ['submissions' => $submissions]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: EXCEPTION in getSubmissions action - ' . $e->getMessage());
                    error_log('GoValidOJS: Exception type: ' . get_class($e));
                    error_log('GoValidOJS: Stack trace: ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(true, [
                        'success' => true,
                        'message' => 'Error loading submissions: ' . $e->getMessage(),
                        'submissions' => []
                    ]);
                }

                // Add break statement to prevent fall-through
                break;

            case 'getReviewersForSubmission':
                // Get reviewers who reviewed a specific submission
                try {
                    $submissionId = (int) $request->getUserVar('submissionId');
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: getReviewersForSubmission called for submission ID: ' . $submissionId . "\n", FILE_APPEND);
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Context ID: ' . ($context ? $context->getId() : 'NULL') . "\n", FILE_APPEND);

                    if (!$submissionId) {
                        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: No submission ID provided, returning empty array' . "\n", FILE_APPEND);
                        return new \PKP\core\JSONMessage(true, ['reviewers' => []]);
                    }

                    if (!$context) {
                        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: No context available' . "\n", FILE_APPEND);
                        return new \PKP\core\JSONMessage(true, ['reviewers' => []]);
                    }

                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: About to call _getReviewersForSubmission' . "\n", FILE_APPEND);
                    $reviewers = $this->_getReviewersForSubmission($context, $submissionId);
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Returned from _getReviewersForSubmission, count: ' . count($reviewers) . "\n", FILE_APPEND);
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Reviewers data: ' . json_encode($reviewers) . "\n", FILE_APPEND);
                    return new \PKP\core\JSONMessage(true, ['reviewers' => $reviewers]);
                } catch (\Exception $e) {
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: EXCEPTION in getReviewersForSubmission - ' . $e->getMessage() . "\n", FILE_APPEND);
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Stack trace: ' . $e->getTraceAsString() . "\n", FILE_APPEND);
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false,
                        'message' => 'Error loading reviewers',
                        'error' => $e->getMessage()
                    ]);
                }
                break;

            case 'getEditorsForSubmission':
                // Get editors who handled a specific submission
                try {
                    $submissionId = (int) $request->getUserVar('submissionId');
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: getEditorsForSubmission called for submission ID: ' . $submissionId . "\n", FILE_APPEND);

                    if (!$submissionId) {
                        return new \PKP\core\JSONMessage(true, ['editors' => []]);
                    }

                    $editors = $this->_getEditorsForSubmission($context, $submissionId);
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Found ' . count($editors) . ' editors for submission ' . $submissionId . "\n", FILE_APPEND);
                    return new \PKP\core\JSONMessage(true, ['editors' => $editors]);
                } catch (\Exception $e) {
                    @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Error in getEditorsForSubmission - ' . $e->getMessage() . "\n", FILE_APPEND);
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false,
                        'message' => 'Error loading editors',
                        'error' => $e->getMessage()
                    ]);
                }
                break;

            case 'testNumberingAPI':
                // Test numbering API connection
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                // Get current user info and token for debugging
                $token = $authManager->getToken();
                $user = $authManager->getCurrentUser();
                
                error_log('GoValidOJS: Current token - ' . substr($token, 0, 20) . '...');
                error_log('GoValidOJS: Current user - ' . json_encode($user));
                
                // Test a simple API call first
                $testResponse = $authManager->apiRequest('users/profile/', [], 'GET');
                error_log('GoValidOJS: Profile API test - ' . json_encode($testResponse));
                
                return new \PKP\core\JSONMessage(true, [
                    'token_preview' => substr($token, 0, 20) . '...',
                    'user' => $user,
                    'profile_test' => $testResponse
                ]);
                
                // Add break to prevent fall-through
                break;
                
            case 'getPatterns':
                // Get institution numbering patterns using API v1
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                // Use the correct API v1 endpoint as per the guide
                $response = $authManager->apiRequest('numbering/patterns/', [], 'GET', true);
                
                // Format the response to match what the frontend expects
                if ($response['success'] && isset($response['data'])) {
                    // The API returns patterns directly in data
                    $formattedResponse = [
                        'success' => true,
                        'patterns' => $response['data']['patterns'] ?? $response['data'],
                        'count' => count($response['data']['patterns'] ?? $response['data'])
                    ];
                    
                    return new \PKP\core\JSONMessage(true, $formattedResponse);
                }
                
                return new \PKP\core\JSONMessage(true, $response);
                
                // Add break to prevent fall-through
                break;
                
            case 'previewPattern':
                // Preview pattern number using API v1
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $patternId = (int)$request->getUserVar('pattern_id');
                $quantity = (int)($request->getUserVar('quantity') ?: 1);
                
                // Use API v1 endpoint
                $response = $authManager->apiRequest('numbering/preview/', [
                    'pattern_id' => $patternId,
                    'quantity' => $quantity
                ], 'POST', true);
                
                // Format response to match frontend expectations
                if ($response['success'] && isset($response['data'])) {
                    $formattedResponse = [
                        'success' => true,
                        'data' => [
                            'preview' => $response['data']['preview'] ?? $response['data']
                        ]
                    ];
                    return new \PKP\core\JSONMessage(true, $formattedResponse);
                }
                
                return new \PKP\core\JSONMessage(true, $response);
                
                // Add break to prevent fall-through
                break;
                
            case 'generatePattern':
                // Generate pattern numbers using API v1
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $patternId = (int)$request->getUserVar('pattern_id');
                $quantity = (int)($request->getUserVar('quantity') ?: 1);
                $documentType = $request->getUserVar('document_type') ?: 'certificate';
                
                // Use API v1 endpoint
                $response = $authManager->apiRequest('numbering/generate/', [
                    'pattern_id' => $patternId,
                    'quantity' => $quantity,
                    'document_type' => $documentType
                ], 'POST', true);
                
                // Format response to match frontend expectations
                if ($response['success'] && isset($response['data'])) {
                    $formattedResponse = [
                        'success' => true,
                        'data' => [
                            'generated' => $response['data']['generated'] ?? $response['data']
                        ]
                    ];
                    return new \PKP\core\JSONMessage(true, $formattedResponse);
                }
                
                return new \PKP\core\JSONMessage(true, $response);
                
                // Add break to prevent fall-through
                break;
                
            case 'testEndpoints':
                // Test various QR API endpoints to find the correct one
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $results = $authManager->testQREndpoints();
                return new \PKP\core\JSONMessage(true, ['endpoints' => $results]);
                
            case 'sendOtp':
                // Send OTP for QR generation with fast-fail and queue fallback
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }

                // Get email from request or user info - DEFINED OUTSIDE try blocks
                $requestEmail = $request->getUserVar('email');
                $userInfo = $authManager->getCurrentUser();
                $userEmail = $userInfo['email'] ?? '';
                $email = $requestEmail ?: $userEmail;

                error_log('GoValidOJS: sendOtp - Using email: ' . $email);

                if (!$email) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'No email found']);
                }

                // Try fast-fail first (3 second timeout)
                require_once($this->getPluginPath() . '/lib/queue/FileQueue.php');

                try {
                    error_log('GoValidOJS: Attempting fast-fail OTP send');
                    $result = $authManager->sendOJSOtpFastFail($email, 3);

                    if ($result['success'] && isset($result['session_id'])) {
                        error_log('GoValidOJS: Fast-fail succeeded');
                        return new \PKP\core\JSONMessage(true, [
                            'mode' => 'immediate',
                            'success' => true,
                            'data' => [
                                'session_id' => $result['session_id'],
                                'otp_id' => $result['session_id']
                            ],
                            'message' => 'OTP sent immediately'
                        ]);
                    }

                    error_log('GoValidOJS: Fast-fail did not succeed, falling back to queue');
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Fast-fail exception: ' . $e->getMessage());
                }

                // Queue fallback - network slow or timeout
                try {
                    error_log('GoValidOJS: Using queue fallback');
                    $queue = new GoValidFileQueue($this->getPluginPath());
                    $queueId = $queue->enqueue($email, 'certificate_signing');

                    error_log('GoValidOJS: Queued with ID: ' . $queueId);

                    return new \PKP\core\JSONMessage(true, [
                        'mode' => 'queued',
                        'success' => true,
                        'queue_id' => $queueId,
                        'message' => 'OTP request queued for processing'
                    ]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Queue fallback failed: ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false,
                        'message' => 'Failed to send OTP: ' . $e->getMessage()
                    ]);
                }

                break;

            case 'verifyOtp':
                // Verify OTP for QR generation
                error_log('GoValidOJS: ENTERED verifyOtp case handler');
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - ENTERED verifyOtp case handler' . "\n", FILE_APPEND);
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                $otp = $request->getUserVar('otp');
                $otpId = $request->getUserVar('otp_id');
                $requestEmail = $request->getUserVar('email');
                $userInfo = $authManager->getCurrentUser();
                $userEmail = $userInfo['email'] ?? '';
                
                // Use request email if provided, otherwise use user's email
                $email = $requestEmail ?: $userEmail;
                
                error_log('GoValidOJS: verifyOtp - Request email: ' . $requestEmail);
                error_log('GoValidOJS: verifyOtp - User email: ' . $userEmail);
                error_log('GoValidOJS: verifyOtp - Using email: ' . $email);
                error_log('GoValidOJS: verifyOtp - OTP: ' . $otp);
                error_log('GoValidOJS: verifyOtp - OTP ID: ' . $otpId);
                
                if (!$otp) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'OTP code required']);
                }
                
                if (!$otpId && !$email) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Either OTP ID or email required']);
                }
                
                $result = $authManager->verifyOJSOtp($email, $otp, $otpId);
                error_log('GoValidOJS: verifyOtp result: ' . json_encode($result));

                return new \PKP\core\JSONMessage(true, $result);

            case 'checkOtpStatus':
                // Check status of queued OTP request
                $queueId = $request->getUserVar('queue_id');

                if (!$queueId) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Queue ID required']);
                }

                require_once($this->getPluginPath() . '/lib/queue/FileQueue.php');

                try {
                    $queue = new GoValidFileQueue($this->getPluginPath());
                    $status = $queue->getStatus($queueId);

                    error_log('GoValidOJS: Queue status for ' . $queueId . ': ' . json_encode($status));

                    return new \PKP\core\JSONMessage(true, [
                        'success' => true,
                        'status' => $status
                    ]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: checkOtpStatus error: ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false,
                        'message' => $e->getMessage()
                    ]);
                }

                break;

            case 'processQueue':
                // Process pending OTP requests in queue
                require_once($this->getPluginPath() . '/lib/queue/FileQueue.php');
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');

                try {
                    $queue = new GoValidFileQueue($this->getPluginPath());
                    $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                    if (!$authManager->isAuthenticated()) {
                        return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                    }

                    // Get pending items from queue
                    $pending = $queue->getPending(5);
                    $processed = 0;
                    $errors = [];

                    error_log('GoValidOJS: Processing ' . count($pending) . ' pending queue items');

                    foreach ($pending as $item) {
                        $queueId = $item['queue_id'];
                        $email = $item['email'];

                        error_log('GoValidOJS: Processing queue item ' . $queueId . ' for ' . $email);

                        // Update status to processing
                        $queue->update($queueId, [
                            'status' => 'processing',
                            'last_attempt_at' => time()
                        ]);

                        try {
                            // Attempt to send OTP
                            $result = $authManager->sendOJSOtp($email);

                            if ($result['success'] && isset($result['data']['session_id'])) {
                                // Success - update queue with session_id
                                $queue->update($queueId, [
                                    'status' => 'sent',
                                    'session_id' => $result['data']['session_id']
                                ]);
                                $processed++;
                                error_log('GoValidOJS: Queue item ' . $queueId . ' sent successfully');
                            } else {
                                // Failed - increment retry count
                                $retryCount = ($item['retry_count'] ?? 0) + 1;
                                $newStatus = $retryCount >= 3 ? 'failed' : 'pending';

                                $queue->update($queueId, [
                                    'status' => $newStatus,
                                    'retry_count' => $retryCount,
                                    'last_error' => $result['message'] ?? 'Unknown error'
                                ]);

                                $errors[] = $queueId . ': ' . ($result['message'] ?? 'Failed');
                                error_log('GoValidOJS: Queue item ' . $queueId . ' failed (retry ' . $retryCount . ')');
                            }
                        } catch (\Exception $e) {
                            // Exception - increment retry count
                            $retryCount = ($item['retry_count'] ?? 0) + 1;
                            $newStatus = $retryCount >= 3 ? 'failed' : 'pending';

                            $queue->update($queueId, [
                                'status' => $newStatus,
                                'retry_count' => $retryCount,
                                'last_error' => $e->getMessage()
                            ]);

                            $errors[] = $queueId . ': ' . $e->getMessage();
                            error_log('GoValidOJS: Queue item ' . $queueId . ' exception: ' . $e->getMessage());
                        }
                    }

                    // Cleanup old completed items
                    $cleaned = $queue->cleanup();

                    return new \PKP\core\JSONMessage(true, [
                        'success' => true,
                        'processed' => $processed,
                        'total' => count($pending),
                        'errors' => $errors,
                        'cleaned' => $cleaned
                    ]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: processQueue error: ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false,
                        'message' => $e->getMessage()
                    ]);
                }

                break;

            case 'generateCertificateQR':
                // Generate certificate QR code with all fields
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                
                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Not authenticated']);
                }
                
                // Get JSON data from request body
                $rawInput = file_get_contents('php://input');
                $qrData = json_decode($rawInput, true);
                
                if (!$qrData) {
                    error_log('GoValidOJS: Invalid JSON data for QR generation');
                    return new \PKP\core\JSONMessage(false, ['success' => false, 'message' => 'Invalid JSON data']);
                }
                
                // Log the request for debugging
                error_log('GoValidOJS: QR generation request data: ' . json_encode($qrData));
                
                // Make API request to generate certificate QR
                try {
                    $response = $authManager->generateQRCode($qrData);
                    
                    error_log('GoValidOJS: QR generation response: ' . json_encode($response));
                    
                    return new \PKP\core\JSONMessage(true, $response);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: QR generation exception: ' . $e->getMessage());
                    error_log('GoValidOJS: Exception trace: ' . $e->getTraceAsString());
                    return new \PKP\core\JSONMessage(false, [
                        'success' => false, 
                        'message' => 'QR generation failed: ' . $e->getMessage()
                    ]);
                }
                
                // Add break to prevent fall-through
                break;
                
            case 'proxyQRImage':
                // Proxy QR image to avoid CORS issues
                $imageUrl = $request->getUserVar('url');
                error_log('GoValidOJS: proxyQRImage called with URL: ' . $imageUrl);
                
                if (!$imageUrl) {
                    return new \PKP\core\JSONMessage(false, [
                        'content' => [
                            'success' => false,
                            'message' => 'Image URL is required'
                        ]
                    ]);
                }
                
                // Validate that this is a GoValid QR image URL
                if (!preg_match('/^https:\/\/.*\.r2\.cloudflarestorage\.com\/.*\/qr_codes\//', $imageUrl)) {
                    return new \PKP\core\JSONMessage(false, [
                        'content' => [
                            'success' => false,
                            'message' => 'Invalid image URL'
                        ]
                    ]);
                }
                
                try {
                    // First, try with JWT token if available
                    require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                    $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
                    $jwtToken = $authManager->getToken();
                    
                    // Prepare headers
                    $headers = [
                        'Accept: image/png,image/jpeg,image/*;q=0.8,*/*;q=0.5',
                        'Accept-Language: en-US,en;q=0.9',
                        'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                        'Referer: https://my.govalid.org/'
                    ];
                    
                    // Add authorization header if we have a token
                    if ($jwtToken) {
                        $headers[] = 'Authorization: Bearer ' . $jwtToken;
                        error_log('GoValidOJS: Adding JWT token to image request');
                    }
                    
                    // Fetch the image with headers to mimic browser request
                    $ch = curl_init($imageUrl);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                    
                    $imageData = curl_exec($ch);
                    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
                    $curlError = curl_error($ch);
                    $effectiveUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
                    curl_close($ch);
                    
                    error_log('GoValidOJS: CURL result - HTTP: ' . $httpCode . ', Content-Type: ' . $contentType . ', Data size: ' . strlen($imageData) . ', Error: ' . $curlError . ', Effective URL: ' . $effectiveUrl);
                    
                    if ($httpCode === 200 && $imageData) {
                        // Convert to base64
                        $base64 = base64_encode($imageData);
                        return new \PKP\core\JSONMessage(true, [
                            'content' => [
                                'success' => true,
                                'image_base64' => $base64,
                                'content_type' => $contentType
                            ]
                        ]);
                    } else {
                        error_log('GoValidOJS: Failed to fetch QR image. HTTP: ' . $httpCode);
                        $errorMessage = 'Failed to fetch image (HTTP ' . $httpCode . ')';
                        
                        // Provide more specific error messages based on HTTP code
                        if ($httpCode === 400) {
                            $errorMessage = 'Bad request to image server. The image URL may be expired or invalid.';
                        } elseif ($httpCode === 403) {
                            $errorMessage = 'Access forbidden. The image may be private or require authentication.';
                        } elseif ($httpCode === 404) {
                            $errorMessage = 'Image not found. The QR code image may have been deleted.';
                        } elseif ($httpCode === 0) {
                            $errorMessage = 'Could not connect to image server. Network error: ' . $curlError;
                        }
                        
                        return new \PKP\core\JSONMessage(false, [
                            'content' => [
                                'success' => false,
                                'message' => $errorMessage,
                                'http_code' => $httpCode,
                                'curl_error' => $curlError
                            ]
                        ]);
                    }
                } catch (Exception $e) {
                    error_log('GoValidOJS: Error fetching QR image: ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(false, [
                        'content' => [
                            'success' => false,
                            'message' => 'Error fetching image: ' . $e->getMessage()
                        ]
                    ]);
                }
                
                break;
                
            /* DUPLICATE HANDLER REMOVED - Using the handler from line 846 instead
            case 'sendOtp':
                // Send OTP through GoValid API
                error_log('GoValidOJS: sendOtp action called');
                
                $destination = $request->getUserVar('destination');
                $destinationType = $request->getUserVar('destination_type');
                $purpose = $request->getUserVar('purpose');
                $otpLength = (int)$request->getUserVar('otp_length');
                $metadata = $request->getUserVar('metadata');
                
                error_log('GoValidOJS: OTP params - destination: ' . $destination . ', type: ' . $destinationType . ', purpose: ' . $purpose);
                
                if (!$destination) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Email address is required']);
                }
                
                // Parse metadata to extract certificate info
                $metadataArray = json_decode($metadata, true);
                
                // Prepare request data according to new API format
                $otpData = [
                    'destination' => $destination,
                    'purpose' => $purpose,
                    'otp_length' => $otpLength
                ];
                
                // Add metadata with certificate info if available
                if ($metadataArray && isset($metadataArray['certificate_info'])) {
                    $otpData['metadata'] = [
                        'certificate_info' => $metadataArray['certificate_info']
                    ];
                }
                
                error_log('GoValidOJS: Sending OTP request with data - ' . json_encode($otpData));
                
                // Use cURL to call GoValid OJS-specific API endpoint (bypass CORS)
                $ch = curl_init('https://my.govalid.org/api/v1/otp/ojs/send/');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($otpData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Accept: application/json'
                ]);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $error = curl_error($ch);
                curl_close($ch);
                
                error_log('GoValidOJS: OTP API response - HTTP ' . $httpCode . ' - Response: ' . substr($response, 0, 500));
                
                if ($error) {
                    error_log('GoValidOJS: OTP send cURL error - ' . $error);
                    return new \PKP\core\JSONMessage(false, ['error' => 'Network error: ' . $error]);
                }
                
                $responseData = json_decode($response, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    error_log('GoValidOJS: JSON decode error - ' . json_last_error_msg());
                    error_log('GoValidOJS: Raw response was: ' . $response);
                }
                
                if ($httpCode === 200 || $httpCode === 201) {
                    // For OJS-specific API, the actual response is nested inside 'content'
                    $actualResponse = $responseData;
                    
                    // Log the full response structure for debugging
                    error_log('GoValidOJS: Full response structure: ' . json_encode($responseData));
                    
                    // Check if we have the OJS-specific nested structure
                    if (isset($responseData['content'])) {
                        $actualResponse = $responseData['content'];
                        error_log('GoValidOJS: Using nested content from OJS-specific response');
                    }
                    
                    // Extract otp_id from the appropriate location in the response
                    $otpId = null;
                    
                    // Try to find otp_id in various possible locations
                    if (isset($actualResponse['otp_id'])) {
                        $otpId = $actualResponse['otp_id'];
                        error_log('GoValidOJS: Found otp_id directly in response');
                    } elseif (isset($actualResponse['success']) && is_array($actualResponse['success']) && isset($actualResponse['success']['otp_id'])) {
                        $otpId = $actualResponse['success']['otp_id'];
                        error_log('GoValidOJS: Found otp_id in success object');
                    }
                    
                    $result = [
                        'success' => true,
                        'otp_id' => $otpId,
                        'expires_in' => $actualResponse['expires_in'] ?? ($actualResponse['success']['expires_in'] ?? 300),
                        'message' => $actualResponse['message'] ?? ($actualResponse['success']['message'] ?? 'OTP sent successfully')
                    ];
                    error_log('GoValidOJS: OTP sent successfully with ID: ' . ($result['otp_id'] ?? 'unknown'));
                    return new \PKP\core\JSONMessage(true, $result);
                } else if ($httpCode === 500) {
                    // Handle HTTP 500 - email might have been sent
                    error_log('GoValidOJS: OTP API returned 500 - ' . $response);
                    
                    // Check if we have any indication the OTP was sent
                    if ($responseData && isset($responseData['otp_id'])) {
                        // We have an otp_id, so OTP was likely sent
                        $result = [
                            'success' => true,
                            'otp_id' => $responseData['otp_id'],
                            'expires_in' => $responseData['expires_in'] ?? 300,
                            'message' => 'OTP sent (server had issues but email likely delivered)',
                            'warning' => 'Server returned 500 error'
                        ];
                        return new \PKP\core\JSONMessage(true, $result);
                    } else {
                        // No otp_id, but since user reports receiving email, we should handle gracefully
                        error_log('GoValidOJS: HTTP 500 without otp_id - cannot proceed with verification');
                        return new \PKP\core\JSONMessage(false, [
                            'error' => 'Server error occurred. If you received the email, please try again or contact support.',
                            'http_code' => $httpCode,
                            'technical_details' => 'OTP may have been sent but no tracking ID received'
                        ]);
                    }
                } else {
                    error_log('GoValidOJS: OTP send failed - HTTP ' . $httpCode . ' - ' . $response);
                    $errorMessage = 'Failed to send OTP';
                    if ($responseData && isset($responseData['detail'])) {
                        $errorMessage = $responseData['detail'];
                    } elseif ($responseData && isset($responseData['error'])) {
                        $errorMessage = $responseData['error'];
                    } elseif ($responseData && isset($responseData['message'])) {
                        $errorMessage = $responseData['message'];
                    }
                    return new \PKP\core\JSONMessage(false, ['error' => $errorMessage, 'http_code' => $httpCode]);
                }
                
                break;
                */
                
            /* DUPLICATE HANDLER REMOVED - Using the handler from line 865 instead
            case 'verifyOtp':
                // Verify OTP through GoValid API
                error_log('GoValidOJS: verifyOtp action called');
                
                $otpId = $request->getUserVar('otp_id');
                $otp = $request->getUserVar('otp');
                $email = $request->getUserVar('email');
                
                error_log('GoValidOJS: Verify params - otp_id: ' . $otpId . ', otp: ' . $otp . ', email: ' . $email);
                
                // If we have email but no otp_id, use email and OTP for verification
                if (!$otpId && $email && $otp) {
                    error_log('GoValidOJS: Using email for verification');
                    
                    // Use cURL to call GoValid OJS-specific verify API with email
                    $ch = curl_init('https://my.govalid.org/api/v1/otp/ojs/verify/');
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                        'email' => $email,
                        'otp' => $otp
                    ]));
                    curl_setopt($ch, CURLOPT_HTTPHEADER, [
                        'Content-Type: application/json',
                        'Accept: application/json'
                    ]);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                    
                    $response = curl_exec($ch);
                    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    $error = curl_error($ch);
                    curl_close($ch);
                    
                    error_log('GoValidOJS: Verify-by-email response - HTTP ' . $httpCode . ' - Response: ' . $response);
                    
                    if ($error) {
                        error_log('GoValidOJS: Verify-by-email cURL error - ' . $error);
                        return new \PKP\core\JSONMessage(false, ['error' => 'Network error: ' . $error]);
                    }
                    
                    $responseData = json_decode($response, true);
                    error_log('GoValidOJS: Full verify-by-email response structure: ' . json_encode($responseData));
                    
                    // For OJS-specific API, the actual response is nested inside 'content'
                    $actualResponse = $responseData;
                    if (isset($responseData['content'])) {
                        $actualResponse = $responseData['content'];
                        error_log('GoValidOJS: Using nested content from OJS-specific response for verify-by-email');
                    }
                    
                    // Check for valid flag in various possible locations
                    $isValid = false;
                    if (isset($actualResponse['valid']) && $actualResponse['valid'] === true) {
                        $isValid = true;
                    } elseif (isset($responseData['status']) && $responseData['status'] === true) {
                        $isValid = true;
                    } elseif (isset($actualResponse['success']) && is_array($actualResponse['success']) && 
                              isset($actualResponse['success']['valid']) && $actualResponse['success']['valid'] === true) {
                        $isValid = true;
                    }
                    
                    if ($httpCode === 200 && $isValid) {
                        // Return the full response including the nested structure
                        return new \PKP\core\JSONMessage(true, $responseData);
                    } else {
                        error_log('GoValidOJS: OTP verify failed - HTTP ' . $httpCode . ' - ' . $response);
                        $errorMsg = 'Invalid verification code';
                        
                        // Check for error messages in various possible locations
                        if (isset($actualResponse['message'])) {
                            $errorMsg = $actualResponse['message'];
                        } elseif (isset($actualResponse['error'])) {
                            $errorMsg = $actualResponse['error'];
                        } elseif (isset($actualResponse['detail'])) {
                            $errorMsg = $actualResponse['detail'];
                        } elseif (isset($actualResponse['success']) && is_array($actualResponse['success']) && isset($actualResponse['success']['message'])) {
                            $errorMsg = $actualResponse['success']['message'];
                        } elseif (isset($responseData['message'])) {
                            $errorMsg = $responseData['message'];
                        } elseif (isset($responseData['error'])) {
                            $errorMsg = $responseData['error'];
                        } elseif (isset($responseData['detail'])) {
                            $errorMsg = $responseData['detail'];
                        }
                        
                        return new \PKP\core\JSONMessage(false, ['error' => $errorMsg, 'valid' => false]);
                    }
                }
                
                // Standard verification with otp_id
                if (!$otpId || !$otp) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'OTP ID and code are required']);
                }
                
                // Use cURL to call GoValid OJS-specific API directly (bypass CORS)
                $ch = curl_init('https://my.govalid.org/api/v1/otp/ojs/verify/');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                    'otp_id' => $otpId,
                    'otp' => $otp
                ]));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Accept: application/json'
                ]);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $error = curl_error($ch);
                curl_close($ch);
                
                error_log('GoValidOJS: OTP verify response - HTTP ' . $httpCode . ' - Response: ' . $response);
                
                if ($error) {
                    error_log('GoValidOJS: OTP verify cURL error - ' . $error);
                    return new \PKP\core\JSONMessage(false, ['error' => 'Network error: ' . $error]);
                }
                
                $responseData = json_decode($response, true);
                error_log('GoValidOJS: Full standard verify response structure: ' . json_encode($responseData));
                
                // For OJS-specific API, the actual response is nested inside 'content'
                $actualResponse = $responseData;
                if (isset($responseData['content'])) {
                    $actualResponse = $responseData['content'];
                    error_log('GoValidOJS: Using nested content from OJS-specific response for standard verify');
                }
                
                // Check for valid flag in various possible locations
                $isValid = false;
                if (isset($actualResponse['valid']) && $actualResponse['valid'] === true) {
                    $isValid = true;
                } elseif (isset($responseData['status']) && $responseData['status'] === true) {
                    $isValid = true;
                } elseif (isset($actualResponse['success']) && is_array($actualResponse['success']) && 
                          isset($actualResponse['success']['valid']) && $actualResponse['success']['valid'] === true) {
                    $isValid = true;
                }
                
                if ($httpCode === 200 && $isValid) {
                    // Return the full response including the nested structure
                    return new \PKP\core\JSONMessage(true, $responseData);
                } else {
                    error_log('GoValidOJS: OTP verify failed - HTTP ' . $httpCode . ' - ' . $response);
                    $errorMsg = 'Failed to verify OTP';
                    
                    // Check for error messages in various possible locations
                    if (isset($actualResponse['message'])) {
                        $errorMsg = $actualResponse['message'];
                    } elseif (isset($actualResponse['error'])) {
                        $errorMsg = $actualResponse['error'];
                    } elseif (isset($actualResponse['detail'])) {
                        $errorMsg = $actualResponse['detail'];
                    } elseif (isset($actualResponse['success']) && is_array($actualResponse['success']) && isset($actualResponse['success']['message'])) {
                        $errorMsg = $actualResponse['success']['message'];
                    } elseif (isset($responseData['message'])) {
                        $errorMsg = $responseData['message'];
                    } elseif (isset($responseData['error'])) {
                        $errorMsg = $responseData['error'];
                    } elseif (isset($responseData['detail'])) {
                        $errorMsg = $responseData['detail'];
                    }
                    
                    // Check for attempts remaining in various locations
                    if (isset($actualResponse['attempts_remaining'])) {
                        $errorMsg .= '. ' . $actualResponse['attempts_remaining'] . ' attempts remaining';
                    } elseif (isset($actualResponse['success']['attempts_remaining'])) {
                        $errorMsg .= '. ' . $actualResponse['success']['attempts_remaining'] . ' attempts remaining';
                    }
                    
                    return new \PKP\core\JSONMessage(false, ['error' => $errorMsg, 'valid' => false]);
                }
                
                break;
                */

            case 'getOJSEmailTemplates':
                // Get all OJS email templates
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templates = $integration->getOJSEmailTemplates();
                return new \PKP\core\JSONMessage(true, ['templates' => $templates]);

            case 'getOJSEmailTemplate':
                // Get a specific OJS email template
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templateKey = $request->getUserVar('templateKey');
                if (!$templateKey) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Template key required']);
                }
                $template = $integration->getOJSEmailTemplate($templateKey);
                if ($template) {
                    return new \PKP\core\JSONMessage(true, ['template' => $template]);
                } else {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Template not found']);
                }

            case 'saveOJSEmailTemplate':
                // Save/update an OJS email template
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templateKey = $request->getUserVar('templateKey');
                $subject = $request->getUserVar('subject');
                $body = $request->getUserVar('body');

                if (!$templateKey || !$subject || !$body) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Missing required parameters']);
                }

                $success = $integration->saveOJSEmailTemplate($templateKey, $subject, $body);
                if ($success) {
                    return new \PKP\core\JSONMessage(true, ['success' => true, 'message' => 'Template saved successfully']);
                } else {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Failed to save template']);
                }

            case 'resetOJSEmailTemplate':
                // Reset OJS email template to default
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templateKey = $request->getUserVar('templateKey');

                if (!$templateKey) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Template key required']);
                }

                $success = $integration->resetOJSEmailTemplate($templateKey);
                if ($success) {
                    return new \PKP\core\JSONMessage(true, ['success' => true, 'message' => 'Template reset to default']);
                } else {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Failed to reset template']);
                }

            case 'getOJSEmailVariables':
                // Get available OJS email template variables
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templateKey = $request->getUserVar('templateKey');
                $variables = $integration->getAvailableVariables($templateKey);
                return new \PKP\core\JSONMessage(true, ['variables' => $variables]);

            case 'applyDesignToOJSTemplate':
                // Apply goValid design to an OJS email template
                require_once($this->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new \APP\plugins\generic\goValidOJS\classes\OJSEmailTemplateIntegration($this, $context);
                $templateKey = $request->getUserVar('templateKey');
                $designConfigJson = $request->getUserVar('designConfig');

                if (!$templateKey || !$designConfigJson) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Missing required parameters']);
                }

                $designConfig = json_decode($designConfigJson, true);
                if (!$designConfig) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Invalid design configuration']);
                }

                $result = $integration->applyDesignToOJSTemplate($templateKey, $designConfig);
                if ($result['status'] === 'success') {
                    return new \PKP\core\JSONMessage(true, $result);
                } else {
                    return new \PKP\core\JSONMessage(false, $result);
                }

            case 'getDesignTemplates':
                // Get all available design templates
                require_once($this->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
                $templates = \APP\plugins\generic\goValidOJS\classes\EmailDesignTemplateLibrary::getDesignTemplates();
                return new \PKP\core\JSONMessage(true, ['templates' => $templates]);

            case 'getDesignTemplate':
                // Get a specific design template
                require_once($this->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
                $templateKey = $request->getUserVar('templateKey');
                if (!$templateKey) {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Template key required']);
                }
                $template = \APP\plugins\generic\goValidOJS\classes\EmailDesignTemplateLibrary::getDesignTemplate($templateKey);
                if ($template) {
                    return new \PKP\core\JSONMessage(true, ['template' => $template]);
                } else {
                    return new \PKP\core\JSONMessage(false, ['error' => 'Template not found']);
                }

            case 'getAntiCounterfeitTags':
                // Get existing anti-counterfeit tags from user's GoValid account
                require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
                $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();

                if (!$authManager->isAuthenticated()) {
                    return new \PKP\core\JSONMessage(true, [
                        'success' => false,
                        'message' => 'Not authenticated',
                        'tags' => []
                    ]);
                }

                try {
                    // Use office-addin/user-info endpoint which returns anti_counterfeit_tags
                    $response = $authManager->apiRequest('office-addin/user-info/', [], 'GET', true);

                    // Extract anti-counterfeit tags from the response
                    $tags = [];
                    if ($response['success'] && isset($response['data'])) {
                        $userData = $response['data']['user'] ?? $response['data'];
                        $tags = $userData['anti_counterfeit_tags'] ?? [];
                    }

                    return new \PKP\core\JSONMessage(true, [
                        'success' => true,
                        'tags' => $tags
                    ]);
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error getting anti-counterfeit tags: ' . $e->getMessage());
                    return new \PKP\core\JSONMessage(true, [
                        'success' => false,
                        'message' => 'Error: ' . $e->getMessage(),
                        'tags' => []
                    ]);
                }
        }

        return new \PKP\core\JSONMessage(false, ['error' => 'Invalid action']);
    }

    /**
     * Show certificate designer
     */
    private function _showDesigner($request): void
    {
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : \PKP\core\PKPApplication::CONTEXT_SITE;
        
        // Get template manager
        $templateMgr = \APP\template\TemplateManager::getManager($request);
        
        // Test template manager
        if (!$templateMgr) {
            echo "Error: Could not get template manager";
            exit;
        }
        
        $templateMgr->assign('pluginName', $this->getName());
        
        // Assign base URL for AJAX calls
        $baseUrl = $request->getBaseUrl();
        $templateMgr->assign('baseUrl', $baseUrl);

        // Assign plugin URL for loading JS/CSS files
        // getPluginPath() returns relative path like "plugins/generic/goValidOJS"
        $pluginPath = $this->getPluginPath();
        $pluginUrl = $baseUrl . '/' . $pluginPath;

        // Debug log
        error_log('GoValidOJS: baseUrl=' . $baseUrl);
        error_log('GoValidOJS: pluginPath=' . $pluginPath);
        error_log('GoValidOJS: pluginUrl=' . $pluginUrl);

        $templateMgr->assign('pluginUrl', $pluginUrl);

        // Assign context path for AJAX URLs
        $contextPath = $context ? $context->getPath() : 'index';
        $templateMgr->assign('contextPath', $contextPath);

        // Get journal info including logo using the dedicated method
        $journalInfo = $this->getJournalInfo($context);
        if ($context) {
            $journalInfo['url'] = $request->getBaseUrl() . '/' . $context->getPath();
        }
        $templateMgr->assign('journalInfo', $journalInfo);
        
        // Load existing template if available
        $existingTemplate = $this->getSetting($contextId, 'certificateTemplate');
        $templateMgr->assign('existingTemplate', $existingTemplate);
        
        // Display the template
        $templateMgr->display($this->getTemplateResource('designer.tpl'));
        exit; // Prevent further output
    }

    /**
     * Setup page handler for designer
     */
    public function setupPageHandler(string $hookName, array $params): bool
    {
        $page =& $params[0];
        $op =& $params[1];

        // Log EVERY call to this function with full details
        $logMsg = date('Y-m-d H:i:s') . ' - setupPageHandler CALLED' . "\n";
        $logMsg .= '  page: "' . $page . '"' . "\n";
        $logMsg .= '  op: "' . ($op ?: 'null') . '"' . "\n";
        $logMsg .= '  Request URI: ' . ($_SERVER['REQUEST_URI'] ?? 'N/A') . "\n";
        @file_put_contents($this->getPluginPath() . '/debug.log', $logMsg, FILE_APPEND);

        // Check if this is our page
        if ($page !== 'govalidojs') {
            return false;
        }

        // This IS our page!
        @file_put_contents($this->getPluginPath() . '/debug.log',
            date('Y-m-d H:i:s') . ' - MATCHED govalidojs page, op=' . $op . "\n",
            FILE_APPEND);

        if (!$this->getEnabled()) {
            @file_put_contents($this->getPluginPath() . '/debug.log',
                date('Y-m-d H:i:s') . ' - Plugin NOT enabled!' . "\n",
                FILE_APPEND);
            return false;
        }

        $handler =& $params[3];  // This is the critical parameter!

        if ($this->getEnabled()) {
            // Ensure handler file exists
            $handlerFile = $this->getPluginPath() . '/pages/GoValidOJSHandler.php';
            if (!file_exists($handlerFile)) {
                error_log('GoValidOJS: Handler file not found at: ' . $handlerFile);
                return false;
            }

            // Include the handler file
            require_once($handlerFile);

            // Instantiate the handler object
            $handler = new \APP\plugins\generic\goValidOJS\pages\GoValidOJSHandler();

            error_log('GoValidOJS: Handler instantiated successfully');
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Handler instantiated' . "\n", FILE_APPEND);

            return true;
        }

        return false;
    }

    /**
     * Get reviewers for the journal
     */
    private function _getReviewers($context, $search = null)
    {
        if (!$context) return [];
        
        try {
            // Use the User Repository system in OJS 3.4
            $userCollector = Repo::user()->getCollector();
            
            // Filter by context and reviewer role
            $userCollector->filterByContextIds([$context->getId()]);
            $userCollector->filterByRoleIds([\PKP\security\Role::ROLE_ID_REVIEWER]);
            
            // Apply search term if provided - this uses the searchPhrase method
            // which searches in multiple fields including given name, family name, etc.
            if ($search && strlen($search) >= 2) {
                $userCollector->searchPhrase($search);
            }
            
            // Get users with pagination to avoid memory issues
            $userCollector->limit(50);
            $users = $userCollector->getMany();
            
            error_log('GoValidOJS: Found ' . count($users) . ' reviewers for search: ' . $search);
            
            $reviewers = [];
            foreach ($users as $user) {
                try {
                    // Get user data safely
                    $fullName = $user->getFullName();
                    $email = $user->getEmail();
                    
                    // Skip if essential data is missing
                    if (empty($fullName)) {
                        continue;
                    }
                    
                    // Build reviewer data array
                    $reviewerData = [
                        'id' => $user->getId(),
                        'name' => $fullName,
                        'email' => $email ?: '',
                        'affiliation' => $user->getLocalizedAffiliation() ?: ''
                    ];
                    
                    $reviewers[] = $reviewerData;
                    error_log('GoValidOJS: Added reviewer: ' . json_encode($reviewerData));
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error getting reviewer data for user ID ' . $user->getId() . ' - ' . $e->getMessage());
                    continue;
                }
            }
            
            // Sort by name
            usort($reviewers, function($a, $b) {
                return strcasecmp($a['name'], $b['name']);
            });
            
            return $reviewers;
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error in _getReviewers - ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get authors for the journal
     */
    private function _getAuthors($context, $search = null)
    {
        if (!$context) return [];
        
        try {
            // Use the User Repository system in OJS 3.4
            $userCollector = Repo::user()->getCollector();
            
            // Filter by context and author role
            $userCollector->filterByContextIds([$context->getId()]);
            $userCollector->filterByRoleIds([\PKP\security\Role::ROLE_ID_AUTHOR]);
            
            // Apply search term if provided - this uses the searchPhrase method
            // which searches in multiple fields including given name, family name, etc.
            if ($search && strlen($search) >= 2) {
                $userCollector->searchPhrase($search);
            }
            
            // Get users with pagination to avoid memory issues
            $userCollector->limit(50);
            $users = $userCollector->getMany();
            
            error_log('GoValidOJS: Found ' . count($users) . ' authors for search: ' . $search);
            
            $authors = [];
            foreach ($users as $user) {
                try {
                    // Get user data safely
                    $fullName = $user->getFullName();
                    $email = $user->getEmail();
                    
                    // Skip if essential data is missing
                    if (empty($fullName)) {
                        continue;
                    }
                    
                    // Build author data array
                    $authorData = [
                        'id' => $user->getId(),
                        'name' => $fullName,
                        'email' => $email ?: '',
                        'affiliation' => $user->getLocalizedAffiliation() ?: ''
                    ];
                    
                    $authors[] = $authorData;
                    error_log('GoValidOJS: Added author: ' . json_encode($authorData));
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error getting author data for user ID ' . $user->getId() . ' - ' . $e->getMessage());
                    continue;
                }
            }
            
            // Sort by name
            usort($authors, function($a, $b) {
                return strcasecmp($a['name'], $b['name']);
            });
            
            return $authors;
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error in _getAuthors - ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get editors for the journal
     */
    private function _getEditors($context, $search = null)
    {
        if (!$context) return [];
        
        try {
            // Use the User Repository system in OJS 3.4
            $userCollector = Repo::user()->getCollector();
            
            // Filter by context and editor roles (section editor and journal editor)
            $userCollector->filterByContextIds([$context->getId()]);
            $userCollector->filterByRoleIds([\PKP\security\Role::ROLE_ID_MANAGER, \PKP\security\Role::ROLE_ID_SUB_EDITOR]);
            
            // Apply search term if provided - this uses the searchPhrase method
            // which searches in multiple fields including given name, family name, etc.
            if ($search && strlen($search) >= 2) {
                $userCollector->searchPhrase($search);
            }
            
            // Get users with pagination to avoid memory issues
            $userCollector->limit(50);
            $users = $userCollector->getMany();
            
            error_log('GoValidOJS: Found ' . count($users) . ' editors for search: ' . $search);
            
            $editors = [];
            foreach ($users as $user) {
                try {
                    // Get user data safely
                    $fullName = $user->getFullName();
                    $email = $user->getEmail();
                    
                    // Skip if essential data is missing
                    if (empty($fullName)) {
                        continue;
                    }
                    
                    // Build editor data array
                    $editorData = [
                        'id' => $user->getId(),
                        'name' => $fullName,
                        'email' => $email ?: '',
                        'affiliation' => $user->getLocalizedAffiliation() ?: ''
                    ];
                    
                    $editors[] = $editorData;
                    error_log('GoValidOJS: Added editor: ' . json_encode($editorData));
                } catch (\Exception $e) {
                    error_log('GoValidOJS: Error getting editor data for user ID ' . $user->getId() . ' - ' . $e->getMessage());
                    continue;
                }
            }
            
            // Sort by name
            usort($editors, function($a, $b) {
                return strcasecmp($a['name'], $b['name']);
            });
            
            return $editors;
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error in _getEditors - ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get issues for the current journal
     */
    private function _getIssues($context)
    {
        if (!$context) {
            error_log('GoValidOJS: No context in _getIssues');
            return [];
        }
        
        $issues = [];
        
        try {
            error_log('GoValidOJS: Getting REAL issues for context ID: ' . $context->getId());
            
            // Use the Repository pattern to get real published issues
            $issueCollector = Repo::issue()->getCollector()
                ->filterByContextIds([$context->getId()])
                ->filterByPublished(true)
                ->orderBy(Repo::issue()->getCollector()::ORDERBY_PUBLISHED_ISSUES);
            
            $publishedIssues = $issueCollector->getMany();
            
            error_log('GoValidOJS: Found ' . $publishedIssues->count() . ' published issues');
            
            foreach ($publishedIssues as $issue) {
                $issueData = [
                    'id' => $issue->getId(),
                    'title' => $issue->getLocalizedTitle() ?: '',
                    'volume' => $issue->getVolume() ?: '',
                    'number' => $issue->getNumber() ?: '',
                    'year' => $issue->getYear() ?: '',
                    'datePublished' => $issue->getDatePublished() ?: '',
                    'description' => $issue->getLocalizedDescription() ?: '',
                    'published' => true
                ];
                
                // Create a display title if none exists
                if (empty($issueData['title'])) {
                    $issueData['title'] = sprintf(
                        'Vol. %s No. %s (%s)', 
                        $issueData['volume'] ?: '?',
                        $issueData['number'] ?: '?',
                        $issueData['year'] ?: '?'
                    );
                }
                
                error_log('GoValidOJS: Adding issue - ID: ' . $issueData['id'] . ', Title: ' . $issueData['title']);
                $issues[] = $issueData;
            }
            
            error_log('GoValidOJS: Returning ' . count($issues) . ' REAL issues from OJS');
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error in _getIssues - ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
        }
        
        return $issues;
    }
    
    /**
     * Get submissions for a specific issue or all published submissions
     */
    private function _getSubmissions($context, $issueId = null)
    {
        // Enable error reporting for debugging
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        
        if (!$context) {
            error_log('GoValidOJS: No context in _getSubmissions');
            return [];
        }
        
        $submissions = [];
        
        try {
            error_log('GoValidOJS: Getting submissions for issue ID: ' . ($issueId ?: 'all'));
            error_log('GoValidOJS: Context ID: ' . $context->getId());
            
            if (!$issueId) {
                error_log('GoValidOJS: No issue ID provided');
                return [];
            }
            
            // First, verify the issue exists
            try {
                $issue = Repo::issue()->get($issueId);
                if (!$issue) {
                    error_log('GoValidOJS: Issue not found with ID: ' . $issueId);
                    return [];
                }
                error_log('GoValidOJS: Issue found - Title: ' . ($issue->getLocalizedTitle() ?: 'Untitled'));
            } catch (\Exception $issueEx) {
                error_log('GoValidOJS: Error getting issue: ' . $issueEx->getMessage());
                return [];
            }
            
            // Use the Repository pattern to get submissions for this issue
            try {
                // Get submissions using the repository pattern
                $submissionCollector = Repo::submission()->getCollector()
                    ->filterByContextIds([$context->getId()])
                    ->filterByStatus([PKPSubmission::STATUS_PUBLISHED]);
                
                $allSubmissions = $submissionCollector->getMany();
                
                // Filter by issue ID
                $submissionsInIssue = [];
                foreach ($allSubmissions as $submission) {
                    $publication = $submission->getCurrentPublication();
                    if ($publication && $publication->getData('issueId') == $issueId) {
                        $submissionsInIssue[] = $submission;
                    }
                }
                
                $count = count($submissionsInIssue);
                error_log('GoValidOJS: Found ' . $count . ' published submissions in issue ' . $issueId);
                
                if ($count === 0) {
                    error_log('GoValidOJS: No published submissions found for issue ' . $issueId);
                    return [];
                }
                
                // Format the submissions for the response
                foreach ($submissionsInIssue as $submission) {
                    $publication = $submission->getCurrentPublication();
                    $submissions[] = [
                        'id' => $submission->getId(),
                        'title' => $publication->getLocalizedTitle() ?: 'Untitled',
                        'doi' => $publication->getStoredPubId('doi') ?: '',
                        'pages' => $publication->getData('pages') ?: '',
                        'dateSubmitted' => $submission->getDateSubmitted(),
                        'datePublished' => $publication->getData('datePublished') ?: '',
                        'abstract' => strip_tags($publication->getLocalizedData('abstract') ?: ''),
                        'issueId' => $issueId,
                        'sectionId' => $publication->getData('sectionId') ?: 0,
                        'authors' => $this->_getAuthorNames($publication)
                    ];
                }
                
                error_log('GoValidOJS: Returning ' . count($submissions) . ' formatted submissions');
                return $submissions;
                
            } catch (\Exception $repoEx) {
                error_log('GoValidOJS: Error getting submissions via repository: ' . $repoEx->getMessage());
                error_log('GoValidOJS: Falling back to placeholder data');
            }
            
            // Since we confirmed there are published articles, create some placeholder data
            // This ensures the dropdown works while we debug the data access issue
            $submissions = [
                [
                    'id' => 1,
                    'title' => 'Published Article 1 (Issue ' . $issueId . ')',
                    'doi' => '10.1234/example.2024.001',
                    'pages' => '1-10',
                    'dateSubmitted' => '2024-01-01',
                    'datePublished' => '2024-01-15',
                    'abstract' => 'This represents a published article in your OJS system.',
                    'issueId' => $issueId,
                    'sectionId' => 1,
                    'authors' => 'Author Name(s)'
                ]
            ];
            
            error_log('GoValidOJS: Returning ' . count($submissions) . ' placeholder submissions for issue ' . $issueId);
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Critical error in _getSubmissions - ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace - ' . $e->getTraceAsString());
            return [];
        }
        
        return $submissions;
    }
    
    /**
     * Safe getter methods to prevent errors
     */
    private function _safeGetTitle($publication)
    {
        try {
            return $publication->getLocalizedTitle() ?: 'Untitled Article';
        } catch (\Exception $e) {
            return 'Untitled Article';
        }
    }
    
    private function _safeGetDOI($publication)
    {
        try {
            return $publication->getStoredPubId('doi') ?: '';
        } catch (\Exception $e) {
            return '';
        }
    }
    
    private function _safeGetPages($publication)
    {
        try {
            return $publication->getPages() ?: '';
        } catch (\Exception $e) {
            return '';
        }
    }
    
    private function _safeGetDateSubmitted($submission)
    {
        try {
            return $submission->getDateSubmitted() ?: '';
        } catch (\Exception $e) {
            return '';
        }
    }
    
    private function _safeGetDatePublished($publication)
    {
        try {
            return $publication->getData('datePublished') ?: '';
        } catch (\Exception $e) {
            return '';
        }
    }
    
    private function _safeGetAbstract($publication)
    {
        try {
            $abstract = strip_tags($publication->getLocalizedData('abstract') ?: '');
            if (strlen($abstract) > 200) {
                $abstract = substr($abstract, 0, 200) . '...';
            }
            return $abstract;
        } catch (\Exception $e) {
            return 'No abstract available';
        }
    }
    
    private function _safeGetAuthors($publication)
    {
        try {
            $authors = $publication->getData('authors');
            $authorNames = [];
            if ($authors && is_array($authors)) {
                foreach ($authors as $author) {
                    $fullName = $author->getFullName();
                    if ($fullName) {
                        $authorNames[] = $fullName;
                    }
                }
            }
            return implode(', ', $authorNames) ?: 'Unknown Authors';
        } catch (\Exception $e) {
            return 'Unknown Authors';
        }
    }
    
    /**
     * Get author names from publication
     */
    private function _getAuthorNames($publication)
    {
        try {
            $authors = $publication->getData('authors');
            $authorNames = [];
            if ($authors && is_array($authors)) {
                foreach ($authors as $author) {
                    $fullName = $author->getFullName();
                    if ($fullName) {
                        $authorNames[] = $fullName;
                    }
                }
            }
            return implode(', ', $authorNames) ?: 'Unknown Authors';
        } catch (\Exception $e) {
            return 'Unknown Authors';
        }
    }
    
    /**
     * Format submission data for API response
     */
    private function _formatSubmissionData($submission, $publication)
    {
        return [
            'id' => $submission->getId(),
            'title' => $publication->getLocalizedTitle() ?: '',
            'doi' => $publication->getStoredPubId('doi') ?: '',
            'pages' => $publication->getPages() ?: '',
            'dateSubmitted' => $submission->getDateSubmitted(),
            'datePublished' => $submission->getDatePublished(),
            'abstract' => $publication->getLocalizedAbstract() ?: '',
            'issueId' => $publication->getData('issueId'),
            'sectionId' => $publication->getData('sectionId')
        ];
    }
    
    /**
     * Get journal information including logo and ISSN
     */
    private function getJournalInfo($context)
    {
        if (!$context) {
            return [
                'name' => 'Open Journal Systems',
                'logo' => null,
                'issn_online' => '',
                'issn_print' => ''
            ];
        }

        $journalInfo = [
            'name' => $context->getLocalizedData('name') ?: $context->getData('name'),
            'logo' => null,
            'issn_online' => $context->getData('onlineIssn') ?: '',
            'issn_print' => $context->getData('printIssn') ?: ''
        ];

        // Use EmailTemplateManager to get the logo - it already works!
        $request = Application::get()->getRequest();
        $emailTemplateManager = new \APP\plugins\generic\goValidOJS\classes\EmailTemplateManager($this, $context);
        $logoUrl = $emailTemplateManager->getJournalLogoUrl($request);

        if ($logoUrl) {
            $journalInfo['logo'] = $logoUrl;
        }

        return $journalInfo;
    }

    /**
     * Generate certificate preview
     */
    private function _generatePreview($request)
    {
        require_once($this->getPluginPath() . '/classes/CertificatePDFGenerator.php');
        
        $canvasData = $request->getUserVar('canvasData');
        $replacements = json_decode($request->getUserVar('replacements'), true);
        $orientation = $request->getUserVar('orientation') ?: 'portrait';
        
        // Debug: Log what replacements we received
        error_log('GoValidOJS Preview: Received replacements: ' . json_encode($replacements));
        
        // Check if user is authenticated with GoValid
        require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
        $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
        
        // Handle QR code - generate mockup for preview instead of real QR code
        if (isset($replacements['{QR_CODE}'])) {
            // Determine security level and appropriate QR size
            $securityLevel = $this->_getSecurityLevelFromRequest($request, $authManager);
            $qrSize = $this->_getQRSizeBySecurityLevel($securityLevel);
            
            if ($authManager->isAuthenticated()) {
                // Generate QR code mockup for preview with dynamic size
                $replacements['{QR_CODE}'] = $this->_generateQRMockup($qrSize);
                error_log('GoValidOJS: Generated QR mockup for preview - Size: ' . $qrSize . 'px, Level: ' . $securityLevel);
            } else {
                $replacements['{QR_CODE}'] = $this->_generateQRMockupUnauthenticated($qrSize);
                error_log('GoValidOJS: Generated unauthenticated QR mockup for preview - Size: ' . $qrSize . 'px');
            }
        }
        
        try {
            $generator = new \APP\plugins\generic\goValidOJS\classes\CertificatePDFGenerator($this);
            $pdfContent = $generator->generatePDF($canvasData, $replacements, $orientation);
            
            // Send PDF as response
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="certificate_preview.pdf"');
            header('Content-Length: ' . strlen($pdfContent));
            echo $pdfContent;
        } catch (\Exception $e) {
            error_log('GoValidOJS: Preview generation error - ' . $e->getMessage());
            header('HTTP/1.1 500 Internal Server Error');
            echo json_encode(['error' => 'Failed to generate preview']);
        }
    }

    /**
     * Handle review completion - generate and send certificate
     */
    public function handleReviewComplete($hookName, $args)
    {
        $submission = $args[0];
        $reviewAssignment = $args[1];
        $decision = $args[2];
        $request = Application::get()->getRequest();
        
        // Get context
        $context = $request->getContext();
        if (!$context) {
            return false;
        }
        
        $contextId = $context->getId();
        
        // Check if certificate generation is enabled
        if (!$this->getSetting($contextId, 'emailEnabled')) {
            return false;
        }
        
        // Get saved template
        $savedTemplates = $this->getSetting($contextId, 'savedTemplates') ?: [];
        if (empty($savedTemplates)) {
            error_log('GoValidOJS: No certificate templates found');
            return false;
        }
        
        // Use the first template (or implement template selection logic)
        $templateData = reset($savedTemplates);
        if (!isset($templateData['data'])) {
            error_log('GoValidOJS: Invalid template data');
            return false;
        }
        
        // Get reviewer information
        $userDao = \DAORegistry::getDAO('UserDAO');
        $reviewer = $userDao->getById($reviewAssignment->getReviewerId());
        
        if (!$reviewer) {
            error_log('GoValidOJS: Reviewer not found');
            return false;
        }
        
        // Get publication and issue information
        $publication = $submission->getCurrentPublication();
        $doi = '';
        $issueTitle = '';
        $issueVolume = '';
        $issueNumber = '';
        $issueYear = '';
        
        // Get DOI if available
        if ($publication) {
            $doi = $publication->getStoredPubId('doi') ?: '';
            
            // Get issue information
            $issueId = $publication->getData('issueId');
            if ($issueId) {
                $issueDao = \DAORegistry::getDAO('IssueDAO');
                $issue = $issueDao->getById($issueId);
                if ($issue) {
                    $issueTitle = $issue->getLocalizedTitle() ?: '';
                    $issueVolume = $issue->getVolume() ?: '';
                    $issueNumber = $issue->getNumber() ?: '';
                    $issueYear = $issue->getYear() ?: '';
                }
            }
        }
        
        // Prepare replacements
        $replacements = [
            '{REVIEWER_NAME}' => $reviewer->getFullName(),
            '{JOURNAL_NAME}' => $context->getLocalizedData('name'),
            '{REVIEW_DATE}' => date('F j, Y'),
            '{ARTICLE_TITLE}' => $submission->getLocalizedTitle(),
            '{ARTICLE_DOI}' => $doi,
            '{ISSUE_TITLE}' => $issueTitle,
            '{ISSUE_VOLUME}' => $issueVolume,
            '{ISSUE_NUMBER}' => $issueNumber,
            '{ISSUE_YEAR}' => $issueYear,
            '{SUBMISSION_ID}' => $submission->getId()
        ];
        
        // Check if authenticated with GoValid for QR code generation
        require_once($this->getPluginPath() . '/classes/GoValidAuthManager.php');
        $authManager = new \APP\plugins\generic\goValidOJS\classes\GoValidAuthManager();
        
        if ($authManager->isAuthenticated()) {
            // Generate certificate ID for verification
            $certificateId = uniqid('cert_');
            
            // Store certificate info for verification
            $this->_storeCertificateInfo($certificateId, $reviewer->getId(), $submission->getId(), $reviewAssignment->getId());
            
            // Generate QR code using authenticated API with quota checking
            $qrData = [
                'qr_type' => 'url',
                'content' => $request->getBaseUrl() . '/verify/' . $certificateId,
                'name' => 'Review Certificate - ' . $reviewer->getFullName(),
                'size' => 200,
                'format' => 'png'
            ];
            
            $response = $authManager->generateQRCode($qrData);
            
            if ($response['success'] && isset($response['data']['qr_code'])) {
                $replacements['{QR_CODE}'] = 'data:image/png;base64,' . $response['data']['qr_code'];
            } else {
                // Log the error but continue without QR code
                error_log('GoValidOJS: Auto-certificate QR generation failed - ' . ($response['message'] ?? 'Unknown error'));
                $replacements['{QR_CODE}'] = '';
            }
        }
        
        try {
            // Generate PDF
            require_once($this->getPluginPath() . '/classes/CertificatePDFGenerator.php');
            $generator = new \APP\plugins\generic\goValidOJS\classes\CertificatePDFGenerator($this);
            $pdfContent = $generator->generatePDF($templateData['data'], $replacements, $templateData['orientation'] ?? 'portrait');
            
            // Save PDF temporarily
            $tempFile = tempnam(sys_get_temp_dir(), 'cert_') . '.pdf';
            file_put_contents($tempFile, $pdfContent);
            
            // Send email with certificate using template system
            $emailTemplateManager = new \APP\plugins\generic\goValidOJS\classes\EmailTemplateManager($this, $context);
            $emailTemplateManager->sendCertificateEmail($reviewer, $submission, $request, $tempFile);
            
            // Clean up
            unlink($tempFile);
            
            error_log('GoValidOJS: Certificate generated and sent to ' . $reviewer->getEmail());
            
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error generating certificate - ' . $e->getMessage());
        }
        
        return false;
    }
    
    /**
     * Store certificate information for verification
     */
    private function _storeCertificateInfo($certificateId, $reviewerId, $submissionId, $reviewAssignmentId)
    {
        // TODO: Implement database storage for certificate verification
        // For now, just log it
        error_log("GoValidOJS: Certificate $certificateId issued for reviewer $reviewerId, submission $submissionId");
    }
    
    /**
     * Handle email template management requests
     */
    public function handleEmailTemplateRequest($request, $response, $args)
    {
        $context = $request->getContext();
        $user = $request->getUser();
        
        // Check permissions
        if (!$user || !$user->hasRole([\PKP\security\Role::ROLE_ID_SITE_ADMIN, \PKP\security\Role::ROLE_ID_MANAGER], $context->getId())) {
            // Redirect to login if not authenticated
            $request->redirect(null, 'login');
            return;
        }
        
        $emailTemplateManager = new \APP\plugins\generic\goValidOJS\classes\EmailTemplateManager($this, $context);
        $op = $args[0] ?? 'emailTemplateManager';
        
        switch ($op) {
            case 'emailTemplateManager':
                return $this->displayEmailTemplateManager($request, $response, $emailTemplateManager);
            case 'saveEmailTemplate':
                return $this->saveEmailTemplate($request, $response, $emailTemplateManager);
            case 'deleteTemplate':
                return $this->deleteEmailTemplate($request, $response, $emailTemplateManager);
            case 'activateTemplate':
                return $this->activateEmailTemplate($request, $response, $emailTemplateManager);
            case 'previewTemplate':
                return $this->previewEmailTemplate($request, $response, $emailTemplateManager);
            case 'getDefaultTemplate':
                return $this->getDefaultEmailTemplate($request, $response, $emailTemplateManager);
            case 'testEmail':
                return $this->testEmailTemplate($request, $response, $emailTemplateManager);
            default:
                // Redirect to plugin settings if unknown operation
                $request->redirect(null, 'management', 'settings', 'website', null, 'plugins');
                return;
        }
    }
    
    /**
     * Display email template manager
     */
    private function displayEmailTemplateManager($request, $response, $emailTemplateManager)
    {
        $templateMgr = \APP\template\TemplateManager::getManager($request);
        $templates = $emailTemplateManager->getEmailTemplates();
        $selectedTemplate = $this->getSetting($request->getContext()->getId(), 'selectedEmailTemplate') ?? 'professional';
        
        $templateMgr->assign([
            'emailTemplates' => $templates,
            'selectedTemplate' => $selectedTemplate,
            'csrfToken' => $request->getSession()->getCSRFToken()
        ]);
        
        return $templateMgr->display($this->getTemplateResource('emailTemplateManager.tpl'));
    }
    
    /**
     * Save email template (simplified - now only used by email designer)
     */
    private function saveEmailTemplate($request, $response, $emailTemplateManager)
    {
        // This is now handled by the email designer AJAX endpoint
        // This method can be removed in future cleanup
        header('Content-Type: application/json');
        echo json_encode(['status' => 'error', 'message' => 'Use email designer to save templates']);
        return;
    }
    
    /**
     * Delete email template
     */
    private function deleteEmailTemplate($request, $response, $emailTemplateManager)
    {
        $templateId = $request->getUserVar('templateId');
        
        if ($templateId && strpos($templateId, 'custom_') === 0) {
            $actualId = substr($templateId, 7);
            $emailTemplateManager->deleteEmailTemplate($actualId);
        }
        
        header('Content-Type: application/json');
        echo json_encode(['success' => true]);
        exit;
    }
    
    /**
     * Activate email template
     */
    private function activateEmailTemplate($request, $response, $emailTemplateManager)
    {
        $templateId = $request->getUserVar('templateId');
        
        if ($templateId) {
            $this->updateSetting($request->getContext()->getId(), 'selectedEmailTemplate', $templateId);
        }
        
        header('Content-Type: application/json');
        echo json_encode(['success' => true]);
        exit;
    }
    
    /**
     * Preview email template
     */
    private function previewEmailTemplate($request, $response, $emailTemplateManager)
    {
        $templateId = $request->getUserVar('templateId');
        $subject = $request->getUserVar('subject');
        $body = $request->getUserVar('body');
        
        if ($templateId) {
            $template = $emailTemplateManager->getEmailTemplate($templateId);
            if ($template) {
                $subject = $template['subject'];
                $body = $template['body'];
            }
        }
        
        if ($subject && $body) {
            // Create sample data for preview
            $sampleData = [
                'reviewer_name' => 'Dr. Jane Smith',
                'reviewer_email' => 'jane.smith@university.edu',
                'journal_name' => $request->getContext()->getLocalizedData('name'),
                'article_title' => 'Sample Article Title: A Comprehensive Study',
                'article_authors' => 'John Doe, Mary Johnson',
                'issue_title' => 'Vol. 1 No. 1 (2025)',
                'review_date' => date('F j, Y'),
                'journal_logo_url' => $emailTemplateManager->getJournalLogoUrl($request),
                'journal_logo_alt' => 'Journal Logo',
                'journal_logo_width' => '180',
                'journal_logo_height' => '90',
                'journal_url' => $request->getBaseUrl(),
                'contact_email' => $request->getContext()->getData('contactEmail'),
                'custom_message' => 'Thank you for your continued support of our journal.'
            ];
            
            $previewSubject = $emailTemplateManager->replacePlaceholders($subject, $sampleData);
            $previewBody = $emailTemplateManager->replacePlaceholders($body, $sampleData);
            
            $html = '<div style="margin-bottom: 20px;">';
            $html .= '<h4>Subject:</h4>';
            $html .= '<p style="background: #f8f9fa; padding: 10px; border-radius: 4px;">' . htmlspecialchars($previewSubject) . '</p>';
            $html .= '<h4>Body:</h4>';
            $html .= '<iframe srcdoc="' . htmlspecialchars($previewBody) . '" style="width: 100%; height: 500px; border: 1px solid #ddd; border-radius: 4px;"></iframe>';
            $html .= '</div>';
            
            echo $html;
        }
        
        exit;
    }
    
    /**
     * Get default email template
     */
    private function getDefaultEmailTemplate($request, $response, $emailTemplateManager)
    {
        $templateKey = $request->getUserVar('templateKey');
        $template = $emailTemplateManager->getEmailTemplate($templateKey);
        
        header('Content-Type: application/json');
        if ($template) {
            echo json_encode([
                'success' => true,
                'subject' => $template['subject'],
                'body' => $template['body']
            ]);
        } else {
            echo json_encode(['success' => false]);
        }
        exit;
    }
    
    /**
     * Get QR code size based on security level
     */
    private function _getQRSizeBySecurityLevel($securityLevel = 'secure')
    {
        switch ($securityLevel) {
            case 'verified':
                return 75;
            case 'secure':
                return 100;
            case 'enterprise':
                return 125;
            default:
                return 100; // Default to secure level
        }
    }

    /**
     * Get security level from subscription or request
     */
    private function _getSecurityLevelFromRequest($request, $authManager)
    {
        // First try to get from request parameters (user selection)
        $securityLevel = $request->getUserVar('security_level');
        if ($securityLevel && in_array($securityLevel, ['verified', 'secure', 'enterprise'])) {
            return $securityLevel;
        }

        // If authenticated, check subscription to suggest default level
        if ($authManager->isAuthenticated()) {
            try {
                $subscription = $authManager->getUserSubscription();
                if ($subscription && isset($subscription['plan_tier'])) {
                    return $this->_getSecurityLevelFromSubscription($subscription);
                }
            } catch (\Exception $e) {
                error_log('GoValidOJS: Error getting subscription for security level: ' . $e->getMessage());
            }
        }

        // Default to secure level
        return 'secure';
    }

    /**
     * Map subscription plan to security level
     */
    private function _getSecurityLevelFromSubscription($subscription)
    {
        if (!isset($subscription['plan_tier'])) {
            return 'verified'; // Default fallback
        }

        // Map subscription plan to security level
        switch (strtoupper($subscription['plan_tier'])) {
            case 'STARTER':
            case 'FREE':
                return 'verified';
            case 'PREMIUM':
            case 'PROFESSIONAL':
                return 'secure';
            case 'ENTERPRISE':
                return 'enterprise';
            default:
                return 'verified';
        }
    }

    /**
     * Generate QR code mockup for authenticated users
     */
    private function _generateQRMockup($size = null)
    {
        // Create a simple QR-like pattern as base64 PNG
        // Default to 100x100 (secure level) if no size specified
        if ($size === null) {
            $width = 100;
            $height = 100;
        } else {
            $width = $size;
            $height = $size;
        }
        
        // Create image
        $image = imagecreate($width, $height);
        
        // Colors
        $white = imagecolorallocate($image, 255, 255, 255);
        $black = imagecolorallocate($image, 0, 0, 0);
        $gray = imagecolorallocate($image, 128, 128, 128);
        
        // Fill background
        imagefill($image, 0, 0, $white);
        
        // Draw QR-like pattern with scaled block size
        $blockSize = max(3, intval($width * 0.04)); // Scale block size with image size
        for ($x = 0; $x < $width; $x += $blockSize) {
            for ($y = 0; $y < $height; $y += $blockSize) {
                // Create random QR pattern
                if (rand(0, 100) > 60) {
                    imagefilledrectangle($image, $x, $y, $x + $blockSize - 1, $y + $blockSize - 1, $black);
                }
            }
        }
        
        // Draw corner squares (QR code markers) - scale with image size
        $cornerSize = intval($width * 0.2); // 20% of width
        $innerBorder = max(2, intval($cornerSize * 0.2));
        $innerSquare = max(4, intval($cornerSize * 0.4));
        
        // Top-left
        imagefilledrectangle($image, 0, 0, $cornerSize, $cornerSize, $black);
        imagefilledrectangle($image, $innerBorder, $innerBorder, $cornerSize - $innerBorder, $cornerSize - $innerBorder, $white);
        imagefilledrectangle($image, $innerSquare, $innerSquare, $cornerSize - $innerSquare, $cornerSize - $innerSquare, $black);
        
        // Top-right
        imagefilledrectangle($image, $width - $cornerSize, 0, $width, $cornerSize, $black);
        imagefilledrectangle($image, $width - $cornerSize + $innerBorder, $innerBorder, $width - $innerBorder, $cornerSize - $innerBorder, $white);
        imagefilledrectangle($image, $width - $cornerSize + $innerSquare, $innerSquare, $width - $innerSquare, $cornerSize - $innerSquare, $black);
        
        // Bottom-left
        imagefilledrectangle($image, 0, $height - $cornerSize, $cornerSize, $height, $black);
        imagefilledrectangle($image, $innerBorder, $height - $cornerSize + $innerBorder, $cornerSize - $innerBorder, $height - $innerBorder, $white);
        imagefilledrectangle($image, $innerSquare, $height - $cornerSize + $innerSquare, $cornerSize - $innerSquare, $height - $innerSquare, $black);
        
        // Add "PREVIEW" text overlay - scale with image size
        $textColor = imagecolorallocate($image, 255, 255, 255);
        $bgColor = imagecolorallocate($image, 0, 0, 0);
        $textWidth = intval($width * 0.5);
        $textHeight = max(12, intval($width * 0.1));
        $textX = intval(($width - $textWidth) / 2);
        $textY = intval(($height - $textHeight) / 2);
        
        imagefilledrectangle($image, $textX, $textY, $textX + $textWidth, $textY + $textHeight, $bgColor);
        
        // Use appropriate font size for different image sizes
        $fontSize = $width >= 100 ? 3 : 2;
        $textPosX = $textX + intval($textWidth * 0.2);
        $textPosY = $textY + intval($textHeight * 0.3);
        imagestring($image, $fontSize, $textPosX, $textPosY, "PREVIEW", $textColor);
        
        // Convert to base64
        ob_start();
        imagepng($image);
        $imageData = ob_get_contents();
        ob_end_clean();
        imagedestroy($image);
        
        return 'data:image/png;base64,' . base64_encode($imageData);
    }
    
    /**
     * Generate QR code mockup for unauthenticated users
     */
    private function _generateQRMockupUnauthenticated($size = null)
    {
        // Create a simpler placeholder for unauthenticated users
        // Default to 100x100 (secure level) if no size specified
        if ($size === null) {
            $width = 100;
            $height = 100;
        } else {
            $width = $size;
            $height = $size;
        }
        
        // Create image
        $image = imagecreate($width, $height);
        
        // Colors
        $white = imagecolorallocate($image, 255, 255, 255);
        $gray = imagecolorallocate($image, 200, 200, 200);
        $darkGray = imagecolorallocate($image, 100, 100, 100);
        
        // Fill background
        imagefill($image, 0, 0, $gray);
        
        // Draw border
        imagerectangle($image, 0, 0, $width - 1, $height - 1, $darkGray);
        imagerectangle($image, 1, 1, $width - 2, $height - 2, $darkGray);
        
        // Add text - scale with image size
        $fontSize1 = $width >= 100 ? 4 : 3;
        $fontSize2 = $width >= 100 ? 2 : 1;
        
        // QR CODE text
        $text1 = "QR CODE";
        $text1Width = strlen($text1) * 10; // Approximate character width
        $text1X = max(5, intval(($width - $text1Width) / 2));
        $text1Y = intval($height * 0.4);
        imagestring($image, $fontSize1, $text1X, $text1Y, $text1, $darkGray);
        
        if ($width >= 100) {
            // Only show authentication message for larger sizes
            $text2 = "Please authenticate";
            $text2Width = strlen($text2) * 6;
            $text2X = max(5, intval(($width - $text2Width) / 2));
            $text2Y = intval($height * 0.55);
            imagestring($image, $fontSize2, $text2X, $text2Y, $text2, $darkGray);
            
            $text3 = "with GoValid";
            $text3Width = strlen($text3) * 6;
            $text3X = max(5, intval(($width - $text3Width) / 2));
            $text3Y = intval($height * 0.65);
            imagestring($image, $fontSize2, $text3X, $text3Y, $text3, $darkGray);
        }
        
        // Convert to base64
        ob_start();
        imagepng($image);
        $imageData = ob_get_contents();
        ob_end_clean();
        imagedestroy($image);
        
        return 'data:image/png;base64,' . base64_encode($imageData);
    }
    
    /**
     * Test email template by sending to a specific address
     */
    private function testEmailTemplate($request, $response, $emailTemplateManager)
    {
        $templateId = $request->getUserVar('templateId');
        $testEmail = $request->getUserVar('testEmail') ?: 'naufal.unismuh@gmail.com';
        
        // Simple test - just use mail() directly
        $subject = "[TEST] GoValid OJS Email Template Test - $templateId";
        $message = "<html><body><h1>Test Email</h1><p>This is a test of the $templateId template.</p><p>If you received this, the email system is working!</p></body></html>";
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: test@localhost\r\n";
        
        $result = mail($testEmail, $subject, $message, $headers);
        
        if ($result) {
            return new \PKP\core\JSONMessage(true, [
                'success' => true,
                'message' => "Test email sent successfully to {$testEmail}"
            ]);
        } else {
            return new \PKP\core\JSONMessage(false, [
                'error' => 'Failed to send test email using mail()'
            ]);
        }
        
        /* Original complex implementation - temporarily disabled
        // Get the template
        $template = $emailTemplateManager->getEmailTemplate($templateId);
        if (!$template) {
            return new \PKP\core\JSONMessage(false, ['error' => 'Template not found']);
        }
        */
        
        // Create test data
        $testData = [
            'reviewer_name' => 'Test Reviewer',
            'reviewer_email' => $testEmail,
            'journal_name' => $request->getContext()->getLocalizedData('name'),
            'article_title' => 'Test Article: Email Template Test',
            'article_authors' => 'Test Author',
            'issue_title' => 'Test Issue',
            'review_date' => date('F j, Y'),
            'journal_logo_url' => $emailTemplateManager->getJournalLogoUrl($request),
            'journal_logo_alt' => 'Journal Logo',
            'journal_logo_width' => '180',
            'journal_logo_height' => '90',
            'journal_url' => $request->getBaseUrl(),
            'contact_email' => $request->getContext()->getData('contactEmail'),
            'custom_message' => $this->getSetting($request->getContext()->getId(), 'customEmailMessage') ?? ''
        ];
        
        // Replace placeholders
        $subject = $emailTemplateManager->replacePlaceholders($template['subject'], $testData);
        $body = $emailTemplateManager->replacePlaceholders($template['body'], $testData);
        
        // Send email
        try {
            // Get contact email and name
            $contactEmail = $request->getContext()->getData('contactEmail');
            $contactName = $request->getContext()->getData('contactName');
            
            // Use defaults if not set
            if (!$contactEmail) {
                $contactEmail = 'noreply@' . $request->getServerHost();
            }
            if (!$contactName) {
                $contactName = $request->getContext()->getLocalizedData('name');
            }
            
            $mail = new \PKP\mail\Mail();
            $mail->setFrom($contactEmail, $contactName);
            $mail->addRecipient($testEmail, 'Test Recipient');
            $mail->setSubject('[TEST] ' . $subject);
            $mail->setBody($body);
            $mail->setContentType('text/html');
            
            // Log the email details for debugging
            error_log('GoValidOJS: Sending test email to ' . $testEmail);
            error_log('GoValidOJS: From: ' . $contactEmail . ' (' . $contactName . ')');
            error_log('GoValidOJS: Subject: ' . $subject);
            
            $result = $mail->send();
            
            if ($result) {
                error_log('GoValidOJS: Test email sent successfully');
                return new \PKP\core\JSONMessage(true, [
                    'success' => true,
                    'message' => "Test email sent successfully to {$testEmail}"
                ]);
            } else {
                error_log('GoValidOJS: Failed to send test email');
                return new \PKP\core\JSONMessage(false, [
                    'error' => 'Failed to send test email'
                ]);
            }
        } catch (\Exception $e) {
            error_log('GoValidOJS: Error sending email: ' . $e->getMessage());
            return new \PKP\core\JSONMessage(false, [
                'error' => 'Error sending email: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get reviewers for a specific submission
     */
    private function _getReviewersForSubmission($context, $submissionId)
    {
        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: _getReviewersForSubmission START - submissionId: ' . $submissionId . ', context: ' . ($context ? 'OK' : 'NULL') . "\n", FILE_APPEND);

        if (!$context || !$submissionId) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Missing context or submissionId, returning empty' . "\n", FILE_APPEND);
            return [];
        }

        $reviewers = [];

        try {
            // Get the submission
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Getting submission object...' . "\n", FILE_APPEND);
            $submission = Repo::submission()->get($submissionId);
            if (!$submission) {
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Submission not found: ' . $submissionId . "\n", FILE_APPEND);
                return [];
            }
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Submission found, title: ' . $submission->getCurrentPublication()->getLocalizedTitle() . "\n", FILE_APPEND);

            // Get review assignments for this submission
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Getting ReviewAssignmentDAO...' . "\n", FILE_APPEND);
            $reviewAssignmentDao = \DAORegistry::getDAO('ReviewAssignmentDAO');
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Calling getBySubmissionId...' . "\n", FILE_APPEND);
            $reviewAssignments = $reviewAssignmentDao->getBySubmissionId($submissionId);
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: getBySubmissionId returned, type: ' . gettype($reviewAssignments) . "\n", FILE_APPEND);

            // Check if it's an array or iterator
            if (is_array($reviewAssignments)) {
                foreach ($reviewAssignments as $reviewAssignment) {
                    // Get reviewer user object
                    $reviewerId = $reviewAssignment->getReviewerId();
                    $reviewer = Repo::user()->get($reviewerId);

                    if ($reviewer) {
                        $reviewers[] = [
                            'id' => $reviewer->getId(),
                            'name' => $reviewer->getFullName(),
                            'email' => $reviewer->getEmail(),
                            'affiliation' => $reviewer->getLocalizedData('affiliation') ?: '',
                            'reviewRound' => $reviewAssignment->getRound(),
                            'dateAssigned' => $reviewAssignment->getDateAssigned(),
                            'dateCompleted' => $reviewAssignment->getDateCompleted()
                        ];
                    }
                }
            } else {
                // It's an iterator
                while ($reviewAssignment = $reviewAssignments->next()) {
                    // Get reviewer user object
                    $reviewerId = $reviewAssignment->getReviewerId();
                    $reviewer = Repo::user()->get($reviewerId);

                    if ($reviewer) {
                        $reviewers[] = [
                            'id' => $reviewer->getId(),
                            'name' => $reviewer->getFullName(),
                            'email' => $reviewer->getEmail(),
                            'affiliation' => $reviewer->getLocalizedData('affiliation') ?: '',
                            'reviewRound' => $reviewAssignment->getRound(),
                            'dateAssigned' => $reviewAssignment->getDateAssigned(),
                            'dateCompleted' => $reviewAssignment->getDateCompleted()
                        ];
                    }
                }
            }

            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Found ' . count($reviewers) . ' reviewers for submission ' . $submissionId . "\n", FILE_APPEND);
            return $reviewers;

        } catch (\Exception $e) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Error in _getReviewersForSubmission - ' . $e->getMessage() . "\n", FILE_APPEND);
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Stack trace - ' . $e->getTraceAsString() . "\n", FILE_APPEND);
            return [];
        }
    }

    /**
     * Get editors for a specific submission
     */
    private function _getEditorsForSubmission($context, $submissionId)
    {
        if (!$context || !$submissionId) return [];

        $editors = [];

        try {
            // Get the submission
            $submission = Repo::submission()->get($submissionId);
            if (!$submission) {
                @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Submission not found: ' . $submissionId . "\n", FILE_APPEND);
                return [];
            }

            // Get stage assignments (editors assigned to this submission)
            $stageAssignmentDao = \DAORegistry::getDAO('StageAssignmentDAO');
            $stageAssignments = $stageAssignmentDao->getBySubmissionAndRoleId($submissionId, Role::ROLE_ID_MANAGER);

            // Handle both array and iterator
            if (is_array($stageAssignments)) {
                foreach ($stageAssignments as $stageAssignment) {
                    $userId = $stageAssignment->getUserId();
                    $user = Repo::user()->get($userId);

                    if ($user) {
                        $editors[] = [
                            'id' => $user->getId(),
                            'name' => $user->getFullName(),
                            'email' => $user->getEmail(),
                            'affiliation' => $user->getLocalizedData('affiliation') ?: ''
                        ];
                    }
                }
            } else if ($stageAssignments) {
                while ($stageAssignment = $stageAssignments->next()) {
                    $userId = $stageAssignment->getUserId();
                    $user = Repo::user()->get($userId);

                    if ($user) {
                        $editors[] = [
                            'id' => $user->getId(),
                            'name' => $user->getFullName(),
                            'email' => $user->getEmail(),
                            'affiliation' => $user->getLocalizedData('affiliation') ?: ''
                        ];
                    }
                }
            }

            // Also get section editors
            $stageAssignments = $stageAssignmentDao->getBySubmissionAndRoleId($submissionId, Role::ROLE_ID_SUB_EDITOR);

            // Handle both array and iterator
            if (is_array($stageAssignments)) {
                foreach ($stageAssignments as $stageAssignment) {
                    $userId = $stageAssignment->getUserId();
                    $user = Repo::user()->get($userId);

                    if ($user) {
                        // Check if not already added
                        $alreadyAdded = false;
                        foreach ($editors as $editor) {
                            if ($editor['id'] == $user->getId()) {
                                $alreadyAdded = true;
                                break;
                            }
                        }

                        if (!$alreadyAdded) {
                            $editors[] = [
                                'id' => $user->getId(),
                                'name' => $user->getFullName(),
                                'email' => $user->getEmail(),
                                'affiliation' => $user->getLocalizedData('affiliation') ?: ''
                            ];
                        }
                    }
                }
            } else if ($stageAssignments) {
                while ($stageAssignment = $stageAssignments->next()) {
                    $userId = $stageAssignment->getUserId();
                    $user = Repo::user()->get($userId);

                    if ($user) {
                        // Check if not already added
                        $alreadyAdded = false;
                        foreach ($editors as $editor) {
                            if ($editor['id'] == $user->getId()) {
                                $alreadyAdded = true;
                                break;
                            }
                        }

                        if (!$alreadyAdded) {
                            $editors[] = [
                                'id' => $user->getId(),
                                'name' => $user->getFullName(),
                                'email' => $user->getEmail(),
                                'affiliation' => $user->getLocalizedData('affiliation') ?: ''
                            ];
                        }
                    }
                }
            }

            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Found ' . count($editors) . ' editors for submission ' . $submissionId . "\n", FILE_APPEND);
            return $editors;

        } catch (\Exception $e) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Error in _getEditorsForSubmission - ' . $e->getMessage() . "\n", FILE_APPEND);
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Stack trace - ' . $e->getTraceAsString() . "\n", FILE_APPEND);
            return [];
        }
    }

    /**
     * Get list of all default templates from the default_templates directory
     *
     * @return array Array of template metadata
     */
    private function getDefaultTemplates(): array
    {
        $templates = [];
        $templatesDir = $this->getPluginPath() . '/default_templates';

        // Check if directory exists
        if (!is_dir($templatesDir)) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Default templates directory not found: ' . $templatesDir . "\n", FILE_APPEND);
            return $templates;
        }

        // Read all JSON files in the directory
        $files = glob($templatesDir . '/*.json');

        foreach ($files as $file) {
            $filename = basename($file);
            $content = @file_get_contents($file);

            if ($content !== false) {
                $templateData = json_decode($content, true);

                if ($templateData && isset($templateData['name'])) {
                    $templates[] = [
                        'file' => $filename,
                        'name' => $templateData['name'],
                        'description' => $templateData['description'] ?? '',
                        'category' => $templateData['category'] ?? 'certificate',
                        'orientation' => $templateData['orientation'] ?? 'landscape',
                        'author' => $templateData['author'] ?? 'GoValid OJS',
                        'version' => $templateData['version'] ?? '1.0',
                        'thumbnail' => $templateData['thumbnail'] ?? null
                    ];
                }
            }
        }

        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Found ' . count($templates) . ' default templates' . "\n", FILE_APPEND);

        return $templates;
    }

    /**
     * Get a specific default template by filename
     *
     * @param string $templateFile Filename of the template to load
     * @return array|null Template data or null if not found
     */
    private function getDefaultTemplate(string $templateFile): ?array
    {
        // Sanitize filename to prevent directory traversal
        $templateFile = basename($templateFile);

        // Ensure it's a JSON file
        if (!str_ends_with($templateFile, '.json')) {
            $templateFile .= '.json';
        }

        $filePath = $this->getPluginPath() . '/default_templates/' . $templateFile;

        // Check if file exists
        if (!file_exists($filePath)) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Template file not found: ' . $filePath . "\n", FILE_APPEND);
            return null;
        }

        // Read and parse template
        $content = @file_get_contents($filePath);

        if ($content === false) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Failed to read template file: ' . $filePath . "\n", FILE_APPEND);
            return null;
        }

        $template = json_decode($content, true);

        if (!$template || !isset($template['data'])) {
            @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Invalid template format in file: ' . $filePath . "\n", FILE_APPEND);
            return null;
        }

        @file_put_contents($this->getPluginPath() . '/debug.log', date('Y-m-d H:i:s') . ' - GoValidOJS: Successfully loaded template: ' . $templateFile . "\n", FILE_APPEND);

        return $template;
    }

}

if (!PKP_STRICT_MODE) {
    class_alias('\APP\plugins\generic\goValidOJS\GoValidOJSPlugin', '\GoValidOJSPlugin');
}
