<?php
/**
 * @file FileQueue.php
 * Simple file-based queue for OTP requests
 */

class GoValidFileQueue {
    private $queueDir;
    const MAX_RETRIES = 3;
    const MAX_QUEUE_SIZE = 100;
    const CLEANUP_AGE = 3600;
    const MAX_QUEUE_PER_EMAIL = 3;

    public function __construct($pluginPath) {
        $this->queueDir = $pluginPath . '/lib/queue/data';

        if (!is_dir($this->queueDir)) {
            mkdir($this->queueDir, 0755, true);
            $this->createSecurityFiles();
        }

        if (!is_writable($this->queueDir)) {
            throw new Exception('Queue directory not writable');
        }
    }

    private function createSecurityFiles() {
        file_put_contents($this->queueDir . '/.htaccess', "Deny from all\n");
        file_put_contents($this->queueDir . '/.gitignore', "*.json\n!.gitignore\n");
        file_put_contents($this->queueDir . '/README.txt', "OTP Queue Storage\n");
    }

    public function enqueue($email, $purpose = 'certificate_signing') {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid email');
        }

        $currentQueueSize = count(glob($this->queueDir . '/otp_*.json'));
        if ($currentQueueSize >= self::MAX_QUEUE_SIZE) {
            throw new Exception('Queue full');
        }

        $emailQueueCount = $this->countPendingForEmail($email);
        if ($emailQueueCount >= self::MAX_QUEUE_PER_EMAIL) {
            throw new Exception('Too many pending requests');
        }

        $queueId = uniqid('otp_', true);
        $queueFile = $this->queueDir . '/' . $queueId . '.json';

        $data = [
            'queue_id' => $queueId,
            'email' => $email,
            'purpose' => $purpose,
            'status' => 'pending',
            'session_id' => null,
            'created_at' => time(),
            'updated_at' => time(),
            'retry_count' => 0,
            'last_error' => null,
            'last_attempt_at' => null
        ];

        file_put_contents($queueFile, json_encode($data, JSON_PRETTY_PRINT));
        error_log("GoValidOJS: Enqueued {$queueId} for {$email}");

        return $queueId;
    }

    public function getStatus($queueId) {
        if (!preg_match('/^otp_[a-f0-9.]+$/', $queueId)) {
            return ['status' => 'invalid'];
        }

        $queueFile = $this->queueDir . '/' . $queueId . '.json';

        if (!file_exists($queueFile)) {
            return ['status' => 'not_found'];
        }

        $data = json_decode(file_get_contents($queueFile), true);
        return $data ?: ['status' => 'error'];
    }

    public function update($queueId, $updates) {
        $queueFile = $this->queueDir . '/' . $queueId . '.json';

        if (!file_exists($queueFile)) {
            return false;
        }

        $data = json_decode(file_get_contents($queueFile), true);
        $data = array_merge($data, $updates);
        $data['updated_at'] = time();

        file_put_contents($queueFile, json_encode($data, JSON_PRETTY_PRINT));
        return true;
    }

    public function getPending($limit = 5) {
        $pending = [];
        $files = glob($this->queueDir . '/otp_*.json');

        if (!$files) return [];

        foreach ($files as $file) {
            if (count($pending) >= $limit) break;

            $data = json_decode(file_get_contents($file), true);
            if ($data && $data['status'] === 'pending' && $data['retry_count'] < self::MAX_RETRIES) {
                $pending[] = $data;
            }
        }

        return $pending;
    }

    public function cleanup() {
        $cleaned = 0;
        $files = glob($this->queueDir . '/otp_*.json');
        $cutoffTime = time() - self::CLEANUP_AGE;

        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);

            if (!$data) {
                unlink($file);
                $cleaned++;
                continue;
            }

            if ($data['created_at'] < $cutoffTime && in_array($data['status'], ['sent', 'failed', 'verified'])) {
                unlink($file);
                $cleaned++;
            }
        }

        return $cleaned;
    }

    private function countPendingForEmail($email) {
        $count = 0;
        $files = glob($this->queueDir . '/otp_*.json');

        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data && $data['email'] === $email && $data['status'] === 'pending') {
                $count++;
            }
        }

        return $count;
    }
}
