/**
 * GoValid Smart Verify Widget Loader
 * Dynamically injects the widget HTML into the page
 */
(function() {
    'use strict';

    // Get config from PHP
    var config = window.GoValidSmartVerifyConfig || {};
    var position = config.position || 'bottom-right';

    // Widget HTML template with Camera tab
    var widgetHtml = `
<div id="govalid-smart-verify-widget" class="govalid-sv-widget govalid-sv-${position}">
    <button id="govalid-sv-toggle" class="govalid-sv-toggle" title="Verify QR Code">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
            <polyline points="22 4 12 14.01 9 11.01"></polyline>
        </svg>
    </button>
    <div id="govalid-sv-panel" class="govalid-sv-panel">
        <div class="govalid-sv-header">
            <div class="govalid-sv-title">
                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
                    <polyline points="22 4 12 14.01 9 11.01"></polyline>
                </svg>
                Smart Verify
            </div>
            <button id="govalid-sv-close" class="govalid-sv-close">&times;</button>
        </div>
        <div class="govalid-sv-body">
            <div class="govalid-sv-tabs">
                <button class="govalid-sv-tab active" data-tab="qr-id">QR ID</button>
                <button class="govalid-sv-tab" data-tab="camera">Camera</button>
                <button class="govalid-sv-tab" data-tab="upload">Upload</button>
            </div>
            <div id="govalid-sv-tab-qr-id" class="govalid-sv-tab-content active">
                <div class="govalid-sv-input-group">
                    <input type="text" id="govalid-sv-qr-input" class="govalid-sv-input" placeholder="Enter QR ID (e.g., 1ACE-DO-abc123)" />
                    <button id="govalid-sv-search-btn" class="govalid-sv-btn-primary">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                    </button>
                </div>
                <div class="govalid-sv-hint">Format: XXXX-XX-xxxxxxxx or XXXXXXxxxxxxxx</div>
            </div>
            <div id="govalid-sv-tab-camera" class="govalid-sv-tab-content">
                <div id="govalid-sv-camera-container" class="govalid-sv-camera-container">
                    <video id="govalid-sv-camera-video" class="govalid-sv-camera-video" playsinline></video>
                    <canvas id="govalid-sv-camera-canvas" style="display:none;"></canvas>
                    <div id="govalid-sv-camera-overlay" class="govalid-sv-camera-overlay">
                        <div class="govalid-sv-scan-region"></div>
                    </div>
                </div>
                <div id="govalid-sv-camera-controls" class="govalid-sv-camera-controls">
                    <button id="govalid-sv-camera-start" class="govalid-sv-btn-primary">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"></path>
                            <circle cx="12" cy="13" r="4"></circle>
                        </svg>
                        Start Camera
                    </button>
                    <button id="govalid-sv-camera-stop" class="govalid-sv-btn-secondary" style="display:none;">
                        Stop
                    </button>
                </div>
                <div id="govalid-sv-camera-status" class="govalid-sv-camera-status"></div>
            </div>
            <div id="govalid-sv-tab-upload" class="govalid-sv-tab-content">
                <div id="govalid-sv-dropzone" class="govalid-sv-dropzone">
                    <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                        <polyline points="17 8 12 3 7 8"></polyline>
                        <line x1="12" y1="3" x2="12" y2="15"></line>
                    </svg>
                    <span>Drop image here or click to upload</span>
                    <input type="file" id="govalid-sv-file-input" accept="image/*" style="display: none;" />
                </div>
                <div id="govalid-sv-preview" class="govalid-sv-preview" style="display: none;">
                    <img id="govalid-sv-preview-img" src="" alt="Preview" />
                    <button id="govalid-sv-clear-preview" class="govalid-sv-btn-secondary">Clear</button>
                </div>
            </div>
            <div id="govalid-sv-results" class="govalid-sv-results" style="display: none;">
                <div id="govalid-sv-loading" class="govalid-sv-loading" style="display: none;">
                    <div class="govalid-sv-spinner"></div>
                    <span>Verifying...</span>
                </div>
                <div id="govalid-sv-result-content"></div>
            </div>
        </div>
        <div class="govalid-sv-footer">
            <a href="https://govalid.org" target="_blank" class="govalid-sv-powered">
                Powered by <strong>GoValid</strong>
            </a>
        </div>
    </div>
</div>`;

    // Camera scanning variables
    var cameraStream = null;
    var scanInterval = null;

    // Inject widget when DOM is ready
    function injectWidget() {
        if (document.getElementById('govalid-smart-verify-widget')) {
            return; // Already injected
        }

        var container = document.createElement('div');
        container.innerHTML = widgetHtml;
        document.body.appendChild(container.firstElementChild);

        // Initialize widget functionality
        initWidget();
    }

    // Initialize widget events and functionality
    function initWidget() {
        var widget = document.getElementById('govalid-smart-verify-widget');
        if (!widget) return;

        var toggleBtn = document.getElementById('govalid-sv-toggle');
        var panel = document.getElementById('govalid-sv-panel');
        var closeBtn = document.getElementById('govalid-sv-close');
        var qrInput = document.getElementById('govalid-sv-qr-input');
        var searchBtn = document.getElementById('govalid-sv-search-btn');
        var dropzone = document.getElementById('govalid-sv-dropzone');
        var fileInput = document.getElementById('govalid-sv-file-input');
        var preview = document.getElementById('govalid-sv-preview');
        var previewImg = document.getElementById('govalid-sv-preview-img');
        var clearPreviewBtn = document.getElementById('govalid-sv-clear-preview');
        var resultsDiv = document.getElementById('govalid-sv-results');
        var loadingDiv = document.getElementById('govalid-sv-loading');
        var resultContent = document.getElementById('govalid-sv-result-content');
        var tabs = document.querySelectorAll('.govalid-sv-tab');

        // Camera elements
        var cameraVideo = document.getElementById('govalid-sv-camera-video');
        var cameraCanvas = document.getElementById('govalid-sv-camera-canvas');
        var cameraStartBtn = document.getElementById('govalid-sv-camera-start');
        var cameraStopBtn = document.getElementById('govalid-sv-camera-stop');
        var cameraStatus = document.getElementById('govalid-sv-camera-status');

        // Toggle panel
        toggleBtn.addEventListener('click', function() {
            panel.classList.toggle('open');
            toggleBtn.classList.toggle('active');
        });

        closeBtn.addEventListener('click', function() {
            panel.classList.remove('open');
            toggleBtn.classList.remove('active');
            stopCamera();
        });

        // Tab switching
        tabs.forEach(function(tab) {
            tab.addEventListener('click', function() {
                var tabId = this.dataset.tab;
                tabs.forEach(function(t) { t.classList.toggle('active', t.dataset.tab === tabId); });
                document.querySelectorAll('.govalid-sv-tab-content').forEach(function(c) {
                    c.classList.toggle('active', c.id === 'govalid-sv-tab-' + tabId);
                });
                hideResults();

                // Stop camera when switching away from camera tab
                if (tabId !== 'camera') {
                    stopCamera();
                }
            });
        });

        // QR ID search
        searchBtn.addEventListener('click', searchByQRId);
        qrInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') searchByQRId();
        });

        // Camera controls
        cameraStartBtn.addEventListener('click', startCamera);
        cameraStopBtn.addEventListener('click', stopCamera);

        // File upload
        dropzone.addEventListener('click', function() { fileInput.click(); });
        fileInput.addEventListener('change', function(e) {
            if (e.target.files[0]) processFile(e.target.files[0]);
        });

        // Drag and drop
        dropzone.addEventListener('dragover', function(e) {
            e.preventDefault();
            dropzone.classList.add('dragover');
        });
        dropzone.addEventListener('dragleave', function(e) {
            e.preventDefault();
            dropzone.classList.remove('dragover');
        });
        dropzone.addEventListener('drop', function(e) {
            e.preventDefault();
            dropzone.classList.remove('dragover');
            var file = e.dataTransfer.files[0];
            if (file && file.type.startsWith('image/')) processFile(file);
        });

        // Clear preview
        clearPreviewBtn.addEventListener('click', function() {
            preview.style.display = 'none';
            dropzone.style.display = 'flex';
            previewImg.src = '';
            fileInput.value = '';
            hideResults();
        });

        // Close on outside click
        document.addEventListener('click', function(e) {
            if (panel.classList.contains('open') && !panel.contains(e.target) && !toggleBtn.contains(e.target)) {
                panel.classList.remove('open');
                toggleBtn.classList.remove('active');
                stopCamera();
            }
        });

        // Camera functions
        function startCamera() {
            if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
                cameraStatus.textContent = 'Camera not supported on this device';
                cameraStatus.style.color = '#c53030';
                return;
            }

            cameraStatus.textContent = 'Starting camera...';
            cameraStatus.style.color = '#666';

            navigator.mediaDevices.getUserMedia({
                video: { facingMode: 'environment' }
            })
            .then(function(stream) {
                cameraStream = stream;
                cameraVideo.srcObject = stream;
                cameraVideo.play();

                cameraStartBtn.style.display = 'none';
                cameraStopBtn.style.display = 'inline-flex';
                cameraStatus.textContent = 'Scanning for QR code...';
                cameraStatus.style.color = '#667eea';

                // Start scanning
                scanInterval = setInterval(scanCameraFrame, 300);
            })
            .catch(function(err) {
                console.error('Camera error:', err);
                cameraStatus.textContent = 'Camera access denied or not available';
                cameraStatus.style.color = '#c53030';
            });
        }

        function stopCamera() {
            if (scanInterval) {
                clearInterval(scanInterval);
                scanInterval = null;
            }

            if (cameraStream) {
                cameraStream.getTracks().forEach(function(track) {
                    track.stop();
                });
                cameraStream = null;
            }

            cameraVideo.srcObject = null;
            cameraStartBtn.style.display = 'inline-flex';
            cameraStopBtn.style.display = 'none';
            cameraStatus.textContent = '';
        }

        function scanCameraFrame() {
            if (!cameraVideo.videoWidth || !cameraVideo.videoHeight) return;

            var canvas = cameraCanvas;
            var ctx = canvas.getContext('2d');
            canvas.width = cameraVideo.videoWidth;
            canvas.height = cameraVideo.videoHeight;
            ctx.drawImage(cameraVideo, 0, 0);

            var imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);

            if (typeof jsQR !== 'undefined') {
                var code = jsQR(imageData.data, imageData.width, imageData.height, {
                    inversionAttempts: 'dontInvert'
                });

                if (!code) {
                    code = jsQR(imageData.data, imageData.width, imageData.height, {
                        inversionAttempts: 'attemptBoth'
                    });
                }

                if (code && code.data) {
                    // QR code found!
                    stopCamera();
                    cameraStatus.textContent = 'QR code detected!';
                    cameraStatus.style.color = '#38a169';

                    if (isGoValidQR(code.data)) {
                        verifyGoValidQR(code.data);
                    } else {
                        showError('This is not a GoValid QR code');
                    }
                }
            }
        }

        function searchByQRId() {
            var qrId = qrInput.value.trim();
            if (!qrId) {
                showError('Please enter a QR ID');
                return;
            }
            showLoading();
            fetch('https://my.govalid.org/qr_codes/api/v1/qr/lookup/?formatted_uuid=' + encodeURIComponent(qrId))
                .then(function(r) { return r.json(); })
                .then(function(data) {
                    if (data.success && data.data) {
                        showQRDetails(data.data);
                    } else {
                        showError(data.message || 'QR code not found');
                    }
                })
                .catch(function() { showError('Failed to search. Please try again.'); });
        }

        function processFile(file) {
            var reader = new FileReader();
            reader.onload = function(e) {
                previewImg.src = e.target.result;
                preview.style.display = 'block';
                dropzone.style.display = 'none';
                scanImage(e.target.result);
            };
            reader.readAsDataURL(file);
        }

        function scanImage(imageSrc) {
            showLoading();
            var img = new Image();
            img.onload = function() {
                var canvas = document.createElement('canvas');
                var ctx = canvas.getContext('2d');
                canvas.width = img.width;
                canvas.height = img.height;
                ctx.drawImage(img, 0, 0);
                var imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);

                var code = null;
                if (typeof jsQR !== 'undefined') {
                    code = jsQR(imageData.data, imageData.width, imageData.height, { inversionAttempts: 'dontInvert' });
                    if (!code) {
                        code = jsQR(imageData.data, imageData.width, imageData.height, { inversionAttempts: 'attemptBoth' });
                    }
                }

                if (code && code.data) {
                    console.log('jsQR decoded:', code.data);
                    if (isGoValidQR(code.data)) {
                        verifyGoValidQR(code.data);
                    } else {
                        showError('This is not a GoValid QR code');
                    }
                } else {
                    // Client-side decoding failed, try server-side API
                    console.log('jsQR failed, trying server-side decode...');
                    decodeImageServerSide(imageSrc);
                }
            };
            img.onerror = function() { showError('Failed to load image'); };
            img.src = imageSrc;
        }

        function decodeImageServerSide(imageSrc) {
            // Extract base64 data
            var base64Data = imageSrc;
            if (imageSrc.includes(',')) {
                base64Data = imageSrc.split(',')[1];
            }

            fetch('https://my.govalid.org/api/v1/ojs/decode-qr-image/', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ image_base64: base64Data })
            })
            .then(function(r) { return r.json(); })
            .then(function(result) {
                var data = result.content || result;
                if (data.success && data.qr_data) {
                    console.log('Server decoded:', data.qr_data);
                    if (data.is_govalid_qr) {
                        verifyGoValidQR(data.qr_data);
                    } else {
                        showError('This is not a GoValid QR code');
                    }
                } else {
                    showError(data.error || 'No QR code detected in the image');
                }
            })
            .catch(function(err) {
                console.error('Server decode error:', err);
                showError('Failed to decode QR code');
            });
        }

        function isGoValidQR(data) {
            // Check for verify-scan URL pattern
            if (data.includes('govalid.org/qr_codes/verify-scan/')) {
                return true;
            }
            // Check if it's a GoValid URL (/v/ signed URL or /verify/)
            if (data.includes('govalid.org/v/') || data.includes('govalid.org/verify/') || data.includes('govalid.org/qr/')) {
                return true;
            }
            // Check if it looks like a GoValid formatted UUID (XXXX-XX-xxxxxxxx pattern)
            if (/^[A-Z0-9]{4}-[A-Z0-9]{2}-[a-zA-Z0-9]+$/i.test(data)) {
                return true;
            }
            return false;
        }

        function verifyGoValidQR(data) {
            showLoading();
            console.log('verifyGoValidQR called with:', data);

            // Check for /v/ signed URL pattern - extract token and verify
            var signedUrlMatch = data.match(/govalid\.org\/v\/([A-Za-z0-9_.-]+)/);
            console.log('signedUrlMatch:', signedUrlMatch);
            if (signedUrlMatch && signedUrlMatch[1]) {
                console.log('Signed URL detected, verifying token...');
                var signedToken = signedUrlMatch[1];

                // Call the verification page to get QR details
                // We'll fetch the verification URL and parse the response
                verifySignedToken(signedToken, data);
                return;
            }

            // Check for verify-scan URL with token params
            var tokenMatch = data.match(/[?&]t=([A-Za-z0-9_-]+)/);
            if (tokenMatch) {
                // Use scan/verify API for token-based verification
                var token = tokenMatch[1];
                var sigMatch = data.match(/[?&]s=([A-Za-z0-9_-]+)/);
                var verMatch = data.match(/[?&]v=([A-Za-z0-9]+)/);

                var payload = { token: token };
                if (sigMatch) payload.signature = sigMatch[1];
                if (verMatch) payload.version = verMatch[1];

                fetch('https://my.govalid.org/api/v1/scan/verify/', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(payload)
                })
                .then(function(r) { return r.json(); })
                .then(function(result) {
                    if (result.success && result.is_verified) {
                        var qrData = {
                            name: result.qr_code ? result.qr_code.name : '',
                            formatted_uuid: result.qr_code ? result.qr_code.uuid : '',
                            template_type: result.qr_code ? result.qr_code.template_type : '',
                            scan_count: result.qr_code ? result.qr_code.scan_count : 0,
                            is_active: result.qr_code ? result.qr_code.is_active : true,
                            creator: result.creator || null,
                            anti_counterfeit: result.anti_counterfeit || null
                        };
                        showQRDetails(qrData, data);
                    } else {
                        showError(result.message || result.error || 'QR code verification failed');
                    }
                })
                .catch(function() {
                    showBasicVerification(data);
                });
                return;
            }

            // Try formatted UUID lookup
            var qrId = extractQRId(data);
            if (qrId && /^[A-Z0-9]{4}-[A-Z0-9]{2}-[a-f0-9]+$/i.test(qrId)) {
                fetch('https://my.govalid.org/qr_codes/api/v1/qr/lookup/?formatted_uuid=' + encodeURIComponent(qrId))
                    .then(function(r) { return r.json(); })
                    .then(function(response) {
                        if (response.success && response.data) {
                            showQRDetails(response.data);
                        } else if (data.includes('govalid.org')) {
                            showBasicVerification(data);
                        } else {
                            showError('QR code not found');
                        }
                    })
                    .catch(function() {
                        if (data.includes('govalid.org')) {
                            showBasicVerification(data);
                        } else {
                            showError('Failed to verify QR code');
                        }
                    });
            } else if (data.includes('govalid.org')) {
                // If it's a GoValid URL but we couldn't extract ID, show basic verification
                showBasicVerification(data);
            } else {
                showError('Unable to verify QR code');
            }
        }

        function verifySignedToken(signedToken, originalUrl) {
            // Use the OJS verify-signed API endpoint to get QR details
            console.log('verifySignedToken called with:', signedToken);
            fetch('https://my.govalid.org/api/v1/ojs/verify-signed/?token=' + encodeURIComponent(signedToken))
                .then(function(r) { return r.json(); })
                .then(function(result) {
                    console.log('verify-signed response:', result);
                    var data = result.content || result;
                    console.log('Parsed data:', data);
                    if (data.success && data.qr_code) {
                        var qrData = {
                            name: data.qr_code.name || '',
                            formatted_uuid: data.qr_code.formatted_uuid || data.qr_code.uuid || '',
                            template_type: data.qr_code.template_type || data.qr_code.qr_type || '',
                            scan_count: data.qr_code.scan_count || 0,
                            is_active: data.qr_code.is_active !== false,
                            creator: data.creator || null,
                            anti_counterfeit: data.anti_counterfeit || null
                        };
                        console.log('Calling showQRDetails with:', qrData);
                        showQRDetails(qrData, originalUrl);
                    } else {
                        console.log('Token lookup failed, showing verification link');
                        // Show basic verification with link to full verification page
                        showBasicVerification(originalUrl);
                    }
                })
                .catch(function(err) {
                    console.error('Signed token verify error:', err);
                    showBasicVerification(originalUrl);
                });
        }

        function showBasicVerification(url) {
            resultsDiv.style.display = 'block';
            loadingDiv.style.display = 'none';
            resultContent.innerHTML = '<div class="govalid-sv-result govalid-sv-result-success"><div class="govalid-sv-result-header"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline></svg><span>GoValid QR Code Detected</span></div><div class="govalid-sv-result-body"><p style="margin: 0; color: #666; font-size: 13px;">Click below to view full verification details on GoValid.</p></div><div class="govalid-sv-result-footer"><a href="' + escapeHtml(url) + '" target="_blank" class="govalid-sv-btn-primary">Verify on GoValid</a></div></div>';
        }

        function extractQRId(data) {
            // If it's a URL, extract the ID from it
            if (data.includes('govalid.org/')) {
                var urlParts = data.split('/');
                var qrId = urlParts[urlParts.length - 1] || urlParts[urlParts.length - 2];
                if (qrId && qrId.includes('?')) {
                    qrId = qrId.split('?')[0];
                }
                return qrId;
            }
            // If it matches formatted UUID pattern, return as is
            if (/^[A-Z0-9]{4}-[A-Z0-9]{2}-[a-zA-Z0-9]+$/i.test(data)) {
                return data;
            }
            // Return the raw data to try direct lookup
            return data;
        }

        function showLoading() {
            resultsDiv.style.display = 'block';
            loadingDiv.style.display = 'flex';
            resultContent.innerHTML = '';
        }

        function hideResults() {
            resultsDiv.style.display = 'none';
            loadingDiv.style.display = 'none';
            resultContent.innerHTML = '';
        }

        function showError(message) {
            resultsDiv.style.display = 'block';
            loadingDiv.style.display = 'none';
            resultContent.innerHTML = '<div class="govalid-sv-error"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><line x1="15" y1="9" x2="9" y2="15"></line><line x1="9" y1="9" x2="15" y2="15"></line></svg><span>' + escapeHtml(message) + '</span></div>';
        }

        function showQRDetails(data, verifyUrl) {
            resultsDiv.style.display = 'block';
            loadingDiv.style.display = 'none';

            var isActive = data.is_active !== false;

            // Check for anti-counterfeit violations (handle both API formats)
            var antiCounterfeit = data.anti_counterfeit || {};
            var hasViolations = antiCounterfeit.has_violations || antiCounterfeit.has_violation || false;
            var acAction = antiCounterfeit.action || antiCounterfeit.violation_action || null;

            // Determine status based on anti-counterfeit
            var statusClass, statusText, statusIcon;
            if (hasViolations && acAction === 'block') {
                statusClass = 'danger';
                statusText = 'Potentially Counterfeit';
                statusIcon = '<circle cx="12" cy="12" r="10"></circle><line x1="15" y1="9" x2="9" y2="15"></line><line x1="9" y1="9" x2="15" y2="15"></line>';
            } else if (hasViolations && acAction === 'warning') {
                statusClass = 'warning';
                statusText = 'Warning: Suspicious Activity';
                statusIcon = '<path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path><line x1="12" y1="9" x2="12" y2="13"></line><line x1="12" y1="17" x2="12.01" y2="17"></line>';
            } else if (!isActive) {
                statusClass = 'warning';
                statusText = 'Inactive';
                statusIcon = '<circle cx="12" cy="12" r="10"></circle><line x1="12" y1="8" x2="12" y2="12"></line><line x1="12" y1="16" x2="12.01" y2="16"></line>';
            } else {
                statusClass = 'success';
                statusText = 'Valid';
                statusIcon = '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline>';
            }

            // Build verification URL if not provided
            var url = verifyUrl || data.verification_url || '';
            if (!url && data.formatted_uuid) {
                url = 'https://my.govalid.org/v/' + data.formatted_uuid;
            }

            // Extract creator info
            var creator = data.creator || {};
            var generatedBy = creator.full_name || creator.username || null;
            var institutionName = creator.institution_name || null;
            var isVerified = creator.is_verified || creator.staff_verified || false;

            // Build verified badge SVG
            var verifiedBadge = isVerified
                ? '<svg width="14" height="14" viewBox="0 0 24 24" fill="#1da1f2" style="margin-left: 4px; vertical-align: middle;"><path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>'
                : '';

            var html = '<div class="govalid-sv-result govalid-sv-result-' + statusClass + '">';
            html += '<div class="govalid-sv-result-header"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' + statusIcon + '</svg><span>' + statusText + ' GoValid QR Code</span></div>';

            // Show anti-counterfeit warning if violations exist
            if (hasViolations) {
                html += '<div class="govalid-sv-warning-box" style="background: ' + (acAction === 'block' ? '#fee2e2' : '#fef3c7') + '; border: 1px solid ' + (acAction === 'block' ? '#f87171' : '#fbbf24') + '; border-radius: 6px; padding: 10px; margin: 10px 0;">';
                html += '<div style="font-weight: 600; color: ' + (acAction === 'block' ? '#dc2626' : '#d97706') + '; margin-bottom: 6px;">⚠️ ' + (antiCounterfeit.violation_message || 'Suspicious activity detected') + '</div>';

                // Handle both formats: violations (array of objects) and violation_types (array of strings)
                var violations = antiCounterfeit.violations || [];
                var violationTypes = antiCounterfeit.violation_types || [];

                if (violations.length > 0) {
                    for (var i = 0; i < violations.length; i++) {
                        var v = violations[i];
                        html += '<div style="font-size: 12px; color: #666; margin: 4px 0;">• ' + escapeHtml(v.title || v.type) + '</div>';
                    }
                } else if (violationTypes.length > 0) {
                    for (var i = 0; i < violationTypes.length; i++) {
                        // Convert violation_type to readable title
                        var vType = violationTypes[i];
                        var vTitle = vType.replace(/_/g, ' ').replace(/\b\w/g, function(l) { return l.toUpperCase(); });
                        html += '<div style="font-size: 12px; color: #666; margin: 4px 0;">• ' + escapeHtml(vTitle) + '</div>';
                    }
                }
                html += '</div>';
            }

            html += '<div class="govalid-sv-result-body">';
            if (data.name) html += '<div class="govalid-sv-field"><label>Name:</label><span>' + escapeHtml(data.name) + '</span></div>';
            if (data.formatted_uuid) html += '<div class="govalid-sv-field"><label>QR ID:</label><span class="govalid-sv-mono">' + escapeHtml(data.formatted_uuid) + '</span></div>';
            if (data.template_type) html += '<div class="govalid-sv-field"><label>Type:</label><span>' + escapeHtml(data.template_type) + '</span></div>';
            if (generatedBy) html += '<div class="govalid-sv-field"><label>Generated by:</label><span>' + escapeHtml(generatedBy) + verifiedBadge + '</span></div>';
            if (institutionName) html += '<div class="govalid-sv-field"><label>Institution:</label><span>' + escapeHtml(institutionName) + '</span></div>';
            if (data.scan_count !== undefined) html += '<div class="govalid-sv-field"><label>Scans:</label><span>' + data.scan_count + '</span></div>';
            html += '</div>';
            if (url) html += '<div class="govalid-sv-result-footer"><a href="' + escapeHtml(url) + '" target="_blank" class="govalid-sv-btn-primary">View Full Details</a></div>';
            html += '</div>';

            resultContent.innerHTML = html;
        }


        function escapeHtml(text) {
            if (!text) return '';
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        console.log('GoValid Smart Verify Widget initialized');
    }

    // Run when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', injectWidget);
    } else {
        injectWidget();
    }
})();
