    function requireAuth(action, callback) {
        if (!goValidAuth.isAuthenticated) {
            console.warn('Action blocked: User not authenticated');
            alert('Please log in to use this feature');
            showLoginModal();
            return false;
        }
        
        // Additional check with server
        checkAuthStatus();
        
        if (goValidAuth.isAuthenticated) {
            return callback();
        }
        return false;
    }
    
    // Disable/enable UI based on auth status
    function updateUIAuthState() {
        const buttons = [
            'addText', 'addImage', 'addQRCode', 'saveTemplate', 
            'previewCertificate', 'clearCanvas', 'addBgImage'
        ];
        
        const isAuth = goValidAuth.isAuthenticated;
        
        // Show/hide overlay - but not during initialization
        const overlay = document.getElementById('authOverlay');
        if (overlay) {
            if (goValidAuth.initializing) {
                // During initialization, hide overlay regardless of auth status
                overlay.style.display = 'none';
            } else {
                // After initialization, show/hide based on auth status
                if (isAuth) {
                    overlay.style.display = 'none';
                } else {
                    overlay.style.display = 'flex';
                    overlay.style.alignItems = 'center';
                    overlay.style.justifyContent = 'center';
                }
            }
        }
        
        buttons.forEach(id => {
            const btn = document.getElementById(id);
            if (btn) {
                btn.disabled = !isAuth;
                if (!isAuth) {
                    btn.classList.add('disabled-auth');
                    btn.title = 'Login required';
                } else {
                    btn.classList.remove('disabled-auth');
                    btn.title = '';
                }
            }
        });
        
        // Disable canvas interactions
        if (!isAuth) {
            canvas.selection = false;
            canvas.forEachObject(function(obj) {
                obj.selectable = false;
            });
        } else {
            canvas.selection = true;
            canvas.forEachObject(function(obj) {
                obj.selectable = true;
            });
        }
        
        // Disable draggable fields when not authenticated
        const draggableFields = document.querySelectorAll('.draggable-field');
        draggableFields.forEach(field => {
            if (!isAuth) {
                field.draggable = false;
                field.style.opacity = '0.5';
                field.style.cursor = 'not-allowed';
                field.title = 'Login required';
            } else {
                field.draggable = true;
                field.style.opacity = '1';
                field.style.cursor = 'grab';
                field.title = '';
            }
        });
        
        // Update auth status display and user info section
        const authStatus = document.getElementById('govalidAuthStatus');
        const userInfoSection = document.getElementById('userInfoSection');
        const authText = document.getElementById('authStatusText');

        if (isAuth) {
            // Hide auth status, show user info
            if (authStatus) authStatus.style.display = 'none';
            if (userInfoSection) userInfoSection.style.display = 'block';

            // Populate user info
            if (goValidAuth.userInfo) {
                const userDisplayName = document.getElementById('userDisplayName');
                const userEmail = document.getElementById('userEmail');
                const userAvatarImage = document.getElementById('userAvatarImage');
                const userAvatarIcon = document.getElementById('userAvatarIcon');

                if (userDisplayName) {
                    const firstName = goValidAuth.userInfo.first_name || '';
                    const lastName = goValidAuth.userInfo.last_name || '';
                    const username = goValidAuth.userInfo.username || '';
                    const displayName = (firstName + ' ' + lastName).trim() || username;
                    userDisplayName.textContent = displayName;
                }

                if (userEmail) {
                    userEmail.textContent = goValidAuth.userInfo.email || goValidAuth.userInfo.username || '';
                }

                // Set profile photo if available
                console.log('GoValid Auth: User info:', goValidAuth.userInfo);
                console.log('GoValid Auth: Profile photo URL:', goValidAuth.userInfo.profile_photo_url);

                if (userAvatarImage && userAvatarIcon && goValidAuth.userInfo.profile_photo_url) {
                    // Check if URL is already absolute (starts with http:// or https://)
                    let photoUrl = goValidAuth.userInfo.profile_photo_url;
                    if (!photoUrl.startsWith('http://') && !photoUrl.startsWith('https://')) {
                        photoUrl = 'https://my.govalid.org' + photoUrl;
                    }
                    console.log('GoValid Auth: Setting avatar image to:', photoUrl);
                    userAvatarImage.src = photoUrl;
                    userAvatarImage.style.display = 'block';
                    userAvatarIcon.style.display = 'none';
                } else if (userAvatarImage && userAvatarIcon) {
                    // No photo, show icon
                    console.log('GoValid Auth: No profile photo, showing default icon');
                    userAvatarImage.style.display = 'none';
                    userAvatarIcon.style.display = 'block';
                }
            }
        } else {
            // Show auth status, hide user info
            if (authStatus) authStatus.style.display = 'block';
            if (userInfoSection) userInfoSection.style.display = 'none';

            if (authText) {
                authText.innerHTML = '<i class="fa fa-lock"></i> Login Required - Click to Sign In';
                authStatus.style.background = 'rgba(239, 68, 68, 0.3)';
            }
        }
    }
    var minZoom = 0.25;
    var maxZoom = 3.0;
    var zoomStep = 0.25;
    
    // Authentication Functions
    function showLoginModal() {
        const modal = document.getElementById('govalidLoginModal');
        if (modal) {
            modal.style.setProperty('display', 'block', 'important');
        }
        setTimeout(() => {
            const usernameField = document.getElementById('govalid_username');
            if (usernameField) {
                usernameField.focus();
            }
        }, 100);
    }
    
    function hideLoginModal() {
        const modal = document.getElementById('govalidLoginModal');
        if (modal) {
            modal.style.setProperty('display', 'none', 'important');
        }
        
        const form = document.getElementById('govalidLoginForm');
        if (form) {
            form.reset();
        }
        
        const errorDiv = document.getElementById('loginError');
        if (errorDiv) {
            errorDiv.style.display = 'none';
        }
        
        // Reset password visibility
        const passwordField = document.getElementById('govalid_password');
        const eyeIcon = document.querySelector('#togglePassword i');
        if (passwordField && eyeIcon) {
            passwordField.type = 'password';
            eyeIcon.className = 'fa fa-eye';
        }
        
    }
    
    // Try to refresh the token if available
    function tryRefreshToken() {
        const refreshToken = localStorage.getItem('govalidRefreshToken');
        if (!refreshToken) {
            console.log('No refresh token available');
            return false;
        }
        
        console.log('Attempting to refresh token');
        const url = buildActionURL('refreshToken');
        
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
                'Accept': 'application/json'
            },
            body: new URLSearchParams({
                refresh_token: refreshToken
            })
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success && data.content && data.content.token) {
                console.log('Token refreshed successfully');
                // Update session with new token data
                const now = Date.now();
                goValidAuth.isAuthenticated = true;
                goValidAuth.sessionStart = now;
                goValidAuth.lastCheck = now;
                goValidAuth.userInfo = data.content.user || {};
                
                // Store new session
                localStorage.setItem('govalidAuthSession', JSON.stringify({
                    sessionStart: now,
                    userInfo: data.content.user,
                    subscription: data.content.subscription
                }));
                
                // Store new refresh token if provided
                if (data.content.refresh) {
                    localStorage.setItem('govalidRefreshToken', data.content.refresh);
                }
                
                // Update UI
                updateAuthStatus({
                    authenticated: true,
                    user: data.content.user,
                    userInfo: data.content.user,
                    subscription: data.content.subscription
                });
                
                return true;
            } else {
                throw new Error('Invalid refresh response');
            }
        })
        .catch(error => {
            console.error('Token refresh failed:', error);
            // Clear invalid refresh token
            localStorage.removeItem('govalidRefreshToken');
            return false;
        });
    }
    
    // Check auth status with server
    function checkAuthStatus(forceCheck = false) {
        // Skip if we checked recently (within 5 minutes) unless forced
        const now = Date.now();
        if (!forceCheck && goValidAuth.lastCheck && (now - goValidAuth.lastCheck) < 300000) {
            console.log('Skipping auth check - using cached status');
            return;
        }
        
        // Also skip if session is still valid (within 1 hour)
        if (!forceCheck && goValidAuth.sessionStart && 
            (now - goValidAuth.sessionStart) < goValidAuth.sessionDuration) {
            console.log('Session still valid, skipping server check');
            return;
        }
        
        // If we're still initializing and have a stored session, don't check yet
        if (goValidAuth.initializing) {
            const storedSession = localStorage.getItem('govalidAuthSession');
            if (storedSession) {
                console.log('Skipping auth check during initialization with stored session');
                return;
            }
        }
        
        // Use clean AJAX URL
        const url = buildActionURL('checkAuth');
        
        fetch(url, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            },
            credentials: 'same-origin'
        })
        .then(response => {
            console.log('Response status:', response.status);
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            return response.text(); // Get as text first to debug
        })
        .then(text => {
            console.log('Raw response:', text);
            try {
                const data = JSON.parse(text);
                const result = data.content || data;
                goValidAuth.lastCheck = now;
                // Always update session data when authenticated
                if (result.authenticated) {
                    // If not already authenticated or session is about to expire, reset the timer
                    if (!goValidAuth.isAuthenticated || 
                        (goValidAuth.sessionStart && (now - goValidAuth.sessionStart) > (goValidAuth.sessionDuration * 0.75))) {
                        goValidAuth.sessionStart = now;
                    }
                    
                    // Always store the latest session data
                    try {
                        localStorage.setItem('govalidAuthSession', JSON.stringify({
                            sessionStart: goValidAuth.sessionStart,
                            userInfo: result.user || result.userInfo,
                            subscription: result.subscription
                        }));
                    } catch (e) {
                        console.log('Failed to store session:', e);
                    }
                }
                updateAuthStatus(result);
            } catch (e) {
                console.error('JSON parse error:', e, 'Raw text:', text);
                updateAuthStatus({ authenticated: false });
            }
        })
        .catch(error => {
            console.error('Auth check failed:', error);
            // If we have cached auth status and the check failed, keep using cached status
            if (goValidAuth.isAuthenticated && goValidAuth.userInfo) {
                console.log('Using cached auth status after failed check');
            } else {
                updateAuthStatus({ authenticated: false });
            }
        });
    }
    
    async function checkBeforeSave() {
        // Check if authenticated
        if (!goValidAuth.isAuthenticated) {
            alert('Please login to save templates');
            return false;
        }
        
        // Check if user info exists
        if (!goValidAuth.userInfo) {
            alert('Authentication error. Please login again.');
            return false;
        }
        
        // Check subscription status
        if (!goValidAuth.subscription) {
            alert('No subscription found. Please check your account status.');
            return false;
        }
        
        // Check if subscription is active
        if (!goValidAuth.subscription.is_active) {
            alert('Your subscription is not active. Please check your account status.');
            return false;
        }
        
        return true;
    }
    
    function updateAuthStatus(result) {
        const authStatusText = document.getElementById('authStatusText');
        const quotaSection = document.getElementById('quotaRow'); // Changed from quotaSection to quotaRow
        const subscriptionRow = document.getElementById('subscriptionRow');

        if (!authStatusText) {
            console.error('Auth status text element not found');
            return;
        }

        if (result.authenticated) {
            goValidAuth.isAuthenticated = true;
            goValidAuth.userInfo = result.user || result.userInfo;
            goValidAuth.subscription = result.subscription;

            // Update status display
            const fullName = result.userInfo?.fullName || result.user?.username || result.user?.email || 'User';
            authStatusText.textContent = `Logged in as: ${fullName}`;
            const authContainer = document.getElementById('govalidAuthStatus');
            if (authContainer) {
                authContainer.classList.add('authenticated');
                authContainer.style.background = '#f0fdf4';
                authContainer.style.borderColor = '#86efac';
                authStatusText.style.color = '#166534';
            }

            // Show subscription info if available
            if (result.subscription) {
                displayQuotaInfo(result.subscription);
                // Rows are shown by displayQuotaInfo function
            }

            // Update QR settings user name fields
            updateUserNameFields();

            // Update institution numbering visibility
            if (window.setupInstitutionNumbering) {
                setupInstitutionNumbering();
            }

            // Update email designer authentication state if available
            if (window.updateEmailDesignerAuth) {
                const quota = result.subscription?.monthly_qr_limit - result.subscription?.qr_codes_used || 0;
                window.updateEmailDesignerAuth({
                    authenticated: true,
                    quota: quota,
                    subscription: result.subscription
                });
            }

            // Update certificate designer header branding if available
            if (window.updateDesignerHeaderBranding) {
                window.updateDesignerHeaderBranding();
            }

            // Update certificate email button state if available
            if (window.updateCertificateEmailButtonState) {
                window.updateCertificateEmailButtonState();
            }
        } else {
            goValidAuth.isAuthenticated = false;
            goValidAuth.userInfo = null;
            goValidAuth.subscription = null;

            authStatusText.innerHTML = '<i class="fa fa-lock"></i> Click here to login to GoValid';
            const authContainer = document.getElementById('govalidAuthStatus');
            if (authContainer) {
                authContainer.classList.remove('authenticated');
                authContainer.style.background = 'white';
                authContainer.style.borderColor = '#e2e8f0';
                authStatusText.style.color = '#333';
            }
            // Hide subscription and quota rows
            if (quotaSection) quotaSection.style.display = 'none';
            if (subscriptionRow) subscriptionRow.style.display = 'none';

            // Clear QR settings user name fields
            updateUserNameFields();

            // Update institution numbering visibility
            if (window.setupInstitutionNumbering) {
                setupInstitutionNumbering();
            }

            // Update email designer authentication state if available
            if (window.updateEmailDesignerAuth) {
                window.updateEmailDesignerAuth({
                    authenticated: false,
                    quota: 0
                });
            }

            // Update certificate email button state if available
            if (window.updateCertificateEmailButtonState) {
                window.updateCertificateEmailButtonState();
            }
        }

        // Update UI state based on authentication - but skip if initializing
        if (!goValidAuth.initializing) {
            updateUIAuthState();
            // Update identifier format visibility based on institution membership
            if (typeof window.updateIdentifierFormatVisibility === 'function') {
                window.updateIdentifierFormatVisibility();
            }
        }
    }
    
    function displayQuotaInfo(subscription) {
        console.log('displayQuotaInfo called with:', subscription);
        const subscriptionInfo = document.getElementById('subscriptionInfo');
        const subscriptionRow = document.getElementById('subscriptionRow');
        const quotaInfo = document.getElementById('quotaInfo');
        const quotaRow = document.getElementById('quotaRow');

        // Also update header versions
        const subscriptionInfoHeader = document.getElementById('subscriptionInfoHeader');
        const subscriptionRowHeader = document.getElementById('subscriptionRowHeader');
        const quotaInfoHeader = document.getElementById('quotaInfoHeader');
        const quotaRowHeader = document.getElementById('quotaRowHeader');

        console.log('Elements found:', {subscriptionInfo, subscriptionRow, quotaInfo, quotaRow});

        if (!quotaInfo) {
            console.error('Quota info element not found');
            return;
        }

        if (subscription.business_model === 'plan') {
            const percentage = Math.round((subscription.qr_codes_used / subscription.monthly_qr_limit) * 100);
            const statusColor = percentage >= 90 ? '#f44336' : (percentage >= 75 ? '#ff9800' : '#4CAF50');

            // Format large numbers
            const formatNumber = (num) => {
                if (num >= 1000) {
                    return (num / 1000).toFixed(1) + 'K';
                }
                return num.toString();
            };

            // Row 2: Subscription Info
            let subscriptionHtml = subscription.plan_name || 'Free Plan';

            if (subscription.subscription_source === 'institution' && subscription.institution_name) {
                subscriptionHtml += ` (Institution - ${subscription.institution_name})`;
            } else if (subscription.subscription_source === 'personal') {
                subscriptionHtml += ' (Personal)';
            } else if (subscription.membership_type && subscription.institution_name) {
                subscriptionHtml += ` (${subscription.institution_name} Member)`;
            }

            if (subscriptionInfo && subscriptionRow) {
                subscriptionInfo.textContent = subscriptionHtml;
                subscriptionRow.style.display = 'block';
            }
            // Update header version too
            if (subscriptionInfoHeader && subscriptionRowHeader) {
                subscriptionInfoHeader.textContent = subscriptionHtml;
                subscriptionRowHeader.style.display = 'block';
            }

            // Row 3: Quota Info - Clean format
            let quotaHtml = `QR: ${formatNumber(subscription.qr_codes_used)}/${formatNumber(subscription.monthly_qr_limit)} `;
            quotaHtml += `<span style="color: ${statusColor};">(${percentage}%)</span>`;

            if (subscription.is_active) {
                quotaHtml += ' <span style="color: #4CAF50;">●</span>';
            } else {
                quotaHtml += ' <span style="color: #f44336;">●</span>';
            }

            quotaInfo.innerHTML = quotaHtml;
            if (quotaRow) {
                quotaRow.style.display = 'block';
            }
            // Update header version too
            if (quotaInfoHeader) {
                quotaInfoHeader.innerHTML = quotaHtml;
            }
            if (quotaRowHeader) {
                quotaRowHeader.style.display = 'block';
            }
        } else if (subscription.business_model === 'credit') {
            const formattedBalance = new Intl.NumberFormat('id-ID').format(subscription.balance_idr);
            const quotaHtml = `Balance: IDR ${formattedBalance} | Bonus QR: ${subscription.bonus_qr_quota}`;

            quotaInfo.innerHTML = quotaHtml;
            if (quotaRow) {
                quotaRow.style.display = 'block';
            }
            // Update header version too
            if (quotaInfoHeader) {
                quotaInfoHeader.innerHTML = quotaHtml;
            }
            if (quotaRowHeader) {
                quotaRowHeader.style.display = 'block';
            }
        } else if (subscription.business_model === 'unknown') {
            // Debug mode for unknown business models
            const debugHtml = `<div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 8px; border-radius: 4px; margin: 4px 0;">
                <strong>🔍 Debug Info:</strong><br>
                Model: ${subscription.detected_model || 'not detected'}<br>
                Endpoint: ${subscription.endpoint_used || 'unknown'}<br>
                <small>Check browser console for raw data</small>
            </div>`;

            quotaInfo.innerHTML = debugHtml;

            // Log raw data for debugging
            console.log('GoValidOJS: Unknown subscription format detected');
            console.log('Raw subscription data:', subscription.raw_data);
            console.log('Detected model:', subscription.detected_model);
            console.log('Endpoint used:', subscription.endpoint_used);
            if (quotaRow) {
                quotaRow.style.display = 'block';
            }
            // Update header version too
            if (quotaInfoHeader) {
                quotaInfoHeader.innerHTML = debugHtml;
            }
            if (quotaRowHeader) {
                quotaRowHeader.style.display = 'block';
            }
        } else {
            // Fallback for unexpected business models
            const fallbackHtml = `<div style="background: #f8d7da; border: 1px solid #f5c6cb; padding: 8px; border-radius: 4px; margin: 4px 0;">
                <strong>⚠️ Unknown subscription type:</strong> ${subscription.business_model}<br>
                <small>Check browser console for details</small>
            </div>`;
            quotaInfo.innerHTML = fallbackHtml;
            console.log('GoValidOJS: Unexpected business model:', subscription.business_model);
            console.log('Full subscription data:', subscription);
            if (quotaRow) {
                quotaRow.style.display = 'block';
            }
            // Update header version too
            if (quotaInfoHeader) {
                quotaInfoHeader.innerHTML = fallbackHtml;
            }
            if (quotaRowHeader) {
                quotaRowHeader.style.display = 'block';
            }
        }
    }
    
    function performLogin() {
        const username = document.getElementById('govalid_username').value.trim();
        const password = document.getElementById('govalid_password').value.trim();
        const remember = document.getElementById('govalid_remember').checked;
        const submitBtn = document.getElementById('loginSubmitBtn');
        const loginText = submitBtn.querySelector('.login-text');
        const spinner = submitBtn.querySelector('.login-spinner');
        const errorDiv = document.getElementById('loginError');
        
        if (!username || !password) {
            errorDiv.textContent = 'Please enter both username and password';
            errorDiv.style.display = 'block';
            return;
        }
        
        // Show loading state
        errorDiv.style.display = 'none';
        submitBtn.disabled = true;
        loginText.style.display = 'none';
        spinner.style.display = 'inline-block';
        
        // Debug URL
        console.log('Original window.location.href:', window.location.href);
        
        // Decode HTML entities in the URL
        const decodedHref = window.location.href.replace(/&amp;/g, '&');
        console.log('Decoded href:', decodedHref);
        
        const baseUrl = decodedHref.split('?')[0];
        const queryString = decodedHref.split('?')[1] || '';
        console.log('Base URL:', baseUrl);
        console.log('Query string:', queryString);
        
        const params = new URLSearchParams(queryString);
        console.log('Params before adding action:', params.toString());
        
        params.set('action', 'login');
        console.log('Params after adding action:', params.toString());
        
        // Ensure URL is not HTML-encoded
        const url = baseUrl + '?' + params.toString();
        console.log('Final Login URL:', url);
        
        // Use clean AJAX URL and append action to URL for OJS compatibility
        const loginUrl = buildActionURL('login');
        console.log('Using clean URL for POST:', loginUrl);
        console.log('Current window.location:', window.location.href);
        console.log('getAjaxURL() returns:', getAjaxURL());

        fetch(loginUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=login&username=${encodeURIComponent(username)}&password=${encodeURIComponent(password)}&remember=${remember ? '1' : '0'}`
        })
        .then(response => {
            console.log('Login response status:', response.status);
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            return response.text();
        })
        .then(text => {
            console.log('Login raw response:', text);
            try {
                const data = JSON.parse(text);
                console.log('Parsed login data:', data);

                // IMPORTANT: Check for OTP/2FA requirements FIRST (before success check)
                // because these responses have success=false
                if (data.content && data.content.otp_required) {
                    // Email OTP verification required (risk-based authentication)
                    console.log('Email OTP verification required (risk-based)');
                    console.log('OTP session token:', data.content.otp_session_token);
                    window.otpSessionToken = data.content.otp_session_token;

                    // Hide login form, show OTP section
                    document.getElementById('govalid_username').parentElement.parentElement.style.display = 'none';
                    document.getElementById('govalid_password').parentElement.parentElement.style.display = 'none';
                    document.querySelector('.form-options').style.display = 'none';
                    document.getElementById('loginSubmitBtn').style.display = 'none';
                    document.querySelector('.divider').style.display = 'none';
                    document.querySelector('.btn-register').style.display = 'none';

                    // Show email OTP section
                    document.getElementById('emailOTPSection').style.display = 'block';
                    errorDiv.style.display = 'none';

                    // Show message about email OTP
                    const otpMessage = document.getElementById('emailOTPMessage');
                    if (otpMessage) {
                        otpMessage.textContent = data.content.message || 'Please check your email for the verification code.';
                    }

                    // Focus first OTP input
                    const firstInput = document.querySelector('.email-otp-input[data-index="0"]');
                    if (firstInput) {
                        firstInput.focus();
                    }

                    // Setup email OTP input handlers
                    setupEmailOTPInputHandlers();
                } else if (data.content && data.content.requires_2fa) {
                    // 2FA verification required
                    console.log('2FA verification required');
                    window.twoFASessionToken = data.content['2fa_session_token'];

                    // Hide login form, show 2FA section
                    document.getElementById('govalid_username').parentElement.parentElement.style.display = 'none';
                    document.getElementById('govalid_password').parentElement.parentElement.style.display = 'none';
                    document.querySelector('.form-options').style.display = 'none';
                    document.getElementById('loginSubmitBtn').style.display = 'none';
                    document.querySelector('.divider').style.display = 'none';
                    document.querySelector('.btn-register').style.display = 'none';

                    // Show 2FA section
                    document.getElementById('twoFASection').style.display = 'block';
                    errorDiv.style.display = 'none';

                    // Focus first 2FA input
                    document.querySelector('.twofa-input[data-index="0"]').focus();

                    // Setup 2FA input handlers
                    setup2FAInputHandlers();
                } else if (data.status && data.content && data.content.success) {
                    console.log('Login successful, hiding modal...');

                    // Store access token
                    if (data.content.access) {
                        localStorage.setItem('govalid_access_token', data.content.access);
                        // Also store in goValidAuth object if available
                        if (typeof goValidAuth !== 'undefined') {
                            goValidAuth.accessToken = data.content.access;
                        }
                    }

                    // Store credentials if remember me is checked
                    if (remember) {
                        try {
                            // Store encrypted credentials
                            const credentials = btoa(JSON.stringify({
                                username: username,
                                timestamp: Date.now(),
                                remember: true
                            }));
                            localStorage.setItem('govalidRememberMe', credentials);

                            // Store refresh token if available
                            if (data.content.refresh) {
                                localStorage.setItem('govalidRefreshToken', data.content.refresh);
                                console.log('Refresh token stored successfully');
                            }
                        } catch (e) {
                            console.log('Failed to store remember me data:', e);
                        }
                    } else {
                        // Clear remember me data if not checked
                        localStorage.removeItem('govalidRememberMe');
                        localStorage.removeItem('govalidRefreshToken');
                    }
                    
                    hideLoginModal();
                    // Set session start time immediately
                    goValidAuth.sessionStart = Date.now();
                    
                    // Store the session immediately after successful login
                    if (data.content.user) {
                        goValidAuth.isAuthenticated = true;
                        goValidAuth.userInfo = data.content.user;
                        
                        // Store session in localStorage
                        try {
                            localStorage.setItem('govalidAuthSession', JSON.stringify({
                                sessionStart: goValidAuth.sessionStart,
                                userInfo: data.content.user,
                                subscription: null // Will be updated by checkAuthStatus
                            }));
                            console.log('Session stored after login');
                        } catch (e) {
                            console.log('Failed to store session:', e);
                        }
                    }
                    
                    checkAuthStatus(true); // Force check to get full user info and subscription
                } else {
                    // Login failed - extract error message
                    console.log('Login failed, data.content:', data.content);
                    let message = 'Login failed. Please check your credentials.';

                    if (data.content) {
                        if (data.content.message) {
                            message = data.content.message;
                        } else if (data.content.error && typeof data.content.error === 'object') {
                            // Handle structured error (from GoValid API)
                            message = data.content.error.details || data.content.error.message || message;
                        } else if (typeof data.content.error === 'string') {
                            message = data.content.error;
                        }
                    }

                    errorDiv.textContent = message;
                    errorDiv.style.display = 'block';
                }
            } catch (e) {
                console.error('JSON parse error:', e);
                console.error('Raw response text:', text);
                errorDiv.textContent = 'Server error: Unable to parse response. Please check console for details.';
                errorDiv.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Login failed:', error);
            errorDiv.textContent = 'Connection error. Please try again.';
            errorDiv.style.display = 'block';
        })
        .finally(() => {
            // Reset loading state
            submitBtn.disabled = false;
            loginText.style.display = 'inline';
            spinner.style.display = 'none';
        });
    }

    // 2FA Input Handlers
    function setup2FAInputHandlers() {
        const inputs = document.querySelectorAll('.twofa-input');

        inputs.forEach((input, index) => {
            // Auto-focus next input
            input.addEventListener('input', function(e) {
                if (this.value.length === 1 && index < inputs.length - 1) {
                    inputs[index + 1].focus();
                }
                // Auto-submit when all filled
                if (index === inputs.length - 1 && this.value.length === 1) {
                    const allFilled = Array.from(inputs).every(inp => inp.value.length === 1);
                    if (allFilled) {
                        verify2FA();
                    }
                }
            });

            // Handle backspace
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0 && index > 0) {
                    inputs[index - 1].focus();
                }
            });

            // Only allow numbers
            input.addEventListener('keypress', function(e) {
                if (!/[0-9]/.test(e.key)) {
                    e.preventDefault();
                }
            });
        });
    }

    function verify2FA() {
        const inputs = document.querySelectorAll('.twofa-input');
        const otpCode = Array.from(inputs).map(inp => inp.value).join('');
        const errorDiv = document.getElementById('twoFAError');
        const verifyBtn = document.getElementById('verify2FABtn');
        const verifyText = document.getElementById('verify2FAText');
        const verifySpinner = document.getElementById('verify2FASpinner');

        if (otpCode.length !== 6) {
            errorDiv.textContent = 'Please enter all 6 digits';
            errorDiv.style.display = 'block';
            return;
        }

        // Show loading state
        errorDiv.style.display = 'none';
        verifyBtn.disabled = true;
        verifyText.style.display = 'none';
        verifySpinner.style.display = 'inline-block';

        const loginUrl = buildActionURL('verify2FA');

        fetch(loginUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=verify2FA&2fa_session_token=${encodeURIComponent(window.twoFASessionToken)}&otp_code=${encodeURIComponent(otpCode)}`
        })
        .then(response => response.json())
        .then(data => {
            console.log('2FA verification response:', data);

            if (data.status && data.content && data.content.success) {
                console.log('2FA verification successful');

                // Store tokens
                if (data.content.access) {
                    localStorage.setItem('govalid_access_token', data.content.access);
                    if (typeof goValidAuth !== 'undefined') {
                        goValidAuth.accessToken = data.content.access;
                    }
                }

                if (data.content.refresh) {
                    localStorage.setItem('govalidRefreshToken', data.content.refresh);
                }

                // Store user session
                if (data.content.user) {
                    goValidAuth.isAuthenticated = true;
                    goValidAuth.userInfo = data.content.user;
                    goValidAuth.sessionStart = Date.now();

                    localStorage.setItem('govalidAuthSession', JSON.stringify({
                        sessionStart: goValidAuth.sessionStart,
                        userInfo: data.content.user,
                        subscription: null
                    }));
                }

                hideLoginModal();
                checkAuthStatus(true);
            } else {
                // Verification failed
                const message = data.content?.message || 'Invalid or expired code';
                errorDiv.textContent = message;
                errorDiv.style.display = 'block';

                // Clear inputs
                inputs.forEach(inp => inp.value = '');
                inputs[0].focus();
            }
        })
        .catch(error => {
            console.error('2FA verification error:', error);
            errorDiv.textContent = 'Network error. Please try again.';
            errorDiv.style.display = 'block';
        })
        .finally(() => {
            verifyBtn.disabled = false;
            verifyText.style.display = 'inline';
            verifySpinner.style.display = 'none';
        });
    }

    function cancel2FA() {
        // Reset 2FA form
        document.getElementById('twoFASection').style.display = 'none';
        document.querySelectorAll('.twofa-input').forEach(inp => inp.value = '');

        // Show login form again
        document.getElementById('govalid_username').parentElement.parentElement.style.display = '';
        document.getElementById('govalid_password').parentElement.parentElement.style.display = '';
        document.querySelector('.form-options').style.display = '';
        document.getElementById('loginSubmitBtn').style.display = '';
        document.querySelector('.divider').style.display = '';
        document.querySelector('.btn-register').style.display = '';

        // Clear any errors
        document.getElementById('twoFAError').style.display = 'none';

        // Reset button state
        const submitBtn = document.getElementById('loginSubmitBtn');
        const loginText = submitBtn.querySelector('.login-text');
        const spinner = submitBtn.querySelector('.login-spinner');
        submitBtn.disabled = false;
        loginText.style.display = 'inline';
        spinner.style.display = 'none';
    }

    // Email OTP Functions (Risk-Based Authentication)
    function setupEmailOTPInputHandlers() {
        const inputs = document.querySelectorAll('.email-otp-input');

        inputs.forEach((input, index) => {
            // Auto-focus next input
            input.addEventListener('input', function(e) {
                if (this.value.length === 1 && index < inputs.length - 1) {
                    inputs[index + 1].focus();
                }
                // Auto-submit when all filled
                if (index === inputs.length - 1 && this.value.length === 1) {
                    const allFilled = Array.from(inputs).every(inp => inp.value.length === 1);
                    if (allFilled) {
                        verifyEmailOTP();
                    }
                }
            });

            // Handle backspace
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0 && index > 0) {
                    inputs[index - 1].focus();
                }
            });

            // Only allow numbers
            input.addEventListener('keypress', function(e) {
                if (!/[0-9]/.test(e.key)) {
                    e.preventDefault();
                }
            });
        });
    }

    function verifyEmailOTP() {
        const inputs = document.querySelectorAll('.email-otp-input');
        const otpCode = Array.from(inputs).map(inp => inp.value).join('');
        const errorDiv = document.getElementById('emailOTPError');
        const verifyBtn = document.getElementById('verifyEmailOTPBtn');
        const verifyText = document.getElementById('verifyEmailOTPText');
        const verifySpinner = document.getElementById('verifyEmailOTPSpinner');

        if (otpCode.length !== 6) {
            errorDiv.textContent = 'Please enter all 6 digits';
            errorDiv.style.display = 'block';
            return;
        }

        // Show loading state
        errorDiv.style.display = 'none';
        verifyBtn.disabled = true;
        verifyText.style.display = 'none';
        verifySpinner.style.display = 'inline-block';

        const verifyUrl = buildActionURL('verifyEmailOTP');

        fetch(verifyUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=verifyEmailOTP&otp_session_token=${encodeURIComponent(window.otpSessionToken)}&otp_code=${encodeURIComponent(otpCode)}`
        })
        .then(response => response.json())
        .then(data => {
            console.log('Email OTP verification response:', data);

            if (data.status && data.content && data.content.success) {
                console.log('Email OTP verification successful');

                // Store tokens
                if (data.content.access) {
                    localStorage.setItem('govalid_access_token', data.content.access);
                    if (typeof goValidAuth !== 'undefined') {
                        goValidAuth.accessToken = data.content.access;
                    }
                }

                if (data.content.refresh) {
                    localStorage.setItem('govalidRefreshToken', data.content.refresh);
                }

                // Store user session
                if (data.content.user) {
                    goValidAuth.isAuthenticated = true;
                    goValidAuth.userInfo = data.content.user;
                    goValidAuth.sessionStart = Date.now();

                    localStorage.setItem('govalidAuthSession', JSON.stringify({
                        sessionStart: goValidAuth.sessionStart,
                        userInfo: data.content.user,
                        subscription: null
                    }));
                }

                hideLoginModal();
                checkAuthStatus(true);
            } else {
                // Verification failed
                const message = data.content?.message || data.content?.error || 'Invalid or expired code';
                errorDiv.textContent = message;
                errorDiv.style.display = 'block';

                // Clear inputs
                inputs.forEach(inp => inp.value = '');
                inputs[0].focus();
            }
        })
        .catch(error => {
            console.error('Email OTP verification error:', error);
            errorDiv.textContent = 'Connection error. Please try again.';
            errorDiv.style.display = 'block';
        })
        .finally(() => {
            verifyBtn.disabled = false;
            verifyText.style.display = 'inline';
            verifySpinner.style.display = 'none';
        });
    }

    function cancelEmailOTP() {
        // Reset email OTP form
        document.getElementById('emailOTPSection').style.display = 'none';
        document.querySelectorAll('.email-otp-input').forEach(inp => inp.value = '');

        // Show login form again
        document.getElementById('govalid_username').parentElement.parentElement.style.display = '';
        document.getElementById('govalid_password').parentElement.parentElement.style.display = '';
        document.querySelector('.form-options').style.display = '';
        document.getElementById('loginSubmitBtn').style.display = '';
        document.querySelector('.divider').style.display = '';
        document.querySelector('.btn-register').style.display = '';

        // Clear any errors
        document.getElementById('emailOTPError').style.display = 'none';

        // Reset button state
        const submitBtn = document.getElementById('loginSubmitBtn');
        const loginText = submitBtn.querySelector('.login-text');
        const spinner = submitBtn.querySelector('.login-spinner');
        submitBtn.disabled = false;
        loginText.style.display = 'inline';
        spinner.style.display = 'none';
    }

    function checkRememberedLogin() {
        try {
            const rememberedData = localStorage.getItem('govalidRememberMe');
            const refreshToken = localStorage.getItem('govalidRefreshToken');
            
            if (rememberedData) {
                const data = JSON.parse(atob(rememberedData));
                
                // Check if remember data is not too old (30 days)
                const thirtyDaysAgo = Date.now() - (30 * 24 * 60 * 60 * 1000);
                if (data.timestamp && data.timestamp > thirtyDaysAgo) {
                    // Pre-fill username and check remember box
                    const usernameField = document.getElementById('govalid_username');
                    const rememberCheckbox = document.getElementById('govalid_remember');
                    
                    if (usernameField && data.username) {
                        usernameField.value = data.username;
                    }
                    if (rememberCheckbox) {
                        rememberCheckbox.checked = true;
                    }
                    
                    // Try to auto-login with refresh token if available
                    if (refreshToken && !goValidAuth.isAuthenticated) {
                        attemptAutoLogin(refreshToken);
                    }
                } else {
                    // Clear old remember data
                    localStorage.removeItem('govalidRememberMe');
                    localStorage.removeItem('govalidRefreshToken');
                }
            }
        } catch (e) {
            console.log('Error checking remembered login:', e);
            // Clear corrupted data
            localStorage.removeItem('govalidRememberMe');
            localStorage.removeItem('govalidRefreshToken');
        }
    }
    
    function attemptAutoLogin(refreshToken) {
        console.log('Attempting auto-login with refresh token...');
        
        // Use clean AJAX URL
        const url = buildActionURL('refreshAuth');
        
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `refresh_token=${encodeURIComponent(refreshToken)}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.status && data.content && data.content.authenticated) {
                console.log('Auto-login successful');
                goValidAuth.sessionStart = Date.now();
                checkAuthStatus(true);
            } else {
                console.log('Auto-login failed, clearing stored tokens');
                localStorage.removeItem('govalidRefreshToken');
            }
        })
        .catch(error => {
            console.error('Auto-login error:', error);
            localStorage.removeItem('govalidRefreshToken');
        });
    }
    
    function performLogout() {
        // Clear stored session and remember me data
        try {
            localStorage.removeItem('govalidAuthSession');
            localStorage.removeItem('govalidRememberMe');
            localStorage.removeItem('govalidRefreshToken');
        } catch (e) {
            console.log('Failed to clear stored session:', e);
        }
        
        // Use clean AJAX URL
        const url = buildActionURL('logout');
        
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: ''
        })
        .then(response => response.json())
        .then(data => {
            // Clear auth state
            goValidAuth.isAuthenticated = false;
            goValidAuth.userInfo = null;
            goValidAuth.subscription = null;
            goValidAuth.sessionStart = 0;
            goValidAuth.lastCheck = 0;
            
            updateAuthStatus({ authenticated: false });
        })
        .catch(error => {
            console.error('Logout failed:', error);
            // Still update status in case of error
            updateAuthStatus({ authenticated: false });
        });
    }

    function setupAuthentication() {
        // Try to restore session from localStorage first
        try {
            const storedSession = localStorage.getItem('govalidAuthSession');
            if (storedSession) {
                const session = JSON.parse(storedSession);
                const now = Date.now();
                // Check if session is still valid
                if (session.sessionStart && (now - session.sessionStart) < goValidAuth.sessionDuration) {
                    console.log('Restoring valid session from localStorage');
                    goValidAuth.isAuthenticated = true;
                    goValidAuth.userInfo = session.userInfo;
                    goValidAuth.subscription = session.subscription;
                    goValidAuth.sessionStart = session.sessionStart;
                    goValidAuth.lastCheck = now;
                    
                    // Update UI immediately
                    updateAuthStatus({
                        authenticated: true,
                        user: session.userInfo,
                        userInfo: session.userInfo,
                        subscription: session.subscription
                    });
                    
                    // Don't show login modal since we have a session
                    const loginModal = document.getElementById('govalidLoginModal');
                    if (loginModal) {
                        loginModal.style.setProperty('display', 'none', 'important');
                    }
                    
                    // Also hide the auth overlay
                    const authOverlay = document.getElementById('authOverlay');
                    if (authOverlay) {
                        authOverlay.style.display = 'none';
                    }
                    
                    // Verify with server in the background after a short delay
                    setTimeout(() => {
                        if (goValidAuth.isAuthenticated) {
                            console.log('Verifying restored session with server...');
                            checkAuthStatus(true);
                        }
                    }, 1000);
                } else {
                    console.log('Stored session expired (age: ' + ((now - session.sessionStart) / 1000) + ' seconds), removing');
                    localStorage.removeItem('govalidAuthSession');
                    // Try to refresh using refresh token if available
                    tryRefreshToken();
                }
            }
        } catch (e) {
            console.log('Failed to restore session:', e);
        }
        
        // Auth status click handler
        const authStatus = document.getElementById('govalidAuthStatus');
        if (authStatus) {
            authStatus.addEventListener('click', function() {
                if (goValidAuth.isAuthenticated) {
                    performLogout();
                } else {
                    showLoginModal();
                }
            });
        }
        
        // Login form submission
        const loginForm = document.getElementById('govalidLoginForm');
        if (loginForm) {
            loginForm.addEventListener('submit', function(e) {
                e.preventDefault();
                performLogin();
            });
        }
        
        // Modal close handlers
        const closeBtn = document.querySelector('#govalidLoginModal .close');
        if (closeBtn) {
            closeBtn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                hideLoginModal();
            });
        }
        
        // Click outside modal to close
        const modal = document.getElementById('govalidLoginModal');
        if (modal) {
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    hideLoginModal();
                }
            });
        }
        
        // Password toggle
        const passwordToggle = document.getElementById('togglePassword');
        if (passwordToggle) {
            passwordToggle.addEventListener('click', function() {
                const passwordField = document.getElementById('govalid_password');
                const eyeIcon = this.querySelector('i');
                
                if (passwordField.type === 'password') {
                    passwordField.type = 'text';
                    eyeIcon.className = 'fa fa-eye-slash';
                } else {
                    passwordField.type = 'password';
                    eyeIcon.className = 'fa fa-eye';
                }
            });
        }
        
        // Delay initial auth check to allow session restoration
        setTimeout(() => {
            // Clear initializing flag
            goValidAuth.initializing = false;
            
            // If not already authenticated through session restoration
            if (!goValidAuth.isAuthenticated) {
                // Check for remembered credentials
                checkRememberedLogin();
                
                // If still not authenticated, check with server
                if (!goValidAuth.isAuthenticated) {
                    checkAuthStatus();
                }
            } else {
                // We already have a session, just update UI
                console.log('Session already restored, updating UI only');
            }
            
            // Update UI state after initialization
            updateUIAuthState();
            // Update identifier format visibility based on institution membership
            if (typeof window.updateIdentifierFormatVisibility === 'function') {
                window.updateIdentifierFormatVisibility();
            }
        }, 500); // Small delay to ensure setupAuthentication completes first
    }
    
