/**
 * GoValid Smart Verify Widget
 * Floating verification widget for OJS public pages
 *
 * Copyright (c) 2025 GoValid
 */

(function () {
    'use strict';

    // API Configuration
    const API_BASE = 'https://my.govalid.org';
    const QR_LOOKUP_API = API_BASE + '/qr_codes/api/v1/qr/lookup/';
    const SCAN_VERIFY_API = API_BASE + '/api/v1/scan/verify/';
    const OJS_DECODE_QR_API = API_BASE + '/api/v1/ojs/decode-qr-image/';

    // DOM Elements
    let widget, toggleBtn, panel, closeBtn;
    let qrInput, searchBtn;
    let dropzone, fileInput, preview, previewImg, clearPreviewBtn;
    let resultsDiv, loadingDiv, resultContent;
    let tabs;

    // Camera variables
    let video, cameraToggleBtn, cameraSwitchBtn, cameraStatus;
    let canvasElement, canvasCtx;
    let isScanning = false;
    let animationFrameId;
    let localStream = null;
    let facingMode = "environment";

    /**
     * Initialize the widget
     */
    function init() {
        widget = document.getElementById('govalid-smart-verify-widget');
        if (!widget) return;

        // Get elements
        toggleBtn = document.getElementById('govalid-sv-toggle');
        panel = document.getElementById('govalid-sv-panel');
        closeBtn = document.getElementById('govalid-sv-close');

        qrInput = document.getElementById('govalid-sv-qr-input');
        searchBtn = document.getElementById('govalid-sv-search-btn');

        dropzone = document.getElementById('govalid-sv-dropzone');
        fileInput = document.getElementById('govalid-sv-file-input');
        preview = document.getElementById('govalid-sv-preview');
        previewImg = document.getElementById('govalid-sv-preview-img');
        clearPreviewBtn = document.getElementById('govalid-sv-clear-preview');

        resultsDiv = document.getElementById('govalid-sv-results');
        loadingDiv = document.getElementById('govalid-sv-loading');
        resultContent = document.getElementById('govalid-sv-result-content');

        tabs = document.querySelectorAll('.govalid-sv-tab');

        // Camera elements
        video = document.getElementById('govalid-sv-video');
        cameraToggleBtn = document.getElementById('govalid-sv-camera-toggle');
        cameraSwitchBtn = document.getElementById('govalid-sv-camera-switch');
        cameraStatus = document.getElementById('govalid-sv-camera-status');

        canvasElement = document.createElement('canvas');
        canvasCtx = canvasElement.getContext('2d');

        // Bind events
        bindEvents();

        // Show widget
        widget.style.display = 'block';

        console.log('GoValid Smart Verify Widget initialized');
    }

    /**
     * Bind event listeners
     */
    function bindEvents() {
        // Toggle panel
        toggleBtn.addEventListener('click', togglePanel);
        closeBtn.addEventListener('click', closePanel);

        // Tab switching
        tabs.forEach(tab => {
            tab.addEventListener('click', () => switchTab(tab.dataset.tab));
        });

        // Camera events
        if (cameraToggleBtn) cameraToggleBtn.addEventListener('click', toggleCamera);
        if (cameraSwitchBtn) cameraSwitchBtn.addEventListener('click', switchCamera);

        // QR ID search
        searchBtn.addEventListener('click', searchByQRId);
        qrInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') searchByQRId();
        });

        // File upload
        dropzone.addEventListener('click', () => fileInput.click());
        fileInput.addEventListener('change', handleFileSelect);

        // Drag and drop
        dropzone.addEventListener('dragover', handleDragOver);
        dropzone.addEventListener('dragleave', handleDragLeave);
        dropzone.addEventListener('drop', handleDrop);

        // Clear preview
        clearPreviewBtn.addEventListener('click', clearPreview);

        // Close panel when clicking outside
        document.addEventListener('click', (e) => {
            if (panel.classList.contains('open') &&
                !panel.contains(e.target) &&
                !toggleBtn.contains(e.target)) {
                closePanel();
            }
        });
    }

    /**
     * Toggle panel visibility
     */
    function togglePanel() {
        panel.classList.toggle('open');
        toggleBtn.classList.toggle('active');
    }

    /**
     * Close panel
     */
    function closePanel() {
        panel.classList.remove('open');
        toggleBtn.classList.remove('active');
    }

    /**
     * Switch between tabs
     */
    function switchTab(tabId) {
        // Update tab buttons
        tabs.forEach(tab => {
            tab.classList.toggle('active', tab.dataset.tab === tabId);
        });

        // Update tab content
        document.querySelectorAll('.govalid-sv-tab-content').forEach(content => {
            content.classList.toggle('active', content.id === 'govalid-sv-tab-' + tabId);
        });

        // Clear results when switching tabs
        hideResults();

        // Stop camera if leaving camera tab
        if (tabId !== 'camera') {
            stopCamera();
        }
    }

    /**
     * Search by QR ID
     */
    async function searchByQRId() {
        const qrId = qrInput.value.trim();
        if (!qrId) {
            showError('Please enter a QR ID');
            return;
        }

        showLoading();

        try {
            const response = await fetch(QR_LOOKUP_API + '?formatted_uuid=' + encodeURIComponent(qrId));
            const data = await response.json();

            if (data.success && data.data) {
                showQRDetails(data.data);
            } else {
                showError(data.message || 'QR code not found');
            }
        } catch (error) {
            console.error('Search error:', error);
            showError('Failed to search. Please try again.');
        }
    }

    /**
     * Handle file selection
     */
    function handleFileSelect(e) {
        const file = e.target.files[0];
        if (file) {
            processFile(file);
        }
    }

    /**
     * Handle drag over
     */
    function handleDragOver(e) {
        e.preventDefault();
        e.stopPropagation();
        dropzone.classList.add('dragover');
    }

    /**
     * Handle drag leave
     */
    function handleDragLeave(e) {
        e.preventDefault();
        e.stopPropagation();
        dropzone.classList.remove('dragover');
    }

    /**
     * Handle drop
     */
    function handleDrop(e) {
        e.preventDefault();
        e.stopPropagation();
        dropzone.classList.remove('dragover');

        const file = e.dataTransfer.files[0];
        if (file && file.type.startsWith('image/')) {
            processFile(file);
        } else {
            showError('Please drop an image file');
        }
    }

    /**
     * Process uploaded file
     */
    function processFile(file) {
        const reader = new FileReader();
        reader.onload = function (e) {
            const imageSrc = e.target.result;

            // Show preview
            previewImg.src = imageSrc;
            preview.style.display = 'block';
            dropzone.style.display = 'none';

            // Scan for QR code
            scanImage(imageSrc);
        };
        reader.readAsDataURL(file);
    }

    /**
     * Scan image for QR code
     */
    function scanImage(imageSrc) {
        showLoading();

        const img = new Image();
        img.onload = function () {
            // Create canvas to extract image data
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            canvas.width = img.width;
            canvas.height = img.height;
            ctx.drawImage(img, 0, 0);

            const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);

            // Try to detect QR code using client-side jsQR
            let code = null;

            // First attempt without inversion
            if (typeof jsQR !== 'undefined') {
                code = jsQR(imageData.data, imageData.width, imageData.height, {
                    inversionAttempts: 'dontInvert'
                });

                // Second attempt with inversion (for stamped/negative QR codes)
                if (!code) {
                    code = jsQR(imageData.data, imageData.width, imageData.height, {
                        inversionAttempts: 'attemptBoth'
                    });
                }
            }

            if (code && code.data) {
                console.log('jsQR decoded:', code.data);
                // Check if it's a GoValid QR code
                if (isGoValidQR(code.data)) {
                    verifyGoValidQR(code.data);
                } else {
                    showError('This is not a GoValid QR code');
                }
            } else {
                // Client-side decoding failed, try server-side API
                console.log('jsQR failed, trying server-side decode...');
                decodeImageServerSide(imageSrc);
            }
        };
        img.onerror = function () {
            showError('Failed to load image');
        };
        img.src = imageSrc;
    }

    /**
     * Decode image using server-side API (fallback)
     * Uses public OJS decode endpoint (rate limited, no auth required)
     */
    async function decodeImageServerSide(imageSrc) {
        try {
            // Extract base64 data (remove data URL prefix if present)
            let base64Data = imageSrc;
            if (imageSrc.includes(',')) {
                base64Data = imageSrc.split(',')[1];
            }

            const response = await fetch(OJS_DECODE_QR_API, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    image_base64: base64Data
                })
            });

            const result = await response.json();

            // Handle OJS API response format (wrapped in content)
            const data = result.content || result;

            if (data.success && data.qr_data) {
                console.log('Server decoded:', data.qr_data);

                if (data.is_govalid_qr) {
                    // Check if it has verify-scan params (t=, s=, v=)
                    const hasVerifyScanParams = data.qr_data.includes('?t=') || data.qr_data.includes('&t=');

                    if (hasVerifyScanParams && data.token) {
                        // Use scan/verify API for verify-scan URLs
                        const payload = { token: data.token };
                        if (data.signature) payload.signature = data.signature;
                        if (data.version) payload.version = data.version;

                        const verifyResponse = await fetch(SCAN_VERIFY_API, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json'
                            },
                            body: JSON.stringify(payload)
                        });

                        const verifyResult = await verifyResponse.json();

                        if (verifyResult.success && verifyResult.is_verified) {
                            const qrData = {
                                name: verifyResult.qr_code?.name,
                                formatted_uuid: verifyResult.qr_code?.uuid,
                                template_type: verifyResult.qr_code?.template_type,
                                scan_count: verifyResult.qr_code?.scan_count,
                                is_active: verifyResult.qr_code?.is_active,
                                verification_url: data.verification_url || data.qr_data,
                                creator: verifyResult.creator,
                                data: verifyResult.data
                            };
                            showQRDetails(qrData, data.verification_url || data.qr_data);
                        } else {
                            showError(verifyResult.message || verifyResult.error || 'QR code verification failed');
                        }
                    } else {
                        // For /v/ signed URLs or other formats, show basic verification with link
                        // The user can click through to verify on GoValid directly
                        showBasicVerification(data.verification_url || data.qr_data);
                    }
                } else {
                    showError('This is not a GoValid QR code');
                }
            } else {
                showError(data.error || data.message || 'No QR code detected in the image');
            }
        } catch (error) {
            console.error('Server-side decode error:', error);
            showError('Failed to decode QR code. Please try again.');
        }
    }

    /**
     * Check if URL is a GoValid verification URL
     */
    function isGoValidQR(data) {
        if (!data) return false;

        // Check for verify-scan URL pattern (primary format)
        // Example: https://my.govalid.org/qr_codes/verify-scan/?t=TOKEN&s=SIGNATURE&v=VERSION
        if (data.includes('govalid.org/qr_codes/verify-scan/')) {
            return true;
        }

        // Check for signed URL pattern (/v/) - used by most QR codes
        // Example: https://govalid.org/v/SIGNED_TOKEN
        if (data.includes('govalid.org/v/')) {
            return true;
        }

        // Check for /qr/ pattern
        if (data.includes('govalid.org/qr/')) {
            return true;
        }

        // Check for /verify/ pattern
        if (data.includes('govalid.org/verify/')) {
            return true;
        }

        // Check for ID format (Simple check: alphanumeric with hyphens)
        // Example: 1ACE-DO-abc123
        const idRegex = /^[A-Z0-9]{4}-?[A-Z0-9]{2}-?[a-zA-Z0-9]{8}$/i;
        if (idRegex.test(data.trim())) return true;

        return false;
    }

    /**
     * Verify GoValid QR code by URL
     */
    async function verifyGoValidQR(data) {
        showLoading();

        try {
            let verifyUrl = data.includes('http') ? data : '';

            // Check for verify-scan URL pattern (primary format)
            // Example: https://my.govalid.org/qr_codes/verify-scan/?t=TOKEN&s=SIGNATURE&v=VERSION
            const verifyScanMatch = data.match(/[?&]t=([A-Za-z0-9_-]+)/);
            const signatureMatch = data.match(/[?&]s=([A-Za-z0-9_-]+)/);
            const versionMatch = data.match(/[?&]v=([A-Za-z0-9]+)/);

            if (verifyScanMatch && verifyScanMatch[1]) {
                // Use the scan/verify API for token-based verification
                const token = verifyScanMatch[1];
                const signature = signatureMatch ? signatureMatch[1] : null;
                const version = versionMatch ? versionMatch[1] : 'V';

                console.log('Verifying QR with token:', token);

                const payload = { token: token };
                if (signature) payload.signature = signature;
                if (version) payload.version = version;

                const response = await fetch(SCAN_VERIFY_API, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(payload)
                });

                const result = await response.json();

                if (result.success && result.is_verified) {
                    // Transform API response to match expected format
                    const qrData = {
                        name: result.qr_code?.name,
                        formatted_uuid: result.qr_code?.uuid,
                        template_type: result.qr_code?.template_type,
                        scan_count: result.qr_code?.scan_count,
                        is_active: result.qr_code?.is_active,
                        verification_url: verifyUrl || data,
                        creator: result.creator,
                        data: result.data
                    };
                    showQRDetails(qrData, verifyUrl || data);
                } else {
                    showError(result.message || result.error || 'QR code verification failed');
                }
                return;
            }

            // Check for /v/ signed URL pattern (most common QR code format)
            // Example: https://govalid.org/v/SIGNED_TOKEN
            const signedUrlMatch = data.match(/govalid\.org\/v\/([A-Za-z0-9_.-]+)/);
            if (signedUrlMatch) {
                // For signed URLs, show basic verification with link to GoValid
                // The verification happens on the GoValid website directly
                console.log('Found signed URL, showing basic verification');
                showBasicVerification(data);
                return;
            }

            // Fallback: Try to extract QR ID using regex (for formatted UUID)
            // Matches: 1ACE-DO-abc123 or 1ACEDOabc123
            const qrIdPattern = /([A-Z0-9]{1,2}[A-Z]{3}-?[A-Z]{2}-?[a-f0-9]{8})/i;
            const match = data.match(qrIdPattern);
            let qrId = null;

            if (match && match[1]) {
                qrId = match[1];
                if (!verifyUrl) verifyUrl = API_BASE + '/v/' + qrId;
            } else if (isGoValidQR(data) && data.includes('http')) {
                // Fallback to URL parsing if regex fails but domain matches
                try {
                    const urlObj = new URL(data);
                    const pathParts = urlObj.pathname.split('/').filter(p => p.length > 0);
                    qrId = pathParts[pathParts.length - 1];
                    verifyUrl = data;
                } catch (e) { /* ignore */ }
            }

            if (qrId) {
                // Call lookup API for ID-based verification
                const response = await fetch(QR_LOOKUP_API + '?formatted_uuid=' + encodeURIComponent(qrId));
                const result = await response.json();

                if (result.success && result.data) {
                    showQRDetails(result.data, verifyUrl);
                } else if (verifyUrl) {
                    showBasicVerification(verifyUrl);
                } else {
                    showError(result.message || 'QR code not found');
                }
            } else if (verifyUrl) {
                showBasicVerification(verifyUrl);
            } else {
                showError('Unable to parse QR code data');
            }
        } catch (error) {
            console.error('Verification error:', error);
            showError('Failed to verify. Please try again.');
        }
    }

    // Camera Functions

    function toggleCamera() {
        if (isScanning) {
            stopCamera();
        } else {
            startCamera();
        }
    }

    function switchCamera() {
        if (isScanning) {
            stopCamera();
            facingMode = facingMode === "user" ? "environment" : "user";
            startCamera();
        }
    }

    async function startCamera() {
        if (isScanning) return;

        try {
            localStream = await navigator.mediaDevices.getUserMedia({
                video: { facingMode: facingMode }
            });

            video.srcObject = localStream;
            video.setAttribute("playsinline", true);
            video.play();

            isScanning = true;
            cameraToggleBtn.textContent = 'Stop Camera';
            cameraSwitchBtn.style.display = 'inline-flex';
            cameraStatus.textContent = 'Scanning...';

            // Start scanning loop
            requestAnimationFrame(tick);

        } catch (err) {
            console.error("Camera error:", err);
            cameraStatus.textContent = 'Error accessing camera: ' + err.message;
            showError('Unable to access camera. Please allow permission.');
        }
    }

    function stopCamera() {
        if (!isScanning) return;

        isScanning = false;

        if (localStream) {
            localStream.getTracks().forEach(track => track.stop());
            localStream = null;
        }

        if (video) {
            video.srcObject = null;
        }

        cameraToggleBtn.textContent = 'Start Camera';
        cameraSwitchBtn.style.display = 'none';
        cameraStatus.textContent = 'Camera stopped';
    }

    function tick() {
        if (!isScanning) return;

        if (video.readyState === video.HAVE_ENOUGH_DATA) {
            canvasElement.height = video.videoHeight;
            canvasElement.width = video.videoWidth;
            canvasCtx.drawImage(video, 0, 0, canvasElement.width, canvasElement.height);

            const imageData = canvasCtx.getImageData(0, 0, canvasElement.width, canvasElement.height);

            let code = null;
            if (typeof jsQR !== 'undefined') {
                code = jsQR(imageData.data, imageData.width, imageData.height, {
                    inversionAttempts: "dontInvert",
                });

                if (!code) {
                    code = jsQR(imageData.data, imageData.width, imageData.height, {
                        inversionAttempts: "attemptBoth",
                    });
                }
            }

            if (code && code.data) {
                // Found a QR code
                console.log("Found QR code", code.data);

                // Draw a box (optional, but good for feedback)
                // drawBox(code.location);

                if (isGoValidQR(code.data)) {
                    // Success!
                    stopCamera(); // Stop scanning once found
                    verifyGoValidQR(code.data);
                    return; // Stop loop
                }
            }
        }

        if (isScanning) {
            requestAnimationFrame(tick);
        }
    }

    /**
     * Clear preview and reset
     */
    function clearPreview() {
        preview.style.display = 'none';
        dropzone.style.display = 'flex';
        previewImg.src = '';
        fileInput.value = '';
        hideResults();
    }

    /**
     * Show loading state
     */
    function showLoading() {
        resultsDiv.style.display = 'block';
        loadingDiv.style.display = 'flex';
        resultContent.innerHTML = '';
    }

    /**
     * Hide results
     */
    function hideResults() {
        resultsDiv.style.display = 'none';
        loadingDiv.style.display = 'none';
        resultContent.innerHTML = '';
    }

    /**
     * Show error message
     */
    function showError(message) {
        resultsDiv.style.display = 'block';
        loadingDiv.style.display = 'none';
        resultContent.innerHTML = `
            <div class="govalid-sv-error">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="15" y1="9" x2="9" y2="15"></line>
                    <line x1="9" y1="9" x2="15" y2="15"></line>
                </svg>
                <span>${escapeHtml(message)}</span>
            </div>
        `;
    }

    /**
     * Show QR code details
     */
    function showQRDetails(data, verifyUrl) {
        resultsDiv.style.display = 'block';
        loadingDiv.style.display = 'none';

        const isActive = data.is_active !== false;
        const statusClass = isActive ? 'success' : 'warning';
        const statusText = isActive ? 'Valid' : 'Inactive';
        const statusIcon = isActive
            ? '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline>'
            : '<circle cx="12" cy="12" r="10"></circle><line x1="12" y1="8" x2="12" y2="12"></line><line x1="12" y1="16" x2="12.01" y2="16"></line>';

        const url = verifyUrl || data.verification_url || '';

        // Extract creator info
        const creator = data.creator || {};
        const generatedBy = creator.full_name || creator.username || null;
        const institutionName = creator.institution_name || null;
        const isVerified = creator.is_verified || creator.staff_verified || false;

        // Build verified badge HTML
        const verifiedBadge = isVerified
            ? '<svg width="14" height="14" viewBox="0 0 24 24" fill="#1da1f2" style="margin-left: 4px; vertical-align: middle;"><path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>'
            : '';

        resultContent.innerHTML = `
            <div class="govalid-sv-result govalid-sv-result-${statusClass}">
                <div class="govalid-sv-result-header">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        ${statusIcon}
                    </svg>
                    <span>${statusText} GoValid QR Code</span>
                </div>
                <div class="govalid-sv-result-body">
                    ${data.name ? `<div class="govalid-sv-field"><label>Name:</label><span>${escapeHtml(data.name)}</span></div>` : ''}
                    ${data.formatted_uuid ? `<div class="govalid-sv-field"><label>QR ID:</label><span class="govalid-sv-mono">${escapeHtml(data.formatted_uuid)}</span></div>` : ''}
                    ${data.template_type ? `<div class="govalid-sv-field"><label>Type:</label><span>${escapeHtml(data.template_type)}</span></div>` : ''}
                    ${generatedBy ? `<div class="govalid-sv-field"><label>Generated by:</label><span>${escapeHtml(generatedBy)}${verifiedBadge}</span></div>` : ''}
                    ${institutionName ? `<div class="govalid-sv-field"><label>Institution:</label><span>${escapeHtml(institutionName)}</span></div>` : ''}
                    ${data.scan_count !== undefined ? `<div class="govalid-sv-field"><label>Scans:</label><span>${data.scan_count}</span></div>` : ''}
                </div>
                ${url ? `
                <div class="govalid-sv-result-footer">
                    <a href="${escapeHtml(url)}" target="_blank" class="govalid-sv-btn-primary">
                        View Full Details
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>
                            <polyline points="15 3 21 3 21 9"></polyline>
                            <line x1="10" y1="14" x2="21" y2="3"></line>
                        </svg>
                    </a>
                </div>
                ` : ''}
            </div>
        `;
    }

    /**
     * Show basic verification (when detailed info is not available)
     */
    function showBasicVerification(url) {
        resultsDiv.style.display = 'block';
        loadingDiv.style.display = 'none';

        resultContent.innerHTML = `
            <div class="govalid-sv-result govalid-sv-result-success">
                <div class="govalid-sv-result-header">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
                        <polyline points="22 4 12 14.01 9 11.01"></polyline>
                    </svg>
                    <span>GoValid QR Code Detected</span>
                </div>
                <div class="govalid-sv-result-body">
                    <p style="margin: 0; color: #666; font-size: 13px;">Click below to view full verification details on GoValid.</p>
                </div>
                <div class="govalid-sv-result-footer">
                    <a href="${escapeHtml(url)}" target="_blank" class="govalid-sv-btn-primary">
                        Verify on GoValid
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>
                            <polyline points="15 3 21 3 21 9"></polyline>
                            <line x1="10" y1="14" x2="21" y2="3"></line>
                        </svg>
                    </a>
                </div>
            </div>
        `;
    }

    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
