// ==========================================
// TEMPLATE GALLERY FUNCTIONS
// ==========================================

// Show Template Gallery
function showTemplateGallery() {
    const modal = document.getElementById('templateGalleryModal');
    if (modal) {
        modal.style.display = 'block';
        // Load default templates by default
        loadDefaultTemplates();
    }
}

// Close Template Gallery
function closeTemplateGallery() {
    const modal = document.getElementById('templateGalleryModal');
    if (modal) {
        modal.style.display = 'none';
    }
}

// Switch between Default and My Templates tabs
function switchGalleryTab(tab) {
    const defaultGrid = document.getElementById('defaultTemplatesGrid');
    const myList = document.getElementById('myTemplatesList');
    const buttons = document.querySelectorAll('.gallery-tab-btn');

    // Update button styles
    buttons.forEach(btn => {
        if (btn.dataset.tab === tab) {
            btn.classList.add('active');
            btn.style.background = '#667eea';
            btn.style.color = 'white';
        } else {
            btn.classList.remove('active');
            btn.style.background = '#e3e8ee';
            btn.style.color = '#555';
        }
    });

    // Show/hide content
    if (tab === 'default') {
        defaultGrid.style.display = 'grid';
        myList.style.display = 'none';
        loadDefaultTemplates();
    } else {
        defaultGrid.style.display = 'none';
        myList.style.display = 'block';
        loadMyTemplatesForGallery();
    }
}

// Load Default Templates from backend
async function loadDefaultTemplates() {
    const grid = document.getElementById('defaultTemplatesGrid');
    if (!grid) return;

    // Show loading state
    grid.innerHTML = '<div style="text-align: center; padding: 40px; grid-column: 1/-1;"><i class="fa fa-spinner fa-spin" style="font-size: 48px; color: #667eea;"></i><p style="margin-top: 20px; color: #666;">Loading templates...</p></div>';

    try {
        const response = await fetch(buildActionURL('listDefaultTemplates'), {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            },
            credentials: 'same-origin'
        });

        const data = await response.json();
        console.log('Default templates response:', data);

        if (data.status && data.content && data.content.templates) {
            displayDefaultTemplates(data.content.templates);
        } else {
            grid.innerHTML = '<div class="empty-gallery-message" style="grid-column: 1/-1;"><i class="fa fa-folder-open-o"></i><h3>No Templates Found</h3><p>No default templates are available.</p></div>';
        }
    } catch (error) {
        console.error('Error loading default templates:', error);
        grid.innerHTML = '<div class="empty-gallery-message" style="grid-column: 1/-1;"><i class="fa fa-exclamation-triangle"></i><h3>Error Loading Templates</h3><p>Failed to load default templates. Please try again.</p></div>';
    }
}

// Display Default Templates in Grid
function displayDefaultTemplates(templates) {
    const grid = document.getElementById('defaultTemplatesGrid');
    if (!grid) return;

    if (!templates || templates.length === 0) {
        grid.innerHTML = '<div class="empty-gallery-message" style="grid-column: 1/-1;"><i class="fa fa-folder-open-o"></i><h3>No Templates Found</h3><p>No default templates are available.</p></div>';
        return;
    }

    grid.innerHTML = '';

    templates.forEach(template => {
        const card = document.createElement('div');
        card.className = 'template-card';

        // Determine icon based on category
        let icon = 'fa-certificate';
        if (template.category === 'diploma') icon = 'fa-graduation-cap';
        if (template.category === 'award') icon = 'fa-trophy';

        const safeName = escapeHtml(template.name || 'Untitled');
        const safeDesc = escapeHtml(template.description || 'No description available');
        const safeCat = escapeHtml(template.category || 'certificate');
        const safeOrient = escapeHtml(template.orientation || 'landscape');
        const safeFile = template.file.replace(/'/g, "\\'");

        card.innerHTML = `
            <div class="template-thumbnail">
                <i class="fa ${icon}"></i>
            </div>
            <div class="template-info">
                <h3>${safeName}</h3>
                <p>${safeDesc}</p>
                <div class="template-meta">
                    <span class="template-category">${safeCat}</span>
                    <span class="template-orientation">${safeOrient}</span>
                </div>
            </div>
            <div class="template-actions">
                <button class="use-template-btn" onclick="useDefaultTemplate('${safeFile}', '${safeName.replace(/'/g, "\\'")}')">
                    <i class="fa fa-download"></i> Use This Template
                </button>
            </div>
        `;

        grid.appendChild(card);
    });
}

// Use a Default Template
async function useDefaultTemplate(templateFile, templateName) {
    try {
        console.log('Loading template:', templateFile);

        const response = await fetch(getAjaxURL(), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
                'Accept': 'application/json',
            },
            credentials: 'same-origin',
            body: `action=loadDefaultTemplate&templateFile=${encodeURIComponent(templateFile)}`
        });

        const data = await response.json();
        console.log('Load template response:', data);

        if (data.status && data.content && data.content.templateData) {
            // Set orientation FIRST, before loading canvas data
            // This prevents the orientation change from re-serializing and reloading the canvas
            if (data.content.orientation && data.content.orientation !== currentOrientation) {
                currentOrientation = data.content.orientation;
                const orientationRadio = document.querySelector(`input[name="orientation"][value="${data.content.orientation}"]`);
                if (orientationRadio) {
                    orientationRadio.checked = true;
                }
                // Set canvas dimensions directly without triggering loadFromJSON
                const A4_WIDTH_PX = 794;
                const A4_HEIGHT_PX = 1123;
                const width = data.content.orientation === 'portrait' ? A4_WIDTH_PX : A4_HEIGHT_PX;
                const height = data.content.orientation === 'portrait' ? A4_HEIGHT_PX : A4_WIDTH_PX;
                canvas.setWidth(width * currentZoom);
                canvas.setHeight(height * currentZoom);
                canvas.setZoom(currentZoom);
            }

            // Parse the template data
            const templateData = typeof data.content.templateData === 'string'
                ? JSON.parse(data.content.templateData)
                : data.content.templateData;

            // Load onto canvas
            canvas.loadFromJSON(templateData, () => {
                // Track objects to remove and add (for journal logo replacement)
                const objectsToRemove = [];
                const logoReplacements = [];

                // Restore placeholder text for objects that have fieldType property
                // and ensure all objects are selectable
                canvas.getObjects().forEach(function (obj) {
                    // Ensure object is selectable and moveable
                    obj.set({
                        selectable: true,
                        evented: true,
                        hasControls: true,
                        hasBorders: true,
                        lockMovementX: false,
                        lockMovementY: false
                    });

                    if (obj.fieldType && typeof obj.fieldType === 'string' && obj.fieldType.startsWith('{')) {
                        // Handle {JOURNAL_LOGO} placeholder - replace with actual logo image
                        if (obj.fieldType === '{JOURNAL_LOGO}') {
                            if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                                // Store position and size for replacement
                                logoReplacements.push({
                                    left: obj.left,
                                    top: obj.top,
                                    width: obj.width * (obj.scaleX || 1),
                                    height: obj.height * (obj.scaleY || 1),
                                    originX: obj.originX || 'left',
                                    originY: obj.originY || 'top'
                                });
                                objectsToRemove.push(obj);
                            }
                            // If no logo configured, keep the textbox placeholder
                        }
                        // Handle text objects
                        else if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
                            obj.set('text', obj.fieldType);
                        }
                        // Handle group objects (like QR code group)
                        else if (obj.type === 'group' && obj._objects) {
                            obj._objects.forEach(function(child) {
                                if (child.type === 'text' && child.text) {
                                    child.set('text', obj.fieldType);
                                }
                            });
                        }
                    }
                });

                // Remove placeholder textboxes for journal logo
                objectsToRemove.forEach(function(obj) {
                    canvas.remove(obj);
                });

                // Load journal logo images at placeholder positions
                logoReplacements.forEach(function(pos) {
                    fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                        if (img) {
                            // Scale to fit the placeholder size while maintaining aspect ratio
                            const targetWidth = pos.width || 150;
                            const aspectRatio = img.height / img.width;
                            const scaleX = targetWidth / img.width;

                            img.set({
                                left: pos.left,
                                top: pos.top,
                                scaleX: scaleX,
                                scaleY: scaleX, // Maintain aspect ratio
                                originX: pos.originX,
                                originY: pos.originY,
                                fieldType: 'journal_logo',
                                selectable: true,
                                evented: true,
                                hasControls: true,
                                hasBorders: true,
                                lockMovementX: false,
                                lockMovementY: false,
                                cornerColor: '#ff6b6b',
                                cornerSize: 8,
                                transparentCorners: false
                            });

                            canvas.add(img);
                            canvas.renderAll();
                        }
                    }, { crossOrigin: 'anonymous' });
                });

                // Load background image if specified
                if (data.content.backgroundImage) {
                    loadBackgroundImageFromUrl(data.content.backgroundImage, templateName);
                } else {
                    canvas.renderAll();
                    console.log('Template loaded successfully');
                    closeTemplateGallery();
                    alert(`Template "${templateName}" loaded successfully!`);
                }
            });
        } else {
            alert('Failed to load template. Please try again.');
        }
    } catch (error) {
        console.error('Error loading template:', error);
        alert('An error occurred while loading the template.');
    }
}

// Helper function to load background image from URL
function loadBackgroundImageFromUrl(imageUrl, templateName) {
    fabric.Image.fromURL(imageUrl, function(img) {
        if (img) {
            // Get canvas dimensions
            const canvasWidth = canvas.getWidth() / currentZoom;
            const canvasHeight = canvas.getHeight() / currentZoom;

            // Scale image to fit canvas
            const scaleX = canvasWidth / img.width;
            const scaleY = canvasHeight / img.height;

            img.set({
                scaleX: scaleX,
                scaleY: scaleY,
                originX: 'left',
                originY: 'top',
                left: 0,
                top: 0
            });

            // Set as background image
            canvas.setBackgroundImage(img, canvas.renderAll.bind(canvas));

            // Show background controls if function exists
            if (typeof restoreBackgroundControls === 'function') {
                restoreBackgroundControls();
            }

            console.log('Template with background loaded successfully');
            closeTemplateGallery();
            alert(`Template "${templateName}" loaded successfully!`);
        } else {
            console.error('Failed to load background image');
            canvas.renderAll();
            closeTemplateGallery();
            alert(`Template "${templateName}" loaded (background image failed to load)`);
        }
    }, { crossOrigin: 'anonymous' });
}

// Load "My Templates" for the gallery
async function loadMyTemplatesForGallery() {
    const list = document.getElementById('myTemplatesList');
    if (!list) return;

    list.innerHTML = '<div style="text-align: center; padding: 40px;"><i class="fa fa-spinner fa-spin" style="font-size: 48px; color: #667eea;"></i><p style="margin-top: 20px; color: #666;">Loading your templates...</p></div>';

    try {
        // Use 'list' action instead of 'loadSavedTemplates'
        const response = await fetch(buildActionURL('list'), {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            },
            credentials: 'same-origin'
        });

        const data = await response.json();
        console.log('My templates response:', data);

        // Handle OJS JSONMessage format - templates are in data.content.templates or data.templates
        const result = data.content || data;
        const templates = result.templates || [];

        if (templates.length > 0) {
            displayMyTemplates(templates);
        } else {
            list.innerHTML = '<div class="empty-gallery-message"><i class="fa fa-folder-open-o"></i><h3>No Saved Templates</h3><p>You haven\'t saved any templates yet. Create a design and click "Save Template" to add it here.</p></div>';
        }
    } catch (error) {
        console.error('Error loading my templates:', error);
        list.innerHTML = '<div class="empty-gallery-message"><i class="fa fa-exclamation-triangle"></i><h3>Error Loading Templates</h3><p>Failed to load your templates. Please try again.</p></div>';
    }
}

// Get the current default template from localStorage
function getDefaultTemplate() {
    return localStorage.getItem('govalidDefaultTemplate') || null;
}

// Set a template as default
function setDefaultTemplate(templateName) {
    const currentDefault = getDefaultTemplate();

    if (currentDefault === templateName) {
        // If clicking on current default, remove it (toggle off)
        localStorage.removeItem('govalidDefaultTemplate');
        alert('Default template cleared. LoA will load on refresh.');
    } else {
        // Set new default
        localStorage.setItem('govalidDefaultTemplate', templateName);
        alert(`"${templateName}" is now set as your default template.`);
    }

    // Refresh the display to update star icons
    loadMyTemplatesForGallery();
}

// Display My Templates
function displayMyTemplates(templates) {
    const list = document.getElementById('myTemplatesList');
    if (!list) return;

    console.log('displayMyTemplates called with:', templates);
    list.innerHTML = '';

    const currentDefault = getDefaultTemplate();

    templates.forEach((template) => {
        const item = document.createElement('div');
        item.className = 'my-template-item';
        // Ensure name is a string
        const templateName = (typeof template.name === 'string') ? template.name : String(template.name || 'Untitled Template');
        const safeName = escapeHtml(templateName);
        const safeNameForJs = templateName.replace(/'/g, "\\'").replace(/"/g, '\\"');
        const isDefault = (currentDefault === templateName);
        const starClass = isDefault ? 'fa-star' : 'fa-star-o';
        const starTitle = isDefault ? 'Remove as default' : 'Set as default template';
        const starActiveClass = isDefault ? 'active' : '';

        item.innerHTML = `
            <div class="my-template-info">
                <button class="default-star-btn ${starActiveClass}" onclick="setDefaultTemplate('${safeNameForJs}')" title="${starTitle}">
                    <i class="fa ${starClass}"></i>
                </button>
                <div>
                    <h4>${safeName}</h4>
                    <p>Saved template${isDefault ? ' • <span style="color: #f59e0b;">Default</span>' : ''}</p>
                </div>
            </div>
            <div class="my-template-actions">
                <button class="my-template-btn load" onclick="loadMyTemplateFromGallery('${safeNameForJs}')">
                    <i class="fa fa-folder-open"></i> Load
                </button>
                <button class="my-template-btn delete" onclick="deleteMyTemplateFromGallery('${safeNameForJs}')">
                    <i class="fa fa-trash"></i> Delete
                </button>
            </div>
        `;
        list.appendChild(item);
    });
}

// Load My Template from Gallery (accepts template name directly)
async function loadMyTemplateFromGallery(templateName) {
    try {
        const response = await fetch(getAjaxURL(), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            credentials: 'same-origin',
            body: 'action=load&templateName=' + encodeURIComponent(templateName)
        });

        const data = await response.json();
        const result = data.content || data;

        if (result.templateData) {
            // Set orientation first if specified
            if (result.orientation) {
                const orientationRadio = document.querySelector(`input[name="orientation"][value="${result.orientation}"]`);
                if (orientationRadio && !orientationRadio.checked) {
                    orientationRadio.checked = true;
                    orientationRadio.dispatchEvent(new Event('change', { bubbles: true }));
                }
            }

            // Clean template data to reset QR placeholders
            const cleanedData = typeof cleanTemplateData === 'function' ? cleanTemplateData(result.templateData) : result.templateData;
            canvas.loadFromJSON(cleanedData, () => {
                // Track journal logo placeholders for replacement
                const objectsToRemove = [];
                const logoReplacements = [];

                // Ensure all objects are selectable and moveable after loading
                canvas.getObjects().forEach(function(obj) {
                    obj.set({
                        selectable: true,
                        evented: true,
                        hasControls: true,
                        hasBorders: true,
                        lockMovementX: false,
                        lockMovementY: false
                    });

                    // Check if this is a {JOURNAL_LOGO} placeholder that needs replacement
                    if (obj.fieldType === '{JOURNAL_LOGO}' &&
                        (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text')) {
                        if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                            logoReplacements.push({
                                left: obj.left,
                                top: obj.top,
                                width: obj.width * (obj.scaleX || 1),
                                height: obj.height * (obj.scaleY || 1),
                                originX: obj.originX || 'left',
                                originY: obj.originY || 'top'
                            });
                            objectsToRemove.push(obj);
                        }
                    }
                });

                // Remove journal logo placeholders
                objectsToRemove.forEach(function(obj) {
                    canvas.remove(obj);
                });

                // Load journal logo images at placeholder positions
                logoReplacements.forEach(function(pos) {
                    fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                        if (img) {
                            const targetWidth = pos.width || 150;
                            const scaleX = targetWidth / img.width;

                            img.set({
                                left: pos.left,
                                top: pos.top,
                                scaleX: scaleX,
                                scaleY: scaleX,
                                originX: pos.originX,
                                originY: pos.originY,
                                fieldType: 'journal_logo',
                                selectable: true,
                                evented: true,
                                hasControls: true,
                                hasBorders: true,
                                lockMovementX: false,
                                lockMovementY: false,
                                cornerColor: '#ff6b6b',
                                cornerSize: 8,
                                transparentCorners: false
                            });

                            canvas.add(img);
                            canvas.renderAll();
                        }
                    }, { crossOrigin: 'anonymous' });
                });

                // Restore background image controls if background image exists
                if (typeof restoreBackgroundControls === 'function') {
                    restoreBackgroundControls();
                }

                // Update template name input
                const templateNameInput = document.getElementById('templateName');
                if (templateNameInput) {
                    templateNameInput.value = templateName;
                }
                const templateNameToolbar = document.getElementById('templateNameToolbar');
                if (templateNameToolbar) {
                    templateNameToolbar.value = templateName;
                }

                canvas.renderAll();
                closeTemplateGallery();
                alert(`Template "${templateName}" loaded successfully!`);
            });
        } else {
            alert('Failed to load template. Template data not found.');
        }
    } catch (error) {
        console.error('Error loading template:', error);
        alert('Failed to load template');
    }
}

// Delete My Template from Gallery
async function deleteMyTemplateFromGallery(templateName) {
    if (!confirm(`Are you sure you want to delete the template "${templateName}"?`)) {
        return;
    }

    try {
        const response = await fetch(getAjaxURL(), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            credentials: 'same-origin',
            body: `action=deleteTemplate&templateName=${encodeURIComponent(templateName)}`
        });

        const data = await response.json();

        if (data.status) {
            alert('Template deleted successfully');
            loadMyTemplatesForGallery(); // Reload the list
        } else {
            alert('Failed to delete template');
        }
    } catch (error) {
        console.error('Error deleting template:', error);
        alert('An error occurred while deleting the template');
    }
}

// HTML escape helper function
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Attach event listener to Template Gallery button
document.addEventListener('DOMContentLoaded', function() {
    const galleryBtn = document.getElementById('templateGalleryBtn');
    if (galleryBtn) {
        galleryBtn.addEventListener('click', showTemplateGallery);
    }
    // Also add listener for toolbar button
    const galleryToolbarBtn = document.getElementById('templateGalleryBtnToolbar');
    if (galleryToolbarBtn) {
        galleryToolbarBtn.addEventListener('click', showTemplateGallery);
    }
});
