function setupContentTab() {
    console.log('Setting up content tab controls...');

    // Search by Submission ID
    const submissionIdInput = document.getElementById('submission_id_search');
    const searchSubmissionBtn = document.getElementById('search_submission_btn');

    console.log('Submission search elements:', {
        submissionIdInput: submissionIdInput ? 'Found' : 'Not found',
        searchSubmissionBtn: searchSubmissionBtn ? 'Found' : 'Not found'
    });

    if (searchSubmissionBtn) {
        console.log('Adding click event listener to searchSubmissionBtn');
        searchSubmissionBtn.addEventListener('click', function () {
            console.log('Search submission button clicked');
            searchSubmissionById();
        });
    }

    if (submissionIdInput) {
        submissionIdInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                console.log('Enter key pressed in submission ID input');
                searchSubmissionById();
            }
        });
    }

    // Search reviewers
    const searchReviewersBtn = document.getElementById('searchReviewers');
    const reviewerNameInput = document.getElementById('content_reviewer_name');
    const reviewerResults = document.getElementById('reviewerSearchResults');

    if (searchReviewersBtn) {
        searchReviewersBtn.addEventListener('click', function () {
            searchReviewers(reviewerNameInput.value);
        });
    }

    if (reviewerNameInput) {
        reviewerNameInput.addEventListener('input', function () {
            if (this.value.length >= 2) {
                searchReviewers(this.value);
            } else {
                reviewerResults.style.display = 'none';
            }
        });
    }

    // Search authors
    const searchAuthorsBtn = document.getElementById('searchAuthors');
    const authorNameInput = document.getElementById('content_author_name');
    const authorResults = document.getElementById('authorSearchResults');

    console.log('Author search elements:', {
        searchAuthorsBtn: searchAuthorsBtn ? 'Found' : 'Not found',
        authorNameInput: authorNameInput ? 'Found' : 'Not found',
        authorResults: authorResults ? 'Found' : 'Not found'
    });

    if (searchAuthorsBtn) {
        console.log('Adding click event listener to searchAuthorsBtn');
        searchAuthorsBtn.addEventListener('click', function () {
            console.log('Search authors button clicked');
            searchAuthors(authorNameInput.value);
        });
    }

    if (authorNameInput) {
        console.log('Adding input event listener to authorNameInput');
        authorNameInput.addEventListener('input', function () {
            console.log('Author name input changed:', this.value);
            if (this.value.length >= 2) {
                console.log('Author name input length >= 2, searching...');
                searchAuthors(this.value);
            } else {
                console.log('Author name input too short, hiding results');
                authorResults.style.display = 'none';
            }
        });
    } else {
        console.warn('Author name input element not found');
    }

    // Search editors
    const searchEditorsBtn = document.getElementById('searchEditors');
    const editorNameInput = document.getElementById('content_editor_name');
    const editorResults = document.getElementById('editorSearchResults');

    console.log('Editor search elements:', {
        searchEditorsBtn: searchEditorsBtn ? 'Found' : 'Not found',
        editorNameInput: editorNameInput ? 'Found' : 'Not found',
        editorResults: editorResults ? 'Found' : 'Not found'
    });

    if (searchEditorsBtn) {
        console.log('Adding click event listener to searchEditorsBtn');
        searchEditorsBtn.addEventListener('click', function () {
            console.log('Search editors button clicked');
            searchEditors(editorNameInput.value);
        });
    }

    if (editorNameInput) {
        console.log('Adding input event listener to editorNameInput');
        editorNameInput.addEventListener('input', function () {
            console.log('Editor name input changed:', this.value);
            if (this.value.length >= 2) {
                console.log('Editor name input length >= 2, searching...');
                searchEditors(this.value);
            } else {
                console.log('Editor name input too short, hiding results');
                editorResults.style.display = 'none';
            }
        });
    } else {
        console.warn('Editor name input element not found');
    }

    // Search articles
    const searchArticlesBtn = document.getElementById('searchArticles');
    const articleTitleInput = document.getElementById('content_article_title');

    if (searchArticlesBtn) {
        searchArticlesBtn.addEventListener('click', function () {
            // TODO: Implement article search
            alert('Article search will be implemented based on your journal setup');
        });
    }

    // Generate certificate
    const generateBtn = document.getElementById('generateCertificate');
    if (generateBtn) {
        generateBtn.addEventListener('click', function () {
            generateCertificateWithContent();
        });
    }

    // Clear content
    const clearBtn = document.getElementById('clearContent');
    if (clearBtn) {
        clearBtn.addEventListener('click', function () {
            document.querySelectorAll('.content-input').forEach(input => {
                if (input.type !== 'date' && !input.value.includes('{$')) {
                    input.value = '';
                }
            });
        });
    }

    // Refresh canvas with current content data
    const refreshBtn = document.getElementById('refreshCanvas');
    if (refreshBtn) {
        refreshBtn.addEventListener('click', function () {
            console.log('Refresh Canvas button clicked');
            updateCanvasWithContentData();
        });
    }
}

// Search reviewers function
function searchReviewers(query) {
    const resultsDiv = document.getElementById('reviewerSearchResults');

    if (!query || query.length < 2) {
        resultsDiv.style.display = 'none';
        return;
    }

    fetch(buildActionURL('searchReviewers') + '&query=' + encodeURIComponent(query))
        .then(response => response.json())
        .then(data => {
            if (data.content && data.content.reviewers) {
                displayReviewerResults(data.content.reviewers);
            }
        })
        .catch(error => {
            console.error('Error searching reviewers:', error);
        });
}

function displayReviewerResults(reviewers) {
    const resultsDiv = document.getElementById('reviewerSearchResults');

    if (reviewers.length === 0) {
        resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">No reviewers found</div>';
    } else {
        resultsDiv.innerHTML = reviewers.map(reviewer => `
                <div class="reviewer-result" style="padding: 8px 10px; cursor: pointer; border-bottom: 1px solid #f0f0f0;" 
                     onclick="selectReviewer('${reviewer.name}', ${reviewer.id})">
                    <div style="font-weight: 500;">${reviewer.name}</div>
                    <div style="font-size: 11px; color: #666;">${reviewer.email}</div>
                </div>
            `).join('');
    }

    resultsDiv.style.display = 'block';
}

window.selectReviewer = function (name, id) {
    document.getElementById('content_reviewer_name').value = name;
    document.getElementById('content_reviewer_name').dataset.reviewerId = id;
    document.getElementById('reviewerSearchResults').style.display = 'none';
}

// Author search functions
function searchAuthors(query) {
    console.log('searchAuthors called with query:', query);
    const resultsDiv = document.getElementById('authorSearchResults');

    if (!resultsDiv) {
        console.error('authorSearchResults element not found');
        return;
    }

    if (!query || query.length < 2) {
        console.log('Query too short, hiding results');
        resultsDiv.style.display = 'none';
        return;
    }

    const url = buildActionURL('searchAuthors') + '&query=' + encodeURIComponent(query);
    console.log('Fetching authors from URL:', url);

    fetch(url)
        .then(response => {
            console.log('Author search response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Author search raw response:', data);

            // Try to extract authors from different possible response formats
            let authors = [];

            if (data.content && Array.isArray(data.content.authors)) {
                // OJS JSONMessage format with content property
                authors = data.content.authors;
                console.log('Found authors in data.content.authors:', authors.length);
            } else if (Array.isArray(data.authors)) {
                // Direct format with authors at top level
                authors = data.authors;
                console.log('Found authors in data.authors:', authors.length);
            } else if (data.status && data.content && Array.isArray(data.content.authors)) {
                // Another possible OJS format
                authors = data.content.authors;
                console.log('Found authors in nested data.content.authors:', authors.length);
            } else {
                // Log the full response structure to help debug
                console.warn('Could not find authors array in response. Response structure:', JSON.stringify(data));
            }

            if (authors && authors.length > 0) {
                displayAuthorResults(authors);
            } else {
                console.warn('No authors found in response');
                resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">No authors found</div>';
                resultsDiv.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Error searching authors:', error);
            resultsDiv.innerHTML = '<div style="padding: 10px; color: #f00;">Error searching authors</div>';
            resultsDiv.style.display = 'block';
        });
}

function displayAuthorResults(authors) {
    const resultsDiv = document.getElementById('authorSearchResults');

    if (authors.length === 0) {
        resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">No authors found</div>';
    } else {
        resultsDiv.innerHTML = authors.map(author => `
                <div class="author-result" style="padding: 8px 10px; cursor: pointer; border-bottom: 1px solid #f0f0f0;" 
                    onclick="selectAuthor('${author.name.replace(/'/g, "\\'")}',${author.id || 0})">
                    <div style="font-weight: 500;">${author.name}</div>
                    <div style="font-size: 12px; color: #666;">${author.email || ''}</div>
                </div>
            `).join('');
    }

    resultsDiv.style.display = 'block';
}

window.selectAuthor = function (name, id) {
    document.getElementById('content_author_name').value = name;
    document.getElementById('content_author_name').dataset.authorId = id;
    document.getElementById('authorSearchResults').style.display = 'none';
}

// Editor search functions
function searchEditors(query) {
    console.log('searchEditors called with query:', query);
    const resultsDiv = document.getElementById('editorSearchResults');

    if (!resultsDiv) {
        console.error('editorSearchResults element not found');
        return;
    }

    if (!query || query.length < 2) {
        console.log('Query too short, hiding results');
        resultsDiv.style.display = 'none';
        return;
    }

    const url = buildActionURL('searchEditors') + '&query=' + encodeURIComponent(query);
    console.log('Fetching editors from URL:', url);

    fetch(url)
        .then(response => {
            console.log('Editor search response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Editor search raw response:', data);

            // Try to extract editors from different possible response formats
            let editors = [];

            if (data.content && Array.isArray(data.content.editors)) {
                // OJS JSONMessage format with content property
                editors = data.content.editors;
                console.log('Found editors in data.content.editors:', editors.length);
            } else if (Array.isArray(data.editors)) {
                // Direct format with editors at top level
                editors = data.editors;
                console.log('Found editors in data.editors:', editors.length);
            } else if (data.status && data.content && Array.isArray(data.content.editors)) {
                // Another possible OJS format
                editors = data.content.editors;
                console.log('Found editors in nested data.content.editors:', editors.length);
            } else {
                // Log the full response structure to help debug
                console.warn('Could not find editors array in response. Response structure:', JSON.stringify(data));
            }

            if (editors && editors.length > 0) {
                displayEditorResults(editors);
            } else {
                console.warn('No editors found in response');
                resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">No editors found</div>';
                resultsDiv.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Error searching editors:', error);
            resultsDiv.innerHTML = '<div style="padding: 10px; color: #f00;">Error searching editors</div>';
            resultsDiv.style.display = 'block';
        });
}

function displayEditorResults(editors) {
    const resultsDiv = document.getElementById('editorSearchResults');

    if (editors.length === 0) {
        resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">No editors found</div>';
    } else {
        resultsDiv.innerHTML = editors.map(editor => `
                <div class="editor-result" style="padding: 8px 10px; cursor: pointer; border-bottom: 1px solid #f0f0f0;" 
                    onclick="selectEditor('${editor.name.replace(/'/g, "\\'")}',${editor.id || 0})">
                    <div style="font-weight: 500;">${editor.name}</div>
                    <div style="font-size: 12px; color: #666;">${editor.email || ''}</div>
                </div>
            `).join('');
    }

    resultsDiv.style.display = 'block';
}

window.selectEditor = function (name, id) {
    document.getElementById('content_editor_name').value = name;
    document.getElementById('content_editor_name').dataset.editorId = id;
    document.getElementById('editorSearchResults').style.display = 'none';
}

// Generate preview with specific data
async function generatePreviewWithData(formData) {
    const statusDiv = document.getElementById('contentStatus');
    statusDiv.style.display = 'block';

    // Check GoValid subscription before generating preview
    const canGenerate = await checkBeforeSave();
    if (!canGenerate) {
        statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Unable to generate preview. Please check your subscription status.</p>';
        return;
    }

    statusDiv.innerHTML = '<p style="color: #666;"><i class="fa fa-spinner fa-spin"></i> Generating preview...</p>';

    // Format replacements
    const replacements = {
        '{REVIEWER_NAME}': formData.reviewer_name || '',
        '{AUTHOR_NAME}': formData.author_name || '',
        '{EDITOR_NAME}': formData.editor_name || '',
        '{JOURNAL_NAME}': formData.journal_name,
        '{REVIEW_DATE}': new Date(formData.review_date).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }),
        '{ARTICLE_TITLE}': formData.article_title,
        '{ARTICLE_DOI}': formData.article_doi || '',
        '{ISSUE_TITLE}': formData.issue_title || '',
        '{ISSUE_VOLUME}': formData.issue_volume || '',
        '{ISSUE_NUMBER}': formData.issue_number || '',
        '{ISSUE_YEAR}': formData.issue_year || '',
        '{SUBMISSION_ID}': formData.submission_id || '',
        '{ISSN_ONLINE}': (typeof goValidJournalInfo !== 'undefined' && goValidJournalInfo.onlineIssn) ? goValidJournalInfo.onlineIssn : '',
        '{ISSN_PRINT}': (typeof goValidJournalInfo !== 'undefined' && goValidJournalInfo.printIssn) ? goValidJournalInfo.printIssn : ''
    };

    // Generate QR code if needed
    const templateString = JSON.stringify(canvas.toJSON());
    if (goValidAuth.isAuthenticated && templateString.includes('{QR_CODE}')) {
        try {
            statusDiv.innerHTML = '<p style="color: #666;"><i class="fa fa-spinner fa-spin"></i> Generating certificate QR code...</p>';
            const qrResult = await generateCertificateQR(replacements);
            if (qrResult.success && qrResult.qr_code) {
                replacements['{QR_CODE}'] = 'data:image/png;base64,' + qrResult.qr_code;
            }
        } catch (error) {
            console.error('QR generation failed:', error);
            replacements['{QR_CODE}'] = '';
        }
    }

    statusDiv.innerHTML = '<p style="color: #666;"><i class="fa fa-spinner fa-spin"></i> Generating PDF...</p>';

    // Prepare canvas data
    const canvasData = JSON.stringify(canvas.toJSON());

    // Send preview request
    fetch(getCleanURL(), {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=preview' +
            '&canvasData=' + encodeURIComponent(canvasData) +
            '&replacements=' + encodeURIComponent(JSON.stringify(replacements)) +
            '&orientation=' + currentOrientation
    })
        .then(response => {
            if (!response.ok) {
                throw new Error('Server error');
            }
            return response.blob();
        })
        .then(blob => {
            // Create download link
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'certificate_' + formData.reviewer_name.replace(/\s+/g, '_') + '.pdf';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);

            statusDiv.innerHTML = '<p style="color: #48bb78;"><i class="fa fa-check-circle"></i> Certificate generated successfully!</p>';
            setTimeout(() => {
                statusDiv.style.display = 'none';
            }, 3000);
        })
        .catch(error => {
            console.error('Preview generation failed:', error);
            statusDiv.innerHTML = '<p style="color: #f56565;"><i class="fa fa-exclamation-triangle"></i> Failed to generate certificate. Please try again.</p>';
        });
}

// Function to select name from Content tab for Awarded To field
function selectAwardedName() {
    const selector = document.getElementById('qr_name_selector');
    const awardedToField = document.getElementById('qr_awarded_to');

    if (selector.value === 'reviewer') {
        const reviewerName = document.getElementById('content_reviewer_name').value;
        if (reviewerName) {
            awardedToField.value = reviewerName;
        } else {
            awardedToField.value = '{REVIEWER_NAME}';
        }
    } else if (selector.value === 'author') {
        const authorName = document.getElementById('content_author_name').value;
        if (authorName) {
            awardedToField.value = authorName;
        } else {
            awardedToField.value = '{AUTHOR_NAME}';
        }
    } else if (selector.value === 'editor') {
        const editorName = document.getElementById('content_editor_name').value;
        if (editorName) {
            awardedToField.value = editorName;
        } else {
            awardedToField.value = '{EDITOR_NAME}';
        }
    }
}

// Generate certificate with content
function generateCertificateWithContent() {
    const contentData = {
        reviewer_name: document.getElementById('content_reviewer_name').value,
        author_name: document.getElementById('content_author_name').value,
        editor_name: document.getElementById('content_editor_name').value,
        journal_name: document.getElementById('content_journal_name').value,
        review_date: document.getElementById('content_review_date').value,
        article_title: document.getElementById('content_article_title').value,
        article_doi: document.getElementById('content_article_doi').value,
        issue_title: document.getElementById('content_issue_title').value,
        issue_volume: document.getElementById('content_issue_volume').value,
        issue_number: document.getElementById('content_issue_number').value,
        issue_year: document.getElementById('content_issue_year').value,
        submission_id: document.getElementById('content_submission_id').value
    };

    // Validate required fields - at least one name must be provided
    if (!contentData.reviewer_name && !contentData.author_name && !contentData.editor_name) {
        alert('Please select at least one person (reviewer, author, or editor)');
        return;
    }

    // Generate certificate with the form data from content tab
    const form = document.createElement('form');
    for (const key in contentData) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = key;
        input.value = contentData[key];
        form.appendChild(input);
    }

    generatePreviewWithData(contentData);
}

// Setup Load Template functionality
function setupLoadTemplate() {
    const loadTemplateBtn = document.getElementById('loadTemplate');
    if (loadTemplateBtn) {
        loadTemplateBtn.addEventListener('click', function () {
            showLoadTemplateModal();
        });
    }
    // Also add listener for toolbar button
    const loadTemplateToolbarBtn = document.getElementById('loadTemplateToolbar');
    if (loadTemplateToolbarBtn) {
        loadTemplateToolbarBtn.addEventListener('click', function () {
            showLoadTemplateModal();
        });
    }
}

// Show load template modal
function showLoadTemplateModal() {
    // Create modal if it doesn't exist
    let modal = document.getElementById('loadTemplateModal');
    if (!modal) {
        modal = createLoadTemplateModal();
        document.body.appendChild(modal);
    }

    // Load templates
    loadTemplatesList();

    // Show modal
    modal.style.display = 'block';
}

// Create load template modal
function createLoadTemplateModal() {
    const modal = document.createElement('div');
    modal.id = 'loadTemplateModal';
    modal.className = 'preview-modal';
    modal.innerHTML = `
            <div class="preview-modal-content" style="max-width: 600px;">
                <div class="preview-modal-header">
                    <h3>Load Template</h3>
                    <span class="preview-close" onclick="document.getElementById('loadTemplateModal').style.display='none'">&times;</span>
                </div>
                <div style="padding: 20px;">
                    <div id="templatesListContainer">
                        <p style="text-align: center; color: #666;">
                            <i class="fa fa-spinner fa-spin"></i> Loading templates...
                        </p>
                    </div>
                </div>
            </div>
        `;
    return modal;
}

// Load templates list
function loadTemplatesList() {
    const container = document.getElementById('templatesListContainer');

    fetch(buildActionURL('list'))
        .then(response => response.json())
        .then(data => {
            console.log('Templates list response:', data);
            if (data.status && data.content && data.content.templates) {
                displayTemplatesList(data.content.templates);
            } else if (data.templates) {
                // Handle direct templates array
                displayTemplatesList(data.templates);
            } else {
                container.innerHTML = '<p style="text-align: center; color: #666;">No templates found.</p>';
            }
        })
        .catch(error => {
            container.innerHTML = '<p style="color: #e74c3c;">Error loading templates</p>';
        });
}

// Display templates list
function displayTemplatesList(templates) {
    const container = document.getElementById('templatesListContainer');

    if (!templates || templates.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #666;">No saved templates found</p>';
        return;
    }

    // Store templates globally for later use
    window.loadedTemplates = templates;

    container.innerHTML = templates.map((template, index) => `
            <div style="padding: 15px; border: 1px solid #ddd; margin-bottom: 10px; border-radius: 4px; position: relative;">
                <div style="cursor: pointer;" onclick="loadSelectedTemplate(${index})">
                    <h4 style="margin: 0 0 5px 0; color: #333;">${template.name || 'Untitled Template'}</h4>
                    <p style="margin: 0; font-size: 12px; color: #666;">
                        Created: ${new Date(template.timestamp || template.created || Date.now()).toLocaleDateString()}
                    </p>
                </div>
                <button onclick="event.stopPropagation(); deleteTemplate('${template.name}')" 
                        style="position: absolute; top: 10px; right: 10px; background: #e74c3c; color: white; 
                        border: none; border-radius: 3px; padding: 3px 8px; cursor: pointer;">
                    Delete
                </button>
            </div>
        `).join('');
}

// Delete template
function deleteTemplate(templateName) {
    if (confirm('Are you sure you want to delete this template?')) {
        console.log('Deleting template:', templateName);

        fetch(buildActionURL('delete') + '&templateName=' + encodeURIComponent(templateName), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            credentials: 'same-origin'
        })
            .then(response => response.json())
            .then(data => {
                console.log('Delete response:', data);
                if (data.status) {
                    // Reload templates list
                    loadTemplatesList();
                    alert('Template deleted successfully!');
                } else {
                    alert('Error deleting template: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Error deleting template:', error);
                alert('Error deleting template');
            });
    }
}

// Check template limit before saving
function checkTemplateLimit(callback) {
    fetch(buildActionURL('list'))
        .then(response => response.json())
        .then(data => {
            if (data.status && data.content && data.content.templates) {
                const templates = data.content.templates;
                const templateName = document.getElementById('templateName').value.trim();

                // Check if template with same name exists (update case)
                const existingTemplate = templates.find(t => t.name === templateName);

                // If this is the currently loaded template, it's an update
                let isUpdate = false;
                if (currentLoadedTemplate && templateName === currentLoadedTemplate) {
                    isUpdate = true;
                } else if (existingTemplate) {
                    isUpdate = true;
                }

                // Allow save if updating or if under limit
                const canSave = isUpdate || templates.length < 10;
                callback(canSave, templates.length, isUpdate);
            } else {
                // No templates yet, so under limit
                callback(true, 0, false);
            }
        })
        .catch(error => {
            console.error('Error checking template limit:', error);
            // Default to allowing save on error
            callback(true, 0, false);
        });
}

// Load selected template
window.loadSelectedTemplate = function (index) {
    if (!window.loadedTemplates || !window.loadedTemplates[index]) {
        alert('Template not found');
        return;
    }

    const template = window.loadedTemplates[index];
    const templateName = template.name;

    console.log('Loading template:', templateName);

    // Now load the specific template
    fetch(getAjaxURL(), {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=load&templateName=' + encodeURIComponent(templateName)
    })
        .then(response => response.json())
        .then(data => {
            const result = data.content || data;
            if (result.templateData) {
                const cleanedData = cleanTemplateData(result.templateData);
                canvas.loadFromJSON(cleanedData, function () {
                    // Fix textBaseline for all text objects after loading
                    canvas.getObjects('text').forEach(function (obj) {
                        fixTextBaseline(obj);
                    });

                    // Find and replace QR code images with placeholder boxes
                    // Also handle {JOURNAL_LOGO} placeholder replacement
                    const objectsToRemove = [];
                    const objectsToAdd = [];
                    const logoReplacements = [];

                    canvas.getObjects().forEach(function (obj) {
                        // Check if this is a {JOURNAL_LOGO} placeholder that needs replacement
                        if (obj.fieldType === '{JOURNAL_LOGO}' &&
                            (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text')) {
                            if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                                // Store position and size for replacement
                                logoReplacements.push({
                                    left: obj.left,
                                    top: obj.top,
                                    width: obj.width * (obj.scaleX || 1),
                                    height: obj.height * (obj.scaleY || 1),
                                    originX: obj.originX || 'left',
                                    originY: obj.originY || 'top'
                                });
                                objectsToRemove.push(obj);
                            }
                        }
                        // Check if this is a QR code image that needs to be reset to placeholder
                        else if (obj.fieldType === 'qr_code_image' ||
                            (obj.type === 'image' && (obj.fieldType === '{QR_CODE}' || obj.fieldType === 'qr_code'))) {

                            // Store position and size of current QR image
                            const qrLeft = obj.left;
                            const qrTop = obj.top;
                            const qrWidth = obj.width * (obj.scaleX || 1);
                            const qrHeight = obj.height * (obj.scaleY || 1);
                            const qrSize = Math.max(qrWidth, qrHeight, 125);

                            // Mark for removal
                            objectsToRemove.push(obj);

                            // Create replacement placeholder box
                            const qrBox = new fabric.Rect({
                                left: 0,
                                top: 0,
                                width: qrSize,
                                height: qrSize,
                                fill: '#f0f0f0',
                                stroke: '#333',
                                strokeWidth: 2,
                                cornerColor: '#ff6b6b',
                                cornerSize: 8,
                                transparentCorners: false
                            });

                            const qrLabel = new fabric.Text('{QR_CODE}', {
                                left: qrSize / 2,
                                top: qrSize / 2,
                                fontSize: 12,
                                fill: '#333',
                                originX: 'center',
                                originY: 'center',
                                textBaseline: 'alphabetic',
                                selectable: false,
                                evented: false
                            });

                            const qrGroup = new fabric.Group([qrBox, qrLabel], {
                                left: qrLeft,
                                top: qrTop,
                                fieldType: '{QR_CODE}',
                                cornerColor: '#ff6b6b',
                                cornerSize: 8,
                                transparentCorners: false
                            });

                            objectsToAdd.push(qrGroup);
                        }
                    });

                    // Remove QR images and journal logo placeholders, add placeholder boxes
                    objectsToRemove.forEach(function(obj) {
                        canvas.remove(obj);
                    });
                    objectsToAdd.forEach(function(obj) {
                        canvas.add(obj);
                    });

                    // Load journal logo images at placeholder positions
                    logoReplacements.forEach(function(pos) {
                        fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                            if (img) {
                                // Scale to fit the placeholder size while maintaining aspect ratio
                                const targetWidth = pos.width || 150;
                                const scaleX = targetWidth / img.width;

                                img.set({
                                    left: pos.left,
                                    top: pos.top,
                                    scaleX: scaleX,
                                    scaleY: scaleX, // Maintain aspect ratio
                                    originX: pos.originX,
                                    originY: pos.originY,
                                    fieldType: 'journal_logo',
                                    selectable: true,
                                    evented: true,
                                    hasControls: true,
                                    hasBorders: true,
                                    lockMovementX: false,
                                    lockMovementY: false,
                                    cornerColor: '#ff6b6b',
                                    cornerSize: 8,
                                    transparentCorners: false
                                });

                                canvas.add(img);
                                canvas.renderAll();
                            }
                        }, { crossOrigin: 'anonymous' });
                    });

                    // Restore placeholder text and styles for objects that have fieldType property
                    // This ensures templates load with placeholders ready for new data

                    // Define default styles for content placeholders
                    const contentPlaceholderDefaults = {
                        fill: '#000000',
                        backgroundColor: '',
                        fontWeight: 'normal',
                        padding: 0
                    };

                    // Define styles for QR-related placeholders
                    const qrPlaceholderDefaults = {
                        fill: '#d97706',
                        backgroundColor: '#fffbea',
                        fontWeight: 'bold',
                        padding: 8
                    };

                    // QR field types that need special styling
                    const qrFieldTypes = ['{QR_ID}', '{QR_IDENTIFIER}', '{QR_ISSUE_DATE}', '{QR_EXPIRE_DATE}'];

                    canvas.getObjects().forEach(function (obj) {
                        if (obj.fieldType && typeof obj.fieldType === 'string' && obj.fieldType.startsWith('{')) {
                            // Handle text objects
                            if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
                                // Reset text to placeholder
                                obj.set('text', obj.fieldType);

                                // Apply appropriate styles based on field type
                                if (qrFieldTypes.includes(obj.fieldType)) {
                                    obj.set('fill', qrPlaceholderDefaults.fill);
                                    obj.set('backgroundColor', qrPlaceholderDefaults.backgroundColor);
                                    obj.set('fontWeight', qrPlaceholderDefaults.fontWeight);
                                    obj.set('padding', qrPlaceholderDefaults.padding);
                                } else if (obj.fieldType !== '{QR_CODE}') {
                                    // Regular content placeholders (not QR_CODE which has its own handling)
                                    obj.set('fill', contentPlaceholderDefaults.fill);
                                    obj.set('backgroundColor', contentPlaceholderDefaults.backgroundColor);
                                    obj.set('fontWeight', contentPlaceholderDefaults.fontWeight);
                                    obj.set('padding', contentPlaceholderDefaults.padding);
                                }
                            }
                            // Handle group objects (like QR code group)
                            else if (obj.type === 'group' && obj._objects) {
                                obj._objects.forEach(function(child) {
                                    if (child.type === 'text' && child.text) {
                                        child.set('text', obj.fieldType);
                                    }
                                });
                            }
                        }
                    });

                    // Ensure all objects (including images) are selectable after loading
                    canvas.getObjects().forEach(function(obj) {
                        obj.set({
                            selectable: true,
                            evented: true,
                            hasControls: true,
                            hasBorders: true,
                            lockMovementX: false,
                            lockMovementY: false
                        });
                    });

                    // Restore background image controls if background image exists
                    restoreBackgroundControls();

                    // Set the currently loaded template name
                    currentLoadedTemplate = templateName;

                    // Update the template name input fields to show which template is loaded
                    if (typeof setTemplateName === 'function') {
                        setTemplateName(templateName);
                    } else {
                        const canvasInput = document.getElementById('templateName');
                        const toolbarInput = document.getElementById('templateNameToolbar');
                        if (canvasInput) canvasInput.value = templateName;
                        if (toolbarInput) toolbarInput.value = templateName;
                    }
                    const canvasInput = document.getElementById('templateName');
                    const toolbarInput = document.getElementById('templateNameToolbar');
                    if (canvasInput) canvasInput.setAttribute('data-original', templateName);
                    if (toolbarInput) toolbarInput.setAttribute('data-original', templateName);

                    // Add visual indicator that we're editing an existing template
                    updateSaveButtonText();

                    canvas.renderAll();
                    document.getElementById('loadTemplateModal').style.display = 'none';
                    alert('Template loaded successfully!');
                });
            } else {
                alert('No template data found');
            }
        })
        .catch(error => {
            console.error('Error loading template:', error);
            alert('Error loading template');
        });
}

// Auto-populated dropdown functions
function loadIssuesOnPageLoad() {
    console.log('=== loadIssuesOnPageLoad called ===');

    // Use the current window URL with added action parameter
    const url = buildActionURL('getIssues');

    console.log('Current window URL:', window.location.href);
    console.log('Loading issues from URL:', url);

    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'
    })
        .then(response => {
            console.log('Issues response status:', response.status);
            console.log('Issues response URL:', response.url);
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.text();
        })
        .then(text => {
            console.log('Issues raw response:', text);
            console.log('Response length:', text.length);

            // Check if response is HTML (error page)
            if (text.trim().startsWith('<!DOCTYPE') || text.trim().startsWith('<html')) {
                console.error('Received HTML instead of JSON - likely an error page');
                const issueSelect = document.getElementById('content_issue_select');
                if (issueSelect) {
                    issueSelect.innerHTML = '<option value="">-- Error Loading Issues --</option>';
                }
                return;
            }

            try {
                const data = JSON.parse(text);
                const result = data.content || data;

                console.log('Issues parsed data:', result);

                if (result.debug) {
                    console.log('Debug info:', result.debug);
                }

                if (result.issues && result.issues.length > 0) {
                    console.log('Found', result.issues.length, 'issues');
                    populateIssuesDropdown(result.issues);
                } else {
                    console.log('No published issues found in response');
                    // Show a message to the user
                    const issueSelect = document.getElementById('content_issue_select');
                    if (issueSelect) {
                        issueSelect.innerHTML = '<option value="">-- No Published Issues Available --</option>';
                    }
                }
            } catch (e) {
                console.error('Error parsing issues response:', e);
                console.error('Raw response was:', text);
                const issueSelect = document.getElementById('content_issue_select');
                if (issueSelect) {
                    issueSelect.innerHTML = '<option value="">-- Error: Invalid Response --</option>';
                }
            }
        })
        .catch(error => {
            console.error('Error loading issues:', error);
            const issueSelect = document.getElementById('content_issue_select');
            if (issueSelect) {
                issueSelect.innerHTML = '<option value="">-- Error Loading Issues --</option>';
            }
        });
}

function populateIssuesDropdown(issues) {
    console.log('=== populateIssuesDropdown called ===');
    console.log('Number of issues:', issues.length);

    const issueSelect = document.getElementById('content_issue_select');
    if (!issueSelect) {
        console.error('ERROR: Issue select element not found! Looking for: content_issue_select');
        // Try to find any select elements in content tab
        const contentTab = document.querySelector('#content-tab');
        const selects = contentTab ? contentTab.querySelectorAll('select') : [];
        console.log('Found', selects.length, 'select elements in content tab');
        selects.forEach((sel, idx) => {
            console.log(`Select ${idx}: id="${sel.id}", class="${sel.className}"`);
        });
        return;
    }

    console.log('Found issue select element:', issueSelect);

    // Clear existing options except the default
    issueSelect.innerHTML = '<option value="">-- Select an Issue --</option>';

    // Store issues data globally for preview
    window.ojsIssuesData = {};

    issues.forEach(issue => {
        console.log('Adding issue to dropdown:', issue);
        const option = document.createElement('option');
        option.value = issue.id;
        const publishedStatus = issue.published === false ? ' [UNPUBLISHED]' : '';
        option.textContent = `Vol ${issue.volume || '?'}, No ${issue.number || '?'} (${issue.year || '?'})${issue.title ? ' - ' + issue.title : ''}${publishedStatus}`;
        option.dataset.volume = issue.volume || '';
        option.dataset.number = issue.number || '';
        option.dataset.year = issue.year || '';
        option.dataset.title = issue.title || '';
        issueSelect.appendChild(option);

        // Store issue data globally
        window.ojsIssuesData[issue.id] = issue;
    });

    console.log('Issue dropdown now has', issueSelect.options.length, 'options');
    console.log('=== populateIssuesDropdown completed ===');
}

/**
 * Update all text objects on canvas with current content data
 * This function finds text objects with placeholders and replaces them with actual data
 */
function updateCanvasWithContentData() {
    if (!canvas) {
        console.log('Canvas not available for update');
        return;
    }

    console.log('=== Updating canvas with content data ===');

    // Get QR Content fields from Tab 2
    const qrIdentifierValue = document.getElementById('qr_identifier_code')?.value;
    const qrIdentifier = qrIdentifierValue && qrIdentifierValue.trim() !== '' ? qrIdentifierValue.trim() : '{QR_IDENTIFIER}';
    const qrStartDateToggle = document.getElementById('qr_start_date_toggle')?.checked;
    const qrEndDateToggle = document.getElementById('qr_end_date_toggle')?.checked;
    const qrStartDate = qrStartDateToggle ? document.getElementById('qr_start_date')?.value : null;
    const qrEndDate = qrEndDateToggle ? document.getElementById('qr_end_date')?.value : null;

    // Build replacements object from current content fields
    const replacements = {
        '{REVIEWER_NAME}': document.getElementById('content_reviewer_name')?.value || '{REVIEWER_NAME}',
        '{AUTHOR_NAME}': document.getElementById('content_author_name')?.value || '{AUTHOR_NAME}',
        '{EDITOR_NAME}': document.getElementById('content_editor_name')?.value || '{EDITOR_NAME}',
        '{JOURNAL_NAME}': document.getElementById('content_journal_name')?.value || '{JOURNAL_NAME}',
        '{REVIEW_DATE}': new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }),
        '{ARTICLE_TITLE}': document.getElementById('content_article_title')?.value || '{ARTICLE_TITLE}',
        '{ARTICLE_DOI}': document.getElementById('content_article_doi')?.value || '{ARTICLE_DOI}',
        '{ISSUE_TITLE}': document.getElementById('content_issue_title')?.value || '{ISSUE_TITLE}',
        '{ISSUE_VOLUME}': document.getElementById('content_issue_volume')?.value || '{ISSUE_VOLUME}',
        '{ISSUE_NUMBER}': document.getElementById('content_issue_number')?.value || '{ISSUE_NUMBER}',
        '{ISSUE_YEAR}': document.getElementById('content_issue_year')?.value || '{ISSUE_YEAR}',
        '{SUBMISSION_ID}': document.getElementById('content_submission_id')?.value || '{SUBMISSION_ID}',
        '{QR_ID}': (typeof getQRSettings === 'function' && getQRSettings().formatted_uuid) ? getQRSettings().formatted_uuid : '{QR_ID}',
        '{QR_IDENTIFIER}': qrIdentifier,
        '{QR_ISSUE_DATE}': qrStartDate ? new Date(qrStartDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
        '{QR_EXPIRE_DATE}': qrEndDate ? new Date(qrEndDate).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
        '{ISSN_ONLINE}': (typeof goValidJournalInfo !== 'undefined' && goValidJournalInfo.onlineIssn) ? goValidJournalInfo.onlineIssn : '{ISSN_ONLINE}',
        '{ISSN_PRINT}': (typeof goValidJournalInfo !== 'undefined' && goValidJournalInfo.printIssn) ? goValidJournalInfo.printIssn : '{ISSN_PRINT}'
    };

    console.log('Replacement data:', replacements);

    let updatedCount = 0;

    // Get all objects on canvas
    const objects = canvas.getObjects();
    console.log('Total objects on canvas:', objects.length);

    // Iterate through all objects
    objects.forEach((obj, index) => {
        console.log(`Object ${index}:`, obj.type, obj.text || obj.objectType);

        // Check if it's a text object
        if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
            const originalText = obj.text;
            console.log(`  Original text: "${originalText}"`);
            let newText = originalText;
            let hasPlaceholder = false;

            // Check if text contains any placeholder
            Object.keys(replacements).forEach(placeholder => {
                if (originalText.includes(placeholder)) {
                    hasPlaceholder = true;
                    console.log(`  Found placeholder: ${placeholder}`);
                    // Replace placeholder with actual value
                    newText = newText.replace(new RegExp(placeholder.replace(/[{}]/g, '\\$&'), 'g'), replacements[placeholder]);
                }
            });

            // Update text if it contained placeholders
            if (hasPlaceholder && newText !== originalText) {
                console.log(`  Updating: "${originalText}" → "${newText}"`);
                obj.set('text', newText);
                updatedCount++;
            } else if (!hasPlaceholder) {
                console.log(`  No placeholders found in text`);
            }
        }
    });

    console.log(`Updated ${updatedCount} text objects on canvas`);

    // Re-render canvas to show changes
    canvas.renderAll();

    // Show feedback to user
    const contentStatus = document.getElementById('contentStatus');
    if (contentStatus) {
        contentStatus.style.display = 'block';
        contentStatus.innerHTML = `<p style="color: #48bb78; font-size: 12px;"><i class="fa fa-check-circle"></i> Canvas refreshed! Updated ${updatedCount} text field(s).</p>`;

        // Hide after 3 seconds
        setTimeout(() => {
            contentStatus.style.display = 'none';
        }, 3000);
    }
}

function handleIssueSelectionChange() {
    const issueSelect = document.getElementById('content_issue_select');
    const selectedOption = issueSelect.options[issueSelect.selectedIndex];

    if (selectedOption.value) {
        // Auto-fill issue fields
        document.getElementById('content_issue_volume').value = selectedOption.dataset.volume || '';
        document.getElementById('content_issue_number').value = selectedOption.dataset.number || '';
        document.getElementById('content_issue_year').value = selectedOption.dataset.year || '';
        document.getElementById('content_issue_title').value = selectedOption.dataset.title || '';

        // Load articles for this issue
        loadArticlesForIssue(selectedOption.value);

        // Auto-refresh canvas with new data
        updateCanvasWithContentData();
    } else {
        // Clear fields
        document.getElementById('content_issue_volume').value = '';
        document.getElementById('content_issue_number').value = '';
        document.getElementById('content_issue_year').value = '';
        document.getElementById('content_issue_title').value = '';

        // Clear article dropdown
        const articleSelect = document.getElementById('content_article_select');
        articleSelect.innerHTML = '<option value="">-- Select an Article --</option>';
        clearArticleFields();

        // Refresh canvas to clear data
        updateCanvasWithContentData();
    }
}

function loadArticlesForIssue(issueId) {
    // Use the current window URL with added action parameter
    const url = buildActionURL('getSubmissions') + '&issueId=' + encodeURIComponent(issueId);

    console.log('Loading articles for issue:', issueId);
    console.log('Articles URL:', url);

    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'  // Include cookies for session
    })
        .then(response => {
            console.log('Articles response status:', response.status);
            // Always get the response text, even on error
            return response.text().then(text => {
                if (!response.ok) {
                    console.error('Error response text:', text);
                    // Try to parse as JSON to get error details
                    try {
                        const errorData = JSON.parse(text);
                        console.error('Error details:', errorData);
                    } catch (e) {
                        console.error('Could not parse error response as JSON');
                    }
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return text;
            });
        })
        .then(text => {
            console.log('Articles raw response:', text);
            console.log('Response length:', text.length);

            // Check if response is HTML (error page)
            if (text.trim().startsWith('<!DOCTYPE') || text.trim().startsWith('<html')) {
                console.error('Received HTML instead of JSON - likely an error page');
                const articleSelect = document.getElementById('content_article_select');
                if (articleSelect) {
                    articleSelect.innerHTML = '<option value="">-- Error Loading Articles --</option>';
                }
                return;
            }

            try {
                const data = JSON.parse(text);
                const result = data.content || data;

                console.log('Articles parsed data:', result);

                if (result.submissions && Array.isArray(result.submissions)) {
                    console.log('Found', result.submissions.length, 'articles');
                    populateArticlesDropdown(result.submissions);
                } else {
                    console.log('No articles found in response');
                    const articleSelect = document.getElementById('content_article_select');
                    if (articleSelect) {
                        articleSelect.innerHTML = '<option value="">-- No Articles in This Issue --</option>';
                    }
                }
            } catch (e) {
                console.error('Error parsing submissions response:', e);
                console.error('Raw response was:', text);
                const articleSelect = document.getElementById('content_article_select');
                if (articleSelect) {
                    articleSelect.innerHTML = '<option value="">-- Error: Invalid Response --</option>';
                }
            }
        })
        .catch(error => {
            console.error('Error loading articles:', error);
            const articleSelect = document.getElementById('content_article_select');
            if (articleSelect) {
                articleSelect.innerHTML = '<option value="">-- Error Loading Articles --</option>';
            }
        });
}

function populateArticlesDropdown(submissions) {
    const articleSelect = document.getElementById('content_article_select');
    if (!articleSelect) return;

    // Clear existing options except the default
    articleSelect.innerHTML = '<option value="">-- Select an Article --</option>';

    // Store articles data globally for preview
    window.ojsArticlesData = {};

    submissions.forEach(submission => {
        const option = document.createElement('option');
        option.value = submission.id;
        option.textContent = `[${submission.id}] ${submission.title}`;
        option.dataset.title = submission.title || '';
        option.dataset.doi = submission.doi || '';
        option.dataset.submissionId = submission.id;
        articleSelect.appendChild(option);

        // Store article data
        window.ojsArticlesData[submission.id] = submission;
    });
}

function handleArticleSelectionChange() {
    const articleSelect = document.getElementById('content_article_select');
    const selectedOption = articleSelect.options[articleSelect.selectedIndex];

    if (selectedOption.value) {
        // Auto-fill article fields
        document.getElementById('content_article_title').value = selectedOption.dataset.title || '';
        document.getElementById('content_article_doi').value = selectedOption.dataset.doi || '';
        document.getElementById('content_submission_id').value = selectedOption.dataset.submissionId || '';

        // Get the full article data from cache
        const submissionId = selectedOption.value;
        const articleData = window.ojsArticlesData ? window.ojsArticlesData[submissionId] : null;

        // Populate author dropdown
        const authorSelect = document.getElementById('content_author_name');
        if (articleData && articleData.authors && Array.isArray(articleData.authors)) {
            console.log('Populating authors dropdown:', articleData.authors);

            authorSelect.innerHTML = '<option value="">-- Select an Author --</option>';

            if (articleData.authors.length === 0) {
                authorSelect.innerHTML = '<option value="">-- No authors found --</option>';
                authorSelect.disabled = true;
            } else {
                articleData.authors.forEach(author => {
                    const option = document.createElement('option');
                    option.value = author.name;
                    option.textContent = author.name + (author.affiliation ? ' (' + author.affiliation + ')' : '');
                    option.dataset.authorId = author.id;
                    option.dataset.email = author.email;
                    authorSelect.appendChild(option);
                });
                authorSelect.disabled = false;

                console.log(`Populated ${articleData.authors.length} authors`);
            }
        } else {
            authorSelect.innerHTML = '<option value="">-- No authors data --</option>';
            authorSelect.disabled = true;
        }

        // Load reviewers and editors for this specific article
        loadReviewersForSubmission(submissionId);
        loadEditorsForSubmission(submissionId);

        // Auto-refresh canvas with new article data
        updateCanvasWithContentData();
    } else {
        clearArticleFields();

        // Clear and disable reviewer/author/editor dropdowns
        const reviewerSelect = document.getElementById('content_reviewer_name');
        const authorSelect = document.getElementById('content_author_name');
        const editorSelect = document.getElementById('content_editor_name');

        reviewerSelect.innerHTML = '<option value="">-- Select article first --</option>';
        authorSelect.innerHTML = '<option value="">-- Select article first --</option>';
        editorSelect.innerHTML = '<option value="">-- Select article first --</option>';
        reviewerSelect.disabled = true;
        authorSelect.disabled = true;
        editorSelect.disabled = true;

        // Refresh canvas to clear article data
        updateCanvasWithContentData();
    }
}

function clearArticleFields() {
    document.getElementById('content_article_title').value = '';
    document.getElementById('content_article_doi').value = '';
    document.getElementById('content_submission_id').value = '';
}

/**
 * Load reviewers who reviewed a specific submission
 */
function loadReviewersForSubmission(submissionId) {
    console.log('Loading reviewers for submission:', submissionId);

    const reviewerSelect = document.getElementById('content_reviewer_name');
    reviewerSelect.innerHTML = '<option value="">-- Loading reviewers... --</option>';
    reviewerSelect.disabled = true;

    const url = buildActionURL('getReviewersForSubmission') + '&submissionId=' + encodeURIComponent(submissionId);

    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'
    })
        .then(response => response.json())
        .then(data => {
            console.log('Reviewers response:', data);

            const reviewers = data.content && data.content.reviewers ? data.content.reviewers : [];

            // Populate dropdown
            reviewerSelect.innerHTML = '<option value="">-- Select a Reviewer --</option>';

            if (reviewers.length === 0) {
                reviewerSelect.innerHTML = '<option value="">-- No reviewers found for this article --</option>';
                reviewerSelect.disabled = true;
            } else {
                reviewers.forEach(reviewer => {
                    const option = document.createElement('option');
                    option.value = reviewer.name;
                    option.textContent = reviewer.name + (reviewer.affiliation ? ' (' + reviewer.affiliation + ')' : '');
                    option.dataset.reviewerId = reviewer.id;
                    option.dataset.email = reviewer.email;
                    reviewerSelect.appendChild(option);
                });
                reviewerSelect.disabled = false;

                console.log(`Populated ${reviewers.length} reviewers`);
            }
        })
        .catch(error => {
            console.error('Error loading reviewers:', error);
            reviewerSelect.innerHTML = '<option value="">-- Error loading reviewers --</option>';
            reviewerSelect.disabled = true;
        });
}

/**
 * Load editors who handled a specific submission
 */
function loadEditorsForSubmission(submissionId) {
    console.log('Loading editors for submission:', submissionId);

    const editorSelect = document.getElementById('content_editor_name');
    editorSelect.innerHTML = '<option value="">-- Loading editors... --</option>';
    editorSelect.disabled = true;

    const url = buildActionURL('getEditorsForSubmission') + '&submissionId=' + encodeURIComponent(submissionId);

    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
        },
        credentials: 'same-origin'
    })
        .then(response => response.json())
        .then(data => {
            console.log('Editors response:', data);

            const editors = data.content && data.content.editors ? data.content.editors : [];

            // Populate dropdown
            editorSelect.innerHTML = '<option value="">-- Select an Editor --</option>';

            if (editors.length === 0) {
                editorSelect.innerHTML = '<option value="">-- No editors found for this article --</option>';
                editorSelect.disabled = true;
            } else {
                editors.forEach(editor => {
                    const option = document.createElement('option');
                    option.value = editor.name;
                    option.textContent = editor.name + (editor.affiliation ? ' (' + editor.affiliation + ')' : '');
                    option.dataset.editorId = editor.id;
                    option.dataset.email = editor.email;
                    editorSelect.appendChild(option);
                });
                editorSelect.disabled = false;

                console.log(`Populated ${editors.length} editors`);
            }
        })
        .catch(error => {
            console.error('Error loading editors:', error);
            editorSelect.innerHTML = '<option value="">-- Error loading editors --</option>';
            editorSelect.disabled = true;
        });
}

/**
 * Helper function to escape HTML to prevent XSS
 */
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Show success feedback after finding a submission
 */
function showSearchSuccess(submission) {
    // Create or get the feedback element
    let feedbackDiv = document.getElementById('submission_search_feedback');
    if (!feedbackDiv) {
        feedbackDiv = document.createElement('div');
        feedbackDiv.id = 'submission_search_feedback';
        const searchField = document.getElementById('submission_id_search').parentElement.parentElement;
        searchField.appendChild(feedbackDiv);
    }

    // Build status info
    const stageInfo = submission.stageName ? submission.stageName : '';
    const statusInfo = submission.statusName ? submission.statusName : '';
    const authorCount = submission.authors ? submission.authors.length : 0;
    const reviewerCount = submission.reviewers ? submission.reviewers.length : 0;

    // Show success message with submission details
    feedbackDiv.innerHTML = `
        <div style="margin-top: 8px; padding: 10px; background: linear-gradient(135deg, #d4edda, #c3e6cb); border: 1px solid #28a745; border-radius: 6px; font-size: 12px;">
            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 6px;">
                <i class="fa fa-check-circle" style="color: #28a745; font-size: 16px;"></i>
                <strong style="color: #155724;">Submission Found!</strong>
            </div>
            <div style="color: #155724; line-height: 1.5;">
                <div><strong>Title:</strong> ${escapeHtml(submission.title) || 'Untitled'}</div>
                <div><strong>Stage:</strong> ${escapeHtml(stageInfo)} | <strong>Status:</strong> ${escapeHtml(statusInfo)}</div>
                <div><strong>Authors:</strong> ${authorCount} | <strong>Reviewers:</strong> ${reviewerCount}</div>
            </div>
        </div>
    `;

    // Auto-hide after 5 seconds
    setTimeout(() => {
        feedbackDiv.innerHTML = '';
    }, 5000);
}

/**
 * Search for a submission by ID (works for unpublished submissions too)
 */
function searchSubmissionById() {
    const submissionIdInput = document.getElementById('submission_id_search');
    const submissionId = submissionIdInput ? submissionIdInput.value.trim() : '';

    if (!submissionId) {
        alert('Please enter a Submission ID');
        return;
    }

    // Show loading state
    const searchBtn = document.getElementById('search_submission_btn');
    const originalBtnText = searchBtn.innerHTML;
    searchBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Searching...';
    searchBtn.disabled = true;

    const url = buildActionURL('getSubmissionById') + '&submissionId=' + encodeURIComponent(submissionId);

    fetch(url)
        .then(response => response.json())
        .then(data => {
            searchBtn.innerHTML = originalBtnText;
            searchBtn.disabled = false;

            if (data.success && data.submission) {
                populateFormFromSubmission(data.submission);
                // Show success feedback
                showSearchSuccess(data.submission);
            } else {
                alert(data.message || 'Submission not found');
            }
        })
        .catch(error => {
            console.error('Error searching submission:', error);
            searchBtn.innerHTML = originalBtnText;
            searchBtn.disabled = false;
            alert('Error searching for submission. Please try again.');
        });
}

/**
 * Populate form fields from submission data returned by getSubmissionById
 */
function populateFormFromSubmission(submission) {
    // Clear and populate issue dropdown if submission has an issue
    const issueSelect = document.getElementById('content_issue_select');
    if (submission.issue) {
        // Add the issue as an option and select it
        const issueOption = document.createElement('option');
        issueOption.value = submission.issue.id;
        issueOption.textContent = submission.issue.title;
        issueOption.selected = true;

        // Check if this issue is already in the dropdown
        let found = false;
        for (let i = 0; i < issueSelect.options.length; i++) {
            if (issueSelect.options[i].value == submission.issue.id) {
                issueSelect.options[i].selected = true;
                found = true;
                break;
            }
        }
        if (!found) {
            issueSelect.appendChild(issueOption);
        }

        // Update issue fields
        document.getElementById('content_issue_volume').value = submission.issue.volume || '';
        document.getElementById('content_issue_number').value = submission.issue.number || '';
        document.getElementById('content_issue_year').value = submission.issue.year || '';
    }

    // Clear and populate article dropdown
    const articleSelect = document.getElementById('content_article_select');
    articleSelect.innerHTML = '<option value="">-- Select an Article --</option>';
    const articleOption = document.createElement('option');
    articleOption.value = submission.id;
    articleOption.textContent = submission.title || 'Untitled (ID: ' + submission.id + ')';
    articleOption.selected = true;
    articleSelect.appendChild(articleOption);

    // Store submission data for later use
    articleSelect.dataset.currentSubmission = JSON.stringify(submission);

    // Update article fields
    document.getElementById('content_article_title').value = submission.title || '';
    document.getElementById('content_article_doi').value = submission.doi || '';
    document.getElementById('content_submission_id').value = submission.id || '';

    // Show stage/status info
    const stageInfo = submission.stageName ? ' [' + submission.stageName + ']' : '';
    const statusInfo = submission.statusName ? ' (' + submission.statusName + ')' : '';
    console.log('Loaded submission: ' + submission.title + stageInfo + statusInfo);

    // Populate authors dropdown
    const authorSelect = document.getElementById('content_author_name');
    authorSelect.innerHTML = '<option value="">-- Select Author --</option>';
    authorSelect.disabled = false;
    if (submission.authors && submission.authors.length > 0) {
        submission.authors.forEach(author => {
            const option = document.createElement('option');
            option.value = author.name;
            option.textContent = author.name;
            option.dataset.email = author.email || '';
            option.dataset.affiliation = author.affiliation || '';
            authorSelect.appendChild(option);
        });
    }

    // Populate reviewers dropdown
    const reviewerSelect = document.getElementById('content_reviewer_name');
    reviewerSelect.innerHTML = '<option value="">-- Select Reviewer --</option>';
    reviewerSelect.disabled = false;
    if (submission.reviewers && submission.reviewers.length > 0) {
        submission.reviewers.forEach(reviewer => {
            const option = document.createElement('option');
            option.value = reviewer.name;
            option.textContent = reviewer.name;
            option.dataset.email = reviewer.email || '';
            reviewerSelect.appendChild(option);
        });
    } else {
        reviewerSelect.innerHTML = '<option value="">-- No reviewers assigned --</option>';
    }

    // Populate editors dropdown
    const editorSelect = document.getElementById('content_editor_name');
    editorSelect.innerHTML = '<option value="">-- Select Editor --</option>';
    editorSelect.disabled = false;
    if (submission.editors && submission.editors.length > 0) {
        submission.editors.forEach(editor => {
            const option = document.createElement('option');
            option.value = editor.name;
            option.textContent = editor.name;
            option.dataset.email = editor.email || '';
            editorSelect.appendChild(option);
        });
    } else {
        editorSelect.innerHTML = '<option value="">-- No editors assigned --</option>';
    }
}

// Make searchSubmissionById available globally
window.searchSubmissionById = searchSubmissionById;

function initializeDropdowns() {
    // Set up event listeners for dropdowns
    const issueSelect = document.getElementById('content_issue_select');
    const articleSelect = document.getElementById('content_article_select');

    if (issueSelect) {
        issueSelect.addEventListener('change', handleIssueSelectionChange);
    }

    if (articleSelect) {
        articleSelect.addEventListener('change', handleArticleSelectionChange);
    }

    // Initialize reviewer/author/editor dropdowns as disabled until article is selected
    const reviewerSelect = document.getElementById('content_reviewer_name');
    const authorSelect = document.getElementById('content_author_name');
    const editorSelect = document.getElementById('content_editor_name');

    if (reviewerSelect) {
        reviewerSelect.disabled = true;
        reviewerSelect.innerHTML = '<option value="">-- Select article first --</option>';
    }

    if (authorSelect) {
        authorSelect.disabled = true;
        authorSelect.innerHTML = '<option value="">-- Select article first --</option>';
    }

    if (editorSelect) {
        editorSelect.disabled = true;
        editorSelect.innerHTML = '<option value="">-- Select article first --</option>';
    }
}

// PIN Verification functionality (replaces OTP)
function setupOTPVerification() {
    const signedByToggle = document.getElementById('qr_signed_by_toggle');
    const signedBySection = document.getElementById('signedBySection');
    const pinNotSet = document.getElementById('pinNotSet');
    const pinInputSection = document.getElementById('pinInputSection');
    const pinLocked = document.getElementById('pinLocked');
    const pinSuccess = document.getElementById('pinSuccess');
    const verifyPinBtn = document.getElementById('verifyPinBtn');
    const pinVerifyStatus = document.getElementById('pinVerifyStatus');

    // PIN input fields
    const pinInputs = [
        document.getElementById('pin-1'),
        document.getElementById('pin-2'),
        document.getElementById('pin-3'),
        document.getElementById('pin-4'),
        document.getElementById('pin-5'),
        document.getElementById('pin-6')
    ];

    let pinVerified = false;

    // Check PIN status when signed by is enabled
    async function checkPINStatus() {
        if (!goValidAuth.isAuthenticated) return;

        try {
            const response = await fetch(buildActionURL('pinStatus') + '&token=' + encodeURIComponent(localStorage.getItem('govalid_access_token')), {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin'
            });

            const data = await response.json();
            const result = data.content || data;

            // Hide all sections first
            if (pinNotSet) pinNotSet.style.display = 'none';
            if (pinInputSection) pinInputSection.style.display = 'none';
            if (pinLocked) pinLocked.style.display = 'none';
            if (pinSuccess) pinSuccess.style.display = 'none';

            if (!result.has_pin) {
                if (pinNotSet) pinNotSet.style.display = 'block';
            } else if (result.is_locked) {
                if (pinLocked) pinLocked.style.display = 'block';
            } else {
                if (pinInputSection) pinInputSection.style.display = 'block';
                if (pinInputs[0]) pinInputs[0].focus();
            }
        } catch (error) {
            console.error('Error checking PIN status:', error);
            if (pinInputSection) pinInputSection.style.display = 'block';
        }
    }

    // Verify PIN
    async function verifyPIN() {
        const pinCode = pinInputs.map(input => input ? input.value : '').join('');

        if (pinCode.length !== 6) {
            if (pinVerifyStatus) {
                pinVerifyStatus.innerHTML = '<div style="background: #fef2f2; border: 1px solid #dc3545; border-radius: 4px; padding: 6px; color: #dc3545; font-size: 11px;"><i class="fa fa-exclamation-triangle"></i> Please enter all 6 digits</div>';
            }
            return;
        }

        if (verifyPinBtn) {
            verifyPinBtn.disabled = true;
            verifyPinBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Verifying...';
        }

        try {
            const response = await fetch(buildActionURL('verifyPin') + '&token=' + encodeURIComponent(localStorage.getItem('govalid_access_token')) + '&pin=' + encodeURIComponent(pinCode), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin'
            });

            const data = await response.json();
            const result = data.content || data;

            // Debug: log the full response structure
            console.log('PIN verification full response:', JSON.stringify(data, null, 2));
            console.log('PIN verification result:', JSON.stringify(result, null, 2));

            if (response.ok && result.success) {
                pinVerified = true;

                // Extract qr_session_token from the backend response
                // The backend now returns a proper server-generated token
                const sessionToken = data.qr_session_token ||
                                    data.session_token ||
                                    result.qr_session_token ||
                                    result.session_token ||
                                    (data.content && data.content.qr_session_token) ||
                                    null;

                if (sessionToken) {
                    window.ojsSessionToken = sessionToken;
                    console.log('PIN verified - using backend qr_session_token:', sessionToken.substring(0, 20) + '...');
                } else {
                    // This should not happen with updated backend
                    console.warn('WARNING: No qr_session_token in PIN verification response. Signed certificates may not work.');
                    window.ojsSessionToken = null;
                }

                // Hide input, show success
                if (pinInputSection) pinInputSection.style.display = 'none';
                if (pinSuccess) pinSuccess.style.display = 'block';

                if (pinVerifyStatus) {
                    pinVerifyStatus.innerHTML = '<div style="background: #f0fdf4; border: 1px solid #10b981; border-radius: 4px; padding: 6px; color: #10b981; font-size: 11px;"><i class="fa fa-check-circle"></i> PIN verified!</div>';
                }

                console.log('PIN verified successfully for certificate signing');
            } else {
                // Show error
                const errorMsg = result.error || 'Invalid PIN';
                if (pinVerifyStatus) {
                    pinVerifyStatus.innerHTML = '<div style="background: #fef2f2; border: 1px solid #dc3545; border-radius: 4px; padding: 6px; color: #dc3545; font-size: 11px;"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>';
                }

                // Check if locked
                if (result.is_locked) {
                    if (pinInputSection) pinInputSection.style.display = 'none';
                    if (pinLocked) pinLocked.style.display = 'block';
                }

                // Clear inputs
                pinInputs.forEach(input => {
                    if (input) input.value = '';
                });
                if (pinInputs[0]) pinInputs[0].focus();

                if (verifyPinBtn) {
                    verifyPinBtn.disabled = false;
                    verifyPinBtn.innerHTML = '<i class="fa fa-check"></i> Verify PIN';
                }
            }
        } catch (error) {
            console.error('Error verifying PIN:', error);
            if (pinVerifyStatus) {
                pinVerifyStatus.innerHTML = '<div style="background: #fef2f2; border: 1px solid #dc3545; border-radius: 4px; padding: 6px; color: #dc3545; font-size: 11px;"><i class="fa fa-exclamation-triangle"></i> Network error</div>';
            }
            if (verifyPinBtn) {
                verifyPinBtn.disabled = false;
                verifyPinBtn.innerHTML = '<i class="fa fa-check"></i> Verify PIN';
            }
        }
    }

    // Reset PIN state
    function resetPINState() {
        pinVerified = false;
        if (pinNotSet) pinNotSet.style.display = 'none';
        if (pinInputSection) pinInputSection.style.display = 'none';
        if (pinLocked) pinLocked.style.display = 'none';
        if (pinSuccess) pinSuccess.style.display = 'none';
        if (pinVerifyStatus) pinVerifyStatus.innerHTML = '';
        pinInputs.forEach(input => {
            if (input) input.value = '';
        });
        if (verifyPinBtn) {
            verifyPinBtn.disabled = false;
            verifyPinBtn.innerHTML = '<i class="fa fa-check"></i> Verify PIN';
        }
    }

    // Toggle Signed By section when checkbox is changed
    if (signedByToggle) {
        // Check initial state on page load
        if (signedByToggle.checked) {
            signedBySection.style.display = 'block';
            // Auto-fill the signed by field with authenticated user name if available
            if (goValidAuth.isAuthenticated && goValidAuth.userInfo && goValidAuth.userInfo.name) {
                document.getElementById('qr_signed_by').value = goValidAuth.userInfo.name;
            }
        } else {
            signedBySection.style.display = 'none';
        }

        // Set up event listener for future changes
        signedByToggle.addEventListener('change', function () {
            if (this.checked) {
                signedBySection.style.display = 'block';
                // Auto-fill the signed by field with authenticated user name if available
                if (goValidAuth.isAuthenticated && goValidAuth.userInfo && goValidAuth.userInfo.name) {
                    document.getElementById('qr_signed_by').value = goValidAuth.userInfo.name;
                }
                // Check PIN status
                checkPINStatus();
            } else {
                signedBySection.style.display = 'none';
                resetPINState();
            }
        });

        // If already checked on load, check PIN status
        if (signedByToggle.checked && goValidAuth.isAuthenticated) {
            checkPINStatus();
        }
    }

    // PIN input handling - auto-advance to next field
    pinInputs.forEach((input, index) => {
        if (input) {
            input.addEventListener('input', function () {
                if (this.value.length === 1) {
                    if (index < pinInputs.length - 1) {
                        pinInputs[index + 1].focus();
                    }
                    // Auto-verify when all inputs are filled
                    if (index === pinInputs.length - 1) {
                        const pinCode = pinInputs.map(inp => inp.value).join('');
                        if (pinCode.length === 6) {
                            setTimeout(() => verifyPIN(), 100);
                        }
                    }
                }
            });

            input.addEventListener('keydown', function (e) {
                if (e.key === 'Backspace' && !this.value && index > 0) {
                    pinInputs[index - 1].focus();
                }
            });
        }
    });

    // Verify button click handler
    if (verifyPinBtn) {
        verifyPinBtn.addEventListener('click', verifyPIN);
    }

    // Expose PIN verification status for form validation
    window.pinVerificationStatus = {
        isSignedByEnabled: () => signedByToggle && signedByToggle.checked,
        isPinVerified: () => pinVerified
    };

    // Keep old name for backward compatibility
    window.otpVerificationStatus = window.pinVerificationStatus;

    // Expose function to reset PIN verification state (called after QR generation)
    window.resetPINVerificationState = function() {
        pinVerified = false;
        // Hide success message and show PIN input section again
        if (pinSuccess) pinSuccess.style.display = 'none';
        if (pinInputSection) pinInputSection.style.display = 'block';
        if (pinVerifyStatus) pinVerifyStatus.innerHTML = '';
        // Clear PIN inputs
        pinInputs.forEach(input => {
            if (input) input.value = '';
        });
        // Reset verify button
        if (verifyPinBtn) {
            verifyPinBtn.disabled = false;
            verifyPinBtn.innerHTML = '<i class="fa fa-check"></i> Verify PIN';
        }
        // Clear session token
        window.ojsSessionToken = null;
        console.log('PIN verification state reset - user must verify PIN again for next QR generation');
    };
}

// Expose setupContentTab to global scope
window.setupContentTab = setupContentTab;

