<?php

/**
 * @file plugins/generic/goValidOJS/classes/CertificateGenerator.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class CertificateGenerator
 *
 * Generates certificates in PDF or JPG format for email attachments
 */


class CertificateGenerator
{
    private $plugin;
    private $context;

    /**
     * Constructor
     */
    public function __construct($plugin, $context)
    {
        $this->plugin = $plugin;
        $this->context = $context;
    }

    /**
     * Generate PDF certificate
     * @param array $canvasData The canvas JSON data
     * @param array $replacements Key-value pairs for placeholder replacement
     * @param string $orientation 'portrait' or 'landscape'
     * @return string PDF binary data
     */
    public function generatePDF($canvasData, $replacements = [], $orientation = 'portrait')
    {
        require_once($this->plugin->getPluginPath() . '/classes/CertificatePDFGenerator.php');

        $pdfGenerator = new CertificatePDFGenerator($this->plugin);

        // Convert canvas array to JSON string if needed
        $canvasJson = is_array($canvasData) ? json_encode($canvasData) : $canvasData;

        // Generate PDF and return binary content
        return $pdfGenerator->generatePDF($canvasJson, $replacements, $orientation);
    }

    /**
     * Generate JPG certificate
     * @param array $canvasData The canvas JSON data
     * @param array $replacements Key-value pairs for placeholder replacement
     * @param string $orientation 'portrait' or 'landscape'
     * @return string JPG binary data
     */
    public function generateJPG($canvasData, $replacements = [], $orientation = 'portrait')
    {
        // For JPG, we'll first generate PDF then convert to image
        // This is the most reliable approach given the existing infrastructure

        try {
            // Generate PDF first
            $pdfData = $this->generatePDF($canvasData, $replacements, $orientation);

            if (!$pdfData) {
                throw new Exception('Failed to generate PDF for JPG conversion');
            }

            // Check if Imagick is available
            if (class_exists('Imagick')) {
                return $this->convertPDFToJPGWithImagick($pdfData);
            }

            // Fallback: Check if GD and Ghostscript are available
            if (function_exists('imagecreatefromstring')) {
                return $this->convertPDFToJPGWithGD($pdfData);
            }

            // If no conversion method available, throw error
            throw new Exception('No image conversion library available (Imagick or GD required)');

        } catch (Exception $e) {
            error_log('CertificateGenerator: Error generating JPG - ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Convert PDF to JPG using Imagick
     */
    private function convertPDFToJPGWithImagick($pdfData)
    {
        try {
            $imagick = new \Imagick();
            $imagick->readImageBlob($pdfData);
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompression(\Imagick::COMPRESSION_JPEG);
            $imagick->setImageCompressionQuality(90);

            // Set resolution for better quality
            $imagick->setResolution(300, 300);

            // Get the first page (index 0)
            $imagick->setIteratorIndex(0);

            $jpgData = $imagick->getImageBlob();
            $imagick->clear();
            $imagick->destroy();

            return $jpgData;

        } catch (Exception $e) {
            error_log('CertificateGenerator: Imagick conversion failed - ' . $e->getMessage());
            throw new Exception('Failed to convert PDF to JPG using Imagick: ' . $e->getMessage());
        }
    }

    /**
     * Convert PDF to JPG using GD (requires Ghostscript)
     */
    private function convertPDFToJPGWithGD($pdfData)
    {
        try {
            // Save PDF to temporary file
            $tempPdfFile = tempnam(sys_get_temp_dir(), 'cert_pdf_');
            $tempJpgFile = tempnam(sys_get_temp_dir(), 'cert_jpg_') . '.jpg';

            file_put_contents($tempPdfFile, $pdfData);

            // Use Ghostscript to convert PDF to JPG
            $gsCommand = sprintf(
                'gs -dSAFER -dBATCH -dNOPAUSE -sDEVICE=jpeg -r300 -sOutputFile=%s %s 2>&1',
                escapeshellarg($tempJpgFile),
                escapeshellarg($tempPdfFile)
            );

            exec($gsCommand, $output, $returnCode);

            if ($returnCode !== 0 || !file_exists($tempJpgFile)) {
                // Cleanup
                @unlink($tempPdfFile);
                @unlink($tempJpgFile);

                throw new Exception('Ghostscript conversion failed. Return code: ' . $returnCode);
            }

            // Read JPG data
            $jpgData = file_get_contents($tempJpgFile);

            // Cleanup
            @unlink($tempPdfFile);
            @unlink($tempJpgFile);

            if (!$jpgData) {
                throw new Exception('Failed to read converted JPG file');
            }

            return $jpgData;

        } catch (Exception $e) {
            // Cleanup on error
            if (isset($tempPdfFile)) @unlink($tempPdfFile);
            if (isset($tempJpgFile)) @unlink($tempJpgFile);

            error_log('CertificateGenerator: GD/Ghostscript conversion failed - ' . $e->getMessage());
            throw new Exception('Failed to convert PDF to JPG using GD/Ghostscript: ' . $e->getMessage());
        }
    }
}
