<?php

/**
 * @file plugins/generic/goValidOJS/classes/CertificatePDFGenerator.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class CertificatePDFGenerator
 *
 * Generates PDF certificates from canvas data
 */


class CertificatePDFGenerator
{
    private $plugin;
    private $fpdf;
    
    /**
     * Constructor
     */
    public function __construct($plugin)
    {
        $this->plugin = $plugin;
        // We'll use FPDF - a lightweight PDF library
        require_once($plugin->getPluginPath() . '/lib/fpdf/fpdf.php');
    }
    
    /**
     * Generate PDF from canvas data
     * @param array $canvasData The canvas JSON data
     * @param array $replacements Key-value pairs for placeholder replacement
     * @param string $orientation 'portrait' or 'landscape'
     * @return string PDF content
     */
    public function generatePDF($canvasData, $replacements = [], $orientation = 'portrait')
    {
        // Create new PDF
        $pdfOrientation = $orientation === 'portrait' ? 'P' : 'L';
        $this->fpdf = new \FPDF($pdfOrientation, 'mm', 'A4');
        $this->fpdf->AddPage();
        
        // Set default font
        $this->fpdf->SetFont('Arial', '', 12);
        
        // Parse canvas data
        $objects = json_decode($canvasData, true);
        
        // Debug canvas data structure
        error_log('PDF Generator - Canvas data keys: ' . json_encode(array_keys($objects ?: [])));
        
        // Check if there are any objects at all
        if (isset($objects['objects']) && is_array($objects['objects'])) {
            error_log('PDF Generator - Found ' . count($objects['objects']) . ' objects in canvas');
            foreach ($objects['objects'] as $i => $obj) {
                if (isset($obj['type']) && $obj['type'] === 'image') {
                    error_log("PDF Generator - Object $i is image: " . json_encode([
                        'type' => $obj['type'],
                        'left' => $obj['left'] ?? 'not set',
                        'top' => $obj['top'] ?? 'not set', 
                        'width' => $obj['width'] ?? 'not set',
                        'height' => $obj['height'] ?? 'not set',
                        'scaleX' => $obj['scaleX'] ?? 'not set',
                        'scaleY' => $obj['scaleY'] ?? 'not set',
                        'src_preview' => isset($obj['src']) ? substr($obj['src'], 0, 50) . '...' : 'no src',
                        'isBackground' => $obj['isBackground'] ?? 'not set'
                    ]));
                }
            }
        } else {
            error_log('PDF Generator - No objects array found in canvas data');
        }
        
        // Handle canvas background image first - check multiple possible locations
        $backgroundRendered = false;
        
        // Method 1: Check backgroundImage property
        if (isset($objects['backgroundImage']) && isset($objects['backgroundImage']['src'])) {
            error_log('PDF Generator - Found backgroundImage property');
            $this->renderCanvasBackground($objects['backgroundImage'], $orientation);
            $backgroundRendered = true;
        }
        
        // Method 2: Check if background is stored as a regular object
        if (!$backgroundRendered && isset($objects['objects'])) {
            foreach ($objects['objects'] as $index => $object) {
                if (isset($object['type']) && $object['type'] === 'image' && 
                    isset($object['isBackground']) && $object['isBackground']) {
                    error_log('PDF Generator - Found background image in objects array at index ' . $index);
                    $this->renderCanvasBackgroundFromObject($object, $orientation);
                    $backgroundRendered = true;
                    break;
                }
            }
        }
        
        // Method 3: Check for any large image that could be a background
        if (!$backgroundRendered && isset($objects['objects'])) {
            foreach ($objects['objects'] as $index => $object) {
                if (isset($object['type']) && $object['type'] === 'image' && 
                    isset($object['width']) && isset($object['height'])) {
                    // Check if it's a large image (likely background)
                    $scaledWidth = $object['width'] * ($object['scaleX'] ?? 1);
                    $scaledHeight = $object['height'] * ($object['scaleY'] ?? 1);
                    if ($scaledWidth > 500 && $scaledHeight > 400) { // Likely a background
                        error_log('PDF Generator - Found potential background image (large image) at index ' . $index . " - size: {$scaledWidth}x{$scaledHeight}");
                        $this->renderCanvasBackgroundFromObject($object, $orientation);
                        $backgroundRendered = true;
                        break;
                    }
                }
            }
        }
        
        if (!$backgroundRendered) {
            error_log('PDF Generator - No background image found in canvas data');
        }
        
        if (isset($objects['objects'])) {
            foreach ($objects['objects'] as $index => $object) {
                // Skip background images since they're already rendered
                if (isset($object['type']) && $object['type'] === 'image') {
                    // Skip if it's marked as background
                    if (isset($object['isBackground']) && $object['isBackground']) {
                        error_log("PDF Generator - Skipping background image (marked) at index $index");
                        continue;
                    }
                    // Skip if it looks like a background (large image)
                    if (isset($object['width']) && isset($object['height'])) {
                        $scaledWidth = $object['width'] * ($object['scaleX'] ?? 1);
                        $scaledHeight = $object['height'] * ($object['scaleY'] ?? 1);
                        if ($scaledWidth > 500 && $scaledHeight > 400) {
                            error_log("PDF Generator - Skipping potential background image at index $index - size: {$scaledWidth}x{$scaledHeight}");
                            continue; // Skip background images
                        }
                    }
                }
                
                $this->renderObject($object, $replacements);
            }
        }
        
        // Output PDF
        return $this->fpdf->Output('S'); // Return as string
    }
    
    /**
     * Render a single canvas object to PDF
     */
    private function renderObject($object, $replacements)
    {
        $type = $object['type'] ?? 'unknown';

        switch ($type) {
            case 'text':
            case 'i-text':
            case 'textbox':
                $this->renderText($object, $replacements);
                break;
            case 'image':
                $this->renderImage($object);
                break;
            case 'rect':
                $this->renderRectangle($object);
                break;
            case 'circle':
                $this->renderCircle($object);
                break;
            case 'ellipse':
                $this->renderEllipse($object);
                break;
            case 'line':
                $this->renderLine($object);
                break;
            case 'polygon':
            case 'polyline':
                $this->renderPolygon($object);
                break;
            case 'path':
                $this->renderPath($object);
                break;
            case 'group':
                $this->renderGroup($object, $replacements);
                break;
            default:
                error_log("PDF Generator - Unsupported object type: {$type}");
                break;
        }
    }
    
    /**
     * Render text object
     */
    private function renderText($object, $replacements)
    {
        $text = $object['text'];
        
        // Check if this is a QR code placeholder
        if (trim($text) === '{QR_CODE}') {
            error_log('PDF Generator - Found QR_CODE placeholder in text object');
            if (isset($replacements['{QR_CODE}'])) {
                error_log('PDF Generator - QR_CODE replacement found: ' . substr($replacements['{QR_CODE}'], 0, 50) . '...');
                if (strpos($replacements['{QR_CODE}'], 'data:image') === 0) {
                    error_log('PDF Generator - Converting QR_CODE text to image');
                    // Handle as QR code image
                    $qrObject = $object;
                    $qrObject['src'] = $replacements['{QR_CODE}'];
                    $qrObject['width'] = 150; // Default QR code size
                    $qrObject['height'] = 150;
                    $qrObject['scaleX'] = 1;
                    $qrObject['scaleY'] = 1;
                    $this->renderImage($qrObject);
                    return;
                } else {
                    error_log('PDF Generator - QR_CODE replacement is not image data: ' . $replacements['{QR_CODE}']);
                }
            } else {
                error_log('PDF Generator - No QR_CODE replacement found in replacements array');
                error_log('PDF Generator - Available replacements: ' . json_encode(array_keys($replacements)));
            }
        }
        
        // Replace placeholders
        foreach ($replacements as $key => $value) {
            if (strpos($value, 'data:image') !== 0) { // Don't replace image data
                $text = str_replace($key, $value, $text);
            }
        }
        
        // Convert canvas coordinates to PDF coordinates (72 DPI to mm)
        $x = $this->pixelToMm($object['left']);
        $y = $this->pixelToMm($object['top']);
        
        // Set font properties
        $fontSize = isset($object['fontSize']) ? $object['fontSize'] : 12;
        $fontStyle = '';
        if (isset($object['fontWeight']) && $object['fontWeight'] === 'bold') {
            $fontStyle .= 'B';
        }
        if (isset($object['fontStyle']) && $object['fontStyle'] === 'italic') {
            $fontStyle .= 'I';
        }
        
        $this->fpdf->SetFont('Arial', $fontStyle, $fontSize);
        
        // Set text color
        if (isset($object['fill'])) {
            $color = $this->hexToRgb($object['fill']);
            $this->fpdf->SetTextColor($color['r'], $color['g'], $color['b']);
        }
        
        // Draw text
        $this->fpdf->SetXY($x, $y);
        $this->fpdf->Cell(0, 0, $text);
        
        // Reset color
        $this->fpdf->SetTextColor(0, 0, 0);
    }
    
    /**
     * Render image object
     */
    private function renderImage($object)
    {
        if (!isset($object['src'])) {
            return;
        }

        $x = $this->pixelToMm($object['left']);
        $y = $this->pixelToMm($object['top']);
        $width = $this->pixelToMm($object['width'] * $object['scaleX']);
        $height = $this->pixelToMm($object['height'] * $object['scaleY']);

        // Handle base64 images
        if (strpos($object['src'], 'data:image') === 0) {
            // Extract base64 data
            $data = explode(',', $object['src']);
            $imageData = base64_decode($data[1]);

            // Save to temp file
            $tempFile = tempnam(sys_get_temp_dir(), 'cert_img_');
            file_put_contents($tempFile, $imageData);

            // Detect image type
            $imageInfo = getimagesize($tempFile);
            $type = $imageInfo[2];

            // Optimize/compress image to reduce PDF size
            $optimizedFile = $this->optimizeImage($tempFile, $type);

            if ($type == IMAGETYPE_JPEG) {
                $this->fpdf->Image($optimizedFile, $x, $y, $width, $height, 'JPG');
            } elseif ($type == IMAGETYPE_PNG) {
                $this->fpdf->Image($optimizedFile, $x, $y, $width, $height, 'PNG');
            }

            if ($optimizedFile !== $tempFile) {
                unlink($optimizedFile);
            }
            unlink($tempFile);
        } else {
            // Regular image URL
            $this->fpdf->Image($object['src'], $x, $y, $width, $height);
        }
    }

    /**
     * Optimize image to reduce file size
     * Returns path to optimized image
     */
    private function optimizeImage($imagePath, $imageType)
    {
        // Check if GD is available
        if (!function_exists('imagecreatefromjpeg')) {
            return $imagePath; // Return original if GD not available
        }

        try {
            $image = null;

            // Load image based on type
            switch ($imageType) {
                case IMAGETYPE_JPEG:
                    $image = imagecreatefromjpeg($imagePath);
                    break;
                case IMAGETYPE_PNG:
                    $image = imagecreatefrompng($imagePath);
                    break;
                default:
                    return $imagePath; // Unsupported type, return original
            }

            if (!$image) {
                return $imagePath;
            }

            // Get original dimensions
            $originalWidth = imagesx($image);
            $originalHeight = imagesy($image);

            // Max dimensions for PDF (reduce very large images)
            $maxWidth = 2000;  // Reasonable max width
            $maxHeight = 2000; // Reasonable max height

            // Calculate new dimensions if image is too large
            $newWidth = $originalWidth;
            $newHeight = $originalHeight;

            if ($originalWidth > $maxWidth || $originalHeight > $maxHeight) {
                $ratio = min($maxWidth / $originalWidth, $maxHeight / $originalHeight);
                $newWidth = (int)($originalWidth * $ratio);
                $newHeight = (int)($originalHeight * $ratio);

                // Create resized image
                $resizedImage = imagecreatetruecolor($newWidth, $newHeight);

                // Preserve transparency for PNG
                if ($imageType == IMAGETYPE_PNG) {
                    imagealphablending($resizedImage, false);
                    imagesavealpha($resizedImage, true);
                    $transparent = imagecolorallocatealpha($resizedImage, 255, 255, 255, 127);
                    imagefilledrectangle($resizedImage, 0, 0, $newWidth, $newHeight, $transparent);
                }

                imagecopyresampled($resizedImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
                imagedestroy($image);
                $image = $resizedImage;
            }

            // Save optimized image
            $optimizedPath = tempnam(sys_get_temp_dir(), 'cert_opt_');

            if ($imageType == IMAGETYPE_JPEG) {
                // JPEG: Quality 85 (good balance between quality and size)
                imagejpeg($image, $optimizedPath, 85);
            } elseif ($imageType == IMAGETYPE_PNG) {
                // PNG: Compression level 6 (0-9, higher = smaller but slower)
                imagepng($image, $optimizedPath, 6);
            }

            imagedestroy($image);

            // Check if optimization actually reduced size
            $originalSize = filesize($imagePath);
            $optimizedSize = filesize($optimizedPath);

            error_log("GoValid Plugin: Image optimization - Original: " . number_format($originalSize / 1024, 2) . " KB, Optimized: " . number_format($optimizedSize / 1024, 2) . " KB");

            // Use optimized version if it's smaller
            if ($optimizedSize < $originalSize) {
                return $optimizedPath;
            } else {
                unlink($optimizedPath);
                return $imagePath;
            }

        } catch (Exception $e) {
            error_log("GoValid Plugin: Image optimization failed: " . $e->getMessage());
            return $imagePath; // Return original on error
        }
    }
    
    /**
     * Render rectangle (background)
     */
    private function renderRectangle($object)
    {
        $x = $this->pixelToMm($object['left']);
        $y = $this->pixelToMm($object['top']);
        $width = $this->pixelToMm($object['width'] * $object['scaleX']);
        $height = $this->pixelToMm($object['height'] * $object['scaleY']);

        // Set fill color
        if (isset($object['fill'])) {
            $color = $this->hexToRgb($object['fill']);
            $this->fpdf->SetFillColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->Rect($x, $y, $width, $height, 'F');
        }

        // Set border color
        if (isset($object['stroke'])) {
            $color = $this->hexToRgb($object['stroke']);
            $this->fpdf->SetDrawColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->SetLineWidth($this->pixelToMm($object['strokeWidth'] ?? 1));
            $this->fpdf->Rect($x, $y, $width, $height, 'D');
        }
    }

    /**
     * Render circle object
     */
    private function renderCircle($object)
    {
        $x = $this->pixelToMm($object['left']);
        $y = $this->pixelToMm($object['top']);
        $radius = $this->pixelToMm($object['radius'] * $object['scaleX']);

        // Set fill color
        $style = '';
        if (isset($object['fill']) && $object['fill'] !== 'transparent') {
            $color = $this->hexToRgb($object['fill']);
            $this->fpdf->SetFillColor($color['r'], $color['g'], $color['b']);
            $style .= 'F';
        }

        // Set stroke
        if (isset($object['stroke']) && $object['stroke'] !== null) {
            $color = $this->hexToRgb($object['stroke']);
            $this->fpdf->SetDrawColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->SetLineWidth($this->pixelToMm($object['strokeWidth'] ?? 1));
            $style .= 'D';
        }

        if ($style) {
            // Approximate circle with ellipse
            $this->fpdf->Ellipse($x + $radius, $y + $radius, $radius, $radius, $style);
        }
    }

    /**
     * Render ellipse object
     */
    private function renderEllipse($object)
    {
        $x = $this->pixelToMm($object['left']);
        $y = $this->pixelToMm($object['top']);
        $rx = $this->pixelToMm($object['rx'] * $object['scaleX']);
        $ry = $this->pixelToMm($object['ry'] * $object['scaleY']);

        // Set fill color
        $style = '';
        if (isset($object['fill']) && $object['fill'] !== 'transparent') {
            $color = $this->hexToRgb($object['fill']);
            $this->fpdf->SetFillColor($color['r'], $color['g'], $color['b']);
            $style .= 'F';
        }

        // Set stroke
        if (isset($object['stroke']) && $object['stroke'] !== null) {
            $color = $this->hexToRgb($object['stroke']);
            $this->fpdf->SetDrawColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->SetLineWidth($this->pixelToMm($object['strokeWidth'] ?? 1));
            $style .= 'D';
        }

        if ($style) {
            $this->fpdf->Ellipse($x + $rx, $y + $ry, $rx, $ry, $style);
        }
    }

    /**
     * Render line object
     */
    private function renderLine($object)
    {
        $x1 = $this->pixelToMm($object['left']);
        $y1 = $this->pixelToMm($object['top']);
        $x2 = $this->pixelToMm($object['left'] + ($object['x2'] - $object['x1']));
        $y2 = $this->pixelToMm($object['top'] + ($object['y2'] - $object['y1']));

        if (isset($object['stroke'])) {
            $color = $this->hexToRgb($object['stroke']);
            $this->fpdf->SetDrawColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->SetLineWidth($this->pixelToMm($object['strokeWidth'] ?? 1));
            $this->fpdf->Line($x1, $y1, $x2, $y2);
        }
    }

    /**
     * Render polygon/polyline object
     */
    private function renderPolygon($object)
    {
        if (!isset($object['points']) || !is_array($object['points'])) {
            error_log('PDF Generator - Polygon has no points');
            return;
        }

        $points = [];
        foreach ($object['points'] as $point) {
            $points[] = [
                'x' => $this->pixelToMm($object['left'] + $point['x']),
                'y' => $this->pixelToMm($object['top'] + $point['y'])
            ];
        }

        if (count($points) < 2) {
            return;
        }

        // Set stroke and fill
        if (isset($object['stroke'])) {
            $color = $this->hexToRgb($object['stroke']);
            $this->fpdf->SetDrawColor($color['r'], $color['g'], $color['b']);
            $this->fpdf->SetLineWidth($this->pixelToMm($object['strokeWidth'] ?? 1));
        }

        if (isset($object['fill']) && $object['fill'] !== 'transparent') {
            $color = $this->hexToRgb($object['fill']);
            $this->fpdf->SetFillColor($color['r'], $color['g'], $color['b']);
        }

        // Draw the polygon by connecting points
        for ($i = 0; $i < count($points) - 1; $i++) {
            $this->fpdf->Line($points[$i]['x'], $points[$i]['y'], $points[$i+1]['x'], $points[$i+1]['y']);
        }

        // Close the polygon if needed
        if ($object['type'] === 'polygon') {
            $this->fpdf->Line($points[count($points)-1]['x'], $points[count($points)-1]['y'], $points[0]['x'], $points[0]['y']);
        }
    }

    /**
     * Render path object (simplified - draws as lines)
     */
    private function renderPath($object)
    {
        // Path rendering is complex; for now just log it
        error_log('PDF Generator - Path object rendering not fully implemented yet');
    }

    /**
     * Render group object (renders all child objects)
     */
    private function renderGroup($object, $replacements)
    {
        if (!isset($object['objects']) || !is_array($object['objects'])) {
            error_log('PDF Generator - Group has no child objects');
            return;
        }

        error_log('PDF Generator - Rendering group with ' . count($object['objects']) . ' child objects');

        // Get group position and scale
        $groupLeft = $object['left'] ?? 0;
        $groupTop = $object['top'] ?? 0;
        $groupScaleX = $object['scaleX'] ?? 1;
        $groupScaleY = $object['scaleY'] ?? 1;

        // Render each child object
        foreach ($object['objects'] as $childObject) {
            // Adjust child object position relative to group
            $childObject['left'] = ($childObject['left'] ?? 0) * $groupScaleX + $groupLeft;
            $childObject['top'] = ($childObject['top'] ?? 0) * $groupScaleY + $groupTop;

            // Apply group scale to child
            if (isset($childObject['scaleX'])) {
                $childObject['scaleX'] *= $groupScaleX;
            }
            if (isset($childObject['scaleY'])) {
                $childObject['scaleY'] *= $groupScaleY;
            }

            // Recursively render the child object
            $this->renderObject($childObject, $replacements);
        }
    }

    /**
     * Convert pixels to millimeters (assuming 96 DPI)
     */
    private function pixelToMm($pixels)
    {
        return $pixels * 25.4 / 96;
    }
    
    /**
     * Convert hex color to RGB
     */
    private function hexToRgb($hex)
    {
        $hex = str_replace('#', '', $hex);
        
        if (strlen($hex) == 3) {
            $r = hexdec(substr($hex, 0, 1) . substr($hex, 0, 1));
            $g = hexdec(substr($hex, 1, 1) . substr($hex, 1, 1));
            $b = hexdec(substr($hex, 2, 1) . substr($hex, 2, 1));
        } else {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
        }
        
        return ['r' => $r, 'g' => $g, 'b' => $b];
    }

    /**
     * Calculate background image position and size based on positioning mode
     * This matches the JavaScript logic exactly
     */
    private function calculateBackgroundPosition($imageWidth, $imageHeight, $pageWidth, $pageHeight, $mode)
    {
        $imageAspect = $imageWidth / $imageHeight;
        $paperAspect = $pageWidth / $pageHeight;
        
        error_log("PDF Generator - Calculating position for mode '$mode': image({$imageWidth}x{$imageHeight}), paper({$pageWidth}x{$pageHeight})");
        
        switch ($mode) {
            case 'fit':
                // Scale to fit entire paper while maintaining aspect ratio (may leave whitespace)
                if ($imageAspect > $paperAspect) {
                    $scaleX = $scaleY = $pageWidth / $imageWidth;
                } else {
                    $scaleX = $scaleY = $pageHeight / $imageHeight;
                }
                $width = $imageWidth * $scaleX;
                $height = $imageHeight * $scaleY;
                $x = ($pageWidth - $width) / 2;
                $y = ($pageHeight - $height) / 2;
                break;
                
            case 'fill':
                // Stretch to fill entire paper (may distort aspect ratio)
                $width = $pageWidth;
                $height = $pageHeight;
                $x = 0;
                $y = 0;
                break;
                
            case 'cover':
                // Scale to cover entire paper while maintaining aspect ratio (may crop image)
                if ($imageAspect > $paperAspect) {
                    $scaleX = $scaleY = $pageHeight / $imageHeight;
                } else {
                    $scaleX = $scaleY = $pageWidth / $imageWidth;
                }
                $width = $imageWidth * $scaleX;
                $height = $imageHeight * $scaleY;
                $x = ($pageWidth - $width) / 2;
                $y = ($pageHeight - $height) / 2;
                break;
                
            case 'tile':
                // Use original size for tiling
                $mmPerPixel = 25.4 / 96; // Convert pixels to mm
                $width = $imageWidth * $mmPerPixel;
                $height = $imageHeight * $mmPerPixel;
                $x = 0;
                $y = 0;
                break;
                
            case 'top':
                // Fit width, align to top
                $width = $pageWidth;
                $height = $pageWidth / $imageAspect;
                $x = 0;
                $y = 0;
                break;
                
            case 'bottom':
                // Fit width, align to bottom
                $width = $pageWidth;
                $height = $pageWidth / $imageAspect;
                $x = 0;
                $y = $pageHeight - $height;
                break;
                
            case 'center':
                // Original size, centered
                $mmPerPixel = 25.4 / 96; // Convert pixels to mm
                $width = $imageWidth * $mmPerPixel;
                $height = $imageHeight * $mmPerPixel;
                $x = ($pageWidth - $width) / 2;
                $y = ($pageHeight - $height) / 2;
                break;
                
            case 'left':
                // Fit height, align to left
                $width = $pageHeight * $imageAspect;
                $height = $pageHeight;
                $x = 0;
                $y = 0;
                break;
                
            case 'right':
                // Fit height, align to right
                $width = $pageHeight * $imageAspect;
                $height = $pageHeight;
                $x = $pageWidth - $width;
                $y = 0;
                break;
                
            default:
                // Default to fit
                if ($imageAspect > $paperAspect) {
                    $width = $pageWidth;
                    $height = $pageWidth / $imageAspect;
                    $x = 0;
                    $y = ($pageHeight - $height) / 2;
                } else {
                    $width = $pageHeight * $imageAspect;
                    $height = $pageHeight;
                    $x = ($pageWidth - $width) / 2;
                    $y = 0;
                }
        }
        
        return [
            'x' => $x,
            'y' => $y,
            'width' => $width,
            'height' => $height
        ];
    }

    /**
     * Render tiled background image
     */
    private function renderTiledBackground($imageFile, $pageWidth, $pageHeight, $renderParams)
    {
        $tileWidth = $renderParams['width'];
        $tileHeight = $renderParams['height'];
        
        // Calculate how many tiles we need
        $tilesX = ceil($pageWidth / $tileWidth);
        $tilesY = ceil($pageHeight / $tileHeight);
        
        // Render the tiles
        for ($x = 0; $x < $tilesX; $x++) {
            for ($y = 0; $y < $tilesY; $y++) {
                $posX = $x * $tileWidth;
                $posY = $y * $tileHeight;
                
                // Clip the tile if it goes beyond page boundaries
                $actualWidth = min($tileWidth, $pageWidth - $posX);
                $actualHeight = min($tileHeight, $pageHeight - $posY);
                
                // Only render if the tile is within bounds
                if ($actualWidth > 0 && $actualHeight > 0) {
                    $this->fpdf->Image($imageFile, $posX, $posY, $actualWidth, $actualHeight);
                }
            }
        }
    }
    
    /**
     * Render canvas background from a regular object
     */
    private function renderCanvasBackgroundFromObject($object, $orientation)
    {
        if (!isset($object['src'])) {
            return;
        }
        
        // Canvas background should cover the entire page
        $pageWidth = $orientation === 'portrait' ? 210 : 297; // A4 mm
        $pageHeight = $orientation === 'portrait' ? 297 : 210; // A4 mm
        
        // Get background positioning mode
        $backgroundMode = $object['backgroundMode'] ?? 'fit';
        
        // Handle base64 images
        if (strpos($object['src'], 'data:image') === 0) {
            try {
                $imageData = base64_decode(preg_replace('#^data:image/[^;]+;base64,#', '', $object['src']));
                $tempFile = tempnam(sys_get_temp_dir(), 'cert_bg_') . '.png';
                file_put_contents($tempFile, $imageData);
                
                // Get image dimensions and apply positioning mode
                $imageInfo = getimagesize($tempFile);
                if ($imageInfo !== false) {
                    $imageWidth = $imageInfo[0];
                    $imageHeight = $imageInfo[1];
                    
                    // Calculate rendering parameters based on background mode
                    $renderParams = $this->calculateBackgroundPosition($imageWidth, $imageHeight, $pageWidth, $pageHeight, $backgroundMode);
                    
                    // Render the image based on mode
                    if ($backgroundMode === 'tile') {
                        $this->renderTiledBackground($tempFile, $pageWidth, $pageHeight, $renderParams);
                    } else {
                        $this->fpdf->Image($tempFile, $renderParams['x'], $renderParams['y'], $renderParams['width'], $renderParams['height']);
                    }
                } else {
                    // Fallback to full page if can't get dimensions
                    $this->fpdf->Image($tempFile, 0, 0, $pageWidth, $pageHeight);
                }
                
                // Clean up
                unlink($tempFile);
                error_log('PDF Generator - Successfully rendered background image from object with mode: ' . $backgroundMode);
            } catch (Exception $e) {
                error_log('PDF Generator - Failed to render background image from object: ' . $e->getMessage());
            }
        } else {
            // Regular URL - try to load if accessible
            try {
                $this->fpdf->Image($object['src'], 0, 0, $pageWidth, $pageHeight);
                error_log('PDF Generator - Successfully rendered background image from URL');
            } catch (Exception $e) {
                error_log('PDF Generator - Failed to load background image from URL: ' . $e->getMessage());
            }
        }
    }
    
    /**
     * Render canvas background image
     */
    private function renderCanvasBackground($backgroundImage, $orientation)
    {
        if (!isset($backgroundImage['src'])) {
            error_log('PDF Generator - Background image has no src property');
            return;
        }
        
        error_log('PDF Generator - Attempting to render canvas background image');
        
        // Canvas background should cover the entire page
        $pageWidth = $orientation === 'portrait' ? 210 : 297; // A4 mm
        $pageHeight = $orientation === 'portrait' ? 297 : 210; // A4 mm
        
        // Get background positioning mode
        $backgroundMode = $backgroundImage['backgroundMode'] ?? 'fit';
        
        // Handle base64 images
        if (strpos($backgroundImage['src'], 'data:image') === 0) {
            try {
                error_log('PDF Generator - Processing base64 background image');
                
                // Extract the base64 data more carefully
                $imageDataParts = explode(',', $backgroundImage['src'], 2);
                if (count($imageDataParts) !== 2) {
                    throw new Exception('Invalid base64 image format');
                }
                
                $imageData = base64_decode($imageDataParts[1]);
                if ($imageData === false) {
                    throw new Exception('Failed to decode base64 image data');
                }
                
                // Determine file extension from data URI
                $mimeType = '';
                if (preg_match('#^data:image/([^;]+)#', $backgroundImage['src'], $matches)) {
                    $mimeType = $matches[1];
                }
                
                $extension = ($mimeType === 'jpeg' || $mimeType === 'jpg') ? '.jpg' : '.png';
                $tempFile = tempnam(sys_get_temp_dir(), 'cert_bg_') . $extension;
                
                $bytesWritten = file_put_contents($tempFile, $imageData);
                if ($bytesWritten === false) {
                    throw new Exception('Failed to write image data to temp file');
                }
                
                error_log("PDF Generator - Created temp file: $tempFile (size: $bytesWritten bytes)");
                
                // Verify the image file
                $imageInfo = getimagesize($tempFile);
                if ($imageInfo === false) {
                    throw new Exception('Invalid image file created');
                }
                
                error_log("PDF Generator - Image info: {$imageInfo[0]}x{$imageInfo[1]}, type: {$imageInfo[2]}");
                
                // Calculate dimensions based on background mode
                $imageWidth = $imageInfo[0];
                $imageHeight = $imageInfo[1];
                
                // Calculate rendering parameters based on background mode
                $renderParams = $this->calculateBackgroundPosition($imageWidth, $imageHeight, $pageWidth, $pageHeight, $backgroundMode);
                
                error_log("PDF Generator - Rendering background with mode '$backgroundMode': {$renderParams['width']}x{$renderParams['height']} at ({$renderParams['x']},{$renderParams['y']})");
                
                // Render the image based on mode
                if ($backgroundMode === 'tile') {
                    $this->renderTiledBackground($tempFile, $pageWidth, $pageHeight, $renderParams);
                } else {
                    $this->fpdf->Image($tempFile, $renderParams['x'], $renderParams['y'], $renderParams['width'], $renderParams['height']);
                }
                
                error_log('PDF Generator - Successfully added background image to PDF');
                
                // Clean up
                unlink($tempFile);
            } catch (Exception $e) {
                error_log('PDF Generator - Failed to render background image: ' . $e->getMessage());
            }
        } else {
            // Regular URL - try to load if accessible
            try {
                error_log('PDF Generator - Processing URL background image: ' . $backgroundImage['src']);
                $this->fpdf->Image($backgroundImage['src'], $x, $y, $pageWidth, $pageHeight);
                error_log('PDF Generator - Successfully added URL background image to PDF');
            } catch (Exception $e) {
                error_log('PDF Generator - Failed to load background image from URL: ' . $e->getMessage());
            }
        }
    }
    
    /**
     * Generate QR code using GoValid API (deprecated - use GoValidAuthManager instead)
     * @param string $data Data to encode in QR code
     * @param string $apiKey GoValid API key
     * @return string|false Base64 encoded image or false on failure
     * @deprecated Use GoValidAuthManager->apiRequest() for authenticated API calls
     */
    public function generateQRCode($data, $apiKey)
    {
        // This method is kept for backward compatibility
        // New code should use GoValidAuthManager for authenticated API requests
        error_log('Warning: CertificatePDFGenerator::generateQRCode is deprecated. Use GoValidAuthManager instead.');
        return false;
    }
}