<?php

/**
 * @file plugins/generic/goValidOJS/pages/GoValidOJSHandler.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class GoValidOJSHandler
 * @ingroup plugins_generic_goValidOJS
 *
 * Handle requests for GoValidOJS plugin pages.
 *
 * Converted for OJS 3.3.x compatibility
 */

import('classes.handler.Handler');

class GoValidOJSHandler extends Handler {
    /** @var GoValidOJSPlugin The plugin associated with this request */
    public $plugin;

    /**
     * Constructor
     */
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        // Load the plugin if not already loaded
        PluginRegistry::loadCategory('generic');
        $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
    }

    /**
     * @copydoc PKPHandler::authorize()
     */
    public function authorize($request, &$args, $roleAssignments)
    {
        // Import required classes for OJS 3.3 compatibility
        import('lib.pkp.classes.security.authorization.RoleBasedHandlerOperationPolicy');

        // Define all operations that this handler supports
        $allOperations = [
            'designer',
            'ajax',
            'saveTemplate',
            'getIssues',
            'getSubmissions',
            'emailTemplateManager',
            'emailTemplateDesigner',
            'saveEmailTemplate',
            'deleteTemplate',
            'activateTemplate',
            'previewTemplate',
            'getDefaultTemplate',
            'getEmailTemplate',
            'sendTestEmail',
            'testEmail',
            'searchUsers',
            'sendCertificateEmail'
        ];

        // Get context and plugin settings to determine allowed roles
        $context = $request->getContext();
        $contextId = $context ? $context->getId() : 0;

        // Build allowed roles based on FAB visibility settings
        // Use global constants for OJS 3.3 compatibility (ROLE_ID_* are defined as global constants)
        $allowedRoles = [ROLE_ID_SITE_ADMIN]; // Site admin always has access

        if ($this->plugin) {
            $showFabForManagers = $this->plugin->getSetting($contextId, 'showFabForManagers') ?? true;
            $showFabForEditors = $this->plugin->getSetting($contextId, 'showFabForEditors') ?? true;
            $showFabForAssistants = $this->plugin->getSetting($contextId, 'showFabForAssistants') ?? true;

            if ($showFabForManagers) {
                $allowedRoles[] = ROLE_ID_MANAGER;
            }
            if ($showFabForEditors) {
                $allowedRoles[] = ROLE_ID_SUB_EDITOR;
            }
            if ($showFabForAssistants) {
                $allowedRoles[] = ROLE_ID_ASSISTANT;
            }
        } else {
            // Fallback: allow all roles if plugin not loaded
            $allowedRoles = [
                ROLE_ID_SITE_ADMIN,
                ROLE_ID_MANAGER,
                ROLE_ID_SUB_EDITOR,
                ROLE_ID_ASSISTANT
            ];
        }

        $this->addPolicy(new RoleBasedHandlerOperationPolicy(
            $request,
            $allowedRoles,
            $allOperations
        ));

        return parent::authorize($request, $args, $roleAssignments);
    }

    /**
     * Display the certificate designer or handle AJAX requests
     * 
     * @param array $args
     * @param PKPRequest $request
     */
    public function designer($args, $request)
    {
        // Check if this is an AJAX request
        $action = $request->getUserVar('action');
        if ($action) {
            // Handle AJAX actions
            return $this->handleAjaxAction($action, $request);
        }
        
        // Make sure plugin is loaded for regular page display
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in page handler');
            // Try alternative method to load plugin
            $pluginPath = 'plugins.generic.goValidOJS.GoValidOJSPlugin';
            $plugin = instantiate($pluginPath, 'Plugin');
            if ($plugin && is_a($plugin, 'Plugin')) {
                $plugin->register('generic', dirname(__FILE__, 2));
                $this->plugin = $plugin;
            }
        }
        
        if (!$this->plugin) {
            // Return a proper error page
            $templateMgr = TemplateManager::getManager($request);
            $templateMgr->assign('message', 'plugins.generic.goValidOJS.error.pluginNotFound');
            $templateMgr->assign('backLink', $request->url(null, 'management', 'settings', 'website'));
            $templateMgr->display('frontend/pages/error.tpl');
            return;
        }

        $templateMgr = TemplateManager::getManager($request);
        $templateMgr->assign('pluginName', $this->plugin->getName());
        $templateMgr->assign('baseUrl', $request->getBaseUrl());

        $context = $request->getContext();
        $contextId = $context ? $context->getId() : CONTEXT_SITE;

        // Assign context path for AJAX URLs
        $contextPath = $context ? $context->getPath() : 'index';
        $templateMgr->assign('contextPath', $contextPath);

        // Simple journal info
        if ($context) {
            // Get logo URL using robust method from EmailTemplateManager
            require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
            $logoUrl = $emailTemplateManager->getJournalLogoUrl($request);
            if (!$logoUrl) $logoUrl = '';

            $journalInfo = [
                'name' => $context->getLocalizedData('name') ?: 'Journal Name',
                'logo' => $logoUrl,
                'url' => $request->getBaseUrl() . '/' . $context->getPath(),
                'onlineIssn' => $context->getData('onlineIssn') ?: '',
                'printIssn' => $context->getData('printIssn') ?: ''
            ];
        } else {
            $journalInfo = ['name' => 'Journal Name', 'logo' => '', 'url' => '', 'onlineIssn' => '', 'printIssn' => ''];
        }
        $templateMgr->assign('journalInfo', $journalInfo);
        
        // Load existing template if available
        $existingTemplate = $this->plugin->getSetting($contextId, 'certificateTemplate');
        $templateMgr->assign('existingTemplate', $existingTemplate);

        // Display the template
        $templateMgr->display($this->plugin->getTemplateResource('designer.tpl'));
    }

    /**
     * Save certificate template
     */
    public function saveTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        if (!$this->plugin) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'Plugin not found']);
            exit;
        }

        $context = $request->getContext();
        $templateData = $request->getUserVar('templateData');

        if (!$templateData) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'No template data provided']);
            exit;
        }

        // Parse the template data
        $config = json_decode($templateData, true);
        if (!$config) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'Invalid template data']);
            exit;
        }

        // Extract template name and subject
        $templateName = $config['name'] ?? 'Untitled Template';
        $subject = $config['subject'] ?? '';

        // Generate a unique template key from the name
        $templateKey = 'CUSTOM_' . strtoupper(preg_replace('/[^a-zA-Z0-9]/', '_', $templateName));

        try {
            // Use OJSEmailTemplateIntegration to save as OJS email template
            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new OJSEmailTemplateIntegration($this->plugin, $context);

            // Build the email body from the content fields
            $greeting = $config['content']['greeting'] ?? '';
            $message = $config['content']['message'] ?? '';
            $articleDetails = $config['content']['articleDetails'] ?? '';
            $closing = $config['content']['closing'] ?? '';

            // Combine content parts into body (content is already safe from designer)
            $bodyContent = '';
            if ($greeting) $bodyContent .= '<p>' . nl2br($greeting) . '</p>';
            if ($message) $bodyContent .= '<p>' . nl2br($message) . '</p>';
            if ($articleDetails) $bodyContent .= '<div style="background: ' . ($config['content']['detailsBgColor'] ?? '#f8f9fa') . '; padding: 15px; margin: 15px 0; border-radius: 4px;">' . nl2br($articleDetails) . '</div>';
            if ($closing) $bodyContent .= '<p>' . nl2br($closing) . '</p>';

            // Get journal logo URL and name
            $logoUrl = '';
            $journalName = 'Journal';
            if ($context) {
                $publicFileManager = new \PublicFileManager();
                $publicFilesDir = $request->getBaseUrl() . '/' . $publicFileManager->getContextFilesPath($context->getId());
                $pageHeaderLogoImage = $context->getLocalizedData('pageHeaderLogoImage');
                if ($pageHeaderLogoImage && isset($pageHeaderLogoImage['uploadName'])) {
                    $logoUrl = $publicFilesDir . '/' . $pageHeaderLogoImage['uploadName'];
                }
                $journalName = $context->getLocalizedName() ?? 'Journal';
            }

            // Generate full HTML email with header and footer using Email Designer structure
            $emailBody = $this->generateEmailDesignerHTML($config, $bodyContent, $logoUrl, $journalName);

            // Save as OJS email template
            $result = $integration->saveCustomTemplate([
                'key' => $templateKey,
                'name' => $templateName,
                'subject' => $subject,
                'body' => $emailBody,
                'description' => 'Custom email template created with Email Designer',
                'designConfig' => json_encode($config)
            ]);

            if ($result['success']) {
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'message' => 'Template saved successfully!', 'templateKey' => $templateKey]);
            } else {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'message' => $result['error'] ?? 'Failed to save template']);
            }
        } catch (Exception $e) {
            error_log('GoValid Plugin: Error saving template: ' . $e->getMessage());
            error_log('GoValid Plugin: Stack trace: ' . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }

        exit;
    }

    /**
     * Generate full HTML email from Email Designer config
     * This matches the structure used in email-designer.js generateEmailHTML() and generateHeaderContent()
     *
     * @param array $config The Email Designer config with header, content, footer, settings
     * @param string $bodyContent The processed body content HTML
     * @param string $logoUrl The journal logo URL
     * @param string $journalName The journal name
     * @return string The complete HTML email
     */
    protected function generateEmailDesignerHTML($config, $bodyContent, $logoUrl, $journalName = 'Journal') {
        // Extract settings
        $settings = $config['settings'] ?? [];
        $header = $config['header'] ?? [];
        $content = $config['content'] ?? [];
        $footer = $config['footer'] ?? [];

        // Settings - all settings from Email Designer
        $fontFamily = $settings['fontFamily'] ?? 'Arial, sans-serif';
        $emailWidth = intval($settings['emailWidth'] ?? 600);
        $fontSize = intval($settings['fontSize'] ?? 16);
        $lineHeight = $settings['lineHeight'] ?? '1.6';
        $outerBgColor = $settings['outerBgColor'] ?? '#f5f5f5';

        // Header - all header settings
        $headerBgType = $header['bgType'] ?? 'gradient';
        $headerBgColor = $header['bgColor'] ?? '#667eea';
        $headerGradientStart = $header['gradientStart'] ?? '#667eea';
        $headerGradientEnd = $header['gradientEnd'] ?? '#764ba2';
        $headerTextColor = $header['textColor'] ?? '#ffffff';
        $headerPadding = intval($header['padding'] ?? 30);
        $headerTitle = $header['title'] ?? '';
        $headerSubtitle = $header['subtitle'] ?? '';
        $showLogo = filter_var($header['showLogo'] ?? true, FILTER_VALIDATE_BOOLEAN);
        $logoPosition = $header['logoPosition'] ?? 'center';
        $logoMaxWidth = intval($header['logoMaxWidth'] ?? 200);
        $showNameBesideLogo = filter_var($header['showNameBesideLogo'] ?? false, FILTER_VALIDATE_BOOLEAN);

        // Content - all content settings
        $contentBgColor = $content['bgColor'] ?? '#ffffff';
        $contentTextColor = $content['textColor'] ?? '#333333';
        $contentPadding = intval($content['padding'] ?? 30);
        $detailsBgColor = $content['detailsBgColor'] ?? '#f8f9fa';

        // Footer - all footer settings
        $footerBgColor = $footer['bgColor'] ?? '#f8f9fa';
        $footerTextColor = $footer['textColor'] ?? '#666666';
        $footerPadding = intval($footer['padding'] ?? 20);
        $footerBorderWidth = intval($footer['borderWidth'] ?? 1);
        $footerBorderColor = $footer['borderColor'] ?? '#eeeeee';
        $footerSignature = $footer['signature'] ?? '';
        $footerSignatureName = $footer['signatureName'] ?? '';
        $footerShowUrl = filter_var($footer['showUrl'] ?? true, FILTER_VALIDATE_BOOLEAN);
        $footerLinkColor = $footer['linkColor'] ?? '#667eea';

        // Build header background style
        if ($headerBgType === 'gradient') {
            $headerBgStyle = "background: linear-gradient(135deg, {$headerGradientStart} 0%, {$headerGradientEnd} 100%);";
        } else {
            $headerBgStyle = "background: {$headerBgColor};";
        }

        // Build logo HTML
        $logoHtml = '';
        if ($showLogo && $logoUrl) {
            $logoHtml = '<img src="' . htmlspecialchars($logoUrl) . '" alt="' . htmlspecialchars($journalName) . '" style="max-width: ' . $logoMaxWidth . 'px; height: auto; display: block;">';
        }

        // Build title HTML
        $titleHtml = '';
        if ($headerTitle) {
            $titleHtml .= '<h1 style="margin: 0; color: ' . htmlspecialchars($headerTextColor) . '; font-size: 28px; font-weight: 600;">' . htmlspecialchars($headerTitle) . '</h1>';
        }
        if ($headerSubtitle) {
            $titleHtml .= '<p style="margin: 10px 0 0 0; color: ' . htmlspecialchars($headerTextColor) . '; font-size: 18px; opacity: 0.9;">' . htmlspecialchars($headerSubtitle) . '</p>';
        }

        // Build header content based on logo position (matching email-designer.js generateHeaderContent)
        $headerContent = '';
        if ($logoPosition === 'center') {
            $headerContent = '
            <div style="text-align: center;">
                ' . ($showLogo && $logoUrl ? '<div style="margin-bottom: 20px; display: flex; justify-content: center;">' . $logoHtml . '</div>' : '') . '
                <div style="flex: 1;">
                    ' . $titleHtml . '
                </div>
            </div>';
        } elseif ($logoPosition === 'left' || $logoPosition === 'right') {
            if ($showNameBesideLogo && $showLogo && $logoUrl) {
                // Logo and name side by side
                $flexDirection = ($logoPosition === 'left') ? 'row' : 'row-reverse';
                $textAlign = ($logoPosition === 'left') ? 'left' : 'right';
                $headerContent = '
                <div style="display: flex; align-items: center; gap: 20px; flex-direction: ' . $flexDirection . ';">
                    <div style="flex-shrink: 0;">
                        ' . $logoHtml . '
                    </div>
                    <div style="flex: 1; text-align: ' . $textAlign . ';">
                        ' . $titleHtml . '
                    </div>
                </div>';
            } else {
                // Logo on top, then text
                $alignItems = ($logoPosition === 'left') ? 'flex-start' : 'flex-end';
                $textAlign = ($logoPosition === 'left') ? 'left' : 'right';
                $headerContent = '
                <div style="display: flex; flex-direction: column; align-items: ' . $alignItems . ';">
                    ' . ($showLogo && $logoUrl ? '<div style="margin-bottom: 20px;">' . $logoHtml . '</div>' : '') . '
                    <div style="text-align: ' . $textAlign . '; width: 100%;">
                        ' . $titleHtml . '
                    </div>
                </div>';
            }
        } else {
            // Fallback to center
            $headerContent = '
            <div style="text-align: center;">
                ' . ($showLogo && $logoUrl ? '<div style="margin-bottom: 20px; display: flex; justify-content: center;">' . $logoHtml . '</div>' : '') . '
                ' . $titleHtml . '
            </div>';
        }

        // Build footer content
        $footerContent = '';
        if ($footerSignature) {
            $footerContent .= '<p style="margin: 0 0 10px 0; font-size: ' . $fontSize . 'px; color: ' . htmlspecialchars($footerTextColor) . ';">' . htmlspecialchars($footerSignature) . '</p>';
        }
        if ($footerSignatureName) {
            $footerContent .= '<p style="margin: 0 0 15px 0; font-size: ' . $fontSize . 'px; color: ' . htmlspecialchars($footerTextColor) . '; font-weight: bold;">' . htmlspecialchars($footerSignatureName) . '</p>';
        }
        if ($footerShowUrl) {
            $footerContent .= '<p style="margin: 0; font-size: 14px; color: ' . htmlspecialchars($footerTextColor) . ';"><a href="{$contextUrl}" style="color: ' . htmlspecialchars($footerLinkColor) . '; text-decoration: none;">{$contextUrl}</a></p>';
        }

        // Build the complete HTML (matching email-designer.js generateEmailHTML structure)
        $html = '
<div style="font-family: ' . htmlspecialchars($fontFamily) . '; max-width: ' . $emailWidth . 'px; margin: 0 auto; background: ' . htmlspecialchars($outerBgColor) . ';">
    <!-- Header -->
    <div style="' . $headerBgStyle . ' color: ' . htmlspecialchars($headerTextColor) . '; padding: ' . $headerPadding . 'px;">
        ' . $headerContent . '
    </div>

    <!-- Content -->
    <div style="padding: ' . $contentPadding . 'px; background: ' . htmlspecialchars($contentBgColor) . '; color: ' . htmlspecialchars($contentTextColor) . '; line-height: ' . $lineHeight . '; font-size: ' . $fontSize . 'px;">
        ' . $bodyContent . '
    </div>

    <!-- Footer -->
    <div style="background: ' . htmlspecialchars($footerBgColor) . '; padding: ' . $footerPadding . 'px; text-align: center; border-top: ' . $footerBorderWidth . 'px solid ' . htmlspecialchars($footerBorderColor) . ';">
        ' . $footerContent . '
    </div>
</div>';

        return trim($html);
    }

    /**
     * Handle AJAX requests for issues
     */
    public function getIssues($args, $request)
    {
        $context = $request->getContext();
        
        try {
            $issues = [];
            
            if ($context) {
                // Use OJS 3.3 DAO pattern to get published issues
                $issueDao = DAORegistry::getDAO('IssueDAO');
                $publishedIssuesIterator = $issueDao->getPublishedIssues($context->getId());
                
                while ($issue = $publishedIssuesIterator->next()) {
                    $issues[] = [
                        'id' => $issue->getId(),
                        'title' => $issue->getLocalizedTitle() ?: 'Vol. ' . $issue->getVolume() . ' No. ' . $issue->getNumber() . ' (' . $issue->getYear() . ')',
                        'volume' => $issue->getVolume() ?: '',
                        'number' => $issue->getNumber() ?: '',
                        'year' => $issue->getYear() ?: '',
                        'datePublished' => $issue->getDatePublished(),
                        'description' => $issue->getLocalizedDescription() ?: '',
                        'published' => true
                    ];
                }
            }
            
            header('Content-Type: application/json');
            echo json_encode(['issues' => $issues]);
            exit;
            
        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Error loading issues: ' . $e->getMessage(),
                'issues' => []
            ]);
            exit;
        }
    }

    /**
     * Handle AJAX requests for submissions
     */
    public function getSubmissions($args, $request)
    {
        $context = $request->getContext();
        $issueId = $request->getUserVar('issueId');

        try {
            $submissions = [];

            if ($context) {
                // Use OJS 3.3 DAO pattern - no PublishedArticleDAO in OJS 3.3
                $submissionDao = DAORegistry::getDAO('SubmissionDAO');

                // Get all published submissions for this context
                $submissionsIterator = $submissionDao->getByContextId($context->getId());

                while ($submission = $submissionsIterator->next()) {
                    // Only get published submissions (STATUS_PUBLISHED = 3)
                    if ($submission->getStatus() == STATUS_PUBLISHED) {
                        $publication = $submission->getCurrentPublication();
                        if ($publication) {
                            // If issueId is specified, filter by issue
                            if ($issueId) {
                                $pubIssueId = $publication->getData('issueId');
                                if ($pubIssueId != (int)$issueId) {
                                    continue; // Skip if not in requested issue
                                }
                            }

                            $authorsData = $this->getAuthorsData($publication);
                            $submissions[] = $this->formatSubmissionData($submission, $publication, $authorsData);
                        }
                    }
                }
            }

            header('Content-Type: application/json');
            echo json_encode(['submissions' => $submissions]);
            exit;

        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Error loading submissions: ' . $e->getMessage(),
                'submissions' => []
            ]);
            exit;
        }
    }

    /**
     * Get authors data from publication
     */
    private function getAuthorsData($publication) {
        $authorsData = [];
        $authors = $publication->getData('authors');
        if ($authors) {
            foreach ($authors as $author) {
                $authorsData[] = [
                    'id' => $author->getId(),
                    'name' => $author->getFullName(),
                    'email' => $author->getEmail(),
                    'affiliation' => $author->getLocalizedData('affiliation') ?: '',
                    'country' => $author->getCountry() ?: ''
                ];
            }
        }
        return $authorsData;
    }

    /**
     * Format submission data for JSON response
     */
    private function formatSubmissionData($submission, $publication, $authorsData) {
        return [
            'id' => $submission->getId(),
            'title' => $publication->getLocalizedTitle() ?: '',
            'doi' => $publication->getStoredPubId('doi') ?: '',
            'pages' => $publication->getData('pages') ?: '',
            'dateSubmitted' => $submission->getDateSubmitted(),
            'datePublished' => $publication->getData('datePublished'),
            'abstract' => $publication->getLocalizedData('abstract') ?: '',
            'issueId' => $publication->getData('issueId'),
            'sectionId' => $publication->getData('sectionId'),
            'authors' => $authorsData
        ];
    }
    
    /**
     * Handle AJAX actions
     * 
     * @param string $action
     * @param PKPRequest $request
     * @return void
     */
    protected function handleAjaxAction($action, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        // Map actions to plugin methods
        switch ($action) {
            case 'getIssues':
                $this->getIssues([], $request);
                break;
                
            case 'getSubmissions':
                $this->getSubmissions([], $request);
                break;
                
            case 'saveTemplate':
                $this->saveTemplate([], $request);
                break;

            case 'sendTestEmail':
                $this->sendTestEmail([], $request);
                break;

            case 'sendCertificateEmail':
                $this->sendCertificateEmail([], $request);
                break;

            case 'searchUsers':
                $this->searchUsers([], $request);
                break;

            case 'pinStatus':
                $this->proxyPinStatus($request);
                break;

            case 'verifyPin':
                $this->proxyVerifyPin($request);
                break;

            case 'list':
            case 'login':
            case 'verify2FA':
            case 'verifyEmailOTP':
            case 'logout':
            case 'checkAuth':
            case 'save':
            case 'load':
            case 'delete':
            case 'preview':
            case 'getSubscription':
            case 'checkCapability':
            case 'getUserInfo':
            case 'getReviewers':
            case 'getReviewersForSubmission':
            case 'getEditorsForSubmission':
            case 'getPatterns':
            case 'previewPattern':
            case 'generatePattern':
            case 'testNumberingAPI':
            case 'generateCertificateQR':
            case 'searchReviewers':
            case 'searchAuthors':
            case 'searchEditors':
            case 'refreshAuth':
            case 'sendOtp':
            case 'verifyOtp':
            case 'testEndpoints':
            case 'proxyQRImage':
            case 'listDefaultTemplates':
            case 'loadDefaultTemplate':
            case 'loadSavedTemplates':
            case 'saveDesignerTemplate':
            case 'deleteTemplate':
            case 'getAntiCounterfeitTags':
                // These actions should be handled by the plugin's designer handler
                if ($this->plugin) {
                    // Call the plugin's _handleDesignerAjax method directly
                    try {
                        $reflectionClass = new \ReflectionClass($this->plugin);
                        $method = $reflectionClass->getMethod('_handleDesignerAjax');
                        $method->setAccessible(true);
                        $result = $method->invoke($this->plugin, $request);

                        // Check for JSONMessage (OJS 3.3 uses non-namespaced class name)
                        if ($result && (is_a($result, 'JSONMessage') || is_a($result, 'PKP\core\JSONMessage'))) {
                            header('Content-Type: application/json');
                            echo $result->getString();
                            exit;
                        }
                        
                        // If no JSONMessage result, the plugin may have already sent output
                        // Just exit to prevent double output
                        exit;
                    } catch (Exception $e) {
                        error_log('Error calling plugin method: ' . $e->getMessage());
                        header('Content-Type: application/json');
                        echo json_encode(['status' => false, 'error' => 'Plugin method error: ' . $e->getMessage()]);
                        exit;
                    }
                } else {
                    // Plugin not loaded, return error
                    header('Content-Type: application/json');
                    echo json_encode(['status' => false, 'error' => 'Plugin not loaded']);
                    exit;
                }
                break;

            // OJS Email Template Integration Actions
            case 'getOJSEmailTemplates':
                $this->getOJSEmailTemplates($request);
                break;

            case 'getOJSEmailTemplate':
                $this->getOJSEmailTemplate($request);
                break;

            case 'saveOJSEmailTemplate':
                $this->saveOJSEmailTemplate($request);
                break;

            case 'resetOJSEmailTemplate':
                $this->resetOJSEmailTemplate($request);
                break;

            case 'getOJSEmailVariables':
                $this->getOJSEmailVariables($request);
                break;

            case 'applyDesignToOJSTemplate':
                $this->applyDesignToOJSTemplate($request);
                break;

            // Custom Email Template Actions
            case 'getCustomEmailTemplates':
                $this->getCustomEmailTemplates($request);
                break;

            case 'getCustomEmailTemplate':
                $this->getCustomEmailTemplate($request);
                break;

            // Design Template Library Actions
            case 'getDesignTemplates':
                $this->getDesignTemplates($request);
                break;

            case 'getDesignTemplate':
                $this->getDesignTemplate($request);
                break;

            default:
                header('Content-Type: application/json');
                echo json_encode(['error' => 'Invalid action: ' . $action]);
                exit;
        }
    }
    
    /**
     * Email template manager
     */
    public function emailTemplateManager($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in emailTemplateManager');
            return false;
        }
        
        return $this->plugin->handleEmailTemplateRequest($request, null, $args);
    }
    
    /**
     * Email template designer (new visual designer)
     */
    public function emailTemplateDesigner($args, $request)
    {
        // Check if this is an AJAX request
        $action = $request->getUserVar('action');
        if ($action) {
            // Handle AJAX actions
            return $this->handleAjaxAction($action, $request);
        }

        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        if (!$this->plugin) {
            return false;
        }

        $templateMgr = TemplateManager::getManager($request);
        $context = $request->getContext();

        // Get template ID if editing
        $templateId = $request->getUserVar('templateId');

        // Get journal logo URL
        require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $logoUrl = $emailTemplateManager->getJournalLogoUrl($request);
        $logoData = $emailTemplateManager->getJournalLogoData();

        $templateMgr->assign([
            'pluginName' => $this->plugin->getName(),
            'baseUrl' => $request->getBaseUrl(),
            'contextPath' => $context->getPath(),
            'templateId' => $templateId ?? '',
            'templateName' => '',
            'journalLogoUrl' => $logoUrl ?? '',
            'journalLogoAlt' => $logoData['altText'] ?? 'Journal Logo',
            'journalLogoWidth' => $logoData['width'] ?? 180,
            'journalLogoHeight' => $logoData['height'] ?? 90,
            'journalName' => $context->getLocalizedData('name') ?? 'Journal Name',
            'lb' => '{{',  // Left brace for Smarty
            'rb' => '}}'   // Right brace for Smarty
        ]);

        // Display the template
        $templateMgr->display($this->plugin->getTemplateResource('emailTemplateDesigner.tpl'));
    }

    /**
     * Get email template for editing
     */
    public function getEmailTemplate($args, $request)
    {
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        $templateId = $request->getUserVar('templateId');
        $context = $request->getContext();

        if (!$templateId || !$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $template = $emailTemplateManager->getJSONTemplate($templateId);

        header('Content-Type: application/json');
        if ($template) {
            echo json_encode(['status' => 'success', 'template' => $template]);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Template not found']);
        }
        exit;
    }

    /**
     * Send test email
     */
    public function sendTestEmail($args, $request)
    {
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        $context = $request->getContext();
        $testEmail = $request->getUserVar('testEmail');
        $subject = $request->getUserVar('subject');
        $htmlContent = $request->getUserVar('htmlContent');
        $cc = $request->getUserVar('cc');
        $bcc = $request->getUserVar('bcc');

        if (!$context || !$testEmail || !$subject || !$htmlContent) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
            exit;
        }

        // Handle file attachments
        $attachments = [];

        // Debug logging to file
        $debugMsg = "\n\n=== ATTACHMENT DEBUG " . date('Y-m-d H:i:s') . " ===\n";
        $debugMsg .= "Checking for attachments...\n";
        $debugMsg .= "\$_FILES content: " . print_r($_FILES, true) . "\n";
        file_put_contents('/tmp/govalid_attachments.log', $debugMsg, FILE_APPEND);

        error_log("GoValid Plugin: Checking for attachments...");
        error_log("GoValid Plugin: \$_FILES content: " . print_r($_FILES, true));

        if (isset($_FILES['attachments']) && is_array($_FILES['attachments']['name'])) {
            $fileCount = count($_FILES['attachments']['name']);
            error_log("GoValid Plugin: Found {$fileCount} attachments");

            for ($i = 0; $i < $fileCount; $i++) {
                // Check if file was uploaded without errors
                if ($_FILES['attachments']['error'][$i] === UPLOAD_ERR_OK) {
                    $tmpPath = $_FILES['attachments']['tmp_name'][$i];
                    $fileName = $_FILES['attachments']['name'][$i];
                    $fileSize = $_FILES['attachments']['size'][$i];
                    $fileType = $_FILES['attachments']['type'][$i];

                    // Validate file size (5MB max)
                    if ($fileSize > 5 * 1024 * 1024) {
                        header('Content-Type: application/json');
                        echo json_encode(['status' => 'error', 'message' => "File {$fileName} exceeds 5MB limit"]);
                        exit;
                    }

                    // Validate file type
                    $allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                    $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                    $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png', 'gif'];

                    if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, $allowedExtensions)) {
                        header('Content-Type: application/json');
                        echo json_encode(['status' => 'error', 'message' => "File {$fileName} has invalid type. Only PDF and images allowed."]);
                        exit;
                    }

                    // Read file contents
                    $fileContents = file_get_contents($tmpPath);

                    if ($fileContents !== false) {
                        $attachments[] = [
                            'name' => $fileName,
                            'data' => $fileContents,
                            'type' => $fileType
                        ];
                        error_log("GoValid Plugin: Added attachment: {$fileName} ({$fileSize} bytes, {$fileType})");
                        file_put_contents('/tmp/govalid_attachments.log', "Added attachment: {$fileName} ({$fileSize} bytes, {$fileType})\n", FILE_APPEND);
                    }
                }
            }
        } else {
            error_log("GoValid Plugin: No attachments found in \$_FILES");
            file_put_contents('/tmp/govalid_attachments.log', "No attachments found in \$_FILES\n", FILE_APPEND);
        }

        error_log("GoValid Plugin: Total attachments prepared: " . count($attachments));
        file_put_contents('/tmp/govalid_attachments.log', "Total attachments prepared: " . count($attachments) . "\n", FILE_APPEND);

        require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
        $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
        $result = $emailTemplateManager->sendTestEmail($testEmail, $subject, $htmlContent, $cc, $bcc, $attachments);

        header('Content-Type: application/json');
        echo json_encode($result);
        exit;
    }

    /**
     * Search users for email autocomplete
     */
    public function searchUsers($args, $request)
    {
        $context = $request->getContext();
        $searchTerm = $request->getUserVar('q');

        if (!$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'No context']);
            exit;
        }

        // Require at least 2 characters to search
        if (empty($searchTerm) || strlen($searchTerm) < 2) {
            header('Content-Type: application/json');
            echo json_encode(['results' => []]);
            exit;
        }

        try {
            // Use OJS 3.3 RoleDAO pattern - search all users with any role in this context
            $roleDao = DAORegistry::getDAO('RoleDAO');
            $results = [];
            $processedIds = [];

            // Search users using RoleDAO which supports search parameter
            // getUsersByRoleId($roleId, $contextId, $searchType, $search, $searchMatch, $dbResultRange)
            // $searchType can be: USER_FIELD_FIRSTNAME, USER_FIELD_LASTNAME, USER_FIELD_USERNAME, USER_FIELD_EMAIL, USER_FIELD_NONE

            // Search by name (includes first and last name)
            $usersIterator = $roleDao->getUsersByRoleId(
                null,                           // All roles
                $context->getId(),              // Context
                USER_FIELD_NONE,                // Search all fields
                $searchTerm,                    // Search term
                'contains'                      // Match type
            );

            while ($user = $usersIterator->next()) {
                if (!in_array($user->getId(), $processedIds)) {
                    $results[] = [
                        'id' => $user->getId(),
                        'email' => $user->getEmail(),
                        'name' => $user->getFullName(),
                        'label' => $user->getFullName() . ' <' . $user->getEmail() . '>'
                    ];
                    $processedIds[] = $user->getId();
                }
                if (count($results) >= 10) break;
            }

            header('Content-Type: application/json');
            echo json_encode(['results' => $results]);
            exit;
        } catch (Exception $e) {
            error_log('GoValidOJS: Error searching users: ' . $e->getMessage());
            error_log('GoValidOJS: Stack trace: ' . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
            exit;
        }
    }

    /**
     * Save email template
     */
    public function saveEmailTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in saveEmailTemplate');
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Plugin not found']);
            exit;
        }

        $context = $request->getContext();
        $templateData = $request->getUserVar('templateData');
        $templateId = $request->getUserVar('templateId');

        // Check if this is a JSON-based template (from new designer)
        if ($templateData) {
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);
            $result = $emailTemplateManager->saveJSONTemplate([
                'templateId' => $templateId,
                'templateData' => $templateData
            ]);

            header('Content-Type: application/json');
            echo json_encode($result);
            exit;
        }

        // Otherwise, use old handler
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['saveEmailTemplate'], $args));
    }
    
    /**
     * Delete email template
     */
    public function deleteTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in deleteTemplate');
            return false;
        }
        
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['deleteTemplate'], $args));
    }
    
    /**
     * Activate email template
     */
    public function activateTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in activateTemplate');
            return false;
        }
        
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['activateTemplate'], $args));
    }
    
    /**
     * Preview email template
     */
    public function previewTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in previewTemplate');
            return false;
        }
        
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['previewTemplate'], $args));
    }
    
    /**
     * Get default template
     */
    public function getDefaultTemplate($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }
        
        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in getDefaultTemplate');
            return false;
        }
        
        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['getDefaultTemplate'], $args));
    }
    
    /**
     * Test email template
     */
    public function testEmail($args, $request)
    {
        // Make sure plugin is loaded
        if (!$this->plugin) {
            PluginRegistry::loadCategory('generic');
            $this->plugin = PluginRegistry::getPlugin('generic', 'govalidojsplugin');
        }

        if (!$this->plugin) {
            error_log('GoValidOJS: Plugin not found in testEmail');
            return false;
        }

        return $this->plugin->handleEmailTemplateRequest($request, null, array_merge(['testEmail'], $args));
    }

    /**
     * Handle AJAX requests for designer
     *
     * @param array $args
     * @param PKPRequest $request
     */
    public function ajax($args, $request)
    {
        // This is essentially the same as designer() but specifically for AJAX
        $action = $request->getUserVar('action');
        if ($action) {
            return $this->handleAjaxAction($action, $request);
        }

        header('Content-Type: application/json');
        echo json_encode(['status' => false, 'error' => 'No action specified']);
        exit;
    }

    // ========================================================================
    // OJS Email Template Integration Methods
    // ========================================================================

    /**
     * Get all OJS email templates
     */
    protected function getOJSEmailTemplates($request)
    {
        try {
            error_log("GoValid Plugin: getOJSEmailTemplates handler called");
            $context = $request->getContext();
            error_log("GoValid Plugin: Context ID: " . ($context ? $context->getId() : 'NULL'));

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            error_log("GoValid Plugin: OJSEmailTemplateIntegration class loaded");

            $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
            error_log("GoValid Plugin: Integration instance created");

            $templates = $integration->getOJSEmailTemplates();
            error_log("GoValid Plugin: Templates loaded - count: " . count($templates));

            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
            exit;
        } catch (Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getOJSEmailTemplates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    /**
     * Get a specific OJS email template
     */
    protected function getOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
        $template = $integration->getOJSEmailTemplate($templateKey);

        if ($template) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['template' => $template]]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
        }
        exit;
    }

    /**
     * Save/update an OJS email template
     */
    protected function saveOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');
        $templateName = $request->getUserVar('templateName'); // Optional - for new templates
        $alternateTo = $request->getUserVar('alternateTo'); // Optional - for alternate templates
        $subject = $request->getUserVar('subject');
        $body = $request->getUserVar('body');

        if (!$templateKey || !$subject || !$body) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Missing required parameters']]);
            exit;
        }

        // Debug: Log what's being saved
        error_log("GoValid Plugin: Saving template '$templateKey'");
        error_log("GoValid Plugin: Subject length: " . strlen($subject));
        error_log("GoValid Plugin: Body length: " . strlen($body));
        error_log("GoValid Plugin: Body first 500 chars: " . substr($body, 0, 500));

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
        $success = $integration->saveOJSEmailTemplate($templateKey, $subject, $body, $templateName, $alternateTo);

        if ($success) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['success' => true, 'message' => 'Template saved successfully']]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Failed to save template']]);
        }
        exit;
    }

    /**
     * Reset OJS email template to default
     */
    protected function resetOJSEmailTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
        $success = $integration->resetOJSEmailTemplate($templateKey);

        if ($success) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['success' => true, 'message' => 'Template reset to default']]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Failed to reset template']]);
        }
        exit;
    }

    /**
     * Get available OJS email template variables
     */
    protected function getOJSEmailVariables($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
        $variables = $integration->getAvailableVariables($templateKey);

        header('Content-Type: application/json');
        echo json_encode(['status' => true, 'content' => ['variables' => $variables]]);
        exit;
    }

    /**
     * Apply design to an OJS email template
     */
    protected function applyDesignToOJSTemplate($request)
    {
        $context = $request->getContext();
        $templateKey = $request->getUserVar('templateKey');
        $designConfigJson = $request->getUserVar('designConfig');

        if (!$templateKey || !$designConfigJson) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Missing required parameters']]);
            exit;
        }

        $designConfig = json_decode($designConfigJson, true);
        if (!$designConfig) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Invalid design configuration']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
        $integration = new OJSEmailTemplateIntegration($this->plugin, $context);
        $result = $integration->applyDesignToOJSTemplate($templateKey, $designConfig);

        if ($result['status'] === 'success') {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => $result]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => $result]);
        }
        exit;
    }

    // ========================================================================
    // Custom Email Template Methods
    // ========================================================================

    /**
     * Get all custom email templates (saved via "Save As")
     */
    protected function getCustomEmailTemplates($request)
    {
        try {
            error_log("GoValid Plugin: getCustomEmailTemplates handler called");
            $context = $request->getContext();

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new OJSEmailTemplateIntegration($this->plugin, $context);

            $templates = $integration->getCustomEmailTemplates();
            error_log("GoValid Plugin: Custom templates loaded - count: " . count($templates));

            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
            exit;
        } catch (Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getCustomEmailTemplates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    /**
     * Get a specific custom email template by ID
     */
    protected function getCustomEmailTemplate($request)
    {
        try {
            $context = $request->getContext();
            $templateId = $request->getUserVar('templateId');

            if (!$templateId) {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'content' => ['error' => 'Template ID required']]);
                exit;
            }

            require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
            $integration = new OJSEmailTemplateIntegration($this->plugin, $context);

            $template = $integration->getCustomEmailTemplate($templateId);

            if ($template) {
                header('Content-Type: application/json');
                echo json_encode(['status' => true, 'content' => ['template' => $template]]);
            } else {
                header('Content-Type: application/json');
                echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
            }
            exit;
        } catch (Exception $e) {
            error_log("GoValid Plugin: EXCEPTION in getCustomEmailTemplate: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => $e->getMessage()]]);
            exit;
        }
    }

    // ========================================================================
    // Design Template Library Methods
    // ========================================================================

    /**
     * Get all design templates
     */
    protected function getDesignTemplates($request)
    {
        require_once($this->plugin->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
        $templates = EmailDesignTemplateLibrary::getDesignTemplates();

        header('Content-Type: application/json');
        echo json_encode(['status' => true, 'content' => ['templates' => $templates]]);
        exit;
    }

    /**
     * Get a specific design template
     */
    protected function getDesignTemplate($request)
    {
        $templateKey = $request->getUserVar('templateKey');

        if (!$templateKey) {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template key required']]);
            exit;
        }

        require_once($this->plugin->getPluginPath() . '/classes/EmailDesignTemplateLibrary.php');
        $template = EmailDesignTemplateLibrary::getDesignTemplate($templateKey);

        if ($template) {
            header('Content-Type: application/json');
            echo json_encode(['status' => true, 'content' => ['template' => $template]]);
        } else {
            header('Content-Type: application/json');
            echo json_encode(['status' => false, 'content' => ['error' => 'Template not found']]);
        }
        exit;
    }

    /**
     * Send certificate via email with attachment
     */
    public function sendCertificateEmail($args, $request)
    {
        $context = $request->getContext();

        // Get parameters
        $templateName = $request->getUserVar('templateName');
        $emailSubject = $request->getUserVar('emailSubject'); // Edited subject from preview
        $emailBody = $request->getUserVar('emailBody'); // Edited body from preview
        $toEmail = $request->getUserVar('toEmail');
        $ccEmail = $request->getUserVar('ccEmail');
        $bccEmail = $request->getUserVar('bccEmail');
        $attachmentFormat = $request->getUserVar('attachmentFormat'); // 'pdf' or 'jpg'
        $certificateImageData = $request->getUserVar('certificateImageData'); // Legacy: Direct canvas rendering (base64)
        $canvasData = $request->getUserVar('canvasData'); // Legacy support
        $replacementsJson = $request->getUserVar('replacements');
        $orientation = $request->getUserVar('orientation');

        // Check for file upload (new method - Blob instead of base64 string)
        $hasCertificateFile = isset($_FILES['certificateImageFile']) && $_FILES['certificateImageFile']['error'] === UPLOAD_ERR_OK;

        if (!$context) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'No context']);
            exit;
        }

        // Check if we have certificate data in any form
        if (empty($toEmail) || (!$hasCertificateFile && empty($certificateImageData) && empty($canvasData)) || empty($replacementsJson)) {
            header('Content-Type: application/json');
            echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
            exit;
        }

        // Use edited content if provided, otherwise load from template
        if (empty($emailSubject) || empty($emailBody)) {
            if (empty($templateName)) {
                header('Content-Type: application/json');
                echo json_encode(['status' => 'error', 'message' => 'Missing email content']);
                exit;
            }
        }

        try {
            // Parse replacements
            $replacements = json_decode($replacementsJson, true);
            if (!$replacements) {
                throw new Exception('Invalid replacements data');
            }

            // Initialize email template manager
            require_once($this->plugin->getPluginPath() . '/classes/EmailTemplateManager.php');
            $emailTemplateManager = new EmailTemplateManager($this->plugin, $context);

            // Use edited email content if provided, otherwise load from template
            if (!empty($emailSubject) && !empty($emailBody)) {
                // Use the edited content from the preview
                $emailHtml = $emailBody;
                $finalEmailSubject = $emailSubject;
            } else {
                // Load from template (fallback)
                require_once($this->plugin->getPluginPath() . '/classes/OJSEmailTemplateIntegration.php');
                $integration = new OJSEmailTemplateIntegration($this->plugin, $context);

                $ojsTemplate = $integration->getTemplateByKey($templateName);
                if (!$ojsTemplate) {
                    throw new Exception('Email template not found: ' . $templateName);
                }

                $emailHtml = $ojsTemplate['body'] ?? $ojsTemplate['html'] ?? '';
                $finalEmailSubject = $ojsTemplate['subject'] ?? 'Certificate';
            }

            // Generate or process certificate file
            $certificateData = null;
            $attachmentFilename = '';
            $mimeType = '';

            if ($hasCertificateFile) {
                // New method: File upload (Blob) - direct binary data
                error_log("GoValid Plugin: Using uploaded certificate file (Blob method)");

                $certificateData = file_get_contents($_FILES['certificateImageFile']['tmp_name']);

                if ($certificateData === false) {
                    throw new Exception('Failed to read uploaded certificate file');
                }

                $attachmentFilename = 'certificate_' . time() . '.jpg';
                $mimeType = 'image/jpeg';

                error_log("GoValid Plugin: Certificate file uploaded successfully, size: " . number_format(strlen($certificateData) / 1024 / 1024, 2) . " MB");

                // Verify it's a valid JPEG by checking magic bytes
                $magicBytes = bin2hex(substr($certificateData, 0, 4));
                error_log("GoValid Plugin: JPEG magic bytes: {$magicBytes} (should start with ffd8ff)");

                if (substr($magicBytes, 0, 6) !== 'ffd8ff') {
                    error_log("GoValid Plugin: WARNING - File does not appear to be a valid JPEG!");
                }

            } elseif (!empty($certificateImageData)) {
                // Legacy method: Direct canvas rendering - decode base64 JPG data
                error_log("GoValid Plugin: Using legacy base64 method (may have encoding issues)");

                // Remove data URL prefix if present (data:image/jpeg;base64,)
                if (strpos($certificateImageData, 'data:image/jpeg;base64,') === 0) {
                    $certificateImageData = substr($certificateImageData, strlen('data:image/jpeg;base64,'));
                } elseif (strpos($certificateImageData, 'data:image/jpg;base64,') === 0) {
                    $certificateImageData = substr($certificateImageData, strlen('data:image/jpg;base64,'));
                }

                $certificateData = base64_decode($certificateImageData);

                if ($certificateData === false) {
                    throw new Exception('Failed to decode certificate image data');
                }

                $attachmentFilename = 'certificate_' . time() . '.jpg';
                $mimeType = 'image/jpeg';

                error_log("GoValid Plugin: Certificate JPG decoded successfully, size: " . number_format(strlen($certificateData) / 1024 / 1024, 2) . " MB");

                // Verify it's a valid JPEG by checking magic bytes
                $magicBytes = bin2hex(substr($certificateData, 0, 4));
                error_log("GoValid Plugin: JPEG magic bytes: {$magicBytes} (should start with ffd8ff)");

            } else {
                // Legacy support: Generate from canvas data (PDF/JPG)
                error_log("GoValid Plugin: Using legacy PDF/JPG generation from canvas data");

                require_once($this->plugin->getPluginPath() . '/classes/CertificateGenerator.php');
                $certificateGenerator = new CertificateGenerator($this->plugin, $context);

                $canvasArray = json_decode($canvasData, true);
                if (!$canvasArray) {
                    throw new Exception('Invalid canvas data');
                }

                // Generate certificate
                if ($attachmentFormat === 'jpg') {
                    $certificateData = $certificateGenerator->generateJPG($canvasArray, $replacements, $orientation);
                    $attachmentFilename = 'certificate_' . time() . '.jpg';
                    $mimeType = 'image/jpeg';
                } else {
                    // Default to PDF
                    $certificateData = $certificateGenerator->generatePDF($canvasArray, $replacements, $orientation);
                    $attachmentFilename = 'certificate_' . time() . '.pdf';
                    $mimeType = 'application/pdf';
                }
            }

            if (!$certificateData) {
                throw new Exception('Failed to generate certificate');
            }

            // Check certificate size
            $certificateSizeMB = strlen($certificateData) / 1024 / 1024;
            error_log("GoValid Plugin: Certificate size: " . number_format($certificateSizeMB, 2) . " MB");

            // Limit: Certificate max 5MB
            if (strlen($certificateData) > 5 * 1024 * 1024) {
                throw new Exception("Certificate file is too large (" . number_format($certificateSizeMB, 2) . " MB). Please simplify your design, reduce image sizes, or remove some elements. Maximum allowed: 5MB");
            }

            // Handle additional attachments
            $additionalAttachments = [];
            $totalAdditionalSize = 0;

            if (isset($_FILES['additionalAttachments']) && is_array($_FILES['additionalAttachments']['name'])) {
                $fileCount = count($_FILES['additionalAttachments']['name']);
                error_log("GoValid Plugin: Found {$fileCount} additional attachments for certificate");

                for ($i = 0; $i < $fileCount; $i++) {
                    if ($_FILES['additionalAttachments']['error'][$i] === UPLOAD_ERR_OK) {
                        $tmpPath = $_FILES['additionalAttachments']['tmp_name'][$i];
                        $fileName = $_FILES['additionalAttachments']['name'][$i];
                        $fileSize = $_FILES['additionalAttachments']['size'][$i];
                        $fileType = $_FILES['additionalAttachments']['type'][$i];

                        // Validate file size (5MB max per file)
                        if ($fileSize > 5 * 1024 * 1024) {
                            throw new Exception("File {$fileName} exceeds 5MB limit");
                        }

                        // Validate file type
                        $allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                        $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                        $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png', 'gif'];

                        if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, $allowedExtensions)) {
                            throw new Exception("File {$fileName} has invalid type. Only PDF and images allowed.");
                        }

                        // Read file contents
                        $fileContents = file_get_contents($tmpPath);

                        if ($fileContents !== false) {
                            $totalAdditionalSize += strlen($fileContents);

                            $additionalAttachments[] = [
                                'name' => $fileName,
                                'data' => $fileContents,
                                'type' => $fileType
                            ];
                            error_log("GoValid Plugin: Added additional attachment: {$fileName} ({$fileSize} bytes, {$fileType})");
                        }
                    }
                }
            }

            // Check total email size (Certificate + Additional Attachments)
            // Limit: Total 10MB for all attachments combined
            $totalSize = strlen($certificateData) + $totalAdditionalSize;
            $totalSizeMB = $totalSize / 1024 / 1024;

            if ($totalSize > 10 * 1024 * 1024) {
                $certSizeMB = number_format(strlen($certificateData) / 1024 / 1024, 2);
                $addSizeMB = number_format($totalAdditionalSize / 1024 / 1024, 2);
                throw new Exception("Total email size is too large (" . number_format($totalSizeMB, 2) . " MB). Certificate: {$certSizeMB} MB + Additional files: {$addSizeMB} MB. Maximum allowed: 10MB total. Please reduce file sizes or remove some attachments.");
            }

            error_log("GoValid Plugin: Total email size: " . number_format($totalSizeMB, 2) . " MB (Certificate: " . number_format($certificateSizeMB, 2) . " MB + Additional: " . number_format($totalAdditionalSize / 1024 / 1024, 2) . " MB)");

            // Send email with certificate attachment and additional attachments
            $result = $emailTemplateManager->sendEmailWithAttachment(
                $toEmail,
                $finalEmailSubject,
                $emailHtml,
                $ccEmail,
                $bccEmail,
                $certificateData,
                $attachmentFilename,
                $mimeType,
                $additionalAttachments
            );

            if ($result['status'] === 'success') {
                header('Content-Type: application/json');
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Certificate sent successfully'
                ]);
            } else {
                throw new Exception($result['message']);
            }

        } catch (\Throwable $e) {
            error_log('GoValidOJS: Error sending certificate email: ' . $e->getMessage());
            header('Content-Type: application/json');
            echo json_encode([
                'status' => 'error',
                'message' => 'Server Error: ' . $e->getMessage()
            ]);
        }

        exit;
    }

    /**
     * Proxy request to GoValid API for PIN status
     */
    private function proxyPinStatus($request)
    {
        $token = $request->getUserVar('token');
        
        if (!$token) {
            // Try to get token from Authorization header
            $headers = getallheaders();
            if (isset($headers['Authorization'])) {
                $token = str_replace('Bearer ', '', $headers['Authorization']);
            }
        }

        if (!$token) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No authentication token provided']);
            exit;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://my.govalid.org/api/v1/ojs/pin-status/');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Content-Type: application/json'
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log('GoValidOJS: CURL error in proxyPinStatus: ' . $error);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'Network error: ' . $error]);
            exit;
        }

        header('Content-Type: application/json');
        http_response_code($httpCode);
        echo $response;
        exit;
    }

    /**
     * Proxy request to GoValid API for PIN verification
     */
    private function proxyVerifyPin($request)
    {
        $token = $request->getUserVar('token');
        $pin = $request->getUserVar('pin');
        
        if (!$token) {
            // Try to get token from Authorization header
            $headers = getallheaders();
            if (isset($headers['Authorization'])) {
                $token = str_replace('Bearer ', '', $headers['Authorization']);
            }
        }

        if (!$token) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No authentication token provided']);
            exit;
        }

        if (!$pin) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'No PIN provided']);
            exit;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://my.govalid.org/api/v1/ojs/verify-pin/');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['pin' => $pin]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Content-Type: application/json'
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log('GoValidOJS: CURL error in proxyVerifyPin: ' . $error);
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'error' => 'Network error: ' . $error]);
            exit;
        }

        // Parse the response to check if PIN was verified successfully
        $responseData = json_decode($response, true);

        // Handle nested response structure from GoValid API
        $content = $responseData['content'] ?? $responseData;
        $isSuccess = ($content['success'] ?? false) || ($responseData['success'] ?? false);

        if ($httpCode >= 200 && $httpCode < 300 && $isSuccess) {
            // PIN verified successfully - set OJS session variables
            import('lib.pkp.classes.session.SessionManager');
            $sessionManager = SessionManager::getManager();
            $session = $sessionManager->getUserSession();

            // Get user email from the request or auth manager
            require_once($this->plugin->getPluginPath() . '/classes/GoValidAuthManager.php');
            $authManager = new GoValidAuthManager();
            $userInfo = $authManager->getCurrentUser();
            $userEmail = $userInfo['email'] ?? '';

            // Get the qr_session_token from the backend response (NEW: backend now provides this)
            $qrSessionToken = $content['qr_session_token'] ?? $responseData['qr_session_token'] ?? null;

            error_log('GoValidOJS: PIN verification response content: ' . json_encode($content));
            error_log('GoValidOJS: qr_session_token from backend: ' . ($qrSessionToken ? substr($qrSessionToken, 0, 20) . '...' : 'NOT PROVIDED'));

            // Set session variables to indicate verification is complete
            $session->setSessionVar('govalid_otp_verified', true);
            $session->setSessionVar('govalid_otp_email', $userEmail);
            $session->setSessionVar('govalid_otp_expiry', time() + 300); // 5 minutes (matches backend)

            // Use the server-provided qr_session_token
            if ($qrSessionToken) {
                $session->setSessionVar('govalid_otp_session_token', $qrSessionToken);
                error_log('GoValidOJS: Using backend-provided qr_session_token');
            } else {
                // Fallback (should not happen with updated backend)
                error_log('GoValidOJS: WARNING - No qr_session_token from backend, PIN verification may not work for signed certs');
            }

            error_log('GoValidOJS: PIN verified successfully, OJS session variables set');

            // Ensure the qr_session_token is in the response for JavaScript
            if ($qrSessionToken) {
                $responseData['session_token'] = $qrSessionToken;
                $responseData['qr_session_token'] = $qrSessionToken;
                if (isset($responseData['content'])) {
                    $responseData['content']['session_token'] = $qrSessionToken;
                    $responseData['content']['qr_session_token'] = $qrSessionToken;
                }
            }
            $response = json_encode($responseData);
        }

        header('Content-Type: application/json');
        http_response_code($httpCode);
        echo $response;
        exit;
    }
}