    // ============================================================
    // UNDO/REDO HISTORY SYSTEM
    // ============================================================

    const historyStack = [];
    let historyIndex = -1;
    const MAX_HISTORY = 50;
    let isUndoRedo = false; // Flag to prevent saving state during undo/redo
    let clipboard = null; // For copy/paste

    // Save current state to history
    function saveHistory() {
        if (isUndoRedo) return; // Don't save during undo/redo operations

        // Get canvas state as JSON
        const state = JSON.stringify(canvas.toJSON([
            'fieldType', 'id', 'selectable', 'evented', 'hasControls',
            'lockMovementX', 'lockMovementY', 'backgroundMode'
        ]));

        // Remove any states after current index (for new branch)
        if (historyIndex < historyStack.length - 1) {
            historyStack.splice(historyIndex + 1);
        }

        // Add new state
        historyStack.push(state);
        historyIndex++;

        // Limit history size
        if (historyStack.length > MAX_HISTORY) {
            historyStack.shift();
            historyIndex--;
        }

        // Update button states
        updateUndoRedoButtons();

        console.log('History saved. Stack size:', historyStack.length, 'Index:', historyIndex);
    }

    // Undo last action
    function undo() {
        if (historyIndex <= 0) {
            console.log('Nothing to undo');
            showHistoryNotification('Nothing to undo');
            return;
        }

        isUndoRedo = true;
        historyIndex--;

        const state = historyStack[historyIndex];
        canvas.loadFromJSON(state, function() {
            canvas.renderAll();
            updateLayerUI();
            selectedObject = null;
            document.getElementById('objectProperties').innerHTML = '<p>Select an object to edit properties</p>';
            isUndoRedo = false;
            updateUndoRedoButtons();
            showHistoryNotification('Undo');
            console.log('Undo performed. Index:', historyIndex);
        });
    }

    // Redo last undone action
    function redo() {
        if (historyIndex >= historyStack.length - 1) {
            console.log('Nothing to redo');
            showHistoryNotification('Nothing to redo');
            return;
        }

        isUndoRedo = true;
        historyIndex++;

        const state = historyStack[historyIndex];
        canvas.loadFromJSON(state, function() {
            canvas.renderAll();
            updateLayerUI();
            selectedObject = null;
            document.getElementById('objectProperties').innerHTML = '<p>Select an object to edit properties</p>';
            isUndoRedo = false;
            updateUndoRedoButtons();
            showHistoryNotification('Redo');
            console.log('Redo performed. Index:', historyIndex);
        });
    }

    // Update undo/redo button states
    function updateUndoRedoButtons() {
        const undoBtn = document.getElementById('undoBtn');
        const redoBtn = document.getElementById('redoBtn');
        const undoBtn2 = document.getElementById('undoBtn2');
        const redoBtn2 = document.getElementById('redoBtn2');

        const undoDisabled = historyIndex <= 0;
        const redoDisabled = historyIndex >= historyStack.length - 1;

        if (undoBtn) {
            undoBtn.disabled = undoDisabled;
            undoBtn.style.opacity = undoDisabled ? '0.5' : '1';
        }
        if (undoBtn2) {
            undoBtn2.disabled = undoDisabled;
            undoBtn2.style.opacity = undoDisabled ? '0.5' : '1';
        }
        if (redoBtn) {
            redoBtn.disabled = redoDisabled;
            redoBtn.style.opacity = redoDisabled ? '0.5' : '1';
        }
        if (redoBtn2) {
            redoBtn2.disabled = redoDisabled;
            redoBtn2.style.opacity = redoDisabled ? '0.5' : '1';
        }
    }

    // Show notification for history actions
    function showHistoryNotification(action) {
        let notification = document.getElementById('history-notification');
        if (!notification) {
            notification = document.createElement('div');
            notification.id = 'history-notification';
            notification.style.cssText = 'position: fixed; bottom: 20px; left: 50%; transform: translateX(-50%); background: #1e293b; color: white; padding: 8px 16px; border-radius: 6px; font-size: 12px; z-index: 10000; opacity: 0; transition: opacity 0.2s;';
            document.body.appendChild(notification);
        }

        notification.textContent = action;
        notification.style.opacity = '1';

        setTimeout(function() {
            notification.style.opacity = '0';
        }, 1500);
    }

    // Copy selected object
    function copyObject() {
        if (!selectedObject) {
            showHistoryNotification('No object selected');
            return;
        }

        selectedObject.clone(function(cloned) {
            clipboard = cloned;
            showHistoryNotification('Copied');
            console.log('Object copied to clipboard');
        }, ['fieldType', 'id']);
    }

    // Paste from clipboard
    function pasteObject() {
        if (!clipboard) {
            showHistoryNotification('Nothing to paste');
            return;
        }

        clipboard.clone(function(cloned) {
            // Offset the pasted object slightly
            cloned.set({
                left: cloned.left + 20,
                top: cloned.top + 20,
                evented: true
            });

            // Generate new ID if it has one
            if (cloned.id) {
                cloned.id = cloned.id + '_copy_' + Date.now();
            }

            canvas.add(cloned);
            canvas.setActiveObject(cloned);
            canvas.renderAll();
            updateLayerUI();
            saveHistory();
            showHistoryNotification('Pasted');
            console.log('Object pasted from clipboard');
        }, ['fieldType', 'id']);
    }

    // Delete selected object with history
    function deleteSelectedObject() {
        if (selectedObject) {
            canvas.remove(selectedObject);
            selectedObject = null;
            document.getElementById('objectProperties').innerHTML = '<p>Select an object to edit properties</p>';
            updateLayerUI();
            saveHistory();
            showHistoryNotification('Deleted');
            console.log('Object deleted');
        } else {
            showHistoryNotification('No object selected');
        }
    }

    // Setup keyboard shortcuts
    function setupKeyboardShortcuts() {
        document.addEventListener('keydown', function(e) {
            // Check if user is typing in an input field
            const activeElement = document.activeElement;
            const isTyping = activeElement && (
                activeElement.tagName === 'INPUT' ||
                activeElement.tagName === 'TEXTAREA' ||
                activeElement.isContentEditable ||
                (activeElement.tagName === 'CANVAS' && canvas.getActiveObject() && canvas.getActiveObject().isEditing)
            );

            // Allow keyboard shortcuts even when typing in some cases
            const isCtrlOrCmd = e.ctrlKey || e.metaKey;

            // Undo: Ctrl+Z
            if (isCtrlOrCmd && e.key === 'z' && !e.shiftKey) {
                if (!isTyping) {
                    e.preventDefault();
                    undo();
                }
                return;
            }

            // Redo: Ctrl+Y or Ctrl+Shift+Z
            if ((isCtrlOrCmd && e.key === 'y') || (isCtrlOrCmd && e.key === 'z' && e.shiftKey)) {
                if (!isTyping) {
                    e.preventDefault();
                    redo();
                }
                return;
            }

            // Copy: Ctrl+C
            if (isCtrlOrCmd && e.key === 'c') {
                if (!isTyping && selectedObject) {
                    e.preventDefault();
                    copyObject();
                }
                return;
            }

            // Paste: Ctrl+V
            if (isCtrlOrCmd && e.key === 'v') {
                if (!isTyping && clipboard) {
                    e.preventDefault();
                    pasteObject();
                }
                return;
            }

            // Delete or Backspace: Delete selected object
            if ((e.key === 'Delete' || e.key === 'Backspace') && !isTyping) {
                e.preventDefault();
                deleteSelectedObject();
                return;
            }
        });

        console.log('Keyboard shortcuts initialized: Ctrl+Z (undo), Ctrl+Y/Ctrl+Shift+Z (redo), Ctrl+C (copy), Ctrl+V (paste), Delete/Backspace (delete)');
    }

    // Initialize history with current state
    function initializeHistory() {
        historyStack.length = 0;
        historyIndex = -1;
        saveHistory(); // Save initial state
        console.log('History system initialized');
    }

    // Make functions globally accessible
    window.undo = undo;
    window.redo = redo;
    window.copyObject = copyObject;
    window.pasteObject = pasteObject;
    window.deleteSelectedObject = deleteSelectedObject;
    window.saveHistory = saveHistory;

    // ============================================================
    // END UNDO/REDO HISTORY SYSTEM
    // ============================================================

    function restoreBackgroundControls() {
        const bgImage = canvas.backgroundImage;
        if (bgImage && bgImage.src) {
            const bgImageModeSelect = document.getElementById('bgImageMode');
            const removeBgImageBtn = document.getElementById('removeBgImage');

            // Store the loaded background image in new format
            currentBackgroundImage = {
                src: bgImage.src,
                width: bgImage.width,
                height: bgImage.height,
                fabricImage: bgImage
            };
            currentBackgroundMode = bgImage.backgroundMode || 'fit';

            console.log('Restored background image controls:', {
                hasSrc: !!currentBackgroundImage.src,
                mode: currentBackgroundMode
            });

            // Show controls and set mode
            if (bgImageModeSelect) {
                bgImageModeSelect.style.display = 'inline-block';
                bgImageModeSelect.value = currentBackgroundMode;
            }
            if (removeBgImageBtn) {
                removeBgImageBtn.style.display = 'inline-block';
            }
        }
    }

    // Load existing template if available
    // This will be handled via AJAX or template data passed from backend

    // Add text functionality
    document.getElementById('addText').addEventListener('click', function() {
        requireAuth('addText', function() {
            var text = new fabric.Textbox('Sample Text', {
                left: 100,
                top: 100,
                width: 300,  // Initial width for text wrapping
                fontFamily: 'Arial',
                fontSize: 20,
                fill: '#000000',
                textBaseline: 'alphabetic',  // Set valid baseline
                editable: true,  // Allow double-click to edit
                splitByGrapheme: true  // Better line breaking
            });
            canvas.add(text);
            canvas.setActiveObject(text);
            updateProperties(text);
        });
    });

    // Add image functionality
    document.getElementById('addImage').addEventListener('click', function() {
        requireAuth('addImage', function() {
            var input = document.createElement('input');
            input.type = 'file';
            input.accept = 'image/*';
            input.onchange = function(e) {
                var file = e.target.files[0];
                var reader = new FileReader();
                reader.onload = function(f) {
                    fabric.Image.fromURL(f.target.result, function(img) {
                        img.set({
                            left: 50,
                            top: 50,
                            scaleX: 0.5,
                            scaleY: 0.5
                        });
                        canvas.add(img);
                        canvas.setActiveObject(img);
                        updateProperties(img);
                    });
                };
                reader.readAsDataURL(file);
            };
            input.click();
        });
    });

    // Get QR size based on security level
    function getQRSizeBasedOnSecurity() {
        const securityLevel = getQRSettings().security_level;

        switch(securityLevel) {
            case 'SMART':
                return { width: 60, height: 60 }; // Ultra-compact for small labels
            case 'SECURE':
                return { width: 125, height: 125 };
            case 'ENTERPRISE':
                return { width: 150, height: 150 };
            default:
                return { width: 125, height: 125 }; // Default to secure size
        }
    }

    // Create QR mockup with dynamic sizing
    function createQRMockup(size) {
        const canvas = document.createElement('canvas');
        canvas.width = size.width;
        canvas.height = size.height;
        const ctx = canvas.getContext('2d');
        
        // Add padding
        const padding = Math.round(size.width * 0.08);
        const innerSize = size.width - (padding * 2);
        
        // Fill with white background
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(0, 0, size.width, size.height);
        
        // Draw border
        ctx.strokeStyle = '#000000';
        ctx.lineWidth = Math.max(1, Math.round(size.width * 0.01));
        ctx.strokeRect(padding, padding, innerSize, innerSize);
        
        // Draw QR code pattern - more structured and professional looking
        ctx.fillStyle = '#000000';
        
        // Scale elements based on size
        const cornerSize = Math.round(innerSize * 0.22);
        const centerSize = Math.round(innerSize * 0.18);
        const patternSize = Math.max(2, Math.round(innerSize * 0.06));
        const patternGap = Math.max(1, Math.round(innerSize * 0.02));
        
        // Position finder patterns (the three large squares in corners)
        // Top-left corner
        ctx.fillRect(padding, padding, cornerSize, cornerSize);
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(padding + patternSize, padding + patternSize, cornerSize - (patternSize * 2), cornerSize - (patternSize * 2));
        ctx.fillStyle = '#000000';
        ctx.fillRect(padding + (patternSize * 2), padding + (patternSize * 2), cornerSize - (patternSize * 4), cornerSize - (patternSize * 4));
        
        // Top-right corner
        ctx.fillRect(padding + innerSize - cornerSize, padding, cornerSize, cornerSize);
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(padding + innerSize - cornerSize + patternSize, padding + patternSize, cornerSize - (patternSize * 2), cornerSize - (patternSize * 2));
        ctx.fillStyle = '#000000';
        ctx.fillRect(padding + innerSize - cornerSize + (patternSize * 2), padding + (patternSize * 2), cornerSize - (patternSize * 4), cornerSize - (patternSize * 4));
        
        // Bottom-left corner
        ctx.fillRect(padding, padding + innerSize - cornerSize, cornerSize, cornerSize);
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(padding + patternSize, padding + innerSize - cornerSize + patternSize, cornerSize - (patternSize * 2), cornerSize - (patternSize * 2));
        ctx.fillStyle = '#000000';
        ctx.fillRect(padding + (patternSize * 2), padding + innerSize - cornerSize + (patternSize * 2), cornerSize - (patternSize * 4), cornerSize - (patternSize * 4));
        
        // Center alignment pattern
        ctx.fillRect(padding + (innerSize - centerSize) / 2, padding + (innerSize - centerSize) / 2, centerSize, centerSize);
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(padding + (innerSize - centerSize) / 2 + patternSize, padding + (innerSize - centerSize) / 2 + patternSize, centerSize - (patternSize * 2), centerSize - (patternSize * 2));
        ctx.fillStyle = '#000000';
        ctx.fillRect(padding + (innerSize - centerSize) / 2 + (patternSize * 2), padding + (innerSize - centerSize) / 2 + (patternSize * 2), centerSize - (patternSize * 4), centerSize - (patternSize * 4));
        
        // Create structured data pattern (more organized than random)
        const startX = padding + cornerSize + patternGap;
        const endX = padding + innerSize - cornerSize - patternGap;
        const startY = padding + cornerSize + patternGap;
        const endY = padding + innerSize - cornerSize - patternGap;
        
        // Create a more structured pattern
        for (let i = startX; i < endX; i += patternSize + patternGap) {
            for (let j = startY; j < endY; j += patternSize + patternGap) {
                // Skip center alignment pattern area
                const centerStartX = padding + (innerSize - centerSize) / 2 - patternGap;
                const centerEndX = padding + (innerSize - centerSize) / 2 + centerSize + patternGap;
                const centerStartY = padding + (innerSize - centerSize) / 2 - patternGap;
                const centerEndY = padding + (innerSize - centerSize) / 2 + centerSize + patternGap;
                
                if (i > centerStartX && i < centerEndX && j > centerStartY && j < centerEndY) {
                    continue;
                }
                
                // Create a more structured pattern based on position
                if ((i + j) % (patternSize * 2) === 0 || 
                    (i * j) % (patternSize * 3) === 0) {
                    ctx.fillRect(i, j, patternSize, patternSize);
                }
            }
        }
        
        // Add timing patterns (the lines connecting the corner squares)
        const timingY = padding + cornerSize / 2;
        const timingX = padding + cornerSize / 2;
        const timingEndX = padding + innerSize - cornerSize / 2;
        const timingEndY = padding + innerSize - cornerSize / 2;
        
        for (let i = padding + cornerSize; i < padding + innerSize - cornerSize; i += patternSize * 2) {
            // Horizontal timing pattern
            ctx.fillRect(i, timingY, patternSize, patternSize);
            // Vertical timing pattern
            ctx.fillRect(timingX, i, patternSize, patternSize);
        }
        
        return canvas.toDataURL();
    }

    // Update existing QR objects when security level changes
    function updateQRObjectsSizes() {
        const currentSecurityLevel = getQRSettings().security_level;
        const newSize = getQRSizeBasedOnSecurity();
        
        canvas.getObjects().forEach(function(obj) {
            if (obj.isQRCode) {
                // Update QR object with new size and mockup
                const qrMockupUrl = createQRMockup(newSize);
                
                fabric.Image.fromURL(qrMockupUrl, function(newImg) {
                    // Preserve original position
                    newImg.set({
                        left: obj.left,
                        top: obj.top,
                        width: newSize.width,
                        height: newSize.height,
                        scaleX: 1,
                        scaleY: 1
                    });
                    
                    // Update metadata
                    newImg.qrSettings = getQRSettings();
                    newImg.isQRCode = true;
                    newImg.securityLevel = currentSecurityLevel;
                    
                    // Replace the old object
                    canvas.remove(obj);
                    canvas.add(newImg);
                    canvas.renderAll();
                });
            }
        });
    }

    // Add QR code placeholder with dynamic sizing
    // QR Code button removed from toolbar - QR codes can be added via drag-and-drop from Content tab
    // Keeping the QR code functionality for backward compatibility with templates
    const addQRCodeBtn = document.getElementById('addQRCode');
    if (addQRCodeBtn) {
        addQRCodeBtn.addEventListener('click', function() {
            requireAuth('addQRCode', function() {
                // Get size based on current security level
                const qrSize = getQRSizeBasedOnSecurity();
                const securityLevel = getQRSettings().security_level;

                // Create QR mockup image with proper size
                const qrMockupUrl = createQRMockup(qrSize);

                fabric.Image.fromURL(qrMockupUrl, function(img) {
                    img.set({
                        left: 650,
                        top: 450,
                        width: qrSize.width,
                        height: qrSize.height,
                        scaleX: 1,
                        scaleY: 1
                    });

                    // Add QR settings metadata to the image
                    img.qrSettings = getQRSettings();
                    img.isQRCode = true;
                    img.securityLevel = securityLevel;

                    canvas.add(img);
                    canvas.setActiveObject(img);
                    updateProperties(img);
                });
            });
        });
    }

    // Helper function to get template name from either input
    function getTemplateName() {
        const toolbarInput = document.getElementById('templateNameToolbar');
        const canvasInput = document.getElementById('templateName');
        // Prefer toolbar input if it has value, otherwise use canvas input
        if (toolbarInput && toolbarInput.value.trim()) {
            return toolbarInput.value.trim();
        }
        if (canvasInput && canvasInput.value.trim()) {
            return canvasInput.value.trim();
        }
        return '';
    }

    // Helper function to set template name in both inputs
    function setTemplateName(name) {
        const toolbarInput = document.getElementById('templateNameToolbar');
        const canvasInput = document.getElementById('templateName');
        if (toolbarInput) toolbarInput.value = name;
        if (canvasInput) canvasInput.value = name;
    }

    // Save template handler function
    async function handleSaveTemplate() {
        var templateName = getTemplateName();
        if (!templateName) {
            alert('Please enter a template name');
            return;
        }

        // Check GoValid subscription before saving
        const canGenerate = await checkBeforeSave();
        if (!canGenerate) {
            return;
        }

        // Check template limit before saving
        checkTemplateLimit(function(canSave, templateCount, isUpdate) {
            if (!canSave) {
                alert('Maximum 10 templates allowed. Please delete an existing template before saving a new one.');
                return;
            }

            // Show confirmation dialog
            let confirmMessage;
            if (isUpdate) {
                // Overwrite warning for existing template
                confirmMessage = `⚠️ Template Already Exists\n\n` +
                                `A template named "${templateName}" already exists.\n\n` +
                                `Do you want to overwrite it?`;
            } else {
                // Confirmation for new template
                confirmMessage = `💾 Save Template\n\n` +
                                `Template Name: ${templateName}\n` +
                                `Orientation: ${currentOrientation.charAt(0).toUpperCase() + currentOrientation.slice(1)}\n\n` +
                                `Do you want to save this template?`;
            }

            // Ask for confirmation
            if (!confirm(confirmMessage)) {
                console.log('Save cancelled by user');
                return;
            }

            // Before saving, temporarily restore placeholder text for any objects that have fieldType
            // This ensures we save the template with placeholders, not replaced values
            // Store original values to restore after saving
            const originalTextValues = [];

            canvas.getObjects().forEach(function(obj, index) {
                if (obj.fieldType && typeof obj.fieldType === 'string' && obj.fieldType.startsWith('{')) {
                    // Handle text objects
                    if (obj.type === 'text' || obj.type === 'textbox' || obj.type === 'i-text') {
                        // Store original value
                        originalTextValues.push({ index: index, type: 'text', originalText: obj.text });
                        obj.set('text', obj.fieldType);
                    }
                    // Handle group objects (like QR code group)
                    else if (obj.type === 'group' && obj._objects) {
                        const groupOriginals = [];
                        obj._objects.forEach(function(child, childIndex) {
                            if (child.type === 'text' && child.text) {
                                // Store original value
                                groupOriginals.push({ childIndex: childIndex, originalText: child.text });
                                // Restore the label text inside the group
                                child.set('text', obj.fieldType);
                            }
                        });
                        if (groupOriginals.length > 0) {
                            originalTextValues.push({ index: index, type: 'group', children: groupOriginals });
                        }
                    }
                }
            });

            var templateData = {
                name: templateName,
                data: JSON.stringify(canvas.toJSON()),
                orientation: currentOrientation,
                timestamp: new Date().toISOString()
            };

            // Restore original text values after getting the JSON for saving
            originalTextValues.forEach(function(item) {
                const obj = canvas.getObjects()[item.index];
                if (!obj) return;

                if (item.type === 'text') {
                    obj.set('text', item.originalText);
                } else if (item.type === 'group' && obj._objects) {
                    item.children.forEach(function(child) {
                        if (obj._objects[child.childIndex]) {
                            obj._objects[child.childIndex].set('text', child.originalText);
                        }
                    });
                }
            });
            canvas.renderAll();

            console.log('Saving template:', templateData);
            saveTemplateToServer(templateData, isUpdate);
        });
    }

    // Add event listeners for both save buttons
    const saveTemplateBtn = document.getElementById('saveTemplate');
    if (saveTemplateBtn) {
        saveTemplateBtn.addEventListener('click', handleSaveTemplate);
    }
    const saveTemplateToolbarBtn = document.getElementById('saveTemplateToolbar');
    if (saveTemplateToolbarBtn) {
        saveTemplateToolbarBtn.addEventListener('click', handleSaveTemplate);
    }

    function checkTemplateLimit(callback) {
        const templateName = getTemplateName();
        
        const cleanUrl = getCleanURL();
        fetch(cleanUrl.split('?')[0] + '?verb=designer&plugin=goValidOJS&category=generic&action=list', {
            method: 'GET'
        })
        .then(response => response.json())
        .then(data => {
            const templates = data.templates || [];
            const existingTemplate = templates.find(t => t.name === templateName);
            const isUpdate = !!existingTemplate;
            const canSave = isUpdate || templates.length < 10;
            
            callback(canSave, templates.length, isUpdate);
        })
        .catch(error => {
            console.error('Error checking template limit:', error);
            callback(true, 0, false); // Allow save if check fails
        });
    }
    
    function saveTemplateToServer(templateData, isUpdate) {
        var saveStatus = document.getElementById('saveStatus');
        
        // Send to server via AJAX
        fetch(getAjaxURL(), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'templateData=' + encodeURIComponent(JSON.stringify(templateData)) + '&action=save'
        })
        .then(response => {
            console.log('Save response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Save response data:', data);
            // Check both data.status and data.content.status for compatibility
            const responseStatus = data.content ? data.content.status : data.status;
            const responseMessage = data.content ? data.content.message : data.message;

            if (responseStatus) {
                saveStatus.className = 'save-status success';
                const action = isUpdate ? 'updated' : 'saved';
                saveStatus.textContent = 'Template "' + templateData.name + '" ' + action + ' successfully!';
                saveStatus.style.display = 'block';
                setTemplateName('');

                // Wait a bit before reloading template list to ensure save is complete
                setTimeout(() => {
                    loadTemplateList();
                }, 500);

                setTimeout(() => {
                    saveStatus.style.display = 'none';
                }, 3000);
            } else {
                // Handle server-side errors (like template limit)
                saveStatus.className = 'save-status error';
                saveStatus.textContent = responseMessage || 'Error saving template';
                saveStatus.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Save error:', error);
            saveStatus.className = 'save-status error';
            saveStatus.textContent = 'Error saving template';
            saveStatus.style.display = 'block';
        });
    }

    // Clear canvas
    document.getElementById('clearCanvas').addEventListener('click', function() {
        if (confirm('Are you sure you want to clear the canvas? This will remove all elements.')) {
            canvas.clear();
            canvas.setBackgroundColor('#ffffff', canvas.renderAll.bind(canvas));
            resetBackgroundControls();
            
            // Reset the currently loaded template
            currentLoadedTemplate = null;
            setTemplateName('');
            const canvasInput = document.getElementById('templateName');
            if (canvasInput) canvasInput.removeAttribute('data-original');
            const toolbarInput = document.getElementById('templateNameToolbar');
            if (toolbarInput) toolbarInput.removeAttribute('data-original');
            updateSaveButtonText();
        }
    });
    
    // Update save button text based on whether we're editing an existing template
    function updateSaveButtonText() {
        const saveButton = document.getElementById('saveTemplate');
        const saveButtonToolbar = document.getElementById('saveTemplateToolbar');
        if (currentLoadedTemplate) {
            if (saveButton) {
                saveButton.innerHTML = '<i class="fa fa-save"></i> Update Template';
                saveButton.classList.add('update-mode');
            }
            if (saveButtonToolbar) {
                saveButtonToolbar.title = 'Update Template';
                saveButtonToolbar.classList.add('update-mode');
            }
        } else {
            if (saveButton) {
                saveButton.innerHTML = '<i class="fa fa-save"></i> Save Template';
                saveButton.classList.remove('update-mode');
            }
            if (saveButtonToolbar) {
                saveButtonToolbar.title = 'Save Template';
                saveButtonToolbar.classList.remove('update-mode');
            }
        }
    }

    // Setup canvas event handlers - called during initialization
    function setupCanvasEventHandlers() {
        // Object selection handling
        canvas.on('selection:created', function(e) {
            selectedObject = e.selected[0];

            // Update properties content but don't expand the panel
            const propertiesContent = document.getElementById('objectProperties');
            if (propertiesContent.classList.contains('expanded')) {
                updateProperties(selectedObject);
            } else {
                // Store that we have a selection but don't show it yet
                // The panel will be updated when it's expanded
            }

            updateLayerUI();
        });

        canvas.on('selection:updated', function(e) {
            selectedObject = e.selected[0];

            // Update properties content but don't expand the panel
            const propertiesContent = document.getElementById('objectProperties');
            if (propertiesContent.classList.contains('expanded')) {
                updateProperties(selectedObject);
            }

            updateLayerUI();
        });

        canvas.on('selection:cleared', function() {
            document.getElementById('objectProperties').innerHTML = '<p>Select an object to edit properties</p>';
            selectedObject = null;
            updateLayerUI();
        });

        // Listen for object modifications (move, scale, rotate)
        canvas.on('object:modified', function(e) {
            if (selectedObject) {
                const propertiesContent = document.getElementById('objectProperties');
                if (propertiesContent.classList.contains('expanded')) {
                    updateProperties(selectedObject);
                }
            }
            updateLayerUI();
            saveHistory(); // Save state after object modification
        });

        // Save history when objects are added
        canvas.on('object:added', function(e) {
            if (!isUndoRedo) {
                saveHistory();
            }
        });

        // Save history when objects are removed
        canvas.on('object:removed', function(e) {
            // Don't save here, we save in deleteSelectedObject
        });

        // Real-time update while moving/scaling/rotating
        canvas.on('object:moving', function(e) {
            if (selectedObject) {
                updatePropertiesRealtime(e.target);
            }
        });

        canvas.on('object:scaling', function(e) {
            if (selectedObject) {
                updatePropertiesRealtime(e.target);
            }
        });

        canvas.on('object:rotating', function(e) {
            if (selectedObject) {
                updatePropertiesRealtime(e.target);
            }
        });
    }

    // Real-time update of position, size, and rotation values during manipulation
    function updatePropertiesRealtime(obj) {
        if (!obj) return;

        // Update shape properties
        const shapeLeft = document.getElementById('shapeLeft');
        const shapeTop = document.getElementById('shapeTop');
        const shapeWidth = document.getElementById('shapeWidth');
        const shapeHeight = document.getElementById('shapeHeight');
        const shapeRotation = document.getElementById('shapeRotation');
        const rotationValue = document.getElementById('rotationValue');
        const radiusValue = document.getElementById('radiusValue');

        if (shapeLeft) shapeLeft.value = Math.round(obj.left);
        if (shapeTop) shapeTop.value = Math.round(obj.top);
        if (shapeWidth) shapeWidth.value = Math.round(obj.width * (obj.scaleX || 1));
        if (shapeHeight) shapeHeight.value = Math.round(obj.height * (obj.scaleY || 1));
        if (shapeRotation) shapeRotation.value = Math.round(obj.angle || 0);
        if (rotationValue) rotationValue.textContent = Math.round(obj.angle || 0) + '°';

        // Update text properties if it's a text object
        const textLeft = document.getElementById('textLeft');
        const textTop = document.getElementById('textTop');
        const textRotation = document.getElementById('textRotation');
        const textRotationValue = document.getElementById('textRotationValue');

        if (textLeft) textLeft.value = Math.round(obj.left);
        if (textTop) textTop.value = Math.round(obj.top);
        if (textRotation) textRotation.value = Math.round(obj.angle || 0);
        if (textRotationValue) textRotationValue.textContent = Math.round(obj.angle || 0) + '°';

        // Update image properties if it's an image object
        const imageLeft = document.getElementById('imageLeft');
        const imageTop = document.getElementById('imageTop');
        const imageWidth = document.getElementById('imageWidth');
        const imageHeight = document.getElementById('imageHeight');
        const imageRotation = document.getElementById('imageRotation');
        const imageRotationValue = document.getElementById('imageRotationValue');

        if (imageLeft) imageLeft.value = Math.round(obj.left);
        if (imageTop) imageTop.value = Math.round(obj.top);
        if (imageWidth) imageWidth.value = Math.round(obj.width * (obj.scaleX || 1));
        if (imageHeight) imageHeight.value = Math.round(obj.height * (obj.scaleY || 1));
        if (imageRotation) imageRotation.value = Math.round(obj.angle || 0);
        if (imageRotationValue) imageRotationValue.textContent = Math.round(obj.angle || 0) + '°';
    }

    // Update properties panel
    function updateProperties(obj) {
        if (!obj) return;

        selectedObject = obj;
        var html = '';

        // Check if this is a shape object
        if (obj.objectType === 'shape' && typeof window.updateShapeProperties === 'function') {
            html = window.updateShapeProperties(obj);
        } else if (obj.type === 'text' || obj.type === 'i-text' || obj.type === 'textbox') {
            html = '<div class="properties-fields">';
            html += '<div class="property-field">';
            html += '<label for="textValue">Text:</label>';
            html += '<input type="text" id="textValue" value="' + obj.text + '" onchange="updateText()">';
            html += '</div>';

            // Font Family and Color in one row
            html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
            html += '<div class="property-field" style="flex: 2; margin-bottom: 0;">';
            html += '<label for="fontFamily">Font:</label>';
            html += '<select id="fontFamily" onchange="updateFontFamily()">';
            html += '<option value="Arial"' + (obj.fontFamily === 'Arial' ? ' selected' : '') + '>Arial</option>';
            html += '<option value="Times New Roman"' + (obj.fontFamily === 'Times New Roman' ? ' selected' : '') + '>Times New Roman</option>';
            html += '<option value="Helvetica"' + (obj.fontFamily === 'Helvetica' ? ' selected' : '') + '>Helvetica</option>';
            html += '<option value="Georgia"' + (obj.fontFamily === 'Georgia' ? ' selected' : '') + '>Georgia</option>';
            html += '<option value="Verdana"' + (obj.fontFamily === 'Verdana' ? ' selected' : '') + '>Verdana</option>';
            html += '<option value="Courier New"' + (obj.fontFamily === 'Courier New' ? ' selected' : '') + '>Courier New</option>';
            html += '<option value="Palatino"' + (obj.fontFamily === 'Palatino' ? ' selected' : '') + '>Palatino</option>';
            html += '<option value="Garamond"' + (obj.fontFamily === 'Garamond' ? ' selected' : '') + '>Garamond</option>';
            html += '<option value="Trebuchet MS"' + (obj.fontFamily === 'Trebuchet MS' ? ' selected' : '') + '>Trebuchet MS</option>';
            html += '<option value="Tahoma"' + (obj.fontFamily === 'Tahoma' ? ' selected' : '') + '>Tahoma</option>';
            html += '<option value="Impact"' + (obj.fontFamily === 'Impact' ? ' selected' : '') + '>Impact</option>';
            html += '<option value="Comic Sans MS"' + (obj.fontFamily === 'Comic Sans MS' ? ' selected' : '') + '>Comic Sans MS</option>';
            html += '<option value="Lucida Console"' + (obj.fontFamily === 'Lucida Console' ? ' selected' : '') + '>Lucida Console</option>';
            html += '<option value="Century Gothic"' + (obj.fontFamily === 'Century Gothic' ? ' selected' : '') + '>Century Gothic</option>';
            html += '<option value="Bookman"' + (obj.fontFamily === 'Bookman' ? ' selected' : '') + '>Bookman</option>';
            html += '<option value="Cambria"' + (obj.fontFamily === 'Cambria' ? ' selected' : '') + '>Cambria</option>';
            html += '<option value="Calibri"' + (obj.fontFamily === 'Calibri' ? ' selected' : '') + '>Calibri</option>';
            html += '<option value="Consolas"' + (obj.fontFamily === 'Consolas' ? ' selected' : '') + '>Consolas</option>';
            html += '<option value="Franklin Gothic Medium"' + (obj.fontFamily === 'Franklin Gothic Medium' ? ' selected' : '') + '>Franklin Gothic Medium</option>';
            html += '<option value="Copperplate"' + (obj.fontFamily === 'Copperplate' ? ' selected' : '') + '>Copperplate</option>';
            html += '</select>';
            html += '</div>';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="textColor">Color:</label>';
            html += '<input type="color" id="textColor" value="' + obj.fill + '" onchange="updateColor()">';
            html += '</div>';
            html += '</div>';

            // Font Size and Text Align in one row
            html += '<div style="display: flex; gap: 10px; margin-bottom: 12px;">';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="fontSize">Size:</label>';
            html += '<input type="number" id="fontSize" value="' + obj.fontSize + '" onchange="updateFontSize()" min="8" max="100">';
            html += '</div>';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="textAlign">Align:</label>';
            html += '<select id="textAlign" onchange="updateTextAlign()">';
            html += '<option value="left"' + (obj.textAlign === 'left' ? ' selected' : '') + '>Left</option>';
            html += '<option value="center"' + (obj.textAlign === 'center' ? ' selected' : '') + '>Center</option>';
            html += '<option value="right"' + (obj.textAlign === 'right' ? ' selected' : '') + '>Right</option>';
            html += '</select>';
            html += '</div>';
            html += '</div>';

            // Bold, Italic, Underline in one row
            html += '<div style="display: flex; gap: 15px; margin-bottom: 12px;">';
            html += '<div class="checkbox-group" style="margin-bottom: 0;">';
            html += '<input type="checkbox" id="fontBold" ' + (obj.fontWeight === 'bold' ? 'checked' : '') + ' onchange="updateFontWeight()">';
            html += '<label for="fontBold">Bold</label>';
            html += '</div>';
            html += '<div class="checkbox-group" style="margin-bottom: 0;">';
            html += '<input type="checkbox" id="fontItalic" ' + (obj.fontStyle === 'italic' ? 'checked' : '') + ' onchange="updateFontStyle()">';
            html += '<label for="fontItalic">Italic</label>';
            html += '</div>';
            html += '<div class="checkbox-group" style="margin-bottom: 0;">';
            html += '<input type="checkbox" id="fontUnderline" ' + (obj.underline ? 'checked' : '') + ' onchange="updateUnderline()">';
            html += '<label for="fontUnderline">Underline</label>';
            html += '</div>';
            html += '</div>';

            // Text Background Color
            html += '<div style="display: flex; gap: 10px; margin-bottom: 12px; align-items: center;">';
            html += '<div class="property-field" style="flex: 1; margin-bottom: 0;">';
            html += '<label for="textBackgroundColor">Background:</label>';
            html += '<input type="color" id="textBackgroundColor" value="' + (obj.backgroundColor || '#ffffff') + '" onchange="updateTextBackgroundColor()">';
            html += '</div>';
            html += '<div class="checkbox-group" style="margin-bottom: 0;">';
            html += '<input type="checkbox" id="textBackgroundToggle" ' + (obj.backgroundColor ? 'checked' : '') + ' onchange="toggleTextBackground()">';
            html += '<label for="textBackgroundToggle">Enable</label>';
            html += '</div>';
            html += '</div>';

            html += '<button type="button" onclick="deleteObject()" class="delete-btn">🗑️ Delete</button>';
            html += '</div>';
        } else if (!obj.objectType || obj.objectType !== 'shape') {
            // For other objects (images, QR codes, etc.)
            html = '<div class="properties-fields">';
            html += '<p>Select a text or shape to edit properties</p>';
            html += '<button type="button" onclick="deleteObject()" class="delete-btn">🗑️ Delete</button>';
            html += '</div>';
        }

        // Add delete button to shapes (already included in updateShapeProperties)

        // Update the properties content
        const propertiesContent = document.getElementById('objectProperties');
        propertiesContent.innerHTML = html;
        
        // Don't automatically expand the panel when an object is selected
        // Keep the panel in its current state (expanded or collapsed)
    }

    // Property update functions
    function isTextObject(obj) {
        return obj && (obj.type === 'text' || obj.type === 'i-text' || obj.type === 'textbox');
    }

    function updateText() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('text', document.getElementById('textValue').value);
            canvas.renderAll();
        }
    }

    function updateFontSize() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('fontSize', parseInt(document.getElementById('fontSize').value));
            canvas.renderAll();
        }
    }

    function updateColor() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('fill', document.getElementById('textColor').value);
            canvas.renderAll();
        }
    }

    function updateFontFamily() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('fontFamily', document.getElementById('fontFamily').value);
            canvas.renderAll();
        }
    }

    function updateFontWeight() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('fontWeight', document.getElementById('fontBold').checked ? 'bold' : 'normal');
            canvas.renderAll();
        }
    }

    function updateFontStyle() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('fontStyle', document.getElementById('fontItalic').checked ? 'italic' : 'normal');
            canvas.renderAll();
        }
    }

    function updateTextAlign() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('textAlign', document.getElementById('textAlign').value);
            canvas.renderAll();
        }
    }

    function updateUnderline() {
        if (isTextObject(selectedObject)) {
            selectedObject.set('underline', document.getElementById('fontUnderline').checked);
            canvas.renderAll();
        }
    }

    function updateTextBackgroundColor() {
        if (isTextObject(selectedObject)) {
            const color = document.getElementById('textBackgroundColor').value;
            const enabled = document.getElementById('textBackgroundToggle').checked;
            if (enabled) {
                selectedObject.set('backgroundColor', color);
            }
            canvas.renderAll();
        }
    }

    function toggleTextBackground() {
        if (isTextObject(selectedObject)) {
            const enabled = document.getElementById('textBackgroundToggle').checked;
            if (enabled) {
                const color = document.getElementById('textBackgroundColor').value;
                selectedObject.set('backgroundColor', color);
            } else {
                selectedObject.set('backgroundColor', null);
            }
            canvas.renderAll();
        }
    }

    function updatePosition() {
        if (selectedObject) {
            selectedObject.set({
                left: parseInt(document.getElementById('objLeft').value),
                top: parseInt(document.getElementById('objTop').value)
            });
            canvas.renderAll();
        }
    }

    function deleteObject() {
        if (selectedObject) {
            canvas.remove(selectedObject);
            selectedObject = null;
            document.getElementById('objectProperties').innerHTML = '<p>Select an object to edit properties</p>';
            updateLayerUI();
            
            // The panel should remain in its current state (expanded or collapsed)
            // after deleting an object, so we don't need to change its state here
        }
    }
    
    // Setup drag and drop functionality
    function setupDragAndDrop() {
        const draggableFields = document.querySelectorAll('.draggable-field');
        const canvasContainer = document.querySelector('.canvas-wrapper');
        
        draggableFields.forEach(field => {
            field.addEventListener('dragstart', function(e) {
                e.dataTransfer.setData('text/plain', JSON.stringify({
                    field: this.dataset.field,
                    label: this.dataset.label
                }));
                this.style.opacity = '0.5';
            });
            
            field.addEventListener('dragend', function(e) {
                this.style.opacity = '1';
            });
        });
        
        // Use canvas container instead of canvas element for better drop zone
        canvasContainer.addEventListener('dragover', function(e) {
            e.preventDefault();
            e.dataTransfer.dropEffect = 'copy';
            this.style.background = '#f0f8ff';
        });
        
        canvasContainer.addEventListener('dragleave', function(e) {
            this.style.background = 'white';
        });
        
        canvasContainer.addEventListener('drop', function(e) {
            e.preventDefault();
            this.style.background = 'white';
            
            const data = JSON.parse(e.dataTransfer.getData('text/plain'));
            
            // Get drop position relative to canvas
            const canvasElement = document.getElementById('certificateCanvas');
            const rect = canvasElement.getBoundingClientRect();
            const x = Math.max(0, e.clientX - rect.left);
            const y = Math.max(0, e.clientY - rect.top);
            
            // Handle Journal Logo field specially - create as image
            if (data.field === '{JOURNAL_LOGO}') {
                if (window.goValidJournalInfo && window.goValidJournalInfo.logo) {
                    // Journal has a logo configured, load it
                    fabric.Image.fromURL(window.goValidJournalInfo.logo, function(img) {
                        if (!img) {
                            console.error('Failed to load journal logo image');
                            return;
                        }
                        // Set default width to 150px and calculate height to maintain aspect ratio
                        const defaultWidth = 150;
                        const aspectRatio = img.height / img.width;
                        const calculatedHeight = defaultWidth * aspectRatio;

                        // Calculate scale factors to achieve the desired width
                        const scaleX = defaultWidth / img.width;
                        const scaleY = scaleX; // Use same scale to maintain aspect ratio

                        img.set({
                            left: x,
                            top: y,
                            scaleX: scaleX,
                            scaleY: scaleY,
                            fieldType: 'journal_logo',
                            selectable: true,
                            evented: true,
                            hasControls: true,
                            hasBorders: true,
                            lockMovementX: false,
                            lockMovementY: false,
                            cornerColor: '#ff6b6b',
                            cornerSize: 8,
                            transparentCorners: false
                        });

                        canvas.add(img);
                        canvas.setActiveObject(img);
                        updateProperties(img);
                        updateLayerUI();
                        saveHistory();

                        console.log(`Journal logo dropped: Original ${img.width}x${img.height}, Scaled to ${defaultWidth}x${Math.round(calculatedHeight)}`);
                    }, { crossOrigin: 'anonymous' });
                } else {
                    // No logo configured, create a placeholder text box
                    const journalName = (window.goValidJournalInfo && window.goValidJournalInfo.name) ? window.goValidJournalInfo.name : 'JOURNAL LOGO';
                    const logoPlaceholder = new fabric.Textbox(journalName, {
                        left: x,
                        top: y,
                        width: 300,
                        fontSize: 24,
                        fill: '#333',
                        fontWeight: 'bold',
                        textAlign: 'center',
                        backgroundColor: '#f0f0f0',
                        padding: 20,
                        textBaseline: 'alphabetic',
                        fieldType: 'journal_logo',
                        cornerColor: '#ff6b6b',
                        cornerSize: 8,
                        transparentCorners: false
                    });

                    canvas.add(logoPlaceholder);
                    canvas.setActiveObject(logoPlaceholder);
                    updateProperties(logoPlaceholder);
                    updateLayerUI();

                    console.log('Journal logo not configured in OJS. Created text placeholder instead.');
                }
            } else if (data.field === '{QR_CODE}') {
                // Handle QR code field specially - create as text object for proper replacement
                const qrText = new fabric.Text('{QR_CODE}', {
                    left: x,
                    top: y,
                    fontSize: 12,
                    fill: '#333',
                    backgroundColor: '#f0f0f0',
                    padding: 10,
                    textAlign: 'center',
                    textBaseline: 'alphabetic',
                    fieldType: '{QR_CODE}',
                    cornerColor: '#ff6b6b',
                    cornerSize: 8,
                    transparentCorners: false
                });

                canvas.add(qrText);
                canvas.setActiveObject(qrText);
                updateProperties(qrText);
            } else if (data.field === '{QR_ID}' || data.field === '{QR_IDENTIFIER}' || data.field === '{QR_ISSUE_DATE}' || data.field === '{QR_EXPIRE_DATE}') {
                // Handle QR data fields with special styling
                const qrDataText = new fabric.Text(data.field, {
                    left: x,
                    top: y,
                    fontSize: 14,
                    fill: '#d97706',
                    backgroundColor: '#fffbea',
                    padding: 8,
                    fontWeight: 'bold',
                    textAlign: 'center',
                    textBaseline: 'alphabetic',
                    fieldType: data.field,
                    cornerColor: '#f59e0b',
                    cornerSize: 8,
                    transparentCorners: false
                });

                canvas.add(qrDataText);
                canvas.setActiveObject(qrDataText);
                updateProperties(qrDataText);
            } else {
                // Add field as editable text box (allows line breaks with Enter key)
                const text = new fabric.Textbox(data.field, {
                    left: x,
                    top: y,
                    width: 400,  // Initial width for text wrapping
                    fontFamily: 'Arial',
                    fontSize: 16,
                    fill: '#000000',
                    textAlign: 'center',  // Default to center alignment
                    fieldType: data.field,
                    textBaseline: 'alphabetic',  // Set valid baseline
                    editable: true,  // Allow double-click to edit
                    splitByGrapheme: true  // Better line breaking
                });

                canvas.add(text);
                canvas.setActiveObject(text);
                updateProperties(text);
            }
            
            updateLayerUI();
        });
    }
    
    // Store current background image data
    let currentBackgroundImage = null;
    let currentBackgroundMode = 'fit';

    // Create tiled background that covers entire paper
    function createTiledBackground(img, paperWidth, paperHeight) {
        // Calculate how many tiles we need
        const tileWidth = img.width;
        const tileHeight = img.height;
        
        // Calculate scale to make tiles a reasonable size (not too small, not too big)
        const maxTileSize = Math.max(paperWidth, paperHeight) / 8; // Max 1/8 of paper dimension
        const minTileSize = Math.max(paperWidth, paperHeight) / 16; // Min 1/16 of paper dimension
        
        let tileScale = 1;
        if (Math.max(tileWidth, tileHeight) > maxTileSize) {
            tileScale = maxTileSize / Math.max(tileWidth, tileHeight);
        } else if (Math.max(tileWidth, tileHeight) < minTileSize) {
            tileScale = minTileSize / Math.max(tileWidth, tileHeight);
        }
        
        const scaledTileWidth = tileWidth * tileScale;
        const scaledTileHeight = tileHeight * tileScale;
        
        // Create a canvas for the tiled pattern
        const patternCanvas = document.createElement('canvas');
        patternCanvas.width = paperWidth;
        patternCanvas.height = paperHeight;
        const ctx = patternCanvas.getContext('2d');
        
        // Create image element from the fabric image
        const imgElement = new Image();
        imgElement.onload = function() {
            // Calculate how many tiles fit
            const tilesX = Math.ceil(paperWidth / scaledTileWidth);
            const tilesY = Math.ceil(paperHeight / scaledTileHeight);
            
            // Draw tiles to cover entire paper
            for (let x = 0; x < tilesX; x++) {
                for (let y = 0; y < tilesY; y++) {
                    const posX = x * scaledTileWidth;
                    const posY = y * scaledTileHeight;
                    ctx.drawImage(imgElement, posX, posY, scaledTileWidth, scaledTileHeight);
                }
            }
            
            // Create a new fabric image from the tiled pattern
            fabric.Image.fromURL(patternCanvas.toDataURL(), function(tiledImg) {
                tiledImg.set({
                    scaleX: 1,
                    scaleY: 1,
                    left: paperWidth / 2,
                    top: paperHeight / 2,
                    originX: 'center',
                    originY: 'center'
                });
                
                tiledImg.backgroundMode = 'tile';
                console.log(`Applied tiled background: ${tilesX}x${tilesY} tiles, tile size: ${scaledTileWidth.toFixed(1)}x${scaledTileHeight.toFixed(1)}, paper: ${paperWidth.toFixed(1)}x${paperHeight.toFixed(1)}`);
                
                canvas.setBackgroundImage(tiledImg, canvas.renderAll.bind(canvas));
            });
        };
        imgElement.src = img.src;
        
        // Return the original image for now (will be replaced by the tiled version)
        return img;
    }

    // Background image positioning functions
    function applyBackgroundImageMode(img, mode) {
        // Get the actual canvas dimensions
        const canvasWidth = canvas.getWidth();
        const canvasHeight = canvas.getHeight();
        
        // Also get the zoom level to understand the actual paper size
        const zoom = canvas.getZoom();
        const actualPaperWidth = canvasWidth / zoom;
        const actualPaperHeight = canvasHeight / zoom;

        // Use the actual paper dimensions for calculations
        const paperWidth = actualPaperWidth;
        const paperHeight = actualPaperHeight;
        const imageAspect = img.width / img.height;
        const paperAspect = paperWidth / paperHeight;
        
        let scaleX, scaleY, left, top, originX, originY;
        
        switch(mode) {
            case 'fit':
                // Scale to fit entire paper while maintaining aspect ratio (may leave whitespace)
                if (imageAspect > paperAspect) {
                    scaleX = scaleY = paperWidth / img.width;
                } else {
                    scaleX = scaleY = paperHeight / img.height;
                }
                left = paperWidth / 2;
                top = paperHeight / 2;
                originX = originY = 'center';
                break;
                
            case 'fill':
                // Stretch to fill entire paper (may distort aspect ratio)
                scaleX = paperWidth / img.width;
                scaleY = paperHeight / img.height;
                left = paperWidth / 2;
                top = paperHeight / 2;
                originX = originY = 'center';
                break;
                
            case 'cover':
                // Scale to cover entire paper while maintaining aspect ratio (may crop image)
                if (imageAspect > paperAspect) {
                    scaleX = scaleY = paperHeight / img.height;
                } else {
                    scaleX = scaleY = paperWidth / img.width;
                }
                left = paperWidth / 2;
                top = paperHeight / 2;
                originX = originY = 'center';
                break;
                
            case 'tile':
                // For tiling, create a pattern that covers the entire paper
                return createTiledBackground(img, paperWidth, paperHeight);
                
            case 'top':
                // Fill width and align to top
                scaleX = scaleY = paperWidth / img.width;
                left = paperWidth / 2;
                top = 0;
                originX = 'center';
                originY = 'top';
                break;
                
            case 'bottom':
                // Fill width and align to bottom
                scaleX = scaleY = paperWidth / img.width;
                left = paperWidth / 2;
                top = paperHeight;
                originX = 'center';
                originY = 'bottom';
                break;
                
            case 'center':
                // Cover paper from center
                if (imageAspect > paperAspect) {
                    scaleX = scaleY = paperHeight / img.height;
                } else {
                    scaleX = scaleY = paperWidth / img.width;
                }
                left = paperWidth / 2;
                top = paperHeight / 2;
                originX = originY = 'center';
                break;
                
            case 'left':
                // Fill height and align to left
                scaleX = scaleY = paperHeight / img.height;
                left = 0;
                top = paperHeight / 2;
                originX = 'left';
                originY = 'center';
                break;
                
            case 'right':
                // Fill height and align to right
                scaleX = scaleY = paperHeight / img.height;
                left = paperWidth;
                top = paperHeight / 2;
                originX = 'right';
                originY = 'center';
                break;
                
            default:
                // Default to cover (fill entire paper)
                if (imageAspect > paperAspect) {
                    scaleX = scaleY = paperHeight / img.height;
                } else {
                    scaleX = scaleY = paperWidth / img.width;
                }
                left = paperWidth / 2;
                top = paperHeight / 2;
                originX = originY = 'center';
        }
        
        img.set({
            scaleX: scaleX,
            scaleY: scaleY,
            originX: originX,
            originY: originY,
            left: left,
            top: top
        });

        // Store the mode for template saving
        img.backgroundMode = mode;

        return img;
    }

    // Setup background controls
    function setupBackgroundControls() {
        const bgColorInput = document.getElementById('bgColor');
        const addBgImageBtn = document.getElementById('addBgImage');
        const bgImageModeSelect = document.getElementById('bgImageMode');
        const removeBgImageBtn = document.getElementById('removeBgImage');

        bgColorInput.addEventListener('change', function() {
            const color = this.value;
            if (!canvas) return;

            // Clear background image if exists
            canvas.setBackgroundImage(null);

            // Set background color
            canvas.backgroundColor = color;
            canvas.renderAll();
        });

        // Also listen to input event for real-time changes
        bgColorInput.addEventListener('input', function() {
            const color = this.value;

            // Clear background image if exists
            canvas.setBackgroundImage(null);

            // Set background color
            canvas.backgroundColor = color;
            canvas.renderAll();
        });
        
        if (addBgImageBtn) {
            addBgImageBtn.addEventListener('click', function() {
                const input = document.createElement('input');
                input.type = 'file';
                input.accept = 'image/*';
                input.onchange = function(e) {
                    const file = e.target.files[0];
                    if (!file) return;

                    const reader = new FileReader();
                    reader.onload = function(f) {
                        const imageSrc = f.target.result;

                        fabric.Image.fromURL(imageSrc, function(img) {
                            // Store original image data with source
                            currentBackgroundImage = {
                                src: imageSrc,
                                width: img.width,
                                height: img.height,
                                fabricImage: img
                            };
                            currentBackgroundMode = bgImageModeSelect ? bgImageModeSelect.value : 'fit';

                            // Apply positioning mode
                            applyBackgroundImageMode(img, currentBackgroundMode);

                            // Set background image and force render
                            canvas.setBackgroundImage(img);
                            canvas.renderAll();
                        });
                    };
                    reader.readAsDataURL(file);
                };
                input.click();
            });
        }
        
        // Background image mode change handler
        if (bgImageModeSelect) {
            bgImageModeSelect.addEventListener('change', function() {
                if (currentBackgroundImage && currentBackgroundImage.src) {
                    currentBackgroundMode = this.value;

                    // Create a new image instance to avoid modifying the original
                    fabric.Image.fromURL(currentBackgroundImage.src, function(img) {
                        applyBackgroundImageMode(img, currentBackgroundMode);

                        // Set the background image and force render
                        canvas.setBackgroundImage(img, function() {
                            canvas.renderAll();
                        });
                    });
                }
            });
        }

        // Remove background image
        if (removeBgImageBtn) {
            removeBgImageBtn.addEventListener('click', function() {
                canvas.setBackgroundImage(null, canvas.renderAll.bind(canvas));
                currentBackgroundImage = null;
                currentBackgroundMode = 'fit';
                if (bgImageModeSelect) {
                    bgImageModeSelect.style.display = 'none';
                    bgImageModeSelect.value = 'fit';
                }
                removeBgImageBtn.style.display = 'none';
            });
        }
    }
    
    // Update layer list when selection changes
    function updateLayerUI() {
        // Update layer list
        updateLayerList();
    }
    
    // Update the layer list with current canvas objects
    function updateLayerList() {
        const layerList = document.getElementById('layerList');
        const objects = canvas.getObjects();
        
        // Clear current list
        layerList.innerHTML = '';
        
        if (objects.length === 0) {
            layerList.innerHTML = '<p class="empty-layer-message">No objects on canvas</p>';
            return;
        }
        
        // Create layer items in reverse order (top to bottom)
        for (let i = objects.length - 1; i >= 0; i--) {
            const obj = objects[i];
            const layerItem = document.createElement('div');
            layerItem.className = 'layer-item';
            if (obj === selectedObject) {
                layerItem.classList.add('selected');
            }
            
            // Set data attribute for object index
            layerItem.setAttribute('data-index', i);
            
            // Add drag attributes
            layerItem.setAttribute('draggable', 'true');
            
            // Determine object type and icon
            let icon = 'fa-object-group';
            let name = 'Object ' + (i + 1);

            if (obj.type === 'text' || obj.type === 'i-text' || obj.type === 'textbox') {
                icon = 'fa-font';
                name = obj.text.substring(0, 20) + (obj.text.length > 20 ? '...' : '');
            } else if (obj.type === 'image') {
                icon = 'fa-image';
                name = 'Image ' + (i + 1);
            } else if (obj.type === 'rect' && obj.isQRCode) {
                icon = 'fa-qrcode';
                name = 'QR Code';
            } else if (obj.objectType === 'shape') {
                // Handle shape objects
                switch(obj.shapeType) {
                    case 'rectangle':
                        icon = 'fa-square-o';
                        name = 'Rectangle';
                        break;
                    case 'square':
                        icon = 'fa-square';
                        name = 'Square';
                        break;
                    case 'circle':
                        icon = 'fa-circle-o';
                        name = 'Circle';
                        break;
                    case 'ellipse':
                        icon = 'fa-circle';
                        name = 'Ellipse';
                        break;
                    case 'triangle':
                        icon = 'fa-play';
                        name = 'Triangle';
                        break;
                    case 'line':
                        icon = 'fa-minus';
                        name = 'Line';
                        break;
                    default:
                        icon = 'fa-shapes';
                        name = 'Shape';
                }
            }
            
            // Create layer item content
            layerItem.innerHTML = `
                <div class="layer-item-drag-handle">
                    <i class="fa fa-bars"></i>
                </div>
                <div class="layer-item-icon">
                    <i class="fa ${icon}"></i>
                </div>
                <div class="layer-item-name">${name}</div>
                <div class="layer-item-controls">
                    <button type="button" class="layer-item-btn toggle-visibility" title="Toggle Visibility">
                        <i class="fa ${obj.visible ? 'fa-eye' : 'fa-eye-slash'}"></i>
                    </button>
                    <button type="button" class="layer-item-btn delete-layer" title="Delete Layer">
                        <i class="fa fa-trash"></i>
                    </button>
                </div>
            `;
            
            // Add click event to select object
            layerItem.addEventListener('click', function(e) {
                // Ignore if clicking on a button
                if (e.target.closest('.layer-item-btn')) return;
                
                const index = parseInt(this.getAttribute('data-index'));
                canvas.setActiveObject(objects[index]);
                selectedObject = objects[index];
                canvas.renderAll();
                updateLayerList();
                // Don't automatically open properties panel
                // updateProperties(selectedObject);
            });
            
            // Add visibility toggle functionality
            const visibilityBtn = layerItem.querySelector('.toggle-visibility');
            visibilityBtn.addEventListener('click', function() {
                const index = parseInt(layerItem.getAttribute('data-index'));
                const obj = objects[index];
                obj.visible = !obj.visible;
                
                // Update icon
                const icon = this.querySelector('i');
                if (obj.visible) {
                    icon.classList.remove('fa-eye-slash');
                    icon.classList.add('fa-eye');
                } else {
                    icon.classList.remove('fa-eye');
                    icon.classList.add('fa-eye-slash');
                }
                
                canvas.renderAll();
            });

            // Add delete layer functionality
            const deleteBtn = layerItem.querySelector('.delete-layer');
            deleteBtn.addEventListener('click', function() {
                const index = parseInt(layerItem.getAttribute('data-index'));
                const obj = objects[index];

                // Remove from canvas
                canvas.remove(obj);

                // Clear selection if this was the selected object
                if (selectedObject === obj) {
                    selectedObject = null;
                    canvas.discardActiveObject();
                }

                canvas.renderAll();
                updateLayerList();
            });

            // Add drag events
            layerItem.addEventListener('dragstart', handleDragStart);
            layerItem.addEventListener('dragover', handleDragOver);
            layerItem.addEventListener('dragleave', handleDragLeave);
            layerItem.addEventListener('drop', handleDrop);
            layerItem.addEventListener('dragend', handleDragEnd);
            
            layerList.appendChild(layerItem);
        }
    }
    
    // Drag and drop functionality for layer reordering
    let draggedItem = null;
    
    function handleDragStart(e) {
        draggedItem = this;
        this.classList.add('dragging');
        e.dataTransfer.effectAllowed = 'move';
        e.dataTransfer.setData('text/plain', this.getAttribute('data-index'));
    }
    
    function handleDragOver(e) {
        if (e.preventDefault) {
            e.preventDefault();
        }
        e.dataTransfer.dropEffect = 'move';
        this.classList.add('drag-over');
        return false;
    }
    
    function handleDragLeave(e) {
        this.classList.remove('drag-over');
    }
    
    function handleDrop(e) {
        if (e.stopPropagation) {
            e.stopPropagation();
        }
        
        if (draggedItem !== this) {
            // Get the indices
            const sourceIndex = parseInt(draggedItem.getAttribute('data-index'));
            const targetIndex = parseInt(this.getAttribute('data-index'));
            const objects = canvas.getObjects();
            
            // Get the actual objects
            const sourceObj = objects[sourceIndex];
            const targetObj = objects[targetIndex];
            
            // Determine if moving up or down in the stack
            if (sourceIndex < targetIndex) {
                // Moving down - place below target
                canvas.moveTo(sourceObj, targetIndex);
            } else {
                // Moving up - place above target
                canvas.moveTo(sourceObj, targetIndex);
            }
            
            canvas.renderAll();
            updateLayerList();
        }
        
        this.classList.remove('drag-over');
        return false;
    }
    
    function handleDragEnd(e) {
        this.classList.remove('dragging');
        document.querySelectorAll('.layer-item').forEach(item => {
            item.classList.remove('drag-over');
        });
    }
    
