<?php

/**
 * @file plugins/generic/goValidOJS/classes/OJSEmailTemplateIntegration.php
 *
 * Copyright (c) 2025 Naufal Naufal, University Of Muhammadiyah Makassar, Indonesia
 * Distributed under the GNU GPL v3. For full terms see the file docs/COPYING.
 *
 * @class OJSEmailTemplateIntegration
 * @ingroup plugins_generic_goValidOJS
 *
 * @brief Integrates goValid email designer with OJS native email templates
 *        Compatible with OJS 3.3.x
 */

import('classes.core.Services');

class OJSEmailTemplateIntegration {

    /** @var object Plugin instance */
    private $plugin;

    /** @var object Context */
    private $context;

    public function __construct($plugin, $context) {
        $this->plugin = $plugin;
        $this->context = $context;
    }

    /**
     * Get all OJS email templates for the current context
     * @return array List of email templates
     */
    public function getOJSEmailTemplates() {
        $templates = [];

        try {
            // OJS 3.3 uses Services::get('emailTemplate')
            $emailTemplateService = Services::get('emailTemplate');
            $contextId = $this->context ? $this->context->getId() : CONTEXT_SITE;
            $locale = $this->context ? $this->context->getPrimaryLocale() : 'en_US';

            error_log("GoValid Plugin: Loading email templates for context ID: $contextId, locale: $locale");

            // Get all templates using the service
            $emailTemplatesIterator = $emailTemplateService->getMany([
                'contextId' => $contextId
            ]);

            foreach ($emailTemplatesIterator as $template) {
                $key = $template->getData('key');
                $body = $template->getLocalizedData('body', $locale) ?? '';
                $isCustom = !empty($template->getData('id'));

                // Debug: Log template body info for templates that might have been edited
                if ($isCustom || strpos($body, '<!-- Header -->') !== false || strpos($body, 'max-width:') !== false) {
                    error_log("GoValid Plugin: Template '$key' isCustom=$isCustom, bodyLength=" . strlen($body) . ", bodyFirst200=" . substr($body, 0, 200));
                }

                $templates[] = [
                    'key' => $key,
                    'name' => $template->getLocalizedData('name', $locale) ?? $key,
                    'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                    'body' => $body,
                    'isCustom' => $isCustom,
                    'description' => $template->getLocalizedData('description', $locale) ?? '',
                    'enabled' => $template->getData('enabled') ?? true,
                    'canDisable' => $template->getData('canDisable') ?? false,
                    'canEdit' => $template->getData('canEdit') ?? true
                ];
            }

            error_log("GoValid Plugin: Loaded " . count($templates) . " OJS email templates");

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error loading OJS email templates: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return $templates;
    }

    /**
     * Get a specific OJS email template by key
     * @param string $templateKey The email template key
     * @return array|null Template data or null if not found
     */
    public function getOJSEmailTemplate($templateKey) {
        try {
            $emailTemplateService = Services::get('emailTemplate');
            $contextId = $this->context ? $this->context->getId() : CONTEXT_SITE;
            $locale = $this->context ? $this->context->getPrimaryLocale() : 'en_US';

            $template = $emailTemplateService->getByKey($contextId, $templateKey);

            if ($template) {
                return [
                    'key' => $template->getData('key'),
                    'name' => $template->getLocalizedData('name', $locale) ?? $templateKey,
                    'subject' => $template->getLocalizedData('subject', $locale) ?? '',
                    'body' => $template->getLocalizedData('body', $locale) ?? '',
                    'isCustom' => !empty($template->getData('id')),
                    'description' => $template->getLocalizedData('description', $locale) ?? '',
                    'enabled' => $template->getData('enabled') ?? true,
                    'canDisable' => $template->getData('canDisable') ?? false,
                    'canEdit' => $template->getData('canEdit') ?? true
                ];
            }

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error loading OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
        }

        return null;
    }

    /**
     * Save/update an OJS email template
     * @param string $templateKey The email template key
     * @param string $subject The email subject
     * @param string $body The email body (HTML)
     * @param string $templateName Optional custom name for new templates
     * @param string $alternateTo Optional - for alternate templates
     * @return bool Success status
     */
    public function saveOJSEmailTemplate($templateKey, $subject, $body, $templateName = null, $alternateTo = null) {
        try {
            $emailTemplateService = Services::get('emailTemplate');
            $contextId = $this->context ? $this->context->getId() : CONTEXT_SITE;
            $locale = $this->context ? $this->context->getPrimaryLocale() : 'en_US';

            // Get existing template
            $template = $emailTemplateService->getByKey($contextId, $templateKey);

            if ($template && $template->getData('id')) {
                // Update existing custom template
                $props = [
                    'subject' => [$locale => $subject],
                    'body' => [$locale => $body]
                ];

                $template = $emailTemplateService->edit($template, $props, Application::get()->getRequest());

            } else {
                // Create new custom template
                $emailTemplateDao = DAORegistry::getDAO('EmailTemplateDAO');
                $newTemplate = $emailTemplateDao->newDataObject();

                $newTemplate->setData('key', $templateKey);
                $newTemplate->setData('contextId', $contextId);
                $newTemplate->setData('subject', $subject, $locale);
                $newTemplate->setData('body', $body, $locale);

                if ($templateName) {
                    $newTemplate->setData('name', $templateName, $locale);
                }

                $emailTemplateDao->insertObject($newTemplate);
            }

            error_log("GoValid Plugin: Successfully saved OJS email template '$templateKey'");
            return true;

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error saving OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Reset OJS email template to default
     * @param string $templateKey The email template key
     * @return bool Success status
     */
    public function resetOJSEmailTemplate($templateKey) {
        try {
            $emailTemplateService = Services::get('emailTemplate');
            $contextId = $this->context ? $this->context->getId() : CONTEXT_SITE;

            $template = $emailTemplateService->getByKey($contextId, $templateKey);

            if ($template && $template->getData('id')) {
                $emailTemplateService->delete($template);
                error_log("GoValid Plugin: Successfully reset OJS email template '$templateKey' to default");
                return true;
            }

            error_log("GoValid Plugin: Template '$templateKey' is already default or not found");
            return false;

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error resetting OJS email template '$templateKey': " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Get available email variables for templates
     * @return array List of available variables
     */
    public function getEmailVariables() {
        // Common OJS email template variables
        return [
            'contextName' => __('emailTemplate.variable.context.contextName'),
            'contextUrl' => __('emailTemplate.variable.context.contextUrl'),
            'principalContactSignature' => __('emailTemplate.variable.context.principalContactSignature'),
            'siteTitle' => __('emailTemplate.variable.site.siteTitle'),
            'senderEmail' => __('emailTemplate.variable.senderEmail'),
            'senderName' => __('emailTemplate.variable.senderName'),
            'recipientEmail' => __('emailTemplate.variable.recipientEmail'),
            'recipientName' => __('emailTemplate.variable.recipientName'),
            'authorName' => __('emailTemplate.variable.submission.authorName'),
            'submissionTitle' => __('emailTemplate.variable.submission.submissionTitle'),
            'submissionId' => __('emailTemplate.variable.submission.submissionId'),
            'submissionAbstract' => __('emailTemplate.variable.submission.submissionAbstract'),
            'reviewerName' => __('emailTemplate.variable.reviewer.reviewerName'),
            'reviewDueDate' => __('emailTemplate.variable.review.reviewDueDate'),
            'responseDueDate' => __('emailTemplate.variable.review.responseDueDate'),
        ];
    }

    /**
     * Apply a design configuration to an OJS email template
     * @param string $templateKey The template key
     * @param array $designConfig The design configuration
     * @return bool Success status
     */
    public function applyDesignToOJSTemplate($templateKey, $designConfig) {
        try {
            $template = $this->getOJSEmailTemplate($templateKey);

            if (!$template) {
                error_log("GoValid Plugin: Template '$templateKey' not found for design application");
                return false;
            }

            // Wrap the body content with the design
            $wrappedBody = $this->wrapContentWithDesign($template['body'], $designConfig);

            // Save the updated template
            return $this->saveOJSEmailTemplate($templateKey, $template['subject'], $wrappedBody);

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error applying design to template '$templateKey': " . $e->getMessage());
            return false;
        }
    }

    /**
     * Wrap email content with a design template
     * @param string $content The email body content
     * @param array $config The design configuration
     * @return string The wrapped HTML content
     */
    private function wrapContentWithDesign($content, $config) {
        $bgColor = $config['backgroundColor'] ?? '#f5f5f5';
        $cardBgColor = $config['cardBackgroundColor'] ?? '#ffffff';
        $textColor = $config['textColor'] ?? '#333333';
        $headerColor = $config['headerColor'] ?? '#667eea';
        $fontFamily = $config['fontFamily'] ?? 'Arial, sans-serif';
        $width = $config['width'] ?? 600;
        $showLogo = $config['showLogo'] ?? true;
        $logoUrl = $config['logoUrl'] ?? '';
        $headerText = $config['headerText'] ?? '';
        $footerText = $config['footerText'] ?? '';

        $logoHtml = '';
        if ($showLogo && $logoUrl) {
            $logoHtml = '<img src="' . htmlspecialchars($logoUrl) . '" alt="Logo" style="max-width: 180px; height: auto; margin-bottom: 20px;">';
        }

        $headerHtml = '';
        if ($headerText) {
            $headerHtml = '<div style="background-color: ' . htmlspecialchars($headerColor) . '; color: #ffffff; padding: 20px; text-align: center; font-size: 18px; font-weight: bold;">' . htmlspecialchars($headerText) . '</div>';
        }

        $footerHtml = '';
        if ($footerText) {
            $footerHtml = '<div style="padding: 20px; text-align: center; font-size: 12px; color: #666666;">' . htmlspecialchars($footerText) . '</div>';
        }

        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin: 0; padding: 0; background-color: ' . htmlspecialchars($bgColor) . '; font-family: ' . htmlspecialchars($fontFamily) . ';">
    <table cellpadding="0" cellspacing="0" width="100%" style="background-color: ' . htmlspecialchars($bgColor) . ';">
        <tr>
            <td align="center" style="padding: 40px 20px;">
                <table cellpadding="0" cellspacing="0" width="' . intval($width) . '" style="background-color: ' . htmlspecialchars($cardBgColor) . '; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
                    ' . ($logoHtml ? '<tr><td align="center" style="padding: 30px 20px 10px 20px;">' . $logoHtml . '</td></tr>' : '') . '
                    ' . ($headerHtml ? '<tr><td>' . $headerHtml . '</td></tr>' : '') . '
                    <tr>
                        <td style="padding: 30px; color: ' . htmlspecialchars($textColor) . '; line-height: 1.6;">
                            ' . $content . '
                        </td>
                    </tr>
                    ' . ($footerHtml ? '<tr><td>' . $footerHtml . '</td></tr>' : '') . '
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';
    }

    /**
     * Save a custom email template (for goValid-specific templates)
     * @param array $templateData Template data including key, name, subject, body
     * @return array Result with success status
     */
    public function saveCustomTemplate($templateData) {
        try {
            $key = $templateData['key'] ?? null;
            $name = $templateData['name'] ?? 'Custom Template';
            $subject = $templateData['subject'] ?? '';
            $body = $templateData['body'] ?? '';
            $description = $templateData['description'] ?? '';

            if (!$key || !$subject || !$body) {
                return ['success' => false, 'error' => 'Missing required template data'];
            }

            $contextId = $this->context ? $this->context->getId() : CONTEXT_SITE;
            $locale = $this->context ? $this->context->getPrimaryLocale() : 'en_US';

            $emailTemplateDao = DAORegistry::getDAO('EmailTemplateDAO');

            // Check if template already exists
            $emailTemplateService = Services::get('emailTemplate');
            $existingTemplate = $emailTemplateService->getByKey($contextId, $key);

            if ($existingTemplate && $existingTemplate->getData('id')) {
                // Update existing
                $props = [
                    'subject' => [$locale => $subject],
                    'body' => [$locale => $body]
                ];
                $emailTemplateService->edit($existingTemplate, $props, Application::get()->getRequest());
            } else {
                // Create new
                $newTemplate = $emailTemplateDao->newDataObject();
                $newTemplate->setData('key', $key);
                $newTemplate->setData('contextId', $contextId);
                $newTemplate->setData('subject', $subject, $locale);
                $newTemplate->setData('body', $body, $locale);
                $newTemplate->setData('name', $name, $locale);

                $emailTemplateDao->insertObject($newTemplate);
            }

            // Store design config as plugin setting if provided
            if (isset($templateData['designConfig'])) {
                $this->plugin->updateSetting($contextId, 'template_design_' . $key, $templateData['designConfig']);
            }

            error_log("GoValid Plugin: Successfully saved custom template '$key'");
            return ['success' => true, 'templateKey' => $key];

        } catch (Exception $e) {
            error_log("GoValid Plugin: Error saving custom template: " . $e->getMessage());
            error_log("GoValid Plugin: Stack trace: " . $e->getTraceAsString());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}
